#ifndef K3DSDK_ISCRIPT_ENGINE_H
#define K3DSDK_ISCRIPT_ENGINE_H

// K-3D
// Copyright (c) 1995-2004, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\brief Declares iscript_engine, an abstract interface implemented by objects that can execute scripts written in a specific scripting language
		\author Tim Shead (tshead@k-3d.com)
*/

#include "iunknown.h"

#include <string>
#include <vector>

namespace k3d
{

// Forward declarations ...
class icommand_node;
class idocument;
class iobject;

/// Abstract interface implemented by objects that can execute scripts written in a specific scripting language
class iscript_engine :
	public virtual iunknown
{
public:
	/**	\brief Returns the human-readable name of the scripting language this engine implements
	*/
	virtual const std::string language() = 0;

	/**	\brief Tests to see if the supplied script is written in the scripting language implemented by this engine
		\param Script The complete source code of the script to be tested
		\return true, iff the supplied script can be executed by the engine
	*/
	virtual bool can_execute(const std::string& Script) = 0;

	/// Defines a collection of objects to pass to a script to define its context
	typedef std::vector<iunknown*> context_t;
	/**	\brief Executes a script with optional document and object contexts
		\param ScriptName A human readable identifier for the script, which should be used in error messages, etc.
		\param Script The complete source code of the script to be executed
		\param Context Optional collection of objects that this script should be bound to - what "bound to" means is implementation-dependent
		\return true, iff the script was successfully executed without errors (either syntax or runtime)
	*/
	virtual bool execute(const std::string& ScriptName, const std::string& Script, const context_t& Context) = 0;

	/**	\brief Requests a cancellation of all running scripts.
		\return true, iff script cancellation is supported by this engine.
		\note Cancellation may be asynchronous, i.e. scripts may still be running when the call returns, and may continue to run for an indefinite period before shutting down, if at all.
	*/
	virtual bool halt() = 0;

	/**	\brief Modifies a script so that it can be recognized by this engine
		\param Script The complete source code of the script to be modified
		\note This method may be called an arbitrary number of times for a single script, so you should not
			naively modify the source without first checking to see if it has already been blessed
	*/
	virtual void bless_script(std::string& Script) = 0;

	/** Converts a command-node command into source code appropriate to this language
		\param CommandNode The command node executing the command to be converted
		\param Command The name of the command to be converted
		\param Arguments The command arguments to be converted
		\param Result Buffer to store the converted source code
		\return true, iff the command was successfully converted into source code
	*/
	virtual bool convert_command(icommand_node& CommandNode, const std::string& Command, const std::string& Arguments, std::string& Result) = 0;

protected:
	iscript_engine() {}
	iscript_engine(const iscript_engine& RHS) {}
	iscript_engine& operator = (const iscript_engine& RHS) { return *this; }
	virtual ~iscript_engine() {}
};

} // namespace k3d

#endif // K3DSDK_ISCRIPT_ENGINE_H


