
/*
 * The JTS Topology Suite is a collection of Java classes that
 * implement the fundamental operations required to validate a given
 * geo-spatial data set to a known topological specification.
 *
 * Copyright (C) 2001 Vivid Solutions
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * For more information, contact:
 *
 *     Vivid Solutions
 *     Suite #1A
 *     2328 Government Street
 *     Victoria BC  V8T 5G5
 *     Canada
 *
 *     (250)385-6040
 *     www.vividsolutions.com
 */
package com.vividsolutions.jts.noding;

import com.vividsolutions.jts.algorithm.LineIntersector;
import com.vividsolutions.jts.geom.Coordinate;

/**
 * Contains a list of consecutive line segments which can be used to node the segments.
 * The line segments are represented by an array of {@link Coordinate}s.
 *
 *
 * @version 1.6
 */
public class SegmentString {

  private SegmentNodeList eiList = new SegmentNodeList(this);
  private Coordinate[] pts;
  private Object context;
  private boolean isIsolated;

  public SegmentString(Coordinate[] pts, Object context)
  {
    this.pts = pts;
    this.context = context;
  }

  public Object getContext() { return context; }
  public SegmentNodeList getIntersectionList() { return eiList; }
  public int size() { return pts.length; }
  public Coordinate getCoordinate(int i) { return pts[i]; }
  public Coordinate[] getCoordinates() { return pts; }

  public void setIsolated(boolean isIsolated)  {    this.isIsolated = isIsolated;  }
  public boolean isIsolated()  {    return isIsolated;  }


  public boolean isClosed()
  {
    return pts[0].equals(pts[pts.length - 1]);
  }

  /**
   * Adds EdgeIntersections for one or both
   * intersections found for a segment of an edge to the edge intersection list.
   */
  public void addIntersections(LineIntersector li, int segmentIndex, int geomIndex)
  {
    for (int i = 0; i < li.getIntersectionNum(); i++) {
      addIntersection(li, segmentIndex, geomIndex, i);
    }
  }
  /**
   * Add an SegmentNode for intersection intIndex.
   * An intersection that falls exactly on a vertex
   * of the SegmentString is normalized
   * to use the higher of the two possible segmentIndexes
   */
  public void addIntersection(LineIntersector li, int segmentIndex, int geomIndex, int intIndex)
  {
    Coordinate intPt = new Coordinate(li.getIntersection(intIndex));
    double dist = li.getEdgeDistance(geomIndex, intIndex);
    addIntersection(intPt, segmentIndex, dist);
  }

  public void OLDaddIntersection(LineIntersector li, int segmentIndex, int geomIndex, int intIndex)
  {
    Coordinate intPt = new Coordinate(li.getIntersection(intIndex));
    int normalizedSegmentIndex = segmentIndex;
    double dist = li.getEdgeDistance(geomIndex, intIndex);
//Debug.println("edge intpt: " + intPt + " dist: " + dist);
    // normalize the intersection point location
    int nextSegIndex = normalizedSegmentIndex + 1;
    if (nextSegIndex < pts.length) {
      Coordinate nextPt = pts[nextSegIndex];
//Debug.println("next pt: " + nextPt);

      // Normalize segment index if intPt falls on vertex
      // The check for point equality is 2D only - Z values are ignored
      if (intPt.equals2D(nextPt)) {
//Debug.println("normalized distance");
          normalizedSegmentIndex = nextSegIndex;
          dist = 0.0;
      }
    }
    /**
    * Add the intersection point to edge intersection list.
    */
    SegmentNode ei = eiList.add(intPt, normalizedSegmentIndex, dist);
//ei.print(System.out);
  }
  /**
   * Add an EdgeIntersection for intersection intIndex.
   * An intersection that falls exactly on a vertex of the edge is normalized
   * to use the higher of the two possible segmentIndexes
   */
  public void addIntersection(Coordinate intPt, int segmentIndex)
  {
    double dist = LineIntersector.computeEdgeDistance(intPt, pts[segmentIndex], pts[segmentIndex + 1]);
    addIntersection(intPt, segmentIndex, dist);
  }

  public void addIntersection(Coordinate intPt, int segmentIndex, double dist)
  {
    int normalizedSegmentIndex = segmentIndex;
//Debug.println("edge intpt: " + intPt + " dist: " + dist);
      // normalize the intersection point location
      int nextSegIndex = normalizedSegmentIndex + 1;
      if (nextSegIndex < pts.length) {
        Coordinate nextPt = pts[nextSegIndex];
//Debug.println("next pt: " + nextPt);

        // Normalize segment index if intPt falls on vertex
        // The check for point equality is 2D only - Z values are ignored
        if (intPt.equals2D(nextPt)) {
//Debug.println("normalized distance");
            normalizedSegmentIndex = nextSegIndex;
            dist = 0.0;
        }
      }
      /**
      * Add the intersection point to edge intersection list.
      */
      SegmentNode ei = eiList.add(intPt, normalizedSegmentIndex, dist);
//ei.print(System.out);

  }

}
