% flyspell.sl  -*- mode: SLang; mode: Fold -*-
%
% Author:        Paul Boekholt <p.boekholt@hetnet.nl>
% 
% $Id: flyspell.sl,v 1.11 2003/07/12 17:26:24 paul Exp paul $
% 
% Copyright (c) 2003 Paul Boekholt.
% Released under the terms of the GNU GPL (version 2 or later).
% 
% This file provides a minor mode for on-the-fly spell checking.  We use
% the _jed_switch_active_buffer_hooks to set the _jed_after_key_hooks
% depending on the flyspell blocal var.  In English: you need JED 0.99.16
% or greater.
% To do: make a separate syntax table for each language.
  
require("ispell_common");
autoload("get_blocal", "sl_utils");
autoload("add_keyword_n", "syntax");
use_namespace("ispell");
% The DFA trick used in this file does not work with "-" as an otherchar,
% so we trim it.
variable flyspell_otherchars = strtrim_beg(ispell_otherchars, "-");
variable flyspell_wordchars = flyspell_otherchars + ispell_letters;


% do you want to use keyword2 to have red misspellings?
custom_variable("flyspell_use_keyword2", 1);
static variable flyspell_chars = " 
";

!if (is_defined("flyspell_process"))
  public variable flyspell_process = -1;

%{{{ Flyspell process

% This will also restart flyspell, through the flyspell_is_dead() function
public define kill_flyspell()
{
   if (-1 != flyspell_process)
     kill_process(flyspell_process);
}

public define flyspell_parse_output (pid, output)
{
   output = strtrim(output);
   if (strlen(output))
     {
	output = strtok(output)[1];
	if (flyspell_use_keyword2)
	  add_keyword_n("Flyspell", output, 2);
	else
	  add_keyword("Flyspell", strtok(output)[1]);
     }
}

static define flyspell_init_process();
public define flyspell_is_dead (pid, flags, status)
{
   if (flags & 14)
     flyspell_process = -1;
}

static define flyspell_init_process ()
{
   variable buf, ibuf = " *flyspell*";
   if (flyspell_process != -1)
     return;
   buf = whatbuf();

   variable args = strtok (ispell_command + " -a");
   setbuf(ibuf);
   erase_buffer;
   message ("starting flyspell process....");
   foreach (args)
     ;
   length (args) - 1;
   flyspell_process = open_process ();

   sleep(0.5);
   get_process_input(2);

   if (flyspell_process == -1)
     error ("could not start ispell");
      bob ();
   if (looking_at_char ('@'))     %  ispell header
     del_through_eol ();
   else if (looking_at("Can't open"))
     {
	flyspell_process = -1;
	verror ("You don't have that dictionary!");
     }
   send_process(flyspell_process, "!\n");
   set_process (flyspell_process, "signal", "flyspell_is_dead");
   set_process (flyspell_process, "output", "flyspell_parse_output");
   process_query_at_exit(flyspell_process, 0);
   setbuf(buf);
}


%}}}

%{{{ Flyspelling

static variable lastword = "", lastpoint = 0;
public define flyspell_word()
{
   variable word, point;
   if (flyspell_process == -1) 
     flyspell_init_process;
   push_spot();
   bskip_chars(ispell_non_letters);
   push_mark();
   bskip_chars(flyspell_wordchars);
   skip_chars(flyspell_otherchars);
   point = _get_point();
   word = bufsubstr();
   if (word == "") return pop_spot();

   EXIT_BLOCK 
     { 
	(lastword, lastpoint) = word, point;
	pop_spot();
     }
   
   if (word == lastword)
   {
      if (point != lastpoint)
	{
	   bskip_chars(ispell_non_letters);
	   bskip_chars(flyspell_wordchars);
	   skip_chars(flyspell_otherchars);

	   if (looking_at(word))
	     {
		message("double word");
		beep();
	     }
	}
      return;
   }
   if (strlen(word) < 3) return;
   
   send_process( flyspell_process, strcat ("^", word, "\n"));
}

static define after_key_hook ()
{
   if (is_substr(flyspell_chars, LASTKEY))
     flyspell_word();
}


%}}}

%{{{ Turning flyspell mode on/off

static define flyspell_switch_active_buffer_hook(oldbuf)
{
   if (get_blocal("flyspell", 0))
     add_to_hook ("_jed_after_key_hooks", &after_key_hook);
   else
     remove_from_hook ("_jed_after_key_hooks", &after_key_hook);
}

static define toggle_local_flyspell()
{
   variable flyspell = not get_blocal("flyspell", 0);
   define_blocal_var("flyspell", flyspell);
   if (flyspell)
     {
	set_status_line(str_replace_all(Status_Line_String, "%m", "%m fly"), 0);
	if (flyspell_current_dictionary != ispell_current_dictionary)
	  {
	     kill_flyspell;
	     flyspell_current_dictionary = ispell_current_dictionary;
	  }
     }
   else
     {
	set_status_line(Status_Line_String, 0);
     }
   flyspell_switch_active_buffer_hook("");  % hook expects an argument
}

%}}}

create_syntax_table("Flyspell");
set_syntax_flags("Flyspell", 0);
define_syntax(ispell_wordchars, 'w', "Flyspell");
if (flyspell_use_keyword2)
  set_color("keyword2", "brightred", get_color("normal"), exch(), pop);

#ifdef HAS_DFA_SYNTAX
%%% DFA_CACHE_BEGIN %%%
static define setup_dfa_callback (name)
{
   % Uncomment this line to build the DFA cache, in that case you must have
   % the same otherchars in all languages.
   % dfa_enable_highlight_cache( "flyspell.dfa", "Flyspell");
   dfa_define_highlight_rule 
     (sprintf("[%s][%s]*[%s]",ispell_letters, flyspell_wordchars, ispell_letters),
      "Kdelimiter", "Flyspell");
   % Kdelimiter means "Keyword delimiter" or something. I found this in html.sl
   % but there it needs fixing.
   dfa_build_highlight_table ("Flyspell");
}
dfa_set_init_callback (&setup_dfa_callback, "Flyspell");
%%% DFA_CACHE_END %%%
#endif

% Turn on / toggle flyspell mode.
public define flyspell_mode()
{
   flyspell_init_process();
   use_syntax_table("Flyspell");
   use_dfa_syntax(1);
   toggle_local_flyspell();
   add_to_hook("_jed_switch_active_buffer_hooks",
	       &flyspell_switch_active_buffer_hook);
}

provide("flyspell");
