/*************************************************************************
 * 
 * irmp3 - Multimedia Audio Jukebox for Linux
 * http://irmp3.sourceforge.net
 *
 * $Source: /cvsroot/irmp3/irmp3/src/irmp3d/mod_mpg123.c,v $
 * -- module for the mpg123 player
 * $Id: mod_mpg123.c,v 1.38 2004/02/23 20:32:28 boucman Exp $
 *
 * Copyright (C) by Andreas Neuhaus <andy@fasta.fh-dortmund.de>
 *
 * Please contact the current maintainer, Jeremy Rosen <jeremy.rosen@enst-bretagne.fr>
 * for information and support regarding irmp3.
 *
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <signal.h>
#include <string.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/stat.h>
#include <sys/wait.h>

#include "config.h"
#include "irmp3tools.h"
#include "irmp3log.h"
#include "irmp3config.h"
#include "irmp3mod.h"
#include "mod_mpg123.h"

/*************************************************************************
 * GLOBALS
 */
int mod_mpg123_pid    = 0;	// PID of mpg123
int mod_mpg123_fdsend = 0;	// fd to send commands to mpg123
int mod_mpg123_fdrecv = 0;	// fd to get responses from mpg123
int mod_mpg123_fderror = 0;	// fd to get responses from mpg123
static char *artist = NULL;
static char *title = NULL;
static char *album = NULL;
static char *year = NULL;
static char *genre = NULL;
static char *comment = NULL;
#define MPG123_STATUS_HALT	0
#define MPG123_STATUS_STOP	1
#define MPG123_STATUS_PLAY	2
#define MPG123_STATUS_PAUSE	3
int  mod_mpg123_status = MPG123_STATUS_HALT;
fd_set mod_mpg123_fdset;
static int lastsecpos = -1;
static int lastsecremain = -1;

/*************************************************************************
 * MODULE INFO
 */
mod_t mod_mpg123 = {
	"mod_mpg123",
	mod_mpg123_deinit,	// deinit
	NULL,			// reload
	&mod_mpg123_fdset,	// watchfd_set
	NULL,			// poll
	NULL,			// update
	mod_mpg123_message,	// message
	mod_mpg123_chld,	// SIGCHLD handler
	mod_mpg123_init,
	NULL,			// avoid warning
};
/*************************************************************************
 * CATCH SIGNAL OF EXITING PLAYER
 */
void mod_mpg123_chld (pid_t pid, int status)
{
	if (pid == mod_mpg123_pid) {
		log_printf(LOG_DEBUG, "mod_mpg123_chld(): player %d  exited with code %d\n", pid,WEXITSTATUS(status));
		mod_mpg123_pid = 0;
		close(mod_mpg123_fdrecv);
		mod_mpg123_fdrecv=0;
		close(mod_mpg123_fdsend);
		mod_mpg123_fdsend=0;
		close(mod_mpg123_fderror);
		mod_mpg123_fderror=0;
	}
}


/*************************************************************************
 * ANSWER A HALT REQUEST
 *
 */
void mod_mpg123_halt (void)
{
	if (mod_mpg123_fdsend)
		close(mod_mpg123_fdsend);
	mod_mpg123_fdsend = 0;
	if (mod_mpg123_fdrecv)
		close(mod_mpg123_fdrecv);
	mod_mpg123_fdrecv = 0;
	if (mod_mpg123_fdrecv)
		close(mod_mpg123_fderror);
	mod_mpg123_fderror = 0;
	if (mod_mpg123_pid) {
		log_printf(LOG_DEBUG, "mod_mpg123_halt(): killed player pid %d\n", mod_mpg123_pid);
		kill(mod_mpg123_pid, SIGTERM);
		mod_mpg123_pid = 0;
	}
	FD_ZERO(&mod_mpg123_fdset);
	mod_mpg123.poll = NULL;
	if(album) {free(album); album = NULL;}
	if(title) {free(title); title = NULL;}
	if(comment) {free(comment); comment = NULL;}
	if(artist) {free(artist); artist = NULL;}
	if(year) {free(year); year = NULL;}
	if(genre) {free(genre); genre = NULL;}
	lastsecpos = -1;
	lastsecremain = -1;
	mod_mpg123_status = MPG123_STATUS_HALT;
}



/*************************************************************************
 * START PLAYER PROCESS
 */
int mod_mpg123_launch (void)
{

	// check if player is already running
	if (mod_mpg123_status != MPG123_STATUS_HALT) {
		log_printf(LOG_DEBUG, "mod_mpg123_start(): mod_mpg123_pid is %d; mpg123 seems to be running!\n", mod_mpg123_pid);
		return 0;
	}



	// fork child process
	// check if player is running
	mod_mpg123_pid = system_noblock(&mod_mpg123_fdsend,&mod_mpg123_fdrecv,&mod_mpg123_fderror,"%s %s -R junk -",config_getstr("mpg123_binary",MPG123_PATH),config_getstr("mpg123_param",""));
	sleep(1);
	if (!mod_mpg123_pid || mod_mpg123_pid == -1) {
		//char s[512];
		mod_mpg123_pid = 0;
		log_printf(LOG_DEBUG, "mod_mpg123_start(): player process didn't start!\n");
		mod_mpg123_halt();
		//while(readline(mod_mpg123_fdrecv,s,512)){log_printf(LOG_ERROR,"%s\n",mod_mpg123_fdrecv);}
		//exit(0);
		return -1;
	}

	FD_SET(mod_mpg123_fdrecv,&mod_mpg123_fdset);
	FD_SET(mod_mpg123_fderror,&mod_mpg123_fdset);
	mod_mpg123.poll = mod_mpg123_poll;
	return 0;
}
/******************************************
 * ANSWER A STOP REQUEST
 *
 */

void mod_mpg123_stop() 
{
	switch(mod_mpg123_status) {
		case MPG123_STATUS_HALT:
		case MPG123_STATUS_STOP:
			return;
		case MPG123_STATUS_PLAY:
			sendtext(mod_mpg123_fdsend, "pause\n");
		case MPG123_STATUS_PAUSE:
			//sendtext(mod_mpg123_fdsend, "JUMP 0\n");
			mod_mpg123_status = MPG123_STATUS_STOP;
			log_printf(LOG_NORMAL, "Playing stopped\n");
			lastsecpos = -1;
			lastsecremain = -1;
			log_printf(LOG_NOISYDEBUG,"mod_mpg123 : entered stop state\n");
			return;
		default:
			log_printf(LOG_ERROR,"mod_mpg123: unknown status %d\n",mod_mpg123_status);
			return;
	}
}
		

/******************************************
 * ANSWER A EOS DETECTION
 * we can't use mod_mpg123_stop because a pause seems to stuck mpg123
 *
 */
void  mod_mpg123_eos() {
	log_printf(LOG_NORMAL, "End of song reached\n");
	mod_sendmsg(MSGTYPE_PLAYER, "endofsong");
	mod_mpg123_status = MPG123_STATUS_STOP;
	log_printf(LOG_NORMAL, "Playing stopped\n");
	lastsecpos = -1;
	lastsecremain = -1;
	log_printf(LOG_NOISYDEBUG,"mod_mpg123 : entered stop state\n");
	return;
}
/******************************************
 * ANSWER A PLAY REQUEST
 *
 */
	
void mod_mpg123_play(char* song) {
	char *tmp;
	switch(mod_mpg123_status) {
		case MPG123_STATUS_HALT:
			if (mod_mpg123_launch()) {
				mod_sendmsg(MSGTYPE_PLAYER, "error");
				return;
			}
			tmp = malloc(strlen(song) + 7);
			lastsecpos = -1;
			lastsecremain = -1;
			sprintf(tmp,"LOAD %s\n",song);
			sendtext(mod_mpg123_fdsend, tmp);
			log_printf(LOG_NORMAL, "Playing '%s'\n", song);
			free(tmp);
			mod_mpg123_status= MPG123_STATUS_PLAY;
			log_printf(LOG_NOISYDEBUG,"mod_mpg123 : entered play state\n");
			return;

		case MPG123_STATUS_PAUSE:
			sendtext(mod_mpg123_fdsend, "PAUSE\n");
		case MPG123_STATUS_STOP:
		case MPG123_STATUS_PLAY:
			tmp = malloc(strlen(song) + 7);
			lastsecpos = -1;
			lastsecremain = -1;
			if(album) {free(album); album = NULL;}
			if(title) {free(title); title = NULL;}
			if(comment) {free(comment); comment = NULL;}
			if(artist) {free(artist); artist = NULL;}
			if(year) {free(year); year = NULL;}
			if(genre) {free(genre); genre = NULL;}
			sprintf(tmp,"LOAD %s\n",song);
			sendtext(mod_mpg123_fdsend, tmp);
			log_printf(LOG_NORMAL, "Playing '%s'\n", song);
			lastsecpos = -1;
			lastsecremain = -1;
			free(tmp);
			mod_mpg123_status= MPG123_STATUS_PLAY;
			log_printf(LOG_NOISYDEBUG,"mod_mpg123 : entered play state\n");
			return;
			
		default:
			log_printf(LOG_ERROR,"mod_mpg123: unknown status %d\n",mod_mpg123_status);
			return;
	}
}

/******************************************
 * ANSWER A PAUSE REQUEST
 *
 */
	
void mod_mpg123_pause() {
	switch(mod_mpg123_status) {
		case MPG123_STATUS_HALT:
		case MPG123_STATUS_STOP:
			return;
			
		case MPG123_STATUS_PAUSE:
			sendtext(mod_mpg123_fdsend, "PAUSE\n");
			log_printf(LOG_NOISYDEBUG,"mod_mpg123 : entered play state\n");
			mod_mpg123_status= MPG123_STATUS_PLAY;
			return;
			
		case MPG123_STATUS_PLAY:
			sendtext(mod_mpg123_fdsend, "PAUSE\n");
			log_printf(LOG_NOISYDEBUG,"mod_mpg123 : entered pause state\n");
			mod_mpg123_status= MPG123_STATUS_PAUSE;
			return;
			
		default:
			log_printf(LOG_ERROR,"mod_mpg123: unknown status %d\n",mod_mpg123_status);
			return;

	}
}
/******************************************
 * ANSWER A SEEK REQUEST
 *
 */
	
void mod_mpg123_seek(char* position) {
	switch(mod_mpg123_status) {
		case MPG123_STATUS_HALT:
		case MPG123_STATUS_STOP:
			return;

		case MPG123_STATUS_PAUSE:
			sendtext(mod_mpg123_fdsend, "PAUSE\n");
			log_printf(LOG_NOISYDEBUG,"mod_mpg123 : entered play state\n");
			mod_mpg123_status= MPG123_STATUS_PLAY;
		case MPG123_STATUS_PLAY:
			if (*position=='-' || *position=='+')
				sendtext(mod_mpg123_fdsend, "JUMP %c%d\n", *position, atoi(position+1)*38);
			else
				sendtext(mod_mpg123_fdsend, "JUMP %d\n", atoi(position)*38);
			return;

		default:
			log_printf(LOG_ERROR,"mod_mpg123: unknown status %d\n",mod_mpg123_status);
			return;
	}
}




/*************************************************************************
 * POLL INPUT DATA
 */
void mod_mpg123_poll (int fd)
{
	static int lastframepos, lastframeremain;
	char s[512], *c1, *c2, *c3, *c4, *c5, buf[32];
	int rc, secpos, secremain;

	// read the response from the player
	rc = readline(fd, s, sizeof(s));
	if (rc < 0) {
		// player closed connection
		log_printf(LOG_DEBUG,"mod_mpg123_poll(): player closed connection with %s.\n",strerror(errno));
		mod_mpg123_halt();
		mod_sendmsg(MSGTYPE_PLAYER, "error");
		return;
	}

	// process player response
	if (!s || !*s)
		return;
	c1 = strtok(s, " \t");
	if(!c1) return;

	if (!strcasecmp(c1, "@R")) {
		// player version received
		c2 = strtok(NULL, "");
		if(!c2) return;
		mod_sendmsgf(MSGTYPE_INFO, "player version %s", c2);

	} else if (!strcasecmp(c1, "@P")) {
		// playing status received
		c2 = strtok(NULL, " \t");
		if(!c2) return;
		if (atoi(c2) == 0) {
			c3 =  strtok(NULL, " \t");
			// There are various ways how the end of the song
			// is notified :-( It should be "@P 0 EOF" in future
			if ( (!c3) || ( !strcasecmp(c3, "EOF")) || (lastframepos > 10 && lastframeremain < 10)) {
				mod_mpg123_eos();
			} else {
				log_printf(LOG_DEBUG,"mod_mpg123_poll(): received unknown mpg123 status %s %s %s.\n",c1,c2,c3);
 				if(!strcasecmp("yes",config_getstr("abort_on_unknown_mpg123_status","no"))) {
					log_printf(LOG_NORMAL, "Playing stopped\n");
					mod_sendmsg(MSGTYPE_PLAYER, "error");
					mod_mpg123_halt();
				}
			}
		} else if (atoi(c2) == 1) {
			log_printf(LOG_NORMAL, "Playing paused\n");
		} else if (atoi(c2) == 2) {
			log_printf(LOG_NORMAL, "Playing continued\n");
		} else if (atoi(c2) == 3) {
			mod_mpg123_eos();
		}

	} else if ( !strcasecmp(c1, "@I")) {
		// song-info received (artist, title, etc)
		c2 = strtok(NULL, "");
		if (!strncasecmp(c2, "ID3:", 4)) {
			// ID3 tag
			c2 += 4;
			strncpy(buf, c2, 30); buf[30]=0; trim(buf); c2 += 30;

			if(title) free(title);
			title = strdup(buf);
			mod_sendmsgf(MSGTYPE_EVENT,"update title %s",title);
			strncpy(buf, c2, 30); buf[30]=0; trim(buf); c2 += 30;
			
			if(artist) free(artist);
			artist = strdup(buf);
			mod_sendmsgf(MSGTYPE_EVENT,"update artist %s",artist);
			strncpy(buf, c2, 30); buf[30]=0; trim(buf); c2 += 30;
			
			if(album) free(album);
			album = strdup(buf);
			mod_sendmsgf(MSGTYPE_EVENT,"update album %s",album);
			strncpy(buf, c2, 4); buf[4]=0; trim(buf); c2 += 4;
			
			if(year) free(year);
			year = strdup(buf);
			mod_sendmsgf(MSGTYPE_EVENT,"update year %s",year);
			strncpy(buf, c2, 30); buf[30]=0; trim(buf); c2 += 30;
			
			if(comment) free(comment);
			comment = strdup(buf);
			mod_sendmsgf(MSGTYPE_EVENT,"update comment %s",comment);
			strncpy(buf, c2, 30); buf[30]=0; trim(buf); c2 += 30;
			
			if(genre) free(genre);
			genre = strdup(buf);
			mod_sendmsgf(MSGTYPE_EVENT,"update genre %s",genre);
		} else {
			if(album) {free(album); album = NULL;}
			if(title) {free(title); title = NULL;}
			if(comment) {free(comment); comment = NULL;}
			if(artist) {free(artist); artist = NULL;}
			if(year) {free(year); year = NULL;}
			if(genre) {free(genre); genre = NULL;}
		}
	} else if (!strcasecmp(c1, "@S")) {
		// stream-info received (mpeg type etc)
		// ignored, because not interesting for us, yet

	} else if (!strcasecmp(c1, "@F")) {
		// frame-info received (current times)
		c2 = strtok(NULL, " \t");
		if(!c2) return;
		c3 = strtok(NULL, " \t");
		if(!c3) return;
		c4 = strtok(NULL, " \t");
		if(!c4) return;
		c5 = strtok(NULL, " \t");
		if(!c5) return;
		lastframepos = atoi(c2);
		lastframeremain = atoi(c3);
		secpos = (int)atof(c4);
		secremain = (int)atof(c5);

		// only send out message if something has changed
		// this should help keeping message traffic low
		if (secpos != lastsecpos || secremain != lastsecremain) {
			if (lastsecpos == -1 && atof(c4) < 0.2 ) {
				mod_sendmsgf(MSGTYPE_PLAYER,"playing");
				lastsecpos = secpos;
				lastsecremain = secremain;
			} else if (lastsecpos != -1){
				mod_sendmsgf(MSGTYPE_PLAYER, "time %d %d", secpos, secremain);
				lastsecpos = secpos;
				lastsecremain = secremain;
			}
			// else, remaining message from previous song, do nothing
		}

	} else if ( !strcasecmp(c1, "Found")) {
	  // ignore mpg123 information about old header
	  c2 = strtok(NULL, "");
	  if(!c2) return;
	  if (!strcasecmp(c2, "old ID3 header")) {
	    log_printf(LOG_DEBUG, "mpg123 reported old ID3 header.\n");
	    if(album) {free(album); album = NULL;}
	    if(title) {free(title); title = NULL;}
	    if(comment) {free(comment); comment = NULL;}
	    if(artist) {free(artist); artist = NULL;}
	    if(year) {free(year); year = NULL;}
	    if(genre) {free(genre); genre = NULL;}
	  }
	} else {
		// unknown message, probably an error
		c2 =  strtok(NULL, "");
		if(c2 && strstr(c2,"No such file or directory")){
			log_printf(LOG_NORMAL, "Unknown file requested\n");
			mod_sendmsg(MSGTYPE_PLAYER, "error");
			mod_mpg123_halt();
		} else {
			log_printf(LOG_VERBOSE, "mpg123 reports: '%s %s'\n", c1, c2);
			if(!strcasecmp("yes",config_getstr("abort_on_unknown_mpg123_status","no"))) {
				log_printf(LOG_NORMAL, "Playing stopped\n");
				mod_sendmsg(MSGTYPE_PLAYER, "error");
				mod_mpg123_halt();
			}
		}
	}
}


/*************************************************************************
 * RECEIVE MESSAGE
 */
void mod_mpg123_message (int msgtype, char *msg,const char __attribute__((unused))*sender)
{
	char *c1, *c2, *c3;

	if (msgtype == MSGTYPE_PLAYER) {
		c1 = strtok(msg, " \t");
		if(!c1) return;

		// see if we can play the requested song
		if ( !strcasecmp(c1, "play")) {
			c2 = strtok(NULL, " \t");
			if(!c2) return;
			if ( !strcasecmp(c2,"mod_mpg123") ) {
				c3 = strtok(NULL, "");
				if(!c3) return;
				mod_mpg123_play(c3);
			} else {
					log_printf(LOG_DEBUG, "mod_mpg123_message(): not for me: '%s'\n", c2);
					mod_mpg123_halt();
			}
		} else if ( !strcasecmp(c1, "stop") && mod_mpg123_pid) {
			mod_mpg123_stop();
		} else if ( !strcasecmp(c1, "pause") && mod_mpg123_pid) {
			mod_mpg123_pause();

		} else if ( !strcasecmp(c1, "seek") && mod_mpg123_pid) {
			c2 = strtok(NULL, " \t");
			if(!c2) return;
			log_printf(LOG_DEBUG, "mod_mpg123_message(): seeking %s\n", c2);
			mod_mpg123_seek(c2);
		}
	} else if (msgtype == MSGTYPE_QUERY) {
		c1 = strtok(msg, " \t");
		if(!c1) return;
		if ( !strcasecmp(c1,"whocanplay")) {
			c2 =  strtok(NULL, "") ;
			if(  !strcasecmp(c2,"mp2")) {
				mod_sendmsgf(MSGTYPE_INFO,"canplay mod_mpg123 %s",c2);
			} else if(  !strcasecmp(c2,"mp3")) {
				mod_sendmsgf(MSGTYPE_INFO,"canplay mod_mpg123 %s",c2);
			} else if(  !strcasecmp(c2,"http")) {
				mod_sendmsgf(MSGTYPE_INFO,"canplay mod_mpg123 %s",c2);
			}
		} else {
			if (mod_mpg123_status != MPG123_STATUS_HALT) {
				if(artist  && !strcasecmp(c1,"artist")) mod_sendmsgf(MSGTYPE_INFO,"artist %s",artist);
				if(album   && !strcasecmp(c1,"album"))  mod_sendmsgf(MSGTYPE_INFO,"album %s",album);
				if(title   && !strcasecmp(c1,"title"))  mod_sendmsgf(MSGTYPE_INFO,"title %s",title);
				if(year    && !strcasecmp(c1,"year"))   mod_sendmsgf(MSGTYPE_INFO,"year %s",year);
				if(genre   && !strcasecmp(c1,"genre"))  mod_sendmsgf(MSGTYPE_INFO,"genre %s",genre);
				if(comment && !strcasecmp(c1,"comment"))mod_sendmsgf(MSGTYPE_INFO,"comment %s",comment);
				if(!strcasecmp(c1,"time"))    mod_sendmsgf( MSGTYPE_INFO,"time %d %d", lastsecpos, lastsecremain);
			}
		}
	}
}


/*************************************************************************
 * MODULE INIT FUNCTION
 */
char *mod_mpg123_init (void)
{
	FD_ZERO(&mod_mpg123_fdset);
	return NULL;
}


/*************************************************************************
 * MODULE DEINIT FUNCTION
 */
void mod_mpg123_deinit (void)
{
	mod_mpg123_halt();
	log_printf(LOG_DEBUG, "mod_mpg123_deinit(): deinitialized\n");
}


/*************************************************************************
 * EOF
 */
