/*************************************************************************
 * $Id: mod_netctl.c,v 1.8 2001/05/18 15:48:24 dpotter Exp $
 *
 * mod_netctl.c - a server module for network control of IRMP3
 *
 * Copyright (C) 2000 by David E. Potter <dp-irmp3@dpotter.com>
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <fcntl.h>
#include <time.h>
#include <dirent.h>
#include <fnmatch.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/stat.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <malloc.h>
#include <regex.h>

#include "tools.h"
#include "config.h"
#include "log.h"
#include "mod.h"
#include "mod_netctl.h"


/*************************************************************************
 * GLOBALS AND STATICS
 */

	fd_set	 mod_netctl_fdset;
static int	 sockfd=0,cli[MOD_NETCTL_MAXCLIENTS];
	char	*sockbuf[MOD_NETCTL_MAXCLIENTS];
	filter_t cli_filter[MOD_NETCTL_MAXCLIENTS];
	int		 cli_sock[MOD_NETCTL_MAXCLIENTS],current_cli=0;
	int	mod_netctl_enable;

#define	MOD_NETCTL_BUFSIZE	512



/*************************************************************************
 * MODULE INFO
 */
mod_t mod_netctl = {
	mod_netctl_deinit,	// deinit
	NULL,				// reload
	&mod_netctl_fdset,	// watch_fdset
	mod_netctl_poll,	// poll
	NULL,				// update
	mod_netctl_message,	// message
	NULL,				// SIGCHLD handler
};

void mod_netctl_message (int msgtype, char *msg)
{
	int	client=0;
	char	buf[MOD_NETCTL_BUFSIZE];

	if(current_cli != 0) {
		bzero(buf,sizeof(buf));
		if (msgtype == MSGTYPE_GENERIC) strcpy(buf,"200: ");
		if (msgtype == MSGTYPE_INPUT)   strcpy(buf,"210: ");
		if (msgtype == MSGTYPE_MIXER)   strcpy(buf,"220: ");
		if (msgtype == MSGTYPE_PLAYER)  strcpy(buf,"230: ");

		if (strlen(msg) < sizeof(buf)-5) strcat(buf,msg);
		else {
			strncat(buf,msg,sizeof(buf)-5);
			buf[MOD_NETCTL_BUFSIZE]='\0';
		}

		for (client=0;client<MOD_NETCTL_MAXCLIENTS;client++) {
			if (cli_sock[client] && ( cli_filter[client].enabled ? !mod_netctl_regex_match(&cli_filter[client].regx,buf) : 1)) {
				write(cli_sock[client],buf,strlen(buf));
				write(cli_sock[client],(char *)"\r\n",(int) 2);
			}
		}
	}
}

/*************************************************************************
 * NETWORK POLL FUNCTION
 */


void mod_netctl_poll(int fd)
{

	struct sockaddr_in	cli_addr;
	int					clilen=0,newsockfd,i=0,result;
	char				buf[MOD_NETCTL_BUFSIZE];
	char				*tmpbuf=NULL,*ctl_char=NULL;
	char				buffer_overflow[]="Buffer overflow. Clearing buffer.\n\0";


	if (fd == sockfd) {		// new connection?
		log_printf(LOG_DEBUG,"mod_netctl_poll(): client connecting...\n");
		clilen = sizeof(cli_addr);
		newsockfd=accept(fd,(struct sockaddr *) &cli_addr,&clilen);

		if (newsockfd <0) 
			log_printf(LOG_ERROR,"mod_netctl_poll(): accept error.\n");
		else 	mod_netctl_new_client(newsockfd);
	} else {
		for(i=0;i<=MOD_NETCTL_MAXCLIENTS;i++) {
			if(cli_sock[i]==fd) {
				
				// cli_sock[i] is our socket, i is our client
				bzero(buf,sizeof(buf));
				result=read(fd,buf,sizeof(buf));
				if (0 >= result)  {
					mod_netctl_disc_client(i);	// bad read, kill the client
				} else {
					log_printf(LOG_NOISYDEBUG,"mod_netctl_poll(): received %d bytes from client %d (fd %d): %s\n",result,i,cli_sock[i],buf);
					if (strlen(sockbuf[i])+result<MOD_NETCTL_BUFSIZE)  // buffer bounds check
						strcat(sockbuf[i],buf);
					else {		// buffer overflow
						write(fd,buffer_overflow,sizeof(buffer_overflow));
						bzero(sockbuf[i],MOD_NETCTL_BUFSIZE);
						log_printf(LOG_DEBUG,"Buffer overflow for network client %d.\n",i);
					}
					while((tmpbuf=memchr(sockbuf[i],'\r',MOD_NETCTL_BUFSIZE)) || (tmpbuf=memchr(sockbuf[i],'\n',MOD_NETCTL_BUFSIZE))) {	
							// user pressed enter
						tmpbuf[0]='\0';
						tmpbuf++;
						while(tmpbuf[0]=='\r' || tmpbuf[0]=='\n') // handle cr, lf, and crlf.
							tmpbuf++;
						while((ctl_char=memchr(sockbuf[i],0xff,strlen(sockbuf[i])))) 
							// remove telnet ctl chars - always come in sets of 3
							memcpy(sockbuf[i],ctl_char+3,MOD_NETCTL_BUFSIZE-(ctl_char-sockbuf[i]));
						log_printf(LOG_DEBUG,"mod_netctl_poll(): got command: %s\n",sockbuf[i]);

						// am i receiving a 'network filter' command?
						// if so, handle it and *do not* send it as a message

						// (the hard-coded 14 and 15 in the code below represents 
						// the number of characters in the command "network filter")

						if (!strncasecmp(sockbuf[i],"network filter",14)) {
							if(cli_filter[i].enabled) regfree(&cli_filter[i].regx);
							if(!regcomp(&cli_filter[i].regx,&(sockbuf[i][15]),REG_EXTENDED)) cli_filter[i].enabled=1;
								else cli_filter[i].enabled=0;
							log_printf(LOG_NOISYDEBUG,"mod_netctl_poll(): set network filter for client %d to %s.\n",i,&(sockbuf[i][15]));
							
						} else
							mod_sendmsg(MSGTYPE_INPUT,sockbuf[i]);

						if(tmpbuf[0] == '\0' ) {
							bzero(sockbuf[i],MOD_NETCTL_BUFSIZE);
						} else strcpy(sockbuf[i],tmpbuf);
					}
				}
			}
		}
	}
}

/*************************************************************************
 * SETUP NEW NETWORK CLIENT BUFFER
 */

void mod_netctl_new_client(int newsockfd)
{
	char 	maxclient[]="Maximum clients already connected.\r\n\0";
	char	welcome[128];
	int		i=0;
	

	if (current_cli >= MOD_NETCTL_MAXCLIENTS) {
		write(newsockfd,maxclient,sizeof(maxclient));
		close(newsockfd);
		log_printf(LOG_DEBUG,"mod_netctl_new_client(): connection rejected. Maximum client count already reached.\n");
	} else {

		current_cli++;

		// find available client number
		for(i=0;cli_sock[i] != 0 && i<=MOD_NETCTL_MAXCLIENTS;i++);
		cli_sock[i]=newsockfd;
		sockbuf[i]=(char *) malloc(MOD_NETCTL_BUFSIZE);
		bzero(sockbuf[i],MOD_NETCTL_BUFSIZE);
		FD_SET(cli_sock[i],&mod_netctl_fdset);

		sprintf(welcome,"100: IRMP3 %s (C) 2000 by Andreas Neuhaus and David Potter.\r\n110:%d You are client %d.\r\n",VERSION,i,i);
		write(newsockfd,welcome,strlen(welcome));
		
		log_printf(LOG_DEBUG,"mod_netctl_new_client(): client connection #%d on fd %d.\n",i,cli_sock[i]);
	}
}
		

/*************************************************************************
 * TEAR DOWN NETWORK CLIENT BUFFER
 */
void mod_netctl_disc_client(int client)
{
	log_printf(LOG_DEBUG,"mod_netctl_disc_client(): Disconnecting client #%d on fd %d.\n",client,cli_sock[client]);

	FD_CLR(cli_sock[client],&mod_netctl_fdset);
	close(cli_sock[client]);
	free(sockbuf[client]);
	cli_sock[client]=0;
	if(cli_filter[client].enabled) {
		regfree(&cli_filter[client].regx);
		cli_filter[client].enabled=0;
	}

	current_cli--; 
	cli_sock[MOD_NETCTL_MAXCLIENTS]=0;	// dont need this in any case.
			
}


/*************************************************************************
 * NETWORK SETUP FUNCTION
 */

int mod_netctl_net_setup(void)
{
	int			sockfd=0,netport=0;
	struct sockaddr_in	serv_addr;
        int reuse_addr = 1; //We can rebind to our address while a prev. conn. is still in wait-state     
	// open a TCP socket

	if ((sockfd=socket(AF_INET,SOCK_STREAM,0))<0) {
		log_printf(LOG_ERROR,"mod_netctl_net_setup(): cannot open stream socket.\n");
		return(-1);
	}

	// read the desired network port from the logfile

	netport=config_getnum("network_port",0);
	if ( !netport) {
		log_printf(LOG_ERROR,"mod_netctl_net_setup(): cannot find network port in irmp3.conf.\n");
		return (-2);
	}
	

	// bind our local address

	bzero((char *) &serv_addr,sizeof(serv_addr));
	serv_addr.sin_family		= AF_INET;
	serv_addr.sin_addr.s_addr	= htonl(INADDR_ANY);
	serv_addr.sin_port		= htons(netport);
        setsockopt(sockfd,SOL_SOCKET, SO_REUSEADDR,&reuse_addr,sizeof(reuse_addr));
	if (bind(sockfd,(struct sockaddr *) &serv_addr,sizeof(serv_addr)) < 0) {
		log_printf(LOG_ERROR,"mod_netctl_net_setup(): cannot bind local address.\n");
		return (-3);
	} else 	log_printf(LOG_DEBUG,"mod_netctl_net_setup(): listening on port %d.\n",netport);

	listen(sockfd,5);

	return sockfd;

}

/*************************************************************************
 * MODULE INIT FUNCTION
 */
char *mod_netctl_init (void)
{
	int i;

	// find out if user wants module to be enabled
	mod_netctl_enable = config_getnum("mod_netctl_enable", 0);

	if (!mod_netctl_enable) {
		log_printf(LOG_DEBUG, "mod_netctl_init(): module disabled at user's request.\n");
		return NULL;
	}

	FD_ZERO(&mod_netctl_fdset);
	memset(cli,0,sizeof(cli));
	for(i=0;i<MOD_NETCTL_MAXCLIENTS;i++) {
		cli_sock[i]=0;
		cli_filter[i].enabled=0;
	}

	// register our module
	sockfd=mod_netctl_net_setup();
	if (0 <= sockfd) {
		log_printf(LOG_DEBUG, "mod_netctl_init(): initializing on fd %d.\n",sockfd);
		FD_SET(sockfd,&mod_netctl_fdset);
		mod_register(&mod_netctl);
	}

	return NULL;
}


/*************************************************************************
 * MODULE DEINIT FUNCTION
 */
void mod_netctl_deinit (void)
{
	log_printf(LOG_DEBUG, "mod_netctl_deinit(): deinitialized\n");
}


/*************************************************************************
 * REGEX MATCHING FUNCTION
 *
 * returns positive if match found
 */
int mod_netctl_regex_match(regex_t *preg, char *string)
{
	int 		result=0,eflags=0;
	regmatch_t	pmatch[1];
	size_t		nmatch=0;

	result=regexec(preg,string,nmatch,pmatch,eflags);
	if(!result) return 1;
	else return 0;
}

/*************************************************************************
 * EOF
 */
