/*************************************************************************
 * $Id: mod_mpg123.c,v 1.18 2001/05/10 02:26:32 dpotter Exp $
 *
 * mod_mpg123.c -- IRMP3 module for MPG123 player
 *
 * Copyright (C) by Andreas Neuhaus <andy@fasta.fh-dortmund.de>
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <signal.h>
#include <string.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/wait.h>

#include "tools.h"
#include "log.h"
#include "config.h"
#include "mod.h"
#include "mod_mpg123.h"

/*************************************************************************
 * GLOBALS
 */
int mod_mpg123_pid = 0;		// PID of mpg123
int mod_mpg123_fdsend = 0;	// fd to send commands to mpg123
int mod_mpg123_fdrecv = 0;	// fd to get responses from mpg123
int mod_mpg123_active = 0;	// if mpg123 is the active player
char artist[30],title[30],album[30],year[4],genre[30],comment[30];
fd_set mod_mpg123_fdset;

#if MOD_NETCTL == 1
extern int cli_sock[MOD_NETCTL_MAXCLIENTS];
#endif

/*************************************************************************
 * MODULE INFO
 */
mod_t mod_mpg123 = {
	mod_mpg123_deinit,	// deinit
	NULL,				// reload
	&mod_mpg123_fdset,	// watchfd_set
	NULL,				// poll
	NULL,				// update
	mod_mpg123_message,	// message
	mod_mpg123_chld,	// SIGCHLD handler
};

/*************************************************************************
 * CATCH SIGNAL OF EXITING PLAYER
 */
void mod_mpg123_chld (pid_t pid, int status)
{
	if (pid == mod_mpg123_pid) {
		log_printf(LOG_DEBUG, "mod_mpg123_chld(): player %d  exited with code %d\n", pid,WEXITSTATUS(status));
		mod_mpg123_pid = 0;
	}
}


/*************************************************************************
 * STOP PLAYER PROCESS
 */
void mod_mpg123_stop (void)
{
	if (mod_mpg123_fdsend)
		close(mod_mpg123_fdsend);
	mod_mpg123_fdsend = 0;
	if (mod_mpg123_fdrecv)
		close(mod_mpg123_fdrecv);
	mod_mpg123_fdrecv = 0;
	if (mod_mpg123_pid) {
		log_printf(LOG_DEBUG, "mod_mpg123_stop(): killed player pid %d\n", mod_mpg123_pid);
		kill(mod_mpg123_pid, SIGTERM);
	}
	FD_ZERO(&mod_mpg123_fdset);
	mod_mpg123.poll = NULL;
	bzero(album,sizeof(album));bzero(title,sizeof(title));
	bzero(comment,sizeof(comment));bzero(artist,sizeof(artist));
	bzero(year,sizeof(year));bzero(genre,sizeof(genre));
}


/*************************************************************************
 * START PLAYER PROCESS
 */
int mod_mpg123_start (void)
{
	int fd_send[2], fd_recv[2];
	char *params,*tmpparams;
	char *args[50];
	int i,j=0;

	// check if player is already running
	if (mod_mpg123_pid)
		return 0;

	// make socketpair to communicate with player
	if (socketpair(AF_UNIX, SOCK_STREAM, 0, fd_send) < 0)
		return -1;
	if (socketpair(AF_UNIX, SOCK_STREAM, 0, fd_recv) < 0)
		return -1;

	// handle signals
	signal(SIGPIPE, SIG_IGN);

	// get additional parameters
	tmpparams = config_getstr("mpg123_param", "");
	params = (char *)malloc(strlen(tmpparams)+1);
	strcpy(params,tmpparams);
	args[0] = (char *)malloc(strlen(MOD_MPG123_BIN)+1);
	strcpy(args[0], MOD_MPG123_BIN);
	args[1] = (char *)malloc(3);
	strcpy(args[1], "-R");
	i = 2;
	args[i] = strtok(params, " ");
	while (args[i])
		args[++i] = strtok(NULL, " ");
	args[i] = "-";
	args[++i] = NULL;

	// fork child process
	mod_mpg123_pid = fork();
	if (mod_mpg123_pid == 0) {
		// pipe in/output through socket to parent
		dup2(fd_send[0], STDIN_FILENO);
		close(fd_send[0]);
		close(fd_send[1]);
		dup2(fd_recv[0], STDOUT_FILENO);
		dup2(fd_recv[0], STDERR_FILENO);
		close(fd_recv[0]);
		close(fd_recv[1]);

#if MOD_NETCTL == 1
	for (j=0;j<=MOD_NETCTL_MAXCLIENTS;j++)
		if(cli_sock[j] != 0 ) 
			close(cli_sock[j]);
#endif
		// spawn player
		execvp(args[0], args);
		// never reached if exec was ok
		exit(-1);
	}

	// parent continues here

	free(params);
	free(args[0]);
	free(args[1]);

	close(fd_send[0]);
	mod_mpg123_fdsend = fd_send[1];
	close(fd_recv[0]);
	mod_mpg123_fdrecv = fd_recv[1];

	// check if player is running
	sleep(1);
	if (!mod_mpg123_pid) {
		log_printf(LOG_DEBUG, "mod_mpg123_start(): player process didn't start!\n");
		return -1;
	}

	FD_SET(mod_mpg123_fdrecv,&mod_mpg123_fdset);
	mod_mpg123.poll = mod_mpg123_poll;
	log_printf(LOG_DEBUG, "mod_mpg123_start(): player spawned pid %d, piped on fd %d and %d\n", mod_mpg123_pid, mod_mpg123_fdsend, mod_mpg123_fdrecv);
	return 0;
}


/*************************************************************************
 * POLL INPUT DATA
 */
void mod_mpg123_poll (int fd)
{
	static int lastframepos, lastframeremain;
	static int lastsecpos, lastsecremain;
	char s[512], *c1, *c2, *c3, *c4, *c5, buf[32];
	int rc, secpos, secremain;

	// read the response from the player
	rc = readline(fd, s, sizeof(s));
	if (rc < 0) {
		// player closed connection
		log_printf(LOG_DEBUG,"mod_mpg123_poll(): player closed connection with %s.\n",strerror(rc));
		mod_mpg123_stop();
		mod_sendmsg(MSGTYPE_PLAYER, "stop");
		return;
	}

	// process player response
	if (!s || !*s)
		return;
	c1 = s ? strtok(s, " \t") : NULL;

	if (c1 && !strcasecmp(c1, "@R")) {
		// player version received
		c2 = c1 ? strtok(NULL, "") : NULL;
		mod_sendmsgf(MSGTYPE_PLAYER, "version %s", c2);

	} else if (c1 && !strcasecmp(c1, "@P")) {
		// playing status received
		c2 = c1 ? strtok(NULL, " \t") : NULL;
		c3 = c2 ? strtok(NULL, " \t") : NULL;
		if (atoi(c2) == 0) {
			// There are various ways how the end of the song
			// is notified :-( It should be "@P 0 EOF" in future
			if ((c3 && !strcasecmp(c3, "EOF")) || (lastframepos > 10 && lastframeremain < 10)) {
				log_printf(LOG_NORMAL, "End of song reached\n");
				mod_sendmsg(MSGTYPE_PLAYER, "endofsong");
			} else {
				log_printf(LOG_DEBUG,"mod_mpg123_poll(): received unknown mpg123 status %s %s %s.\n",c1,c2,c3);
 				if(!strcasecmp("yes",config_getstr("abort_on_unknown_mpg123_status","no"))) {
					log_printf(LOG_NORMAL, "Playing stopped\n");
					mod_sendmsg(MSGTYPE_PLAYER, "stop");
				}
			}
		} else if (atoi(c2) == 1) {
			log_printf(LOG_NORMAL, "Playing paused\n");
			mod_sendmsg(MSGTYPE_PLAYER, "pause");
		} else if (atoi(c2) == 2) {
			log_printf(LOG_NORMAL, "Playing continued\n");
			mod_sendmsg(MSGTYPE_PLAYER, "play");
		} else if (atoi(c2) == 3) {
			log_printf(LOG_NORMAL, "End of song reached\n");
			mod_sendmsg(MSGTYPE_PLAYER, "endofsong");
		}

	} else if (c1 && !strcasecmp(c1, "@I")) {
		// song-info received (artist, title, etc)
		mod_sendmsg(MSGTYPE_PLAYER, "play");
		c2 = c1 ? strtok(NULL, "") : NULL;
		if (!strncasecmp(c2, "ID3:", 4)) {
			// ID3 tag
			c2 += 4;
			strncpy(buf, c2, 30); buf[30]=0; trim(buf); mod_sendmsgf(MSGTYPE_PLAYER, "info title %s", buf); c2 += 30;
			strcpy(title,buf);
			strncpy(buf, c2, 30); buf[30]=0; trim(buf); mod_sendmsgf(MSGTYPE_PLAYER, "info artist %s", buf); c2 += 30;
			strcpy(artist,buf);
			strncpy(buf, c2, 30); buf[30]=0; trim(buf); mod_sendmsgf(MSGTYPE_PLAYER, "info album %s", buf); c2 += 30;
			strcpy(album,buf);
			strncpy(buf, c2, 4); buf[4]=0; trim(buf); mod_sendmsgf(MSGTYPE_PLAYER, "info year %s", buf); c2 += 4;
			strcpy(year,buf);
			strncpy(buf, c2, 30); buf[30]=0; trim(buf); mod_sendmsgf(MSGTYPE_PLAYER, "info comment %s", buf); c2 += 30;
			strcpy(comment,buf);
			strncpy(buf, c2, 30); buf[30]=0; trim(buf); mod_sendmsgf(MSGTYPE_PLAYER, "info genre %s", buf); c2 += 30;
			strcpy(genre,buf);
		} else {
			// try to get info from filename
			// try format "(artist) title"
			c3 = strchr(c2, '(');
			c4 = strchr(c2, ')');
			if (c3 && c4 && c3<c4) {
				*c4++ = 0;
				while (*c4==' ') c4++;
				mod_sendmsgf(MSGTYPE_PLAYER, "info title %s", c4);
				strncpy(title,c4,sizeof(title));
				mod_sendmsgf(MSGTYPE_PLAYER, "info artist %s", c3+1);
				strncpy(artist,c3+1,sizeof(artist));
			} else {
				// try format "artist - title"
				c3 = strchr(c2, '-');
				if (c3) {
					*c3++ = 0;
					while (*c3==' ') c3++;
					mod_sendmsgf(MSGTYPE_PLAYER, "info title %s", c2);
					strncpy(title,c2,sizeof(title));
					mod_sendmsgf(MSGTYPE_PLAYER, "info artist %s", c3);
					strncpy(artist,c3,sizeof(title));
				} else {
					// unable to determine :-(
					mod_sendmsgf(MSGTYPE_PLAYER, "info title %s", c2);
					strncpy(title,c2,sizeof(title));
					mod_sendmsgf(MSGTYPE_PLAYER, "info artist (unknown artist)");
				}
			}
		}
	} else if (c1 && !strcasecmp(c1, "@S")) {
		// stream-info received (mpeg type etc)
		// ignored, because not interesting for us, yet

	} else if (c1 && !strcasecmp(c1, "@F")) {
		// frame-info received (current times)
		c2 = c1 ? strtok(NULL, " \t") : NULL;
		c3 = c2 ? strtok(NULL, " \t") : NULL;
		c4 = c3 ? strtok(NULL, " \t") : NULL;
		c5 = c4 ? strtok(NULL, " \t") : NULL;
		lastframepos = atoi(c2);
		lastframeremain = atoi(c3);
		secpos = (int)atof(c4);
		secremain = (int)atof(c5);

		// only send out message if something has changed
		// this should help keeping message traffic low
		if (secpos != lastsecpos || secremain != lastsecremain) {
			mod_sendmsgf(MSGTYPE_PLAYER, "time %d %d", secpos, secremain);
			lastsecpos = secpos;
			lastsecremain = secremain;
		}
	} else {
		// unknown message, probably an error
		c2 = c1 ? strtok(NULL, "") : NULL;
		log_printf(LOG_VERBOSE, "mpg123 reports: '%s %s'\n", c1, c2);
		if(!strcasecmp("yes",config_getstr("abort_on_unknown_mpg123_status","no"))) {
			log_printf(LOG_NORMAL, "Playing stopped\n");
			mod_sendmsg(MSGTYPE_PLAYER, "stop");
		}
	}
}


/*************************************************************************
 * RECEIVE MESSAGE
 */
void mod_mpg123_message (int msgtype, char *msg)
{
	char *c1, *c2;

	// handle input messages
	if (msgtype == MSGTYPE_INPUT) {
		c1 = msg ? strtok(msg, " \t") : NULL;
		c2 = c1 ? strtok(NULL, "") : NULL;

		// see if we can play the requested song
		if (c1 && !strcasecmp(c1, "play")) {
			mod_mpg123_active = 0;
			if (c2 && strlen(c2)>4 && (!strcasecmp(c2+strlen(c2)-4, ".mpg") || !strcasecmp(c2+strlen(c2)-4, ".mp2") || !strcasecmp(c2+strlen(c2)-4, ".mp3") || !strncasecmp(c2, "http://", 7)))
				mod_mpg123_active = 1;
			else
				log_printf(LOG_DEBUG, "mod_mpg123_message(): ignoring unknown song '%s'\n", c2);
		}

		// make sure we are stopped if we can't handle the current song
		if (!mod_mpg123_active) {
			if (mod_mpg123_pid)
				mod_mpg123_stop();
			return;
		}

		// handle player commands
		if (c1 && !strcasecmp(c1, "play")) {
			if (!mod_mpg123_pid)
				if (mod_mpg123_start())
					log_printf(LOG_ERROR, "Failed starting mpg123!\n");
			log_printf(LOG_NORMAL, "Playing '%s'\n", c2);
			sendtext(mod_mpg123_fdsend, "LOAD %s\n", c2);

		} else if (c1 && !strcasecmp(c1, "stop") && mod_mpg123_pid) {
			sendtext(mod_mpg123_fdsend, "STOP\n");
			mod_mpg123_stop();
			log_printf(LOG_NORMAL, "Playing stopped\n");
			mod_sendmsg(MSGTYPE_PLAYER, "stop");

		} else if (c1 && !strcasecmp(c1, "pause") && mod_mpg123_pid) {
			sendtext(mod_mpg123_fdsend, "PAUSE\n");

		} else if (c1 && !strcasecmp(c1, "seek") && mod_mpg123_pid) {
			log_printf(LOG_DEBUG, "mod_mpg123_message(): seeking %s\n", c2);
			if (*c2=='-' || *c2=='+')
				sendtext(mod_mpg123_fdsend, "JUMP %c%d\n", *c2, atoi(c2+1)*38);
			else
				sendtext(mod_mpg123_fdsend, "JUMP %d\n", atoi(c2)*38);
		} else  if (c1 && !strcasecmp(c1, "query") && mod_mpg123_pid) {
				if(c2 && !strcasecmp(c2,"artist")) mod_sendmsgf(MSGTYPE_PLAYER,"info artist %s",artist);
				if(c2 && !strcasecmp(c2,"album")) mod_sendmsgf(MSGTYPE_PLAYER,"info album %s",album);
				if(c2 && !strcasecmp(c2,"title")) mod_sendmsgf(MSGTYPE_PLAYER,"info title %s",title);
				if(c2 && !strcasecmp(c2,"year")) mod_sendmsgf(MSGTYPE_PLAYER,"info year %s",year);
				if(c2 && !strcasecmp(c2,"genre")) mod_sendmsgf(MSGTYPE_PLAYER,"info genre %s",genre);
				if(c2 && !strcasecmp(c2,"comment")) mod_sendmsgf(MSGTYPE_PLAYER,"info comment %s",comment);
			}
	}
}


/*************************************************************************
 * MODULE INIT FUNCTION
 */
char *mod_mpg123_init (void)
{
	log_printf(LOG_DEBUG, "mod_mpg123_init(): initializing\n");
	FD_ZERO(&mod_mpg123_fdset);

	// register our module
	mod_register(&mod_mpg123);

	return NULL;
}


/*************************************************************************
 * MODULE DEINIT FUNCTION
 */
void mod_mpg123_deinit (void)
{
	mod_mpg123_stop();
	log_printf(LOG_DEBUG, "mod_mpg123_deinit(): deinitialized\n");
}


/*************************************************************************
 * EOF
 */
