/*************************************************************************
 * $Id: mod_lirc.c,v 1.10 2000/06/20 00:59:41 dpotter Exp $
 *
 * mod_lirc.c -- LIRC input module
 *
 * Copyright (C) by Andreas Neuhaus <andy@fasta.fh-dortmund.de>
 *
 */

#include <stdio.h>
#include <stdlib.h>

#ifdef MOD_LIRC_V06
#include <lirc/lirc_client.h>
#else
#include <lirc_client.h>
#endif

#include "log.h"
#include "config.h"
#include "mod.h"
#include "mod_lirc.h"


/*************************************************************************
 * GLOBALS
 */
fd_set			 mod_lirc_fdset;
struct lirc_config 	*mod_lirc_irconfig;


/*************************************************************************
 * MODULE INFO
 */
mod_t mod_lirc = {
	mod_lirc_deinit,	// deinit
	mod_lirc_reload,	// reload
	&mod_lirc_fdset,	// watch_fdset
	mod_lirc_poll,		// poll
	NULL,			// update
	NULL,			// message
	NULL,			// SIGCHLD handler
};


/*************************************************************************
 * POLL INPUT DATA
 */
void mod_lirc_poll (int fd)
{
#ifdef MOD_LIRC_V06
	char *code, *command;
	int rc;

	if (!lirc_nextcode(&code)) {
		while ((rc = lirc_code2char(mod_lirc_irconfig, code, &command)) == 0 && command) {
			log_printf(LOG_NOISYDEBUG, "mod_lirc_poll(): got command '%s'\n", command);
			mod_sendmsg(MSGTYPE_INPUT, command);
		}
		free(code);
#else
	char *ir, *command;

	if ((ir = lirc_nextir()) != NULL) {
		while ((command = lirc_ir2char(mod_lirc_irconfig, ir)) != NULL) {
			log_printf(LOG_NOISYDEBUG, "mod_lirc_poll(): got command '%s'\n", command);
			mod_sendmsg(MSGTYPE_INPUT, command);
		}
		free(ir);
		
#endif
	} else {	// lirc has died
			log_printf(LOG_NORMAL, "mod_lirc_poll(): lirc read returned error .\n");
			FD_ZERO(&mod_lirc_fdset);
	}
}


/*************************************************************************
 * MODULE INIT FUNCTION
 */
char *mod_lirc_init (void)
{
	int		mod_lirc_fd=0;

	FD_ZERO(&mod_lirc_fdset);

	// connect to LIRC
#ifdef MOD_LIRC_V06
	if ((mod_lirc_fd = lirc_init("irmp3", 0)) < 0)
#else
	if ((mod_lirc_fd = lirc_init("irmp3")) < 0)
#endif
		return "Unable to connect to LIRC";

	log_printf(LOG_DEBUG, "mod_lirc_init(): lirc connection on fd %d\n", mod_lirc_fd);

	// read LIRC config
	if (lirc_readconfig(config_getstr("lirc_config", NULL), &mod_lirc_irconfig, NULL)) {
		lirc_deinit();
		return "Unable to read LIRC config";
	}

	// register our module
	FD_SET(mod_lirc_fd,&mod_lirc_fdset);
	mod_register(&mod_lirc);

	return NULL;
}


/*************************************************************************
 * MODULE DEINIT FUNCTION
 */
void mod_lirc_deinit (void)
{
	// free LIRC config
	lirc_freeconfig(mod_lirc_irconfig);

	// disconnect from LIRC
	lirc_deinit();

	log_printf(LOG_DEBUG, "mod_lirc_deinit(): closed lirc connection\n");
	FD_ZERO(&mod_lirc_fdset);
}


/*************************************************************************
 * MODULE RELOAD FUNCTION
 */
char *mod_lirc_reload (void)
{
	log_printf(LOG_DEBUG, "mod_lirc_reload(): reloading lirc config\n");

	// free LIRC config
	lirc_freeconfig(mod_lirc_irconfig);

	// read LIRC config
	lirc_readconfig(config_getstr("lirc_config", NULL), &mod_lirc_irconfig, NULL);

	return NULL;
}


/*************************************************************************
 * EOF
 */
