#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <signal.h>
#include <string.h>
#include <fcntl.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/ioctl.h>
#include <sys/wait.h>
#include <sys/mount.h>
#include <linux/cdrom.h>

#include "tools.h"
#include "log.h"
#include "config.h"
#include "mod.h"
#include "mod_cd.h"


#define MOD_CD_NONE 0
#define MOD_CD_AUDIO   1
#define MOD_CD_DATA  2



/*************************************************************************
 * GLOBALS
 */

char *mod_cd_device;
char *mod_cd_mountpoint;
char *mod_cd_scandir_pattern;
int mod_cd_fd = -1;
int mod_cd_cdmode = MOD_CD_NONE;	// NONE, AUDIO, DATA
int mod_cd_externmount = 1;
int mod_cd_chld_pid;
int mod_cd_chld_exited;
int mod_cd_chld_status;

typedef struct trkindex {
    int start;
    int length;
} trkindex_t;    

trkindex_t mod_cd_trkindex[100];

int mod_cd_current_track = 0;
 

/*************************************************************************
 * MODULE INFO
 */
mod_t mod_cd = {
	mod_cd_deinit,		// deinit
	NULL,			// reload
	0,			// watchfd
	NULL,			// poll
	NULL,			// update
	mod_cd_message,		// message
	NULL,			// SIGCHLD handler
};


/*************************************************************************/
int mod_cd_play(int trk)
{
	struct cdrom_ti ti;
        struct cdrom_msf msf;

        int start = mod_cd_trkindex[trk].start;
	int stop = start + mod_cd_trkindex[trk].length;
	
	mod_cd_current_track = trk;
	
        /* try CDROMPLAYMSF */
	msf.cdmsf_min0 = start / (75 * 60);
	msf.cdmsf_sec0 = (start % (75 * 60)) / 75;
	msf.cdmsf_frame0 = start % 75;
	
	msf.cdmsf_min1 = stop / (75 * 60);
	msf.cdmsf_sec1 = (stop % (75 * 60)) / 75;
	msf.cdmsf_frame1 = stop % 75;
	
	if (ioctl(mod_cd_fd, CDROMPLAYMSF, &msf) == -1) {
		/* CDROMPLAYMSF failed, try CDROMPLAYTRKIND*/
		ti.cdti_trk0 = trk;
		ti.cdti_trk1 = trk;

		ti.cdti_ind0 = 1;
		ti.cdti_ind1 = 99;

		if (ioctl(mod_cd_fd, CDROMPLAYTRKIND, &ti) == -1) {
			log_printf(LOG_ERROR,
				"mod_cd_play(): failed both ioctls,"
				" CDROMPLAYMSF and CDROMPLAYTRKIND \n");
			return 1;
		}
	}	
	log_printf(LOG_DEBUG, "mod_cd_play(): playing track %d\n", trk);

	mod_cd.update = mod_cd_update;
	mod_sendmsg(MSGTYPE_PLAYER, "play");
	mod_sendmsgf(MSGTYPE_PLAYER, "info artist Audio-CD / Track: %02d", trk);
	return 0;
}

int mod_cd_stop()
{
	struct cdrom_subchnl subchnl;
	mod_cd.update = NULL;

	if (mod_cd_cdmode != MOD_CD_NONE) {
	       subchnl.cdsc_format = CDROM_MSF;

	       if (ioctl(mod_cd_fd, CDROMSUBCHNL, &subchnl) == -1) {
		       log_printf(LOG_ERROR,
				  "mod_cd_stop(): ioctl cdromsubchnl\n");
		       return 1;
	       }

	       if (subchnl.cdsc_audiostatus != CDROM_AUDIO_PLAY) {
   		        log_printf(LOG_DEBUG, "mod_cd_stop(): not playing\n");
			return 0;
	       }

	       if (ioctl(mod_cd_fd, CDROMSTOP) == -1) {
		       log_printf(LOG_ERROR, "mod_cd_stop(): ioctl cdromstop\n");
		       return 1;
	       }
	       mod_sendmsg(MSGTYPE_PLAYER, "stop");
	}
	return 0;
}

int mod_cd_pause()
{
	struct cdrom_subchnl subchnl;

	if (mod_cd_cdmode != MOD_CD_NONE) {
	       subchnl.cdsc_format = CDROM_MSF;

	       if (ioctl(mod_cd_fd, CDROMSUBCHNL, &subchnl) == -1) {
  		       log_printf(LOG_ERROR,
				  "mod_cd_pause(): ioctl cdromsubchnl\n");
		       return 1;
	       }

	       switch (subchnl.cdsc_audiostatus) {
	       case CDROM_AUDIO_PAUSED:
		       if (ioctl(mod_cd_fd, CDROMRESUME) == -1) {
			       log_printf(LOG_ERROR,
					  "mod_cd_pause(): ioctl cdromresume\n");
			       return 1;
		       }
		       mod_sendmsg(MSGTYPE_PLAYER, "play");
		       return 0;
	       case CDROM_AUDIO_PLAY:
		        if (ioctl(mod_cd_fd, CDROMPAUSE) == -1) {
			        log_printf(LOG_ERROR,
					   "mod_cd_pause(): ioctl cdrompause\n");
				return 1;
			}
			mod_sendmsg(MSGTYPE_PLAYER, "pause");
			return 0;
	       default:		// ignore
	       }
	}
	return 0;
}

int mod_cd_seek_trkind(int seek, int abs)
{
	struct cdrom_ti ti;
	struct cdrom_subchnl subchnl;

	subchnl.cdsc_format = CDROM_MSF;

	if (ioctl(mod_cd_fd, CDROMSUBCHNL, &subchnl) == -1) {
		log_printf(LOG_ERROR,
			   "mod_cd_seek(): ioctl cdromsubchnl\n");
		return 1;
	}

	if (subchnl.cdsc_audiostatus != CDROM_AUDIO_PLAY) {
		log_printf(LOG_DEBUG, "mod_cd_seek(): not playing\n");
		return 0;
	}

	ti.cdti_trk0 = subchnl.cdsc_trk;
	ti.cdti_trk1 = subchnl.cdsc_trk;

	if (abs)
		ti.cdti_ind0 = seek;
	else
		ti.cdti_ind0 = subchnl.cdsc_ind + seek;

	if (ti.cdti_ind0 < 1)
		ti.cdti_ind0 = 1;
	else if (ti.cdti_ind0 > 99)
		ti.cdti_ind0 = 99;

	ti.cdti_ind1 = 99;

	log_printf(LOG_DEBUG, "mod_cd_seek(): seek to %d\n", ti.cdti_ind0);

	if (ioctl(mod_cd_fd, CDROMPLAYTRKIND, &ti) == -1) {
		log_printf(LOG_ERROR,
			   "mod_cd_seek(): ioctl cdromplaytrkind\n");
		return 1;
	}
	return 0;
}

int mod_cd_seek_msf(int seek, int abs)
{
	struct cdrom_subchnl subchnl;
        struct cdrom_msf msf;

        int start, stop, pos;
	subchnl.cdsc_format = CDROM_MSF;

	if (ioctl(mod_cd_fd, CDROMSUBCHNL, &subchnl) == -1) {
		log_printf(LOG_ERROR,
			   "mod_cd_seek(): ioctl cdromsubchnl\n");
		return 1;
	}

	if (subchnl.cdsc_audiostatus != CDROM_AUDIO_PLAY) {
		log_printf(LOG_DEBUG, "mod_cd_seek(): not playing\n");
		return 0;
	}

        start = mod_cd_trkindex[subchnl.cdsc_trk].start;
	stop = start + mod_cd_trkindex[subchnl.cdsc_trk].length;
	
	if (abs)
		pos = start + seek * 75;
	else 
		pos = subchnl.cdsc_absaddr.msf.minute * 75 * 60 +
		      subchnl.cdsc_absaddr.msf.second * 75 +
		      subchnl.cdsc_absaddr.msf.frame +
		      seek * 75;
		      
        if (pos < start) 
		pos = start;

        if (pos > stop) 
		pos = stop - 75; /* 1 second before the end of track */

	msf.cdmsf_min0 = pos / (75 * 60);
	msf.cdmsf_sec0 = (pos % (75 * 60)) / 75;
	msf.cdmsf_frame0 = pos % 75;
	
	msf.cdmsf_min1 = stop / (75 * 60);
	msf.cdmsf_sec1 = (stop % (75 * 60)) / 75;
	msf.cdmsf_frame1 = stop % 75;
	
	if (ioctl(mod_cd_fd, CDROMPLAYMSF, &msf) == -1) {

		log_printf(LOG_ERROR,
			   "mod_cd_seek(): ioctl cdromplaymsf\n");
		return 1;
	}
	return 0;
}

int mod_cd_seek(int seek, int abs)
{
	log_printf(LOG_DEBUG, "mod_cd_seek(): seek: %d  abs: %d\n", seek,
		   abs);
	if (mod_cd_seek_msf(seek, abs))
		if (mod_cd_seek_trkind(seek, abs)) {
			log_printf(LOG_ERROR,                                           
	                  "mod_cd_seek(): both cdromplaytrkind"
			  " and cdromplaymsf failed\n");
		return 1;
		}
	return 0;
}

int mod_cd_toc()
{
	struct cdrom_tochdr tochdr;
	struct cdrom_tocentry te;

	int i;
	int type = MOD_CD_NONE;
	int time = 0, last_time = 0;

	if (ioctl(mod_cd_fd, CDROMREADTOCHDR, &tochdr) == -1) {
		return MOD_CD_NONE;
	}


	for (i = tochdr.cdth_trk0; i < tochdr.cdth_trk1 + 1; i++) {
		te.cdte_track = i;
		te.cdte_format = CDROM_MSF;
		if (ioctl(mod_cd_fd, CDROMREADTOCENTRY, &te) == -1) {
			return MOD_CD_NONE;
		}

		time =
			te.cdte_addr.msf.minute * 60 * 75 + 
			te.cdte_addr.msf.second * 75 +
			te.cdte_addr.msf.frame;
			
		if (i > 0) {
			mod_cd_trkindex[i - 1].start  = last_time;
			mod_cd_trkindex[i - 1].length = time - last_time;
		}
		last_time = time;

		if (type == MOD_CD_NONE) {
			if (te.cdte_ctrl == CDROM_DATA_TRACK)
				type = MOD_CD_DATA;
			else
				type = MOD_CD_AUDIO;
		}
	}

	te.cdte_track = CDROM_LEADOUT;
	te.cdte_format = CDROM_MSF;


	if (ioctl(mod_cd_fd, CDROMREADTOCENTRY, &te) == -1) {
		return type;
	}

	time =
		te.cdte_addr.msf.minute * 60 * 75 + 
		te.cdte_addr.msf.second * 75 +
		te.cdte_addr.msf.frame;

	mod_cd_trkindex[i - 1].start  = last_time;
	mod_cd_trkindex[i - 1].length = time - last_time;

	return type;
}


void mod_cd_sigchld(pid_t childpid, int status)
{
	if (mod_cd_chld_pid == childpid) {
		mod_cd_chld_status = status;
		mod_cd_chld_exited = 1;
	}
}


int mod_cd_system(char *command)
{
	int pid;

	if (command == 0)
		return 1;
	pid = fork();
	if (pid == -1)
		return -1;
	if (pid == 0) {
		char *argv[4];
		argv[0] = "sh";
		argv[1] = "-c";
		argv[2] = command;
		argv[3] = 0;
		execv("/bin/sh", argv);
		exit(127);
	}
	mod_cd_chld_pid = pid;
	mod_cd_chld_exited = 0;
	mod_cd.chld = mod_cd_sigchld;
	do {
		usleep(50000);
	} while (!mod_cd_chld_exited);
	mod_cd.chld = NULL;
	return mod_cd_chld_status;

}


int mod_cd_eject(int force)
{
	if (mod_cd_cdmode == MOD_CD_DATA) {
		log_printf(LOG_DEBUG,
			   "mod_cd_eject(): eject force=%d MP3\n", force);
		usleep(200000);
		if (mod_cd_externmount) {
			char tmp[1000];
			snprintf(tmp, 1000,
				 "umount %s >/dev/null 2>/dev/null",
				 mod_cd_mountpoint);
			if (mod_cd_system(tmp) != 0) {
				log_printf(LOG_DEBUG,
					   "mod_cd_eject(): umount failed\n");
				if (force) {
					mod_sendmsg(MSGTYPE_INPUT, "stop");	/* try to stop player before umount */
					mod_sendmsg(MSGTYPE_INPUT,
						    "cd eject2");
				}
				return 1;
			}
		} else {
			if (umount(mod_cd_mountpoint) != 0) {
				log_printf(LOG_DEBUG,
					   "mod_cd_eject(): umount failed\n");

				if (force) {
					mod_sendmsg(MSGTYPE_INPUT, "stop");
					mod_sendmsg(MSGTYPE_INPUT,
						    "cd eject2");
				}
				return 1;
			}
		}
	} else if (mod_cd_cdmode == MOD_CD_AUDIO) {
		log_printf(LOG_DEBUG,
			   "mod_cd_eject(): eject force=%d CD\n", force);
		mod_cd.update = NULL;
		mod_cd_stop();
		if (ioctl(mod_cd_fd, CDROMSTOP) == -1) {
			log_printf(LOG_ERROR,
				   "mod_cd_stop(): ioctl cdromstop\n");
		}
	} else {
		log_printf(LOG_DEBUG,
			   "mod_cd_eject(): eject force=%d NONE\n", force);
	}
	if (mod_cd_fd >= 0)
		close(mod_cd_fd);
	if ((mod_cd_fd = open(mod_cd_device, O_RDONLY | O_NONBLOCK)) < 0) {
		log_printf(LOG_DEBUG, "mod_cd_eject(): open %s",
			   mod_cd_device);
		return 1;
	}

	if (ioctl(mod_cd_fd, CDROMEJECT) == -1) {
		log_printf(LOG_DEBUG,
			   "mod_cd_eject(): ioctl cdromeject %s\n",
			   strerror(errno));
	}

	if (mod_cd_fd >= 0)
		close(mod_cd_fd);
	mod_cd_fd = -1;
	mod_cd_cdmode = MOD_CD_NONE;
	return 0;
}

int mod_cd_load()
{
	log_printf(LOG_DEBUG, "mod_cd_load():\n");
	if (mod_cd_fd >= 0)
		close(mod_cd_fd);
      	if ((mod_cd_fd = open(mod_cd_device,  O_RDONLY | O_NONBLOCK)) < 0) {
	        mod_cd_cdmode = MOD_CD_NONE;
		log_printf(LOG_DEBUG, "mod_cd_load(): no cd in %s, (%i:\"%s\")\n", mod_cd_device, errno, sys_errlist[errno]);
		return 1;
	}
	if (ioctl(mod_cd_fd, CDROMCLOSETRAY, NULL) == -1) {
		log_printf(LOG_ERROR,
			   "mod_cd_load(): ioctl cdromclosetray\n");
		return 1;
	}
	mod_cd_cdmode = mod_cd_toc();
	switch (mod_cd_cdmode) {
	    case MOD_CD_DATA: {
		log_printf(LOG_DEBUG, "mod_cd_load(): mp3, mounting\n");
		if (mod_cd_externmount) {
			char tmp[1000];
			snprintf(tmp, 1000,
				 "mount %s >/dev/null 2>/dev/null",
				 mod_cd_mountpoint);
			mod_cd_system(tmp);
		} else {
			mount(mod_cd_device, mod_cd_mountpoint, "iso9660",
			      MS_RDONLY | MS_MGC_VAL, NULL);
		}

		mod_sendmsgf(MSGTYPE_INPUT, "playlist loaddir %s",
			     mod_cd_scandir_pattern);
		break;
	    } 
	    case MOD_CD_AUDIO: {
		struct cdrom_tochdr tochdr;
		int i;
		log_printf(LOG_DEBUG, "mod_cd_load(): audio cd\n");

		if (ioctl(mod_cd_fd, CDROMREADTOCHDR, &tochdr) == -1) {
			log_printf(LOG_ERROR,
				   "mod_cd_load(): ioctl cdromreadtochdr\n");
			return 1;
		}

		mod_sendmsg(MSGTYPE_INPUT, "playlist clear");
		for (i = tochdr.cdth_trk0; i < tochdr.cdth_trk1 + 1; i++) {
			struct cdrom_tocentry te;

			te.cdte_track = i;
			te.cdte_format = CDROM_MSF;
			if (ioctl(mod_cd_fd, CDROMREADTOCENTRY, &te) == -1) {
				log_printf(LOG_ERROR,
					   "mod_cd_load(): ioctl cdromreadtocentry\n");
				return 1;
			}
			if (te.cdte_ctrl == CDROM_DATA_TRACK)
				continue;	/* skip data tracks */

			mod_sendmsgf(MSGTYPE_INPUT,
				     "playlist add AUDIOCD_TRACK# %d", i);
		}
		mod_sendmsg(MSGTYPE_INPUT, "playlist done");
		break;
	    }
	    default:	//no cd
	        log_printf(LOG_DEBUG, "mod_cd_load(): no cd\n"); 
        }
	return 0;
}


void mod_cd_update(void)
{
	struct cdrom_subchnl subchnl;
	int pos, remain;

	subchnl.cdsc_format = CDROM_MSF;

	if (ioctl(mod_cd_fd, CDROMSUBCHNL, &subchnl) == -1) {
		log_printf(LOG_ERROR,
			   "mod_cd_update(): ioctl cdromsubchnl\n");
		return;
	}
	pos =
	    subchnl.cdsc_reladdr.msf.minute * 60 +
	    subchnl.cdsc_reladdr.msf.second;
	remain = (mod_cd_trkindex[subchnl.cdsc_trk].length +74) / 75  - pos;
	mod_sendmsgf(MSGTYPE_PLAYER, "time %d %d", pos, remain);

	if (subchnl.cdsc_audiostatus == CDROM_AUDIO_COMPLETED ||
	    subchnl.cdsc_audiostatus == CDROM_AUDIO_NO_STATUS) {
		mod_sendmsg(MSGTYPE_PLAYER, "endofsong");
		mod_cd.update = NULL;
	}
}

/*************************************************************************
 * RECEIVE MESSAGE
 */
void mod_cd_message(int msgtype, char *msg)
{
	char *c1, *c2, *c3;
	// handle input messages
	if (msgtype == MSGTYPE_INPUT) {
		c1 = msg ? strtok(msg, " \t") : NULL;
		c2 = c1 ? strtok(NULL, " \t") : NULL;
		c3 = c2 ? strtok(NULL, "") : NULL;

		if (c1 && !strcasecmp(c1, "cd")) {
			if (c2 && !strcasecmp(c2, "load"))
				mod_cd_load();
			else if (c2 && !strcasecmp(c2, "eject"))
				mod_cd_eject(1);
			else if (c2 && !strcasecmp(c2, "eject2"))
				mod_cd_eject(0);
			else
				log_printf(LOG_DEBUG,
					   "mod_cd_message(): ignoring '%s'\n",
					   msg);
			return;
		}
		// see if we can play the requested song
		if (c1 && !strcasecmp(c1, "play")) {

			if (mod_cd_cdmode == MOD_CD_AUDIO && c2
			    && !strcasecmp(c2, "AUDIOCD_TRACK#") && c3) {
				int trk = atoi(c3);
				mod_cd_play(trk);
			} else {
				log_printf(LOG_DEBUG,
					   "mod_cd_message(): unable to play '%s'\n",
					   c2);
				mod_cd_stop();

			}
		} else if (c1 && !strcasecmp(c1, "seek")
			   && mod_cd_cdmode == MOD_CD_AUDIO) {
			if (*c2 == '-' || *c2 == '+')
				mod_cd_seek(atoi(c2), 0);	//relative
			else
				mod_cd_seek(atoi(c2), 1);

		} else if (c1 && !strcasecmp(c1, "stop")
			   && mod_cd_cdmode == MOD_CD_AUDIO) {
			mod_cd_stop();
		} else if (c1 && !strcasecmp(c1, "pause")
			   && mod_cd_cdmode == MOD_CD_AUDIO) {
			mod_cd_pause();
		}	       
	}
}


/*************************************************************************
 * MODULE INIT FUNCTION
 */
char *mod_cd_init(void)
{
	log_printf(LOG_DEBUG, "mod_cd_init(): initializing\n");

	mod_cd_device = config_getstr("cd_device", "/dev/cdrom");
	mod_cd_mountpoint = config_getstr("cd_mountpoint", "/mnt/cdrom");
	mod_cd_scandir_pattern =
	    config_getstr("cd_scandir_pattern", "/cdrom/*.[Mm][Pp]3");
	mod_cd_externmount =
	    (strcasecmp("yes", config_getstr("cd_externmount", "yes")) ==
	     0);


	// register our module
	mod_register(&mod_cd);

	return NULL;
}


/*************************************************************************
 * MODULE DEINIT FUNCTION
 */
void mod_cd_deinit(void)
{
	mod_cd_stop();
	log_printf(LOG_DEBUG, "mod_cd_deinit(): deinitialized\n");
}


/*************************************************************************
 * EOF
 */
