/*
 * Transparent proxy support for Linux/iptables
 *
 * Copyright (c) 2002 BalaBit IT Ltd.
 * Author: Balzs Scheidler
 *
 * This code is under the GPLv2.
 *
 */
#include <linux/module.h>
#include <linux/skbuff.h>
#include <linux/ip.h>
#include <net/checksum.h>

#include <linux/netfilter_ipv4/ip_tables.h>
#include <linux/netfilter_ipv4/ipt_TPROXY.h>

static unsigned int
target(struct sk_buff **pskb,
       const struct net_device *in,
       const struct net_device *out,
       unsigned int hooknum,
       const void *targinfo,
       void *userinfo)
{
	const struct ipt_tproxy_target_info *tproxyinfo = targinfo;
	struct ipt_tproxy_user_info *tproxy_ui = userinfo;

	if (userinfo) {
		tproxy_ui->changed = 1;
		tproxy_ui->lport = tproxyinfo->lport;
		tproxy_ui->laddr = tproxyinfo->laddr;
		return NF_ACCEPT;
	}
	else
		return NF_DROP;
}

static int
checkentry(const char *tablename,
	   const struct ipt_entry *e,
           void *targinfo,
           unsigned int targinfosize,
           unsigned int hook_mask)
{
	if (targinfosize != IPT_ALIGN(sizeof(struct ipt_tproxy_target_info))) {
		printk(KERN_WARNING "TPROXY: targinfosize %u != %Zu\n",
		       targinfosize,
		       IPT_ALIGN(sizeof(struct ipt_tproxy_target_info)));
		return 0;
	}

	if (strcmp(tablename, "tproxy") != 0) {
		printk(KERN_WARNING "TPROXY: can only be called from \"tproxy\" table, not \"%s\"\n", tablename);
		return 0;
	}

	return 1;
}

static struct ipt_target ipt_tproxy_reg = {
	.name		= "TPROXY",
	.target		= target,
	.checkentry	= checkentry,
	.me		= THIS_MODULE,
};

static int __init init(void)
{
	if (ipt_register_target(&ipt_tproxy_reg))
		return -EINVAL;

	return 0;
}

static void __exit fini(void)
{
	ipt_unregister_target(&ipt_tproxy_reg);
}

module_init(init);
module_exit(fini);
MODULE_LICENSE("GPL");
MODULE_AUTHOR("Balzs Scheidler <bazsi@balabit.hu>");
MODULE_DESCRIPTION("Netfilter transparent proxy TPROXY target module.");
