/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2006 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "icalculator.h"
#include "icalculatorkit.h"
#include "ierror.h"
#include "imath.h"
#include "istring.h"


#define ICALCULATOR_FUN_1V_V(_name_,_in_t_,_out_t_) \
	template<class T> class _name_##_ : public Function_NV_V<T,_in_t_,_out_t_,false> \
	{ \
	public: \
		_name_##_(int loc) : Function_NV_V<T,_in_t_,_out_t_,false>(1,loc,#_name_){} \
		virtual int Evaluate(Variable **args) \
		{ \
			int i, n = this->mResult->Dim(); \
            T *arg1 = static_cast<DataVariable<T>*>(args[0])->Data(); \
            T *res = static_cast<DataVariable<T>*>(this->mResult)->Data(); \
			for(i=0; i<n; i++) res[i] = T(_name_(arg1[i])); \
			return 0; \
		} \
	}

#define ICALCULATOR_TRAP_FUN_1V_V(_name_,_in_t_,_out_t_,_assert_) \
	template<class T> class trap_##_name_##_ : public Function_NV_V<T,_in_t_,_out_t_,false> \
	{ \
	public: \
		trap_##_name_##_(int loc) : Function_NV_V<T,_in_t_,_out_t_,false>(1,loc,#_name_){} \
		virtual int Evaluate(Variable **args) \
		{ \
			int i, n = this->mResult->Dim(); \
            T *arg1 = static_cast<DataVariable<T>*>(args[0])->Data(); \
            T *res = static_cast<DataVariable<T>*>(this->mResult)->Data(); \
			for(i=0; i<n; i++) \
			{ \
				if(_assert_) return 2; \
				res[i] = T(_name_(arg1[i])); \
			} \
			return 0; \
		} \
	}

#define ICALCULATOR_OPR_1V_V(_name_,_in_t_,_out_t_,_op_) \
	template<class T> class _name_##_ : public Function_NV_V<T,_in_t_,_out_t_,false> \
	{ \
	public: \
		_name_##_(int loc) : Function_NV_V<T,_in_t_,_out_t_,false>(1,loc,#_name_){} \
		virtual int Evaluate(Variable **args) \
		{ \
			int i, n = this->mResult->Dim(); \
            T *arg1 = static_cast<DataVariable<T>*>(args[0])->Data(); \
            T *res = static_cast<DataVariable<T>*>(this->mResult)->Data(); \
			for(i=0; i<n; i++) res[i] = _op_ arg1[i]; \
			return 0; \
		} \
	}

#define ICALCULATOR_TRAP_OPR_1V_V(_name_,_in_t_,_out_t_,_op_,_assert_) \
	template<class T> class trap_##_name_##_ : public Function_NV_V<T,_in_t_,_out_t_,false> \
	{ \
	public: \
		trap_##_name_##_(int loc) : Function_NV_V<T,_in_t_,_out_t_,false>(1,loc,#_name_){} \
		virtual int Evaluate(Variable **args) \
		{ \
			int i, n = this->mResult->Dim(); \
            T *arg1 = static_cast<DataVariable<T>*>(args[0])->Data(); \
            T *res = static_cast<DataVariable<T>*>(this->mResult)->Data(); \
			for(i=0; i<n; i++) \
			{ \
				if(_assert_) return 2; \
				res[i] = _op_ arg1[i]; \
			} \
			return 0; \
		} \
	}

#define ICALCULATOR_FUN_2V_V(_name_,_in_t_,_out_t_) \
	template<class T> class _name_##_ : public Function_NV_V<T,_in_t_,_out_t_,false> \
	{ \
	public: \
		_name_##_(int loc) : Function_NV_V<T,_in_t_,_out_t_,false>(2,loc,#_name_){} \
		virtual int Evaluate(Variable **args) \
		{ \
			int i, n = this->mResult->Dim(); \
            T *arg1 = static_cast<DataVariable<T>*>(args[0])->Data(); \
            T *arg2 = static_cast<DataVariable<T>*>(args[1])->Data(); \
            T *res = static_cast<DataVariable<T>*>(this->mResult)->Data(); \
			for(i=0; i<n; i++) res[i] = T(_name_(arg1[i],arg2[i])); \
			return 0; \
		} \
	}

#define ICALCULATOR_TRAP_FUN_2V_V(_name_,_in_t_,_out_t_,_assert_) \
	template<class T> class trap_##_name_##_ : public Function_NV_V<T,_in_t_,_out_t_,false> \
	{ \
	public: \
		trap_##_name_##_(int loc) : Function_NV_V<T,_in_t_,_out_t_,false>(2,loc,#_name_){} \
		virtual int Evaluate(Variable **args) \
		{ \
			int i, n = this->mResult->Dim(); \
            T *arg1 = static_cast<DataVariable<T>*>(args[0])->Data(); \
            T *arg2 = static_cast<DataVariable<T>*>(args[1])->Data(); \
            T *res = static_cast<DataVariable<T>*>(this->mResult)->Data(); \
			for(i=0; i<n; i++) \
			{ \
				if(_assert_) return 2; \
				res[i] = T(_name_(arg1[i],arg2[i])); \
			} \
			return 0; \
		} \
	}

#define ICALCULATOR_OPR_2V_V(_name_,_in_t_,_out_t_,_op_) \
	template<class T> class _name_##_ : public Function_NV_V<T,_in_t_,_out_t_,false> \
	{ \
	public: \
		_name_##_(int loc) : Function_NV_V<T,_in_t_,_out_t_,false>(2,loc,#_name_){} \
		virtual int Evaluate(Variable **args) \
		{ \
			int i, n = this->mResult->Dim(); \
            T *arg1 = static_cast<DataVariable<T>*>(args[0])->Data(); \
            T *arg2 = static_cast<DataVariable<T>*>(args[1])->Data(); \
            T *res = static_cast<DataVariable<T>*>(this->mResult)->Data(); \
			for(i=0; i<n; i++) res[i] = arg1[i] _op_ arg2[i]; \
			return 0; \
		} \
	}

#define ICALCULATOR_TRAP_OPR_2V_V(_name_,_in_t_,_out_t_,_op_,_assert_) \
	template<class T> class trap_##_name_##_ : public Function_NV_V<T,_in_t_,_out_t_,false> \
	{ \
	public: \
		trap_##_name_##_(int loc) : Function_NV_V<T,_in_t_,_out_t_,false>(2,loc,#_name_){} \
		virtual int Evaluate(Variable **args) \
		{ \
			int i, n = this->mResult->Dim(); \
            T *arg1 = static_cast<DataVariable<T>*>(args[0])->Data(); \
            T *arg2 = static_cast<DataVariable<T>*>(args[1])->Data(); \
            T *res = static_cast<DataVariable<T>*>(this->mResult)->Data(); \
			for(i=0; i<n; i++) \
			{ \
				if(_assert_) return 2; \
				res[i] = arg1[i] _op_ arg2[i]; \
			} \
			return 0; \
		} \
	}

#define ICALCULATOR_FUN_SV_V(_name_,_in_t_,_out_t_) \
	template<class T> class _name_##_ : public Function_SV_V<T,_in_t_,_out_t_> \
	{ \
	public: \
		_name_##_(int loc) : Function_SV_V<T,_in_t_,_out_t_>(loc,#_name_){} \
		virtual int Evaluate(Variable **args) \
		{ \
			int i, n = this->mResult->Dim(); \
            T *arg1 = static_cast<DataVariable<T>*>(args[0])->Data(); \
            T *arg2 = static_cast<DataVariable<T>*>(args[1])->Data(); \
            T *res = static_cast<DataVariable<T>*>(this->mResult)->Data(); \
			for(i=0; i<n; i++) res[i] = T(_name_(arg1[this->mScalar1?0:i],arg2[this->mScalar2?0:i])); \
			return 0; \
		} \
	}

#define ICALCULATOR_TRAP_FUN_SV_V(_name_,_in_t_,_out_t_,_assert_) \
	template<class T> class trap_##_name_##_ : public Function_SV_V<T,_in_t_,_out_t_> \
	{ \
	public: \
		trap_##_name_##_(int loc) : Function_SV_V<T,_in_t_,_out_t_>(loc,#_name_){} \
		virtual int Evaluate(Variable **args) \
		{ \
			int i, n = this->mResult->Dim(); \
            T *arg1 = static_cast<DataVariable<T>*>(args[0])->Data(); \
            T *arg2 = static_cast<DataVariable<T>*>(args[1])->Data(); \
            T *res = static_cast<DataVariable<T>*>(this->mResult)->Data(); \
			for(i=0; i<n; i++) \
			{ \
				if(_assert_) return 2; \
				res[i] = T(_name_(arg1[this->mScalar1?0:i],arg2[this->mScalar2?0:i])); \
			} \
			return 0; \
		} \
	}

#define ICALCULATOR_OPR_SV_V(_name_,_in_t_,_out_t_,_op_) \
	template<class T> class _name_##_ : public Function_SV_V<T,_in_t_,_out_t_> \
	{ \
	public: \
		_name_##_(int loc) : Function_SV_V<T,_in_t_,_out_t_>(loc,#_name_){} \
		virtual int Evaluate(Variable **args) \
		{ \
			int i, n = this->mResult->Dim(); \
            T *arg1 = static_cast<DataVariable<T>*>(args[0])->Data(); \
            T *arg2 = static_cast<DataVariable<T>*>(args[1])->Data(); \
            T *res = static_cast<DataVariable<T>*>(this->mResult)->Data(); \
			for(i=0; i<n; i++) res[i] = arg1[this->mScalar1?0:i] _op_ arg2[this->mScalar2?0:i]; \
			return 0; \
		} \
	}
		
#define ICALCULATOR_TRAP_OPR_SV_V(_name_,_in_t_,_out_t_,_op_,_assert_) \
	template<class T> class trap_##_name_##_ : public Function_SV_V<T,_in_t_,_out_t_> \
	{ \
	public: \
		trap_##_name_##_(int loc) : Function_SV_V<T,_in_t_,_out_t_>(loc,#_name_){} \
		virtual int Evaluate(Variable **args) \
		{ \
			int i, n = this->mResult->Dim(); \
            T *arg1 = static_cast<DataVariable<T>*>(args[0])->Data(); \
            T *arg2 = static_cast<DataVariable<T>*>(args[1])->Data(); \
            T *res = static_cast<DataVariable<T>*>(this->mResult)->Data(); \
			for(i=0; i<n; i++) \
			{ \
				if(_assert_) return 2; \
				res[i] = arg1[this->mScalar1?0:i] _op_ arg2[this->mScalar2?0:i]; \
			} \
			return 0; \
		} \
	}
		
#define ICALCULATOR_FUN_MINMAX(_name_,_in_t_,_out_t_,_op_) \
	template<class T> class _name_##_ : public Function_NV1V_VS<T,_in_t_,_out_t_> \
	{ \
	public: \
		_name_##_(int loc) : Function_NV1V_VS<T,_in_t_,_out_t_>(loc,#_name_){} \
		virtual int Evaluate(Variable **args) \
		{ \
			int i, n; \
            T *arg = static_cast<DataVariable<T>*>(args[0])->Data(); \
            T *res = static_cast<DataVariable<T>*>(this->mResult)->Data(); \
			if(this->mNumArgs > 1) \
			{ \
				int j, narg = this->mNumArgs; \
				n = this->mResult->Dim(); \
				memcpy(res,arg,n*sizeof(T)); \
				for(j=1; j<narg; j++) \
				{ \
					arg = static_cast<DataVariable<T>*>(args[j])->Data(); \
					for(i=0; i<n; i++) if(res[i] _op_ arg[i]) res[i] = arg[i]; \
				} \
			} \
			else \
			{ \
				int n = args[0]->Dim(); \
				res[0] = arg[0]; \
				for(i=1; i<n; i++) if(res[0] _op_ arg[i]) res[0] = arg[i]; \
			} \
			return 0; \
		} \
	}

#define ICALCULATOR_SELECT_FUN(_name_) \
	if(str == #_name_) return new iCalculatorKit::_name_##_<T>(loc)

#define ICALCULATOR_SELECT_TRAP_FUN(_name_) \
	if(str == #_name_) return new iCalculatorKit::trap_##_name_##_<T>(loc)


namespace iCalculatorKit
{
	//
	//  These are already defined in some versions of Linux and in ISO C99, so we keep them
	//  inside the namespace
	//
	inline double asinh(double v){ return log(v+sqrt(v*v+1.0)); }
	inline double acosh(double v){ return log(v+sqrt(v*v-1.0)); }
	inline double atanh(double v){ return 0.5*log((1.0+v)/(1.0-v));	}
	inline double sign(double v){ return (v < 0.0) ? -1.0 : 1.0; }

	//
	//  useful helper functions
	//
	template<class T>
	T ConvertFromString(const iString &str, bool &ok);

	template<>
	inline float ConvertFromString<float>(const iString &str, bool &ok)
	{
		return str.ToFloat(ok);
	}

	template<>
	inline double ConvertFromString<double>(const iString &str, bool &ok)
	{
		return str.ToDouble(ok);
	}


	template<class T>
	T ConvertToBoolRepresentation(T v);

	template<>
	inline float ConvertToBoolRepresentation<float>(float v)
	{
		return (fabs(v) < iMath::_FloatTolerance ? 0.0f : 1.0f);
	}

	template<>
	inline double ConvertToBoolRepresentation<double>(double v)
	{
		return (fabs(v) < iMath::_DoubleTolerance ? 0.0 : 1.0);
	}


	//
	//  **********************************************************************
	//
	//  More specialized abstract implementations for variables
	//
	//  **********************************************************************
	//
	template<class T>
	class DataVariable : public Variable
	{

	public:

		DataVariable(int dim, int use, const iString &name) : Variable(dim,use,name), mData(0)
		{
		}

		inline T* const Data() const { return this->mData; }

	protected:

		virtual void CopyBody(const Variable &var)
		{
			if(this->mData != 0) memcpy(this->mData,static_cast<const DataVariable<T>&>(var).mData,this->Dim()*sizeof(T));
		}

		T* mData;
	};


	template<class T, bool morphable>
	class GenericDataVariable : public DataVariable<T>
	{

	public:

		GenericDataVariable(int dim, int use, const iString &name) : DataVariable<T>(dim,use,name)
		{
		}

	protected:

		virtual bool MorphBody(int dim, int use)
		{
			if(morphable)
			{
				if(dim < 1) return false;
				if(dim != this->Dim()) this->ChangeDimension(dim);
				if(use == 1)
				{
					int i;
					for(i=0; i<dim; i++) this->mData[i] = ConvertToBoolRepresentation<T>(this->mData[i]);
				}
				return true;
			}
			else return false;
		}

		virtual void ChangeDimension(int dim) = 0;
	};


	//
	//  **********************************************************************
	//
	//  Concrete implementations for variables
	//
	//  **********************************************************************
	//

	//
	//  Vector that owns the data ("access by value").
	//
	template<class T, bool morphable>
	class var_ : public GenericDataVariable<T,morphable>
	{

	public:

		var_(int dim, int use, const iString &name) : GenericDataVariable<T,morphable>(dim,use,name)
		{
			this->mData = new T[dim]; IERROR_ASSERT(this->mData);
		}

		virtual ~var_()
		{
			delete [] this->mData;
		}

	protected:

		virtual void ChangeDimension(int dim)
		{
			delete [] this->mData;
			this->mData = new T[dim]; IERROR_ASSERT(this->mData);
		}
	};

	//
	//  Vector that references the data ("access by reference")
	//
	template<class T, bool morphable>
	class ptr_ : public GenericDataVariable<T,morphable>
	{

	public:

		ptr_(T* ptr, int dim, int use, const iString &name) : GenericDataVariable<T,morphable>(dim,use,name)
		{
			this->mData = ptr; IERROR_ASSERT(this->mData);
		}

	protected:

		virtual void ChangeDimension(int dim)
		{
			//
			//  Just reset the pointer length - which is done by the parent
			//
		}
	};


	//
	//  **********************************************************************
	//
	//  More specialized abstract implementations for functions
	//
	//  **********************************************************************
	//

	//
	//  Function with vector result and n arguments
	//
	template<class T>
	class GenericFunction : public Function
	{

	public:

		GenericFunction(int narg, int loc, const iString &name) : Function(new var_<T,true>(1,0,name),narg,loc)
		{
			IERROR_ASSERT(this->mResult);
		}

		virtual ~GenericFunction()
		{
			delete this->mResult;
		}
	};


	//
	//  vector-to-vector multi-argument function
	//
	template<class T, int in_t, int out_t, bool scalar>
	class Function_NV_V : public GenericFunction<T>
	{

	public:

		Function_NV_V(int narg, int loc, const iString &name) : GenericFunction<T>(narg,loc,name){}

		virtual bool CheckArguments(int *dims, int *uses, iString &err)
		{
			int i;
			for(i=0; i<this->mNumArgs; i++)
			{
				if(uses[i] != in_t)
				{
					err = "argument #" + iString::FromNumber(i+1) + " has an incompatible usage patterns";
					return false;
				}
				if(dims[i] < 1)
				{
					err = "argument #" + iString::FromNumber(i+1) + " has an invalid dimension < 1";
					return false;
				}
				if(dims[i] != dims[0])
				{
					err = "argument #" + iString::FromNumber(i+1) + " has an incompatible dimension of " + iString::FromNumber(dims[i]) + " instead of " + iString::FromNumber(dims[0]);
					return false;
				}
			}
			if(!this->mResult->Morph(scalar?1:dims[0],out_t))
			{
				err = "output (dim=" + iString::FromNumber(this->mResult->Dim()) + ",use=" + iString::FromNumber(this->mResult->Use()) + ") is not compatible with required dimension of " + iString::FromNumber(dims[0]) + " and usage pattern of " + iString::FromNumber(out_t);
				return false;
			}
			return true;
		}
	};


	//
	//  vector+scalar-to-vector 2-argument function
	//
	template<class T, int in_t, int out_t>
	class Function_SV_V : public GenericFunction<T>
	{

	public:

		Function_SV_V(int loc, const iString &name) : GenericFunction<T>(2,loc,name){}

		virtual bool CheckArguments(int *dims, int *uses, iString &err)
		{
			int i;
			for(i=0; i<2; i++)
			{
				if(uses[i] != in_t)
				{
					err = "argument #" + iString::FromNumber(i+1) + " has an incompatible usage pattern";
					return false;
				}
				if(dims[i] < 1)
				{
					err = "argument #" + iString::FromNumber(i+1) + " has an invalid dimension < 1";
					return false;
				}
			}

			int dim = (dims[0] > dims[1]) ? dims[0] : dims[1];
			for(i=0; i<2; i++)
			{
				if(dims[i]!=1 && dims[i]!=dim)
				{
					err = "argument #" + iString::FromNumber(i+1) + " has an incompatible dimension of " + iString::FromNumber(dims[i]) + " instead of 1 or " + iString::FromNumber(dim);
					return false;
				}
			}

			if(!this->mResult->Morph(dim,out_t))
			{
				err = "output (dim=" + iString::FromNumber(this->mResult->Dim()) + ",use=" + iString::FromNumber(this->mResult->Use()) + ") is not compatible with required dimension of " + iString::FromNumber(dim) + " and usage pattern of " + iString::FromNumber(out_t);
				return false;
			}
			this->mScalar1 = (dims[0]==1 && dim>1);
			this->mScalar2 = (dims[1]==1 && dim>1);
			return true;
		}

	protected:

		bool mScalar1, mScalar2;
	};


	//
	//  an operation on 2 spatial points
	//
	template<class T, bool scalar>
	class Function_2P : public GenericFunction<T>
	{

	public:

		Function_2P(int loc, const iString &name) : GenericFunction<T>(2,loc,name){}

		virtual bool CheckArguments(int *dims, int *uses, iString &err)
		{
			int i;
			for(i=0; i<this->mNumArgs; i++)
			{
				if(uses[i] != 0)
				{
					err = "argument #" + iString::FromNumber(i+1) + " has an incompatible usage pattern";
					return false;
				}
				if(dims[i] != 3)
				{
					err = "argument #" + iString::FromNumber(i+1) + " has an invalid dimension; a dimension of a point must be 3";
					return false;
				}
			}
			if(!this->mResult->Morph(scalar?1:3,0))
			{
				err = "output (dim=" + iString::FromNumber(this->mResult->Dim()) + ",use=" + iString::FromNumber(this->mResult->Use()) + ") is not compatible with required dimension of " + (scalar?"1":"3") + " and usage pattern of 0";
				return false;
			}
			return true;
		}
	};


	//
	//  min/max type functions - do vector-to-vector for narg>1, vector-to-scalar for n=1
	//
	template<class T, int in_t, int out_t>
	class Function_NV1V_VS : public GenericFunction<T>
	{

	public:

		Function_NV1V_VS(int loc, const iString &name) : GenericFunction<T>(-1,loc,name){}

		virtual bool CheckArguments(int *dims, int *uses, iString &err)
		{
			int i;
			for(i=0; i<this->mNumArgs; i++)
			{
				if(uses[i] != in_t)
				{
					err = "argument #" + iString::FromNumber(i+1) + " has an incompatible usage pattern";
					return false;
				}
				if(dims[i] < 1)
				{
					err = "argument #" + iString::FromNumber(i+1) + " has an invalid dimension < 1";
					return false;
				}
				if(dims[i] != dims[0])
				{
					err = "argument #" + iString::FromNumber(i+1) + " has an incompatible dimension of " + iString::FromNumber(dims[i]) + " instead of " + iString::FromNumber(dims[0]);
					return false;
				}
			}
			if(!this->mResult->Morph(this->mNumArgs==1?1:dims[0],out_t))
			{
				err = "output (dim=" + iString::FromNumber(this->mResult->Dim()) + ",use=" + iString::FromNumber(this->mResult->Use()) + ") is not compatible with required dimension of " + iString::FromNumber(this->mNumArgs==1?1:dims[0]) + " and usage pattern of " + iString::FromNumber(out_t);
				return false;
			}
			return true;
		}
	};


	//
	//  **********************************************************************
	//
	//  Concrete implementations for functions
	//
	//  **********************************************************************
	//

	//
	//  Mathematical operators
	//
	ICALCULATOR_OPR_SV_V(mul,0,0,*);
	ICALCULATOR_OPR_SV_V(add,0,0,+);
	ICALCULATOR_OPR_SV_V(sub,0,0,-);
	ICALCULATOR_OPR_1V_V(neg,0,0,-);

	ICALCULATOR_FUN_SV_V(pow,0,0);
	ICALCULATOR_OPR_SV_V(div,0,0,/);
	ICALCULATOR_TRAP_FUN_SV_V(pow,0,0,arg1[i]<0.0 || (arg1[i]==0 && arg2[i]<0.0));
	ICALCULATOR_TRAP_OPR_SV_V(div,0,0,/,arg2[i] == 0.0);

	//
	//  Logical operators
	//
	ICALCULATOR_OPR_SV_V(ltn,0,1,<);
	ICALCULATOR_OPR_SV_V(gtn,0,1,>);
	ICALCULATOR_OPR_SV_V(leq,0,1,<=);
	ICALCULATOR_OPR_SV_V(geq,0,1,>=);
	ICALCULATOR_OPR_SV_V(eql,0,1,==);
	ICALCULATOR_OPR_SV_V(neq,0,1,!=);
	ICALCULATOR_OPR_SV_V(and,1,1,&&);
	ICALCULATOR_OPR_SV_V(lor,1,1,||);
	ICALCULATOR_OPR_1V_V(not,1,1,!);

	//
	//  One-argument functions
	//
	ICALCULATOR_FUN_1V_V(sin,0,0);
	ICALCULATOR_FUN_1V_V(cos,0,0);

	ICALCULATOR_FUN_1V_V(atan,0,0);
	
	ICALCULATOR_FUN_1V_V(sinh,0,0);
	ICALCULATOR_FUN_1V_V(cosh,0,0);
	ICALCULATOR_FUN_1V_V(tanh,0,0);
	
	ICALCULATOR_FUN_1V_V(asinh,0,0);

	ICALCULATOR_FUN_1V_V(pow10,0,0);
	ICALCULATOR_FUN_1V_V(exp,0,0);
	ICALCULATOR_FUN_1V_V(floor,0,0);
	ICALCULATOR_FUN_1V_V(ceil,0,0);
	ICALCULATOR_FUN_1V_V(fabs,0,0);
	ICALCULATOR_FUN_1V_V(sign,0,0);

	ICALCULATOR_FUN_1V_V(tan,0,0);
	ICALCULATOR_FUN_1V_V(asin,0,0);
	ICALCULATOR_FUN_1V_V(acos,0,0);
	ICALCULATOR_TRAP_FUN_1V_V(tan,0,0,cos(arg1[i]) == 0.0);
	ICALCULATOR_TRAP_FUN_1V_V(asin,0,0,arg1[i]<-1.0 || arg1[i]>1.0);
	ICALCULATOR_TRAP_FUN_1V_V(acos,0,0,arg1[i]<-1.0 || arg1[i]>1.0);

	ICALCULATOR_FUN_1V_V(acosh,0,0);
	ICALCULATOR_FUN_1V_V(atanh,0,0);
	ICALCULATOR_TRAP_FUN_1V_V(acosh,0,0,arg1[i] < 1.0);
	ICALCULATOR_TRAP_FUN_1V_V(atanh,0,0,arg1[i]<-1.0 || arg1[i]>1.0);

	ICALCULATOR_FUN_1V_V(log,0,0);
	ICALCULATOR_FUN_1V_V(log10,0,0);
	ICALCULATOR_FUN_1V_V(sqrt,0,0);
	ICALCULATOR_TRAP_FUN_1V_V(log,0,0,arg1[i] <= 0.0);
	ICALCULATOR_TRAP_FUN_1V_V(log10,0,0,arg1[i] <= 0.0);
	ICALCULATOR_TRAP_FUN_1V_V(sqrt,0,0,arg1[i] < 0.0);

	//
	//  Vector operators
	//
	template<class T> class dot_ : public Function_2P<T,true>
	{
	public:
		dot_(int loc) : Function_2P<T,true>(loc,"dot"){}
		virtual int Evaluate(Variable **args)
		{
			int i;
            T *arg1 = static_cast<DataVariable<T>*>(args[0])->Data();
            T *arg2 = static_cast<DataVariable<T>*>(args[1])->Data();
            T *res = static_cast<DataVariable<T>*>(this->mResult)->Data();
			res[0] = T(0);
			for(i=0; i<3; i++) res[0] += arg1[i]*arg2[i];
			return 0;
		}
	};

	template<class T> class xrs_ : public Function_2P<T,false>
	{
	public:
		xrs_(int loc) : Function_2P<T,false>(loc,"xrs"){}
		virtual int Evaluate(Variable **args)
		{
            T *arg1 = static_cast<DataVariable<T>*>(args[0])->Data();
            T *arg2 = static_cast<DataVariable<T>*>(args[1])->Data();
            T *res = static_cast<DataVariable<T>*>(this->mResult)->Data();
			res[0] = arg1[1]*arg2[2] - arg1[2]*arg2[1];
			res[1] = arg1[2]*arg2[0] - arg1[0]*arg2[2];
			res[2] = arg1[0]*arg2[1] - arg1[1]*arg2[0];
			return 0;
		}
	};

	//
	//  Special operators: vector construct, vector component
	//
	template<class T>
	class vec_ : public GenericFunction<T>
	{

	public:

		vec_(int loc) : GenericFunction<T>(-1,loc,"vec"){}

		virtual bool CheckArguments(int *dims, int *uses, iString &err)
		{
			int i;
			for(i=0; i<this->mNumArgs; i++)
			{
				if(uses[i] != uses[0])
				{
					err = "argument #" + iString::FromNumber(i+1) + " has an incompatible usage pattern";
					return false;
				}
				if(dims[i] != 1)
				{
					err = "argument #" + iString::FromNumber(i+1) + " has an invalid dimension; all arguments must have dimension of 1";
					return false;
				}
			}
			if(!this->mResult->Morph(this->mNumArgs,uses[0]))
			{
				err = "output (dim=" + iString::FromNumber(this->mResult->Dim()) + ",use=" + iString::FromNumber(this->mResult->Use()) + ") is not compatible with required dimension of " + iString::FromNumber(this->mNumArgs) + " and usage pattern of " + iString::FromNumber(uses[0]);
				return false;
			}
			return true;
		}

		virtual int Evaluate(Variable **args)
		{
			int i, n = this->mResult->Dim();
            T *res = static_cast<DataVariable<T>*>(this->mResult)->Data();
			for(i=0; i<n; i++)
			{
				res[i] = static_cast<DataVariable<T>*>(args[i])->Data()[0];
			}
			return 0;
		}
	};

	template<class T>
	class arg_ : public GenericFunction<T>
	{

	public:

		arg_(int loc) : GenericFunction<T>(2,loc,"arg"){}

		virtual bool CheckArguments(int *dims, int *uses, iString &err)
		{
			if(uses[1] != 0)
			{
				err = "argument #2 must be numeric (i.e. have a usage pattern of 0)";
				return false;
			}
			if(dims[1] != 1)
			{
				err = "argument #2 must be a scalar (i.e. have a dimension of 1)";
				return false;
			}
			if(dims[0] < 2)
			{
				err = "argument #1 must be a true vector (i.e. have a dimension of 2 and more)";
				return false;
			}
			if(!this->mResult->Morph(1,uses[0]))
			{
				err = "output (dim=" + iString::FromNumber(this->mResult->Dim()) + ",use=" + iString::FromNumber(this->mResult->Use()) + ") is not compatible with required dimension of 1 and usage pattern of " + iString::FromNumber(uses[0]);
				return false;
			}
			return true;
		}

		virtual int Evaluate(Variable **args)
		{
            int ind = int(static_cast<DataVariable<T>*>(args[1])->Data()[0]);
            DataVariable<T> *arg1 = static_cast<DataVariable<T>*>(args[0]);
			if(ind>0 && ind<=arg1->Dim())
			{
                static_cast<DataVariable<T>*>(this->mResult)->Data()[0] = arg1->Data()[ind-1];
				return 0;
			}
			else return 1;
		}
	};

	//
	//  min/max type functions - do vector-to-vector for narg>1, vector-to-scalar for n=1
	//
	ICALCULATOR_FUN_MINMAX(min,0,0,>);
	ICALCULATOR_FUN_MINMAX(max,0,0,<);

	template<class T> class sum_ : public Function_NV1V_VS<T,0,0>
	{
	public:
		sum_(int loc) : Function_NV1V_VS<T,0,0>(loc,"sum"){}
		virtual int Evaluate(Variable **args)
		{
			int i, n;
            T *arg = static_cast<DataVariable<T>*>(args[0])->Data();
            T *res = static_cast<DataVariable<T>*>(this->mResult)->Data();
			if(this->mNumArgs > 1)
			{
				int j, narg = this->mNumArgs;
				n = this->mResult->Dim();
				memcpy(res,arg,n*sizeof(T));
				for(j=1; j<narg; j++)
				{
					arg = static_cast<DataVariable<T>*>(args[j])->Data();
					for(i=0; i<n; i++) res[i] += arg[i];
				}
			}
			else
			{
				int n = args[0]->Dim();
				res[0] = arg[0];
				for(i=1; i<n; i++) res[0] += arg[i];
			}
			return 0;
		}
	};

	//
	//  Return the dimension (size) of the array
	//
	template<class T>
	class dim_ : public GenericFunction<T>
	{

	public:

		dim_(int loc) : GenericFunction<T>(1,loc,"dim"){}

		virtual bool CheckArguments(int *dims, int *uses, iString &err)
		{
			if(!this->mResult->Morph(1,0))
			{
				err = "output (dim=" + iString::FromNumber(this->mResult->Dim()) + ",use=" + iString::FromNumber(this->mResult->Use()) + ") is not compatible with required dimension of 1 and usage pattern of 0";
				return false;
			}
			return true;
		}

		virtual int Evaluate(Variable **args)
		{
            static_cast<DataVariable<T>*>(this->mResult)->Data()[0] = args[0]->Dim();
			return 0;
		}
	};

	//
	//  User-friendly Variable types that auto-add to the calculator:
	//  1. Vector that owns the data ("access by value").
	//
	template<class T, int use>
	class var : public var_<T,false>
	{

	public:

		var(const iString &name, iCalculator<T> *c = 0) : var_<T,false>(1,use,name)
		{
			if(c != 0) c->AddVariable(this);
		}

		var(const iString &name, int dim, iCalculator<T> *c = 0) : var_<T,false>(dim,use,name)
		{
			if(c != 0) c->AddVariable(this);
		}

		inline T& operator[](int i){ return this->mData[i]; }
		inline const T& operator[](int i) const { return this->mData[i]; }
		inline operator T*(){ return this->mData; }
		inline operator const T*() const { return this->mData; }
	};


	//
	//  2. Vector that references the data ("access by reference")
	//
	template<class T, int use>
	class ptr : public ptr_<T,false>
	{

	public:

		ptr(T* p, const iString &name, iCalculator<T> *c = 0) : ptr_<T,false>(p,1,use,name)
		{
			if(c != 0) c->AddVariable(this);
		}

		ptr(T* p, const iString &name, int dim, iCalculator<T> *c = 0) : ptr_<T,false>(p,dim,use,name)
		{
			if(c != 0) c->AddVariable(this);
		}

		inline T& operator[](int i){ return this->mData[i]; }
		inline const T& operator[](int i) const { return this->mData[i]; }
		inline operator T*(){ return this->mData; }
		inline operator const T*() const { return this->mData; }

		inline T* operator++(){ this->mData += this->Dim(); return this->mData; }
	};
};


//
//  **********************************************************************
//
//  Default template calculator (that works with the default parser)
//
//  **********************************************************************
//
template<class T>
iCalculator<T>::iCalculator(const iCalculatorBase *driver) : iCalculatorBase(driver)
{
}


template<class T>
const T* iCalculator<T>::GetResultData() const
{
	if(this->Result() != 0) return static_cast<iCalculatorKit::DataVariable<T>*>(this->Result())->Data(); else return 0;
}


template<class T>
int iCalculator<T>::GetResultSize() const
{
	if(this->Result() != 0) return this->Result()->Dim(); else return 0;
}


template<class T>
void iCalculator<T>::AddVariable(const result_t *var)
{
	this->iCalculatorBase::AddVariable(var);
}


#ifdef I_DEBUG
template<class T>
void iCalculator<T>::PrintStack(int iptr, int narg, int jmax, iCalculatorKit::Variable** stack)
{
	iConsole::Display(iConsole::_Info,"\nCall:\n");
	iConsole::Display(iConsole::_Info,(iString::FromNumber(iptr)+","+iString::FromNumber(narg)+"\n").ToCharPointer());
	int i, j;
	iString w;
	for(j=0; j<=jmax; j++)
	{
		w = "stack["+iString::FromNumber(j)+"]: " + stack[j]->Name() + " ";
		for(i=0; i<stack[j]->Dim(); i++) w += iString::FromNumber(static_cast<iCalculatorKit::DataVariable<T>*>(stack[j])->Data()[i])+" ";
		w += "\n";
		iConsole::Display(iConsole::_Info,w.ToCharPointer());
	}
}
#endif

//
//  Overwritable method for extending
//
template<class T>
iCalculatorKit::Variable* iCalculator<T>::CreateLiteral(const iString &str) const
{
	bool ok;
	T v = iCalculatorKit::ConvertFromString<T>(str,ok); if(!ok) return 0;
	iCalculatorKit::var_<T,false> *ret = new iCalculatorKit::var_<T,false>(1,0,"$$c"); if(ret == 0) return 0;
	ret->Data()[0] = v;
	return ret;
}


template<class T>
iCalculatorKit::Function* iCalculator<T>::CreateFunction(const iString &str, int loc) const
{
	//
	//  Mathematical operators
	//
	ICALCULATOR_SELECT_FUN(mul);
	ICALCULATOR_SELECT_FUN(add);
	ICALCULATOR_SELECT_FUN(sub);
	ICALCULATOR_SELECT_FUN(neg);

	if(this->GetTrapFPE())
	{
		ICALCULATOR_SELECT_TRAP_FUN(pow);
		ICALCULATOR_SELECT_TRAP_FUN(div);
	}
	else
	{
		ICALCULATOR_SELECT_FUN(pow);
		ICALCULATOR_SELECT_FUN(div);
	}

	//
	//  Logical operators
	//
	ICALCULATOR_SELECT_FUN(ltn);
	ICALCULATOR_SELECT_FUN(gtn);
	ICALCULATOR_SELECT_FUN(leq);
	ICALCULATOR_SELECT_FUN(geq);
	ICALCULATOR_SELECT_FUN(eql);
	ICALCULATOR_SELECT_FUN(neq);
	ICALCULATOR_SELECT_FUN(and);
	ICALCULATOR_SELECT_FUN(lor);
	ICALCULATOR_SELECT_FUN(not);

	//
	//  One-argument functions
	//
	ICALCULATOR_SELECT_FUN(sin);
	ICALCULATOR_SELECT_FUN(cos);

	ICALCULATOR_SELECT_FUN(atan);

	ICALCULATOR_SELECT_FUN(tanh);
	ICALCULATOR_SELECT_FUN(sinh);
	ICALCULATOR_SELECT_FUN(cosh);

	ICALCULATOR_SELECT_FUN(asinh);

	ICALCULATOR_SELECT_FUN(pow10);
	ICALCULATOR_SELECT_FUN(exp);
	ICALCULATOR_SELECT_FUN(floor);
	ICALCULATOR_SELECT_FUN(ceil);
	ICALCULATOR_SELECT_FUN(fabs);
	ICALCULATOR_SELECT_FUN(sign);

	if(this->GetTrapFPE())
	{
		ICALCULATOR_SELECT_TRAP_FUN(tan);
		ICALCULATOR_SELECT_TRAP_FUN(asin);
		ICALCULATOR_SELECT_TRAP_FUN(acos);
		ICALCULATOR_SELECT_TRAP_FUN(acosh);
		ICALCULATOR_SELECT_TRAP_FUN(atanh);
		ICALCULATOR_SELECT_TRAP_FUN(log);
		ICALCULATOR_SELECT_TRAP_FUN(log10);
		ICALCULATOR_SELECT_TRAP_FUN(sqrt);
	}
	else
	{
		ICALCULATOR_SELECT_FUN(tan);
		ICALCULATOR_SELECT_FUN(asin);
		ICALCULATOR_SELECT_FUN(acos);
		ICALCULATOR_SELECT_FUN(acosh);
		ICALCULATOR_SELECT_FUN(atanh);
		ICALCULATOR_SELECT_FUN(log);
		ICALCULATOR_SELECT_FUN(log10);
		ICALCULATOR_SELECT_FUN(sqrt);
	}

	//
	//  Vector operators
	//
	ICALCULATOR_SELECT_FUN(dot);
	ICALCULATOR_SELECT_FUN(xrs);
	//
	//  Special operators: vector construct, vector component
	//
	ICALCULATOR_SELECT_FUN(vec);
	ICALCULATOR_SELECT_FUN(arg);

	//
	//  min/max type functions - do vector-to-vector for narg>1, vector-to-scalar for n=1
	//
	ICALCULATOR_SELECT_FUN(min);
	ICALCULATOR_SELECT_FUN(max);
	ICALCULATOR_SELECT_FUN(sum);

	//
	//  Return the dimension (size) of the array
	//
	ICALCULATOR_SELECT_FUN(dim);

	return 0;
}

