/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2006 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "irangecollection.h"


#include "ierror.h"
#include "imath.h"

//
//  Templates
//
#include "iarraytemplate.h"

//
// iRangeCollection class
//
iRangeCollection::iRangeCollection(float min, float max)
{
	mTiled = false;
	mGlobalMin = min;
	mGlobalMax = max;
	mArr.Add(Range(min,max));
}


iRangeCollection::~iRangeCollection()
{
}


void iRangeCollection::SetGlobalRange(float min, float max)
{
	if(min < max)
	{
		//
		//   Scale ranges fist
		//
		int i;
		float oldStart = this->ApplyStretch(mGlobalMin,false);
		float newStart = this->ApplyStretch(min,false);
		float scale = (this->ApplyStretch(max,true)-newStart)/(this->ApplyStretch(mGlobalMax,true)-oldStart);

		for(i=0; i<mArr.Size(); i++)
		{
			mArr[i].Min = this->ResetStretch(newStart+scale*(this->ApplyStretch(mArr[i].Min,false)-oldStart));
			mArr[i].Max = this->ResetStretch(newStart+scale*(this->ApplyStretch(mArr[i].Max,true)-oldStart));
		}
		mGlobalMin = min;
		mGlobalMax = max;
	}
}


void iRangeCollection::SetRange(int i, float min, float max)
{
	if(i<0 || i>=mArr.Size()) return;

	if(mTiled && i>0)
	{
		if(min < mArr[i-1].Min) min = mArr[i-1].Min;
		mArr[i].Min = mArr[i-1].Max = min;
	}
	else if(!mTiled)
	{
		if(min < mGlobalMin) min = mGlobalMin;
		mArr[i].Min = min;
	}

	if(mTiled && i<mArr.MaxIndex())
	{
		if(max > mArr[i+1].Max) max = mArr[i+1].Max;
		mArr[i].Max = mArr[i+1].Min = max;
	}
	else if(!mTiled)
	{
		if(max > mGlobalMax) max = mGlobalMax;
		mArr[i].Max = max;
	}
}


void iRangeCollection::AddRange(float min, float max)
{
	mArr.Add(Range(min,max));
}


void iRangeCollection::AddRange()
{
	int n = mArr.Size() - 1;
	float mid = this->ResetStretch(0.5f*(this->ApplyStretch(mArr[n].Min,false)+this->ApplyStretch(mArr[n].Max,true)));
	float max = mArr[n].Max;
	mArr[n].Max = mid;
	this->AddRange(mid,max);
}


bool iRangeCollection::RemoveRange(int n)
{
	if(mArr.Size()>1 && n>=0 && n<mArr.Size())
	{
		if(mTiled)
		{
			if(n > 0)
			{
				mArr[n-1].Max = mArr[n].Max;
			}
			else
			{
				mArr[1].Min = mGlobalMin;
			}
		}
		mArr.Remove(n);
		return true;
	}
	else return false;
}


void iRangeCollection::Reset()
{
	while(mArr.Size() > 1) mArr.RemoveLast();
	mArr[0].Min = mGlobalMin;
	mArr[0].Max = mGlobalMax;
}


void iRangeCollection::SetTiled(bool s)
{
	int i, j;
	Range tmp;

	if(mTiled == s) return;
	mTiled = s;
	if(!s) return;
	
	//
	//  bubble sort - since the number of pieces is likely to be small
	//
	for(i=1; i<mArr.Size(); i++)
	{
		for(j=0; j<i; j++)
		{
			if(mArr[i].Min < mArr[j].Min)
			{
				tmp = mArr[i];
				mArr[i] = mArr[j];
				mArr[j] = tmp;
			}
		}
	}

	for(i=0; i<mArr.MaxIndex(); i++) mArr[i].Max = mArr[i+1].Min;
	
	mArr[0].Min = mGlobalMin;
	mArr.Last().Max = mGlobalMax;
}


void iRangeCollection::Copy(const iRangeCollection *f)
{
	if(this==f || f==0) return;

	int i;
	for(i=0; i<mArr.Size() && i<f->mArr.Size(); i++) mArr[i] = f->mArr[i];

	while(mArr.Size() < f->mArr.Size()) mArr.Add(f->mArr[mArr.Size()]);
	while(mArr.Size() > f->mArr.Size()) mArr.RemoveLast();

	mGlobalMin = f->mGlobalMin;
	mGlobalMax = f->mGlobalMax;

	mTiled = f->mTiled;

	this->SetStretch(f->GetStretch());
}


