/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2006 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "ivolumeviewsubject.h"


#include "iclipplane.h"
#include "icolorbars.h"
#include "icontrolmodule.h"
#include "idatalimits.h"
#include "idatareader.h"
#include "idatasubject.h"
#include "ierror.h"
#include "ieventobserver.h"
#include "ifunctionmapping.h"
#include "ihistogrammaker.h"
#include "ipalette.h"
#include "ipiecewisefunction.h"
#include "ireplicatedgriddata.h"
#include "ireplicatedvolume.h"
#include "iviewmodule.h"
#include "iviewsubjectobserver.h"
#include "iviewsubjectreplicator.h"
#include "ivolumedataconverter.h"
#include "ivtk.h"

#include <vtkStructuredPoints.h>
#include <vtkVolumeProperty.h>
#include <vtkVolumeRayCastCompositeFunction.h>
#include <vtkVolumeRayCastMIPFunction.h>
#include <vtkVolumeRayCastMapper.h>
#include <vtkVolumeTextureMapper2D.h>
#ifndef IVTK_4
#include <vtkVolumeTextureMapper3D.h>
#endif

//
//  Templates
//
#include "iarraytemplate.h"


using namespace iParameter;


IVIEWSUBJECT_DEFINE_TYPE(iVolumeViewSubject,Volume,w);

IOBJECT_DEFINE_KEY(iVolumeViewSubject,BlendMode,bm,Int,1);
IOBJECT_DEFINE_KEY(iVolumeViewSubject,CompositeMethod,cm,Int,1);
IOBJECT_DEFINE_KEY(iVolumeViewSubject,InterpolationType,it,Int,1);
IOBJECT_DEFINE_KEY(iVolumeViewSubject,Method,m,Int,1);
IOBJECT_DEFINE_KEY(iVolumeViewSubject,OpacityFunction,of,Any,1);
IOBJECT_DEFINE_KEY(iVolumeViewSubject,Palette,p,Int,1);
IOBJECT_DEFINE_KEY(iVolumeViewSubject,Var,v,OffsetInt,1);
IOBJECT_DEFINE_KEY(iVolumeViewSubject,XYResample,rxy,Float,1);
IOBJECT_DEFINE_KEY(iVolumeViewSubject,ZResample,rz,Float,1);

//
//  Inherited keys
//
IVIEWSUBJECT_DEFINE_INHERITED_KEYS_COMMON(iVolumeViewSubject);
IVIEWSUBJECT_DEFINE_INHERITED_KEYS_SHADING(iVolumeViewSubject);
IVIEWSUBJECT_DEFINE_INHERITED_KEYS_REPLICATING(iVolumeViewSubject);


//
// iVolumeViewSubject class
// 
iVolumeViewSubject::iVolumeViewSubject(iViewModule *vm, const iDataType &type, const iString &name) : iViewSubject(vm,type,name,0U)
{
	mObjectType = _ObjectTypeVolume;

	mVar = 0;
	mPalette = 1;
	
#ifdef VTK_USE_VOLUMEPRO
	mMethod = _VolumeMethodVolumePro;
#else
	mMethod = _VolumeMethod2DTexture;
#endif
	
	mOF = iPiecewiseFunction::New(0.0,1.0); IERROR_ASSERT(mOF);
	
	mBlendMode = 0;
	mCompositeMethod = 0;
	mInterpolationType = 1; 

	mResXY = mResZ = 1.0;
	mBlendMode = 0;
	
	mVolume = iReplicatedVolume::New(this->GetReplicator()); IERROR_ASSERT(mVolume);
	mVolume->SetPosition(0.0,0.0,0.0);
	
	mVolume->GetProperty()->SetShade(mShading?1:0);
	mVolume->GetProperty()->SetScalarOpacity(mOF->GetVTKFunction());
	mVolume->GetProperty()->SetColor(this->GetViewModule()->GetControlModule()->GetPalette(iMath::Abs(mPalette)-1)->GetColorTransferFunction(mPalette < 0));
	mVolume->GetProperty()->SetInterpolationTypeToLinear();
	this->UpdateMaterialProperties();
	
	mDataConverter = iVolumeDataConverter::New(this); IERROR_ASSERT(mDataConverter);
	mDataReplicated = iReplicatedGridData::New(this); IERROR_ASSERT(mDataReplicated);
	mHistogramMakers = this->GetViewModule()->GetReader()->GetSubject(this->GetDataType())->CreateHistogramMaker(); IERROR_ASSERT(mHistogramMakers);
	mFunctionMappings = new iFunctionMapping(mOF,mHistogramMakers,0); IERROR_ASSERT(mFunctionMappings);
	
	mDataReplicated->SetInput(mDataConverter->GetOutput());

	mHistogramMakers->SetInput(this->GetData(),this->GetVar());

	iAbortRenderEventObserver *obsAbortRender = this->GetViewModule()->GetAbortRenderEventObserver();

	mRaycastMapper = vtkVolumeRayCastMapper::New(); IERROR_ASSERT(mRaycastMapper);
	mRaycastMapper->AddObserver(vtkCommand::ProgressEvent,obsAbortRender);
//	mRaycastMapper->AutoAdjustSampleDistancesOff();
	mCF = vtkVolumeRayCastCompositeFunction::New(); IERROR_ASSERT(mCF);
	mCF->SetCompositeMethodToInterpolateFirst();
	mMF = vtkVolumeRayCastMIPFunction::New(); IERROR_ASSERT(mMF);

	switch(mBlendMode)
	{
	case 0: { mRaycastMapper->SetVolumeRayCastFunction(mCF); break; }
	case 1: { mRaycastMapper->SetVolumeRayCastFunction(mMF); break; }
	}
	//  mRaycastMapper->SetMinimumImageSampleDistance(0.125);
	//  mRaycastMapper->SetImageSampleDistance(1.0);
	
	m2DTextureMapper = vtkVolumeTextureMapper2D::New(); IERROR_ASSERT(m2DTextureMapper);
	m2DTextureMapper->AddObserver(vtkCommand::ProgressEvent,obsAbortRender);

#ifdef IVTK_4
	m3DTextureMapper = 0;
#else
	m3DTextureMapper = vtkVolumeTextureMapper3D::New(); IERROR_ASSERT(m3DTextureMapper);
	m3DTextureMapper->AddObserver(vtkCommand::ProgressEvent,obsAbortRender);
#endif

	mDataConverter->AddObserver(vtkCommand::ProgressEvent,obsAbortRender);
	mDataReplicated->AddObserver(vtkCommand::ProgressEvent,obsAbortRender);

#ifdef VTK_USE_VOLUMEPRO
	mVolumeProMapper = iVolumeProMapper::New(); IERROR_ASSERT(mVolumeProMapper);

	mVolumeProMapper->IntermixIntersectingGeometryOn();
	mVolumeProMapper->UseImageClipperOff();
	mVolumeProMapper->AddObserver(vtkCommand::ProgressEvent,obsAbortRender);
#else
	mVolumeProMapper = 0;
#endif

#ifdef IVTK_4
	mRaycastMapper->UseImageClipperOff();
	m2DTextureMapper->UseImageClipperOff();
#ifdef VTK_USE_VOLUMEPRO
	mVolumeProMapper->UseImageClipperOff();
#endif
#endif

	m2DTextureMapper->SetInput(mDataReplicated->GetOutput());
#ifndef IVTK_4
	m3DTextureMapper->SetInput(mDataReplicated->GetOutput());
#endif
	mRaycastMapper->SetInput(mDataReplicated->GetOutput());
#ifdef VTK_USE_VOLUMEPRO
	mVolumeProMapper->SetInput(mDataReplicated->GetOutput());
#endif

	//
	//  Add observer to keep information about this object
	//
	mVolume->AddObserver(vtkCommand::UserEvent,mObjectObserver);

	mVolume->SetVisibility(false);
	this->GetViewModule()->AddObject(mVolume);
}


iVolumeViewSubject::~iVolumeViewSubject()
{
	this->GetViewModule()->RemoveObject(mVolume);
	mOF->Delete();
	mVolume->Delete();
	mDataConverter->Delete();
	mDataReplicated->Delete();

	mHistogramMakers->Delete();
	delete mFunctionMappings;

	mRaycastMapper->Delete();
	m2DTextureMapper->Delete();
#ifndef IVTK_4
	m3DTextureMapper->Delete();
#endif
#ifdef VTK_USE_VOLUMEPRO
	mVolumeProMapper->Delete();
#endif
	mCF->Delete();
	mMF->Delete();
}


void iVolumeViewSubject::ConfigureBody()
{
}


void iVolumeViewSubject::FinishInitialization()
{
	this->SetMethod(mMethod);
}


void iVolumeViewSubject::Reset()
{
	mHistogramMakers->SetInput(this->GetData(),this->GetVar());

	mDataConverter->SetInput(vtkImageData::SafeDownCast(this->GetData()));
	this->SyncWithData(this->RequestAll());

	float r = mResZ;
	mResZ = -1.0;
	this->SetZResample(r);

	//
	//  Un-initialize, if needed
	//
	if(mIsInitialized)
	{
		this->ShowColorBars(false);
		mVolume->SetVisibility(false);
		this->GetViewModule()->RemoveObject(mVolume);
		mIsInitialized = false;
	}
}


void iVolumeViewSubject::ShowClipPlane(bool s)
{
	if(s && !mClipPlaneOn)
	{
		mRaycastMapper->AddClippingPlane(this->GetViewModule()->GetClipPlane());
		m2DTextureMapper->AddClippingPlane(this->GetViewModule()->GetClipPlane());
#ifndef IVTK_4
		m3DTextureMapper->AddClippingPlane(this->GetViewModule()->GetClipPlane());
#endif
#ifdef VTK_USE_VOLUMEPRO
		mVolumeProMapper->AddClippingPlane(this->GetViewModule()->GetClipPlane());
#endif
		mClipPlaneOn = true;
	} 
	if(!s && mClipPlaneOn)
	{
		mRaycastMapper->RemoveClippingPlane(this->GetViewModule()->GetClipPlane());
		m2DTextureMapper->RemoveClippingPlane(this->GetViewModule()->GetClipPlane());
#ifndef IVTK_4
		m3DTextureMapper->RemoveClippingPlane(this->GetViewModule()->GetClipPlane());
#endif
#ifdef VTK_USE_VOLUMEPRO
		mVolumeProMapper->RemoveClippingPlane(this->GetViewModule()->GetClipPlane());
#endif
		mClipPlaneOn = false;
	}
	this->ClearCache();
}


void iVolumeViewSubject::ShowColorBars(bool show)
{
	if(!this->IsVisible()) return;
	this->GetViewModule()->GetColorBars()->ShowBar(_ColorBarsPriorityVolume,this->GetVar(),this->GetDataType(),mPalette,show);
	this->ClearCache();
}


void iVolumeViewSubject::SetVar(int v)
{
	if(v>=0 && v<this->GetLimits()->GetNumVars())
	{
		this->ShowColorBars(false);
		mVar = v;
		this->ShowColorBars(true);
		this->UpdateVar();
		this->SyncWithData(this->Request(mVar));
		this->ClearCache();
	}
}


void iVolumeViewSubject::UpdateVar()
{
    mFunctionMappings->AttachToLimits(this->GetLimits(),this->GetVar());
	mDataConverter->SetCurrentVar(this->GetVar());
	mHistogramMakers->SetInputComponent(this->GetVar());
}


void iVolumeViewSubject::SetPalette(int p)
{ 
	if(p!=0 && p>-this->GetViewModule()->GetControlModule()->GetNumberOfPalettes() && p<this->GetViewModule()->GetControlModule()->GetNumberOfPalettes())
	{
		this->ShowColorBars(false);
		mPalette = p;
		this->ShowColorBars(true);
		mVolume->GetProperty()->SetColor(this->GetViewModule()->GetControlModule()->GetPalette(iMath::Abs(mPalette)-1)->GetColorTransferFunction(mPalette < 0));
		this->ClearCache();
	}
}


void iVolumeViewSubject::UpdateMaterialProperties()
{
	if(mShading) mVolume->GetProperty()->SetShade(1); else mVolume->GetProperty()->SetShade(0);

	mVolume->GetProperty()->SetAmbient(mAmbient);
	mVolume->GetProperty()->SetDiffuse(mDiffuse);
	mVolume->GetProperty()->SetSpecular(mSpecular);
	mVolume->GetProperty()->SetSpecularPower(mSpecularPower);
	this->ClearCache();
}


void iVolumeViewSubject::ShowBody(bool show)
{
	if(show)
	{
		mVolume->SetVisibility(true);
		this->ShowColorBars(true);
	} 
	else 
	{
		this->ShowColorBars(false);
		mVolume->SetVisibility(false);
	}
}


int iVolumeViewSubject::SetMethod(int m)
{
	if(m>=0 && m<__NumVolumeMethods)
	{
		int mo = mMethod;
		mMethod = m;
		switch(m) 
		{
		case _VolumeMethodRayCast: 
			{
				if(mIsInitialized) 
				{
					vtkImageData *input = vtkImageData::SafeDownCast(this->GetData());
					if(input != 0)
					{
						double *s = input->GetSpacing();
						mRaycastMapper->SetSampleDistance(s[2]*mResZ);
					}
				}
				mVolume->SetMapper(mRaycastMapper);
				break; 
			}
		case _VolumeMethod2DTexture:
			{
				mVolume->SetMapper(m2DTextureMapper);
				break; 
			}
		case _VolumeMethod3DTexture:
			{
#ifndef IVTK_4
				if(!mIsInitialized) 
				{
					vtkImageData *input = vtkImageData::SafeDownCast(this->GetData());
					if(input != 0)
					{
						double *s = input->GetSpacing();
						mRaycastMapper->SetSampleDistance(s[2]*mResZ);
					}
				}
				mVolume->SetMapper(m3DTextureMapper);
#else
				mMethod = mo;
#endif
				break; 
			}
		case _VolumeMethodVolumePro: 
			{
#ifdef VTK_USE_VOLUMEPRO
				mVolume->SetMapper(mVolumeProMapper);
#else
				mMethod = mo;
#endif
				break; 
			}
		default: 
			{
				mMethod = mo; 
			}
		}      			
		this->ClearCache();
	}
	return mMethod;
}


bool iVolumeViewSubject::HasVolumePro() const
{
#ifdef VTK_USE_VOLUMEPRO
	return true;
#else
	return false;
#endif
}


bool iVolumeViewSubject::Has3DTexture() const
{
#ifdef IVTK_4
	return false;
#else
	return m3DTextureMapper->IsRenderSupported(mVolume->GetProperty()) != 0;
#endif
}


void iVolumeViewSubject::SetCompositeMethod(int m) 
{
	if(m>=0 && m<2) 
	{
		mCompositeMethod = m;
		switch(m)
		{
		case 0:
			{ 
				mCF->SetCompositeMethodToInterpolateFirst();
				break;
			}
		case 1:
			{
				mCF->SetCompositeMethodToClassifyFirst();
				break;
			}
		}
		this->ClearCache();
	}
}


void iVolumeViewSubject::SetInterpolationType(int m) 
{
	if(m>=0 && m<2) 
	{
		mInterpolationType = m;
		switch(m)
		{
		case 0:
			{
				mVolume->GetProperty()->SetInterpolationTypeToNearest();
				break;
			}
		case 1:
			{
				mVolume->GetProperty()->SetInterpolationTypeToLinear();
				break;
			}
		}
		this->ClearCache();
	}
}


void iVolumeViewSubject::SetZResample(float m) 
{
	mResZ = m;

	double s = 0.01;
	if(mIsInitialized)
	{
		vtkImageData *input = vtkImageData::SafeDownCast(this->GetData());
		if(input != 0) s = input->GetSpacing()[2];
	}

	mRaycastMapper->SetSampleDistance(s*mResZ);
#ifndef IVTK_4
	m3DTextureMapper->SetSampleDistance(s*mResZ);
#endif
#ifdef VTK_USE_VOLUMEPRO
	mVolumeProMapper->SetSuperSamplingFactor(mResXY,mResXY,mResZ);
#endif

	this->ClearCache();
}


void iVolumeViewSubject::SetXYResample(float m) 
{
	mResXY = m;
	
	mRaycastMapper->SetImageSampleDistance(mResXY);
#ifdef VTK_USE_VOLUMEPRO
	mVolumeProMapper->SetSuperSamplingFactor(mResXY,mResXY,mResZ);
#endif

	this->ClearCache();
}


void iVolumeViewSubject::SetBlendMode(int m) 
{
	if(m<0 || m>1) return;
	
	mBlendMode = m;

	switch(mBlendMode)
	{
	case 0:
		{
			mRaycastMapper->SetVolumeRayCastFunction(mCF);
			break;
		}
	case 1:
		{
			mRaycastMapper->SetVolumeRayCastFunction(mMF);
			break;
		}
	}

#ifndef IVTK_4
	m3DTextureMapper->SetBlendMode(mBlendMode);
#endif
#ifdef VTK_USE_VOLUMEPRO
	mVolumeProMapper->SetBlendMode(mBlendMode);
#endif

	this->ClearCache();
}


bool iVolumeViewSubject::IsVisible() const
{
	return (mVolume->GetVisibility() == 1);
}


float iVolumeViewSubject::GetMemorySize() const
{
	float s = 0.0;
	s += mDataConverter->GetMemorySize();
	s += mDataReplicated->GetMemorySize();
	return s;
}
//
//  Two functions used in saving/restoring the state and in creating new instances with
//
void iVolumeViewSubject::ViewSubjectPackStateBody(iString &s) const
{
	this->PackValue(s,KeyVar(),this->GetVar());
	this->PackValue(s,KeyMethod(),mMethod);
	this->PackValue(s,KeyPalette(),mPalette);
	this->PackValue(s,KeyBlendMode(),mBlendMode);
	this->PackValue(s,KeyCompositeMethod(),mCompositeMethod);
	this->PackValue(s,KeyInterpolationType(),mInterpolationType);

	this->PackValue(s,KeyXYResample(),mResXY);
	this->PackValue(s,KeyZResample(),mResZ);

	this->PackValuePiecewiseFunction(s,KeyOpacityFunction(),mOF);
}


void iVolumeViewSubject::ViewSubjectUnPackStateBody(const iString &s)
{
	int i; float f;
		
	if(this->UnPackValue(s,KeyVar(),i)) this->SetVar(i);
	if(this->UnPackValue(s,KeyMethod(),i)) this->SetMethod(i);
	if(this->UnPackValue(s,KeyPalette(),i)) this->SetPalette(i);
	if(this->UnPackValue(s,KeyBlendMode(),i)) this->SetBlendMode(i);
	if(this->UnPackValue(s,KeyCompositeMethod(),i)) this->SetCompositeMethod(i);
	if(this->UnPackValue(s,KeyInterpolationType(),i)) this->SetInterpolationType(i);
	
	if(this->UnPackValue(s,KeyXYResample(),f)) this->SetXYResample(f);
	if(this->UnPackValue(s,KeyZResample(),f)) this->SetZResample(f);

	this->UnPackValuePiecewiseFunction(s,KeyOpacityFunction(),mOF); // it resets the function only if successfull
}


vtkImageData* iVolumeViewSubject::GetMapperInput()
{
	switch(mMethod) 
	{
	case _VolumeMethodRayCast:
		{
			return mRaycastMapper->GetInput();
		}
	case _VolumeMethod2DTexture:
		{
			return m2DTextureMapper->GetInput();
		}
	case _VolumeMethod3DTexture:
		{
#ifndef IVTK_4
			return m2DTextureMapper->GetInput();
#endif
		}
	case _VolumeMethodVolumePro: 
		{
#ifdef VTK_USE_VOLUMEPRO
			return mVolumeProMapper->GetInput();
#endif
		}
	}      			
	
	return 0;
}


iFunctionMapping* iVolumeViewSubject::GetOpacityFunction() const
{
	return mFunctionMappings;
}


bool iVolumeViewSubject::CanBeShown() const
{
	return (this->GetVar() < this->GetLimits()->GetNumVars());
}


void iVolumeViewSubject::ViewSubjectSyncWithData(const iDataSyncRequest &)
{
	vtkImageData *input = vtkImageData::SafeDownCast(this->GetData());
	if(input != 0)
	{
		double *s = input->GetSpacing();
		mRaycastMapper->SetSampleDistance(s[2]*mResZ);
	}

    mFunctionMappings->AttachToLimits(this->GetLimits(),this->GetVar());
}

