/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2006 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "iorthopolygonplanefilter.h"


#include "ierror.h"
#include "iorthoslicer.h"
#include "itune.h"
#include "ivtk.h"

#include <vtkCellArray.h>
#include <vtkFloatArray.h>
#include <vtkPointData.h>
#include <vtkPolyData.h>
#include <vtkStructuredPoints.h>

//
//  Templates
//
#include "igenericfiltertemplate.h"


#ifndef IVTK_QUAD_ORDER
#error Misconfiguration: missing IVTK_QUAD_ORDER define.
!TERMINATE!
#endif


iOrthoPolygonPlaneFilter::iOrthoPolygonPlaneFilter(iViewSubject *vo) : iGenericFilter<vtkStructuredPointsToPolyDataFilter,vtkImageData,vtkPolyData>(vo,1,true,false)
{
	mCorrectFlat = false;

	mQuadMode = 1;
	mOldAxis = 0; 
	mOldDim[0] = mOldDim[1] = mOldDim[2] = 0;

	mNewStrips = vtkCellArray::New(); IERROR_ASSERT(mNewStrips);
	mNewQuads = vtkCellArray::New(); IERROR_ASSERT(mNewQuads);
}


iOrthoPolygonPlaneFilter::~iOrthoPolygonPlaneFilter()
{
	mNewStrips->Delete();
	mNewQuads->Delete();
}


void iOrthoPolygonPlaneFilter::SetCorrectForFlatShading(bool s)
{
	mCorrectFlat = s;
	this->Modified();
}


void iOrthoPolygonPlaneFilter::ProduceOutput()
{
	int dim[3];
	double org[3], spa[3];

	vtkImageData *input = this->GetInput();
	vtkPolyData *output = this->GetOutput();

	output->Initialize();

	input->GetOrigin(org);
    input->GetSpacing(spa);
    input->GetDimensions(dim);

	int i, j, Axis = -1;
	for(i=0; i<3; i++)
	{
		if(dim[i] == 1) Axis = i;
	}
	if(Axis == -1) return;

    int Uidx, Vidx;
	iOrthoSlicer::GetUV(Axis,Uidx,Vidx);

	vtkIdType numPts = (vtkIdType)dim[Uidx]*dim[Vidx];
    vtkPoints *newPts = vtkPoints::New(VTK_FLOAT); IERROR_ASSERT(newPts);
    newPts->SetNumberOfPoints(numPts);
    
    int ijk[3];
	float xyz[3];
	xyz[Axis] = org[Axis];
	float *p = (float *)newPts->GetVoidPointer(0);
	for(ijk[Vidx]=0; ijk[Vidx]<dim[Vidx]; ijk[Vidx]++)
	{
		this->UpdateProgress((float)ijk[Vidx]/dim[Vidx]);
		if(this->GetAbortExecute()) break;
				
		xyz[Vidx] = org[Vidx] + spa[Vidx]*ijk[Vidx];
		for (ijk[Uidx]=0; ijk[Uidx]<dim[Uidx]; ijk[Uidx]++)
		{
			xyz[Uidx] = org[Uidx] + spa[Uidx]*ijk[Uidx];
			for(j=0; j<3; j++) p[j+3*(ijk[Uidx]+ijk[Vidx]*dim[Uidx])] = xyz[j];
		}
	}

	if(Axis!=mOldAxis || dim[0]!=mOldDim[0] || dim[1]!=mOldDim[1] || dim[2]!=mOldDim[2]) 
	{
		if (mQuadMode == 0)
		{
			mNewStrips->Delete();
			mNewStrips = vtkCellArray::New(); IERROR_ASSERT(mNewStrips);
			mNewStrips->Allocate(mNewStrips->EstimateSize(dim[Vidx]-1,2*dim[Uidx]));
			for (int ii=0;ii<dim[Vidx]-1;ii++)
			{
				this->UpdateProgress((float)ii/(dim[Vidx]-0.99));
				if(this->GetAbortExecute()) break;

				mNewStrips->InsertNextCell(2*dim[Uidx]);
				for (int jj=0;jj<dim[Uidx];jj++)
				{ 
					mNewStrips->InsertCellPoint(jj+ii*dim[Uidx]);
					mNewStrips->InsertCellPoint(jj+(ii+1)*dim[Uidx]);
				}
			}
		}
		else 
		{
			mNewQuads->Delete();
			mNewQuads = vtkCellArray::New(); IERROR_ASSERT(mNewQuads);
			mNewQuads->Allocate(mNewQuads->EstimateSize(dim[Vidx]-1,4*(dim[Uidx]-1)));
			for (int ii=0; ii<dim[Vidx]-1; ii++)
			{
				this->UpdateProgress((float)ii/(dim[Vidx]-0.99));
				if(this->GetAbortExecute()) break;

				for (int jj=0; jj<dim[Uidx]-1; jj++)
				{ 
					mNewQuads->InsertNextCell(4);
					if(IVTK_QUAD_ORDER == 0) mNewQuads->InsertCellPoint(jj+ii*dim[Uidx]);
					mNewQuads->InsertCellPoint(jj+(ii+1)*dim[Uidx]);
					mNewQuads->InsertCellPoint((jj+1)+(ii+1)*dim[Uidx]);
					mNewQuads->InsertCellPoint((jj+1)+ii*dim[Uidx]);
					if(IVTK_QUAD_ORDER == 3) mNewQuads->InsertCellPoint(jj+ii*dim[Uidx]);
				}
			}
		}
    }
    
	if (mQuadMode == 0)
	{
		output->SetStrips(mNewStrips);
	}
	else 
	{
		output->SetPolys(mNewQuads);
	}

	mOldAxis = Axis;
	mOldDim[0] = dim[0];
	mOldDim[1] = dim[1];
	mOldDim[2] = dim[2];

    output->SetPoints(newPts);
    newPts->Delete();

	//
	//  Correct scalars for flat shading to preserve symmetry - for quads only
	//
	if(mCorrectFlat && mQuadMode!=0)
	{
		if(!this->ScalarsInit(input->GetPointData())) return;
		
		int k;
		float *pl, *ql;
		vtkIdType du = dim[Uidx];
		for(j=0; j<dim[Vidx]-1; j++)
		{
			pl = wScalarPtrIn + wScalarDimIn*j*du;
			ql = wScalarPtrOut + wScalarDimOut*j*du;
			for(i=0; i<du-1; i++)
			{
				for(k=0; k<wScalarDimIn; k++)
				{
					ql[k] = 0.25*(pl[k]+pl[wScalarDimIn+k]+pl[wScalarDimIn*du+k]+pl[wScalarDimIn*(du+1)+k]);
				}
				pl += wScalarDimIn;
				ql += wScalarDimOut;
			}
		}

		this->ScalarsDone(output->GetPointData());
	}
	else
	{
		output->GetPointData()->SetScalars(input->GetPointData()->GetScalars());
	}
}

