/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2006 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "ieventobserver.h"


#include "icontrolmodule.h"
#include "idatareader.h"
#include "ierror.h"
#include "imarker.h"
#include "imarkerfamily.h"
#include "iparallel.h"
#include "ipicker.h"
#include "ishellfactory.h"
#include "iviewmodule.h"
#include "iviewobjectfamily.h"

#include <vtkCriticalSection.h>
#include <vtkObject.h>
#include <vtkPointWidget.h>
#include <vtkProcessObject.h>
#include <vtkRenderWindow.h>
#include <vtkTimerLog.h>

//
//  Templates
//
#include "iarraytemplate.h"


bool iEventObserver::mBlock = false;
bool iAbortRenderEventObserver::mBlockThis = false;

//
//  Parent class for all event observers
//
iEventObserver::iEventObserver()
{
	//
	//  Use this mutex to support multi-threaded execution
	//
	mMutex = vtkCriticalSection::New(); IERROR_ASSERT(mMutex);
}


iEventObserver::~iEventObserver()
{
	mMutex->Delete();
}


//
//  iProgressEventObserver class
//
iProgressEventObserver* iProgressEventObserver::New(iViewModule *vm)
{
	return iPointerCast<iProgressEventObserver,iEventObserver>(iShellFactory::CreateEventObserver("Progress",vm));
}


iProgressEventObserver::iProgressEventObserver(iViewModule *vm) : iViewModuleComponent(vm)
{
	mAborted = false;
	mMode = _Default;
}


void iProgressEventObserver::Execute(vtkObject *, unsigned long event, void *callData)
{
	if(mBlock) return;

	switch(event)
	{
	case vtkCommand::StartEvent:
		{
			this->Started(mMode);
			break;
		}
	case vtkCommand::ProgressEvent:
		{
			float f;
			if(callData == 0) f = 0.0; else f = *((float *)callData);
			this->SetProgress(f);
			break;
		}
	case vtkCommand::EndEvent:
		{
			this->Finished();
			break;
		}
	}
}


void iProgressEventObserver::Started(ProgressMode m)
{
	mMode = m;
	mAborted = false;
	mMutex->Lock();
	this->ChildStarted();
	mMutex->Unlock();
}


void iProgressEventObserver::SetProgress(float f)
{
	mAborted = this->CheckAbort();
	if(!mAborted)
	{
		mMutex->Lock();
		this->SetProgressBody(f);
		mMutex->Unlock();
	}
}


void iProgressEventObserver::Finished()
{
	mMutex->Lock();
	this->ChildFinished();
	mMutex->Unlock();
}


void iProgressEventObserver::SetMode(ProgressMode m)
{
	mMode = m;
}


//
//  iAbortRenderEventObserver class
//
iAbortRenderEventObserver* iAbortRenderEventObserver::New(iViewModule *vm)
{
	return iPointerCast<iAbortRenderEventObserver,iEventObserver>(iShellFactory::CreateEventObserver("AbortRender",vm));
}


iAbortRenderEventObserver::iAbortRenderEventObserver(iViewModule *vm) : iViewModuleComponent(vm)
{
	mCancelled = mInProgress = false;
	mRenderTimer = vtkTimerLog::New(); IERROR_ASSERT(mRenderTimer);
	mRenderTime = 0.0f;
}


iAbortRenderEventObserver::~iAbortRenderEventObserver()
{
	mRenderTimer->Delete();
}


void iAbortRenderEventObserver::Execute(vtkObject *caller, unsigned long event, void *)
{
	if(mBlock || mBlockThis) return;

//	mMutex->Lock();

	switch(event)
	{
	case vtkCommand::StartEvent:
		{
			if(!mInProgress)
			{
				mInProgress = true;
				this->Started();
				mCancelled = false;
				mRenderTimer->StartTimer();
			}
			break;
		}
	case vtkCommand::AbortCheckEvent:
	case vtkCommand::ProgressEvent:
		{
			if(mInProgress)
			{
				mCancelled = this->CheckAbort();
				if(mCancelled && caller!=0)
				{
					if(caller->IsA("vtkRenderWindow") == 1) 
					{
						iRequiredCast<vtkRenderWindow>(caller)->SetAbortRender(1);
					}
					if(caller->IsA("vtkProcessObject") == 1) 
					{
						iRequiredCast<vtkProcessObject>(caller)->SetAbortExecute(1);
					}
				}
			}
			break;
		}
	case vtkCommand::EndEvent:
		{
			if(mInProgress)
			{
				mRenderTimer->StopTimer();
				mRenderTime = mRenderTimer->GetElapsedTime();
				mInProgress = false;
				this->Finished();
				if(mCancelled)
				{
					//
					//  If rendering was cancelled, then different parts of the pipeline
					//  remain in different stages of completion, which confuses VTK.
					//  To avoid that we modify the time stamp of the data, so that next
					//  rendering starts from scratch. It is not an elegant solution,
					//  but it works.
					//
					this->GetViewModule()->GetReader()->ResetPipeline();
				}
			}
			break;
		}
	case vtkCommand::UserEvent:
		{
			this->PostFinished();
			break;
		}
	}

//	mMutex->Unlock();
}


//
//  iSlaveAbortRenderEventObserver class
//
iSlaveAbortRenderEventObserver* iSlaveAbortRenderEventObserver::New(iAbortRenderEventObserver *master)
{
	return new iSlaveAbortRenderEventObserver(master);
}


iSlaveAbortRenderEventObserver::iSlaveAbortRenderEventObserver(iAbortRenderEventObserver *m) : iViewModuleComponent(m->GetViewModule())
{
	mMaster = m;
	mMaster->Register(this);
}


iSlaveAbortRenderEventObserver::~iSlaveAbortRenderEventObserver()
{
	mMaster->UnRegister();
}


void iSlaveAbortRenderEventObserver::Execute(vtkObject *caller, unsigned long event, void *)
{
	if(mBlock || iAbortRenderEventObserver::AbortRenderEventObserversBlocked()) return;

	switch(event)
	{
	case vtkCommand::AbortCheckEvent:
	case vtkCommand::ProgressEvent:
		{
			if(mMaster!=0 && mMaster->IsCancelled())
			{
				if(caller!=0 && caller->IsA("vtkProcessObject")) 
				{
					iRequiredCast<vtkProcessObject>(caller)->SetAbortExecute(1);
				}
			}
			break;
		}
	}
}


//
//  iPickEventObserver class
//
iPickEventObserver* iPickEventObserver::New(iViewModule *vm)
{
	return iPointerCast<iPickEventObserver,iEventObserver>(iShellFactory::CreateEventObserver("Pick",vm));
}


iPickEventObserver::iPickEventObserver(iViewModule *vm) : iViewModuleComponent(vm)
{
}


void iPickEventObserver::Execute(vtkObject *caller, unsigned long eventId, void *)
{
	if(mBlock) return;

	switch (eventId)
	{
	case vtkCommand::StartPickEvent: 
		{ 
			if(caller!=0 && caller->IsA("iPicker")) 
			{
				this->Started();
			}
			break; 
		}
	case vtkCommand::EndPickEvent: 
		{ 
			if(caller!=0 && caller->IsA("iPicker")) 
			{
				iPicker *p = iPicker::SafeDownCast(caller);
				if(p != 0)
				{
					p->UpdateReport(); 
					this->Finished(p);
				}
			}
			break; 
		}
	}
}


//
//  iRecordEventObserver class
//
iRecordEventObserver* iRecordEventObserver::New(iViewModule *vm)
{
	return iPointerCast<iRecordEventObserver,iEventObserver>(iShellFactory::CreateEventObserver("Record",vm));
}


iRecordEventObserver::iRecordEventObserver(iViewModule *vm) : iViewModuleComponent(vm)
{
}


void iRecordEventObserver::Execute(vtkObject *, unsigned long eventId, void *)
{
	if(mBlock) return;
	if(eventId == vtkCommand::InteractionEvent) this->UpdateInteraction(); 
}


//
//  iParallelUpdateEventObserver class
//
iParallelUpdateEventObserver* iParallelUpdateEventObserver::New(iControlModule *cm)
{
	return iPointerCast<iParallelUpdateEventObserver,iEventObserver>(iShellFactory::CreateEventObserver("ParallelUpdate",cm));
}


iParallelUpdateEventObserver::iParallelUpdateEventObserver(iControlModule *cm) : iControlModuleComponent(cm)
{
}


void iParallelUpdateEventObserver::Execute(vtkObject *, unsigned long eventId, void *)
{
	if(mBlock) return;
	if(eventId == iParallel::InformationEvent) this->UpdateInformation(); 
}


//
//  iAnimatorEventObserver class
//
iAnimatorEventObserver* iAnimatorEventObserver::New(iViewModule *vm)
{
	return iPointerCast<iAnimatorEventObserver,iEventObserver>(iShellFactory::CreateEventObserver("Animator",vm));
}


iAnimatorEventObserver::iAnimatorEventObserver(iViewModule *vm) : iViewModuleComponent(vm)
{
}


void iAnimatorEventObserver::Execute(vtkObject *, unsigned long eventId, void *)
{
	if(mBlock) return;
	if(eventId == 0UL) this->OnCameraPath(0);
}


//
//  iMarkerEventObserver class
//
iMarkerEventObserver* iMarkerEventObserver::New(iViewModule *vm)
{
	return new iMarkerEventObserver(vm);
}


iMarkerEventObserver::iMarkerEventObserver(iViewModule *vm) : iViewModuleComponent(vm)
{
}


void iMarkerEventObserver::Execute(vtkObject *caller, unsigned long eventId, void *)
{
	if(mBlock) return;

	switch (eventId)
	{
	case vtkCommand::InteractionEvent:
		{
			if(caller->IsA("vtkPointWidget"))
			{
				double *x = ((vtkPointWidget *)caller)->GetPosition();
				iPosition p(this->GetViewModule());
				p = x;
				this->GetViewModule()->GetMarkerFamily()->GetCurrentMember()->SetPosition(p);
			}
		}
	}
}

