/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2006 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "iarray.h"


#include "ierror.h"


//
//  iArray class
//
template<class T>
iArray<T>::iArray(const T &n, int inc)
{
	this->mNull = n;
	this->mNum = 0;
	this->mInc = (inc<1) ? 1 : inc;
	this->mLen = this->mInc;
	this->mArr = new T[this->mLen]; IERROR_ASSERT(this->mArr);
}


template<class T>
iArray<T>::~iArray()
{
	delete [] this->mArr;
}


template<class T>
T& iArray<T>::operator[](int i)
{
	if(i>=0 && i<this->mNum) return this->mArr[i]; else
	{
		IERROR_FINISH("Accessing iArray outside of boundaries.");
		return this->mNull;
	}
}


template<class T>
const T& iArray<T>::operator[](int i) const
{
	if(i>=0 && i<this->mNum) return this->mArr[i]; else return this->mNull;
}


template<class T>
void iArray<T>::Extend(int len)
{
	int i, n, inc = this->mLen/4;
	if(inc+this->mLen < len) inc = len - this->mLen;
	this->mLen += (inc<this->mInc) ? this->mInc : inc;

	T *tmp, *arr;
	tmp = new T[this->mLen]; IERROR_ASSERT(tmp);

	n = this->mNum;
	arr = this->mArr;  // dereference for speed
	for(i=0; i<n; i++) tmp[i] = arr[i]; // we cannot use memcpy here because = is not necessarily a simple memory copy
	delete [] this->mArr;
	this->mArr = tmp;
}


template<class T>
void iArray<T>::Add(const T &val)
{
	if(this->mNum == this->mLen) this->Extend(0);
	this->mArr[this->mNum++] = val;
}


template<class T>
void iArray<T>::Remove(int n)
{
	if(n<0 || n>=this->mNum)
	{
		IERROR_FINISH("Accessing iArray outside of boundaries.");
	}

	int i, m = this->mNum-1;
	T *arr = this->mArr;  // dereference for speed
	for(i=n; i<m; i++) arr[i] = arr[i+1];
	arr[--this->mNum] = this->mNull;
}


template<class T>
T iArray<T>::RemoveLast()
{
	if(this->mNum == 0)
	{
		IERROR_FINISH("Accessing iArray outside of boundaries.");
		return this->mNull;
	}

	this->mNum--;
	return this->mArr[this->mNum];
}


template<class T>
void iArray<T>::Clear()
{
	this->mNum = 0;
}


template<class T>
void iArray<T>::Resize(int n)
{
	if(n > this->mLen) this->Extend(n);
	if(n > this->mNum) this->mNum = n;
	if(n < this->mNum)
	{
		if(n < 0) this->mNum = 0; else this->mNum = n;
	}
}


//
//  iSearchableArray class
//
template<class T>
iSearchableArray<T>::iSearchableArray(const T &n, int inc) : iArray<T>(n,inc)
{
}


template<class T>
int iSearchableArray<T>::Find(const T &val) const
{
	int i, n = this->mNum;
	T *arr = this->mArr;  // dereference for speed
	for(i=0; i<n; i++) if(arr[i] == val) return i;
	return -1;
}


template<class T>
void iSearchableArray<T>::AddUnique(const T &val)
{
	if(this->Find(val) == -1) this->Add(val);
}


template<class T>
bool iSearchableArray<T>::Remove(const T &val)
{
	int i = this->Find(val);
	if(i > -1)
	{
		this->iArray<T>::Remove(i);
		return true;
	}
	else return false;
}


//
//  iOrderedArray class
//
template<class T>
iOrderedArray<T>::iOrderedArray(const T &n, int inc) : iSearchableArray<T>(n,inc)
{
}


template<class T>
int iOrderedArray<T>::Find(const T &val) const
{
	if(this->mNum == 0) return -1;  // empty array

	T *arr = this->mArr;  // dereference for speed
	//
	//  Binary search of the ordered array
	//
	int ia, ib, ic;
	ia = 0;
	ib = this->mNum - 1;
	while(ib > ia+1)
	{
		ic = (ia+ib)/2;
		if(this->Less(arr[ic],val)) ia = ic; else if(this->Less(val,arr[ic])) ib = ic; else return ic;
	}
	if(arr[ia] == val) return ia; else if(arr[ib] == val) return ib; else return -1;
}


template<class T>
void iOrderedArray<T>::Add(const T &val)
{
	int i, ia, ib;

	if(this->mNum == this->mLen) this->Extend(0);
	T *arr = this->mArr;  // dereference for speed

	if(this->mNum==0 || arr[this->mNum-1]<val)
	{
		arr[this->mNum++] = val;
		return;
	}

	if(this->Less(val,arr[0]))
	{
		ib = 0;
	}
	else
	{
		ia = 0;
		ib = this->mNum - 1;
		while(ib > ia+1)
		{
			i = (ia+ib)/2;
			if(this->Less(arr[i],val)) ia = i; else if(this->Less(val,arr[i])) ib = i; else ia = ib = i;
		}
	}
	for(i=this->mNum; i>ib; i--) arr[i] = arr[i-1]; 
	arr[ib] = val;
	this->mNum++;
}

