/*
 * gbpserial.c: communicate with a GemPC410 smart card reader
 * using GemCore protocol
 * Copyright (C) 2001,2002 Ludovic Rousseau
 * 
 * This program is free software; you can redistribute it and/or modify it 
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2 of the License, or (at your
 * option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License along 
 * with this program; if not, write to the Free Software Foundation, Inc., 
 * 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA 
 */

/*
 * $Id: gbpserial.c,v 1.13 2002/03/10 23:09:48 rousseau Exp $
 */

#include <stdio.h>
#include <stdlib.h>
#include <fcntl.h>
#include <unistd.h>
#include <termios.h>
#include <string.h>
#include <errno.h>

#include "pcscdefines.h"
#include "GemCore.h"
#include "gbpserial.h"
#include "Config.h"
#include "GCdebug.h"
#include "GCCmds.h"

/*
 * GBP (Gemplus Bloc Protocol) format
 *
 * +-----+-----+-----+---------+-----+
 * | NAD | PCB | LEN | DAT ... | EDC |
 * +-----+-----+-----+---------+-----+
 *
 * NAD: source and destination indentifier
 * PCB: block type
 *  I-Block (Information)
 * bit 7   6   5    4-0
 *   +---+---+---+--------+
 *   | 0 | S | 0 | unused |
 *   +---+---+---+--------+
 *   S = Sequence bit
 *
 *  R-Block (Repeat)
 * bit 7   6   5   4   3   2   1   0
 *   +---+---+---+---+---+---+---+---+
 *   | 1 | 0 | 0 | S | 0 | 0 | E | V |
 *   +---+---+---+---+---+---+---+---+
 *   V = Error verified by EDC
 *   E = Another error
 *
 *  S-Block (Synchro)
 * bit 7   6   5   4   3   2   1   0
 *   +---+---+---+---+---+---+---+---+
 *   | 1 | 1 | R | 0 | 0 | 0 | 0 | 0 |
 *   +---+---+---+---+---+---+---+---+
 *   R = Resync
 *     0 = request
 *     1 = response
 *
 * LEN: number of bytes in DAT
 * DAT: data transmitted
 * EDC: xor on NAD, PCB, LEN and DAT bytes
 *
 */

// indexes in GbpBuffer[]
#define NAD 0
#define PCB 1
#define LEN 2
#define DAT 3

// NAD values
#define NAD_HOST2IFD 0x42
#define NAD_IFD2HOST 0x24

/*
 * Maximum number of readers handled 
 */
#define IFDH_MAX_READERS 4

/*
 * Maximum number of slots per reader handled 
 */
#define IFDH_MAX_SLOTS 2

/*
 * Global variables containing the file handle of the serial port
 */
typedef struct {
	int fd;
	char bSeq;
} intrFace;

static int iInitialized = FALSE;
static intrFace gbpDevice[PCSCLITE_MAX_CHANNELS];

// complete buffer + 3 bytes used by GBP
static UCHAR GbpBuffer[CMD_BUF_SIZE + 3];

int explain_gbp(const unsigned char buffer[], const int rv);

/*****************************************************************************
 * 
 *				WriteGBP: Send bytes to the card reader
 *
 * remark: buffer already contains the byte length in [0]
 *
 *****************************************************************************/
RESPONSECODE WriteGBP(DWORD lun, DWORD length, PUCHAR buffer)
{
	int rv, len, i;
	UCHAR edc;

	// PDU length = length + NAD + PCB + LEN
	len = length + 3;

	// NAD byte
	GbpBuffer[NAD] = NAD_HOST2IFD;

	// PCB byte
	GbpBuffer[PCB] = (gbpDevice[lun].bSeq << 6);

	// copy LEN and DATA
	memcpy(GbpBuffer + LEN, buffer, length);

	// checksum
	edc = 0;
	for (i = 0; i < len - 1; i++)
		edc ^= GbpBuffer[i];

	// EDC byte
	GbpBuffer[len - 1] = edc;

	// increment Seq: 0 -> 1 or 1 -> 0
	gbpDevice[lun].bSeq = (gbpDevice[lun].bSeq + 1) % 2;

#ifdef DEBUG_LEVEL_COMM
	DEBUG_XXD("-> ", GbpBuffer, len);
#endif
	rv = write(gbpDevice[lun].fd, GbpBuffer, len);

	if (rv < 0)
	{
		DEBUG_CRITICAL2("WriteGBP: write error: %s", strerror(errno));
		return STATUS_UNSUCCESSFUL;
	}

	return STATUS_SUCCESS;
} /* WriteGBP */


/*****************************************************************************
 * 
 *				ReadGBP: Receive bytes from the card reader
 *
 * remark: buffer contains the byte length in [0]
 *
 *****************************************************************************/
RESPONSECODE ReadGBP(DWORD lun, PDWORD length, PUCHAR buffer)
{
	int rv, len, to_read, already_read;
	int i, pcb, edc;
	DWORD buffer_size;

	len = sizeof(GbpBuffer);
	buffer_size = *length;

	// error by default
	*length = 0;

	rv = read(gbpDevice[lun].fd, GbpBuffer, len);
	if (rv < 0)
		return STATUS_UNSUCCESSFUL;

	// too short
	if (rv < 4)
	{
#ifdef DEBUG_LEVEL_COMM
		DEBUG_XXD("<- ", GbpBuffer, rv);
		explain_gbp(GbpBuffer, rv);
#endif
		DEBUG_COMM2("ReadGBP: only %d byte(s) read", rv);

		return STATUS_UNSUCCESSFUL;
	}

	// NAD byte
	if (GbpBuffer[NAD] != NAD_IFD2HOST)
	{
#ifdef DEBUG_LEVEL_COMM
		DEBUG_XXD("<- ", GbpBuffer, rv);
		explain_gbp(GbpBuffer, rv);
#endif
		DEBUG_COMM2("ReadGBP: wrond NAD byte %02X", GbpBuffer[NAD]);

		return STATUS_UNSUCCESSFUL;
	}

	// PCB byte
	pcb = GbpBuffer[PCB];

	if (pcb & 0xA0)
	{
#ifdef DEBUG_LEVEL_COMM
		DEBUG_XXD("<- ", GbpBuffer, rv);
		explain_gbp(GbpBuffer, rv);
#endif
		DEBUG_COMM("ReadGBP: PCB error");

		return STATUS_PCB_ERROR;
	}

	// LEN byte (+4 bytes of protocol)
	to_read = GbpBuffer[LEN] + 4;

	// Nb of bytes already read
	already_read = rv;

	// Nb of bytes until the end of the buffer
	len -= rv;

	while (already_read < to_read)
	{
		rv = read(gbpDevice[lun].fd, GbpBuffer + already_read, len);
		if (rv < 0)
		{
			DEBUG_COMM2("ReadGBP: read error: %s", strerror(errno));
			return STATUS_UNSUCCESSFUL;
		}

		already_read += rv;
		len -= rv;
	}

	// calculate checksum
	edc = 0;
	for (i = 0; i < to_read; i++)
		edc ^= GbpBuffer[i];

#ifdef DEBUG_LEVEL_COMM
	DEBUG_XXD("<- ", GbpBuffer, to_read);
#endif

	// verify checksum
	if (edc != 0)
	{
		DEBUG_COMM("ReadGBP: wrong EDC");
		return STATUS_UNSUCCESSFUL;
	}
	// copy up to buffer_size bytes
	if (buffer_size > GbpBuffer[LEN])
		*length = GbpBuffer[LEN] + 1;
	else
		*length = buffer_size;

	memcpy(buffer, GbpBuffer + LEN, *length);

	return STATUS_SUCCESS;
} /* ReadGBP */


/*****************************************************************************
 * 
 *				OpenGBP: open the port
 *
 *****************************************************************************/
RESPONSECODE OpenGBP(DWORD lun, DWORD channel)
{
	char dev_name[FILENAME_MAX];
	struct termios current_termios;
	DWORD ctn, slot;
	int ospeed;

	ctn = ((unsigned short) (lun & 0x0000FFFF)) % IFDH_MAX_READERS;
	slot = ((unsigned short) (lun >> 8)) % IFDH_MAX_SLOTS;

	DEBUG_COMM3("OpenGBP: Lun: %d, Channel: %d", lun, channel);

	/*
	 * Conversion of old-style ifd-hanler 1.0 CHANNELID 
	 */
	if (channel == 0x0103F8)
		channel = 1;
	else
		if (channel == 0x0102F8)
			channel = 2;
		else
			if (channel == 0x0103E8)
				channel = 3;
			else
				if (channel == 0x0102E8)
					channel = 4;

	if (iInitialized == FALSE)
	{
		int i;

		for (i=0; i<PCSCLITE_MAX_CHANNELS; i++)
		{
			gbpDevice[i].fd = -1;
			gbpDevice[i].bSeq = -1;
		}

		iInitialized = TRUE;
	}

	if (channel < 0)
	{
		DEBUG_CRITICAL2("wrong port number: %d", (int) channel);
		return STATUS_UNSUCCESSFUL;
	}

	sprintf(dev_name, "/dev/pcsc/%d", (int) channel);

	gbpDevice[lun].fd = open(dev_name, O_RDWR | O_NOCTTY);

	if (gbpDevice[lun].fd <= 0)
	{
		DEBUG_CRITICAL3("open %s: %s", dev_name, sys_errlist[errno]);
		return STATUS_UNSUCCESSFUL;
	}

	/* empty in and out serial buffers */
	if (tcflush(gbpDevice[lun].fd, TCIOFLUSH))
			DEBUG_INFO2("tcflush() function error: %s", sys_errlist[errno]);

	/* get config attributes */
	if (tcgetattr(gbpDevice[lun].fd, &current_termios) == -1)
	{
		DEBUG_INFO2("tcgetattr() function error: %s", sys_errlist[errno]);
		close(gbpDevice[lun].fd);
		gbpDevice[lun].fd = -1;

		return STATUS_UNSUCCESSFUL;
	}

	// actual serial port baud rate
	ospeed = cfgetospeed(&current_termios);

	current_termios.c_iflag = 0;
	current_termios.c_oflag = 0;	// Raw output modes
	current_termios.c_cflag = 0;	// Raw output modes

	// Do not echo characters because if you connect
	// to a host it or your
	// modem will echo characters for you.
	// Don't generate signals.
	current_termios.c_lflag = 0;

	// control flow - enable receiver - ignore modem
	// control lines
	current_termios.c_cflag = CREAD | CLOCAL;

	current_termios.c_cflag |= CS8;

	// This is Non Canonical mode set.
	// by setting c_cc[VMIN]=0 and c_cc[VTIME] = 10
	// each read operation will wait for 10*1/10 = 1 sec
	// If single byte is received, read function returns.
	// if timer expires, read() returns 0.
	// Minimum bytes to read
	current_termios.c_cc[VMIN] = 0;
	// Time between two bytes read (VTIME x 0.10 s)
	// 10 seconds may not be the correct maximum value
	// If the card becomes mute the reader will tell us
	// If the reader becomes mute (crash or serial comm pb.) we won't know
	current_termios.c_cc[VTIME] = 100;

	// The serial port is not yet configured to 38400 bauds?
	printf("%d %d\n", ospeed, B38400);
	if (ospeed != B38400)
	{
		// default speed of GemPC410 is 9600 bauds
		cfsetospeed(&current_termios, B9600);
		cfsetispeed(&current_termios, 0); /* 0 corresponds to output speed */

		DEBUG_INFO("Set serial port baudrate to 9600");
		if (tcsetattr(gbpDevice[lun].fd, TCSANOW, &current_termios) == -1)
		{
			close(gbpDevice[lun].fd);
			gbpDevice[lun].fd = -1;
			DEBUG_INFO2("tcsetattr error: %s", sys_errlist[errno]);

			return STATUS_UNSUCCESSFUL;
		}

		// Configure baud rate
		if (GCCmdConfigureSIOLine(lun, 38400) != STATUS_SUCCESS)
		{
			DEBUG_CRITICAL("GCCmdConfigureSIOLine failed");
			GCCmdConfigureSIOLine(lun, 38400);
			// return STATUS_UNSUCCESSFUL;
		}

		// actual speed is 38400 bauds
		cfsetospeed(&current_termios, B38400);
		cfsetispeed(&current_termios, 0); /* 0 corresponds to output speed */

		DEBUG_INFO("Set serial port baudrate to 38400");
		if (tcsetattr(gbpDevice[lun].fd, TCSANOW, &current_termios) == -1)
		{
			close(gbpDevice[lun].fd);
			gbpDevice[lun].fd = -1;
			DEBUG_INFO2("tcsetattr error: %s", sys_errlist[errno]);

			return STATUS_UNSUCCESSFUL;
		}
	}
	else
	{
		DEBUG_INFO("Serial port baudrate alread set to 38400");

		// speed is 38400 bauds
		cfsetospeed(&current_termios, B38400);
		cfsetispeed(&current_termios, 0); /* 0 corresponds to output speed */

		DEBUG_INFO("Set serial port timeout and other config");
		if (tcsetattr(gbpDevice[lun].fd, TCSANOW, &current_termios) == -1)
		{
			close(gbpDevice[lun].fd);
			gbpDevice[lun].fd = -1;
			DEBUG_INFO2("tcsetattr error: %s", sys_errlist[errno]);

			return STATUS_UNSUCCESSFUL;
		}
	}

	return STATUS_SUCCESS;
} /* OpenGBP */


/*****************************************************************************
 * 
 *				CloseGBP: close the port
 *
 *****************************************************************************/
RESPONSECODE CloseGBP(DWORD lun)
{
	close(gbpDevice[lun].fd);

	gbpDevice[lun].fd = -1;
	gbpDevice[lun].bSeq = -1;

	return STATUS_SUCCESS;
} /* CloseGBP */


/*****************************************************************************
 * 
 *				CloseGBP: close the port
 *
 *****************************************************************************/
int explain_gbp(const unsigned char buffer[], const int rv)
{
	int ret = 0;

	if (rv < 4)
	{
		DEBUG_COMM2("GBP bloc too short: %d instead of min 4", rv);
		return STATUS_DEVICE_PROTOCOL_ERROR;
	}

	switch (buffer[0])
	{
		case 0x42:
			DEBUG_COMM("  NAD: host to gemcore");
			break;
		case 0x24:
			DEBUG_COMM("  NAD: gemcore to host");
			break;
		default:
			DEBUG_COMM2("  NAD: UNKNOWN value %02X", buffer[0]);
	}

	switch (buffer[1])
	{
		case 0x00:
			DEBUG_COMM("  PCB: I-block S=0");
			break;
		case 0x40:
			DEBUG_COMM("  PCB: I-block S=1");
			break;
		case 0x80:
			DEBUG_COMM("  PCB: R-block S=0, EDC error=0, another error=0");
			ret = STATUS_PCB_REP_0;
			break;
		case 0x90:
			DEBUG_COMM("  PCB: R-block S=1, EDC error=0, another error=0");
			ret = STATUS_PCB_REP_1;
			break;
		case 0x81:
			DEBUG_COMM("  PCB: R-block S=0, EDC error=1, another error=0");
			ret = STATUS_PCB_REP_0;
			break;
		case 0x91:
			DEBUG_COMM("  PCB: R-block S=1, EDC error=1, another error=0");
			ret = STATUS_PCB_REP_1;
			break;
		case 0x82:
			DEBUG_COMM("  PCB: R-block S=0, EDC error=0, another error=1");
			ret = STATUS_PCB_REP_0;
			break;
		case 0x92:
			DEBUG_COMM("  PCB: R-block S=1, EDC error=0, another error=1");
			ret = STATUS_PCB_REP_1;
			break;
		case 0x83:
			DEBUG_COMM("  PCB: R-block S=0, EDC error=1, another error=1");
			ret = STATUS_PCB_REP_0;
			break;
		case 0x93:
			DEBUG_COMM("  PCB: R-block S=1, EDC error=1, another error=1");
			ret = STATUS_PCB_REP_1;
			break;
		case 0xB0:
			DEBUG_COMM("  PCB: S-block resynch request");
			ret = STATUS_PCB_SYNC_REQ;
			break;
		case 0xE0:
			DEBUG_COMM("  PCB: S-block resynch response");
			ret = STATUS_PCB_SYNC_RESP;
			break;
		default:
			DEBUG_COMM2("  PCB: UNKNOWN value %02X", buffer[1]);
			break;
	}

	DEBUG_COMM2("  LEN: %02X bytes", buffer[2]);

	if (rv > 4)
		switch (buffer[3])
		{
			case 0x00:
				DEBUG_COMM("  S: no error");
				break;
			case 0x01:
				DEBUG_COMM("  S: unknown driver");
				break;
			case 0x02:
				DEBUG_COMM("  S: operation impossible with this driver");
				break;
			case 0x03:
				DEBUG_COMM("  S: incorrect number of arguments");
				break;
			case 0x04:
				DEBUG_COMM("  S: reader command unknown");
				break;
			case 0x05:
				DEBUG_COMM("  S: response too long for the buffer");
				break;
			case 0x10:
				DEBUG_COMM("  S: resonse error at the card reset");
				break;
			case 0x12:
				DEBUG_COMM("  S: message too long");
				break;
			case 0x13:
				DEBUG_COMM("  S: byte reading error returned by an asynchronous cadr");
				break;
			case 0x15:
				DEBUG_COMM("  S: card powered down");
				break;
			case 0x1B:
				DEBUG_COMM("  S: a command has been sent with an incorrect number of parameters");
				break;
			case 0x1C:
				DEBUG_COMM("  S: overlap on writting flash memory");
				break;
			case 0x1D:
				DEBUG_COMM("  S: the TCK check byte is incorrect in a micropocessor card ATR");
				break;
			case 0x1E:
				DEBUG_COMM("  S: an attempt has been made to write a write-protected external memory");
				break;
			case 0x1F:
				DEBUG_COMM("  S: incorrect data has been send to external memory");
				break;
			case 0xA0:
				DEBUG_COMM("  S: error in the card reset response");
				break;
			case 0xA1:
				DEBUG_COMM("  S: card protocol error");
				break;
			case 0xA2:
				DEBUG_COMM("  S: card malfuntion");
				break;
			case 0xA3:
				DEBUG_COMM("  S: parity error");
				break;
			case 0xA4:
				DEBUG_COMM("  S: card has been chaining (T=1)");
				break;
			case 0xA5:
				DEBUG_COMM("  S: reader has aborted chaining (T=1)");
				break;
			case 0xA6:
				DEBUG_COMM("  S: resynch successfully performed by GemCore");
				break;
			case 0xA7:
				DEBUG_COMM("  S: protocol type selection (PTS) error");
				break;
			case 0xB0:
				DEBUG_COMM("  S: GemCP410 command not supported");
				break;
			case 0xCF:
				DEBUG_COMM("  S: other key already pressed");
				break;
			case 0xE4:
				DEBUG_COMM("  S: the card has just send an invalid procedure byte");
				break;
			case 0xE5:
				DEBUG_COMM("  S: the card has interrupted an exchange");
				break;
			case 0xE7:
				DEBUG_COMM("  S: error returned by the card");
				break;
			case 0xF7:
				DEBUG_COMM("  S: card removed");
				break;
			case 0xF8:
				DEBUG_COMM("  S: the cadr is consuming too much electricity");
				break;
			case 0xFB:
				DEBUG_COMM("  S: card missing");
				break;
			default:
				DEBUG_COMM2("  S: UNKNOWN ERROR %02X", buffer[3]);
				break;
		}

	return ret;
} /* explain_gbp */

