/* -*- Mode: C; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*- */
/*
//
// bool-proc.c
//
// specialized markup processors
//
// Copyright (c) 1995-96 Jim Nelson.  Permission to distribute
// granted by the author.  No warranties are made on the fitness of this
// source code.
//
*/

#include "while-proc.h"

#include "defs.h"
#include "def-proc.h"
#include "htp-files.h"
#include "macro.h"

BOOL ConditionValue(TASK *task, HTML_ATTRIBUTE *attrib, BOOL notTagFound) 
{
    char name[MAX_VARNAME_LEN];
    BOOL quoted;
    uint changeCount;

    BOOL condTrue;
    const char* value;
    char attribValue[MAX_VARVALUE_LEN];
    uint type;

    if (ExpandMacrosInString(task, 
                             attrib->name, name, MAX_VARNAME_LEN,
                             &quoted, &changeCount) == FALSE) {
        return ERROR;
    }

    HtpMsg(MSG_INFO, task->infile, "while loop check %s", 
           changeCount ? name : attrib->name);
    /* get the macros associated value (NULL if macro not defined) */
    if((value = GetVariableValue(task->varstore, 
                                 changeCount ? name : attrib->name)) != NULL)
    {
        type = GetVariableType(task->varstore, 
                               changeCount ? name : attrib->name);
    }
    else
    {
        type = VAR_TYPE_SET_MACRO;
    }

    /* if only a name is specified, only care if macro is defined */
    if(attrib->value == NULL)
    {
        condTrue = (value != NULL) ? TRUE : FALSE;
    }
    else
    {
        /* macro value comparison */
        if(type == VAR_TYPE_SET_MACRO)
        {
            /* macro comparison (case-sensitive) */
            if (ExpandMacrosInString(task, attrib->value, attribValue, 
                                     MAX_VARVALUE_LEN, &quoted, &changeCount)
                == FALSE)
            {
                return ERROR;
            }
            
            condTrue = (strcmp(value, 
                               changeCount ? attribValue : attrib->value) == 0) 
                ? TRUE : FALSE;
        }
        else
        {
            /* block macro, comparisons not allowed */
            condTrue = FALSE;
        }
    }

    /* reverse conditional if NOT attribute found */
    if(notTagFound == TRUE)
    {
        condTrue = (condTrue == TRUE) ? FALSE : TRUE;
    }

    return condTrue;
}


BOOL ProcessWhileBlock(TASK *task, STREAM* blockStream)
{
    TASK newTask;
    BOOL result;

    HtpMsg(MSG_INFO, task->infile, "expanding WHILE loop");

    /* build a new task structure */
    newTask.infile = blockStream;
    newTask.outfile = task->outfile;
    newTask.sourceFilename = task->sourceFilename;
    newTask.varstore = task->varstore;

    /* process the new input file */
    result = ProcessTask(&newTask);
    CloseStream(blockStream);

    return result;
}

uint WhileProcessor(TASK *task, HTML_MARKUP *htmlMarkup, char **newPlaintext)
{
    static uint whileLevel = 0;
    BOOL condTrue;
    BOOL looped;
    HTML_ATTRIBUTE *attrib;
    BOOL notTagFound;
    STREAM blockFile;
    STREAM inFile;

    char expanded[MAX_VARVALUE_LEN];
    BOOL quoted;
    uint changeCount;

    UNREF_PARAM(newPlaintext);

    condTrue = FALSE;

    /* conditionalLevel keeps track of boolean depth */
    if(whileLevel == 0)
    {
        if((IsMarkupTag(htmlMarkup, "/WHILE")))
        {
            HtpMsg(MSG_ERROR, task->infile, "while block must start with WHILE tag");
            return MARKUP_ERROR;
        }
    }

    if(IsMarkupTag(htmlMarkup, "WHILE"))
    {
        whileLevel++;
        /* this is an ugly way to handle the IF-IF NOT test, but will need */
        /* be cleaned up in the future */

        /* should either be one or two attributes in markup */
        if(htmlMarkup->attribCount == 0)
        {
            HtpMsg(MSG_ERROR, task->infile, "no conditional to test");
            return MARKUP_ERROR;
        }

        if(htmlMarkup->attribCount > 2)
        {
            HtpMsg(MSG_ERROR, task->infile, 
                   "too many items in conditional expression");
            return MARKUP_ERROR;
        }

        /* find the attribute to evaluate and search for NOT attribute */
        notTagFound = FALSE;
        attrib = NULL;
        if (! ExpandMacrosInString(task, htmlMarkup->attrib[0].name, expanded,
                                   MAX_VARVALUE_LEN, &quoted, &changeCount)) 
        {
            return MARKUP_ERROR;
        }
        if(stricmp(changeCount ? expanded : htmlMarkup->attrib[0].name,
                   "NOT") == 0)
        {
            /* check to make sure the second attribute is present */
            if(htmlMarkup->attribCount == 1)
            {
                HtpMsg(MSG_ERROR, task->infile, 
                       "NOT listed, no conditional to test");
                return MARKUP_ERROR;
            }

            notTagFound = TRUE;
            attrib = &htmlMarkup->attrib[1];
        }
        else if(htmlMarkup->attribCount == 2)
        {
            if (! ExpandMacrosInString(task, htmlMarkup->attrib[1].name, 
                                       expanded, MAX_VARVALUE_LEN, 
                                       &quoted, &changeCount)) {
                return MARKUP_ERROR;
            }
            if(stricmp(changeCount ? expanded : htmlMarkup->attrib[0].name, 
                       "NOT") == 0)
            {
                notTagFound = TRUE;
                attrib = &htmlMarkup->attrib[0];
            }
            else
            {
                /* this should have been the NOT expression */
                HtpMsg(MSG_ERROR, task->infile, "too many conditionals to test");
                return MARKUP_ERROR;
            }
        }
        else
        {
            attrib = &htmlMarkup->attrib[0];
        }

        condTrue = ConditionValue(task, attrib, notTagFound);
        if (condTrue == ERROR) {
            return MARKUP_ERROR;
        }
        looped = condTrue;

        if(ReadinBlock(task, htmlMarkup, &blockFile) == FALSE)
        {
            return MARKUP_ERROR;
        }

        while (condTrue) {
            /* condition is true, so we have to expand the block */
            CreateBufferReader(&inFile, &blockFile);
            if(! ProcessWhileBlock(task, &inFile)) {
                return MARKUP_ERROR;
            }

            /* check condition */
            condTrue = ConditionValue(task, attrib, notTagFound);
            if (condTrue == ERROR) {
                return MARKUP_ERROR;
            }
        }

        CloseStream(&blockFile);
        return DISCARD_MARKUP;

    }
    else
    {
        /* end of while */
        assert(whileLevel > 0);
        whileLevel--;
    }

    return DISCARD_MARKUP;
}   

