/* -*- Mode: C; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*- */
/*
//
// htp.c
//
// main(), major functionality modules
//
// Copyright (c) 1995-96 Jim Nelson.  Permission to distribute
// granted by the author.  No warranties are made on the fitness of this
// source code.
//
*/

#include "defs.h"

#include "htp-files.h"
#include "macro.h"
#include "option.h"

/*
// HTML file stream functions
*/


BOOL ProcessTask(TASK *task)
{
    char *newPlaintext;
    uint markupResult;
    HTML_MARKUP htmlMarkup;
    BOOL result;
    uint markupType;

    assert(task != NULL);
    assert(task->infile != NULL);
    assert(task->outfile != NULL);
    assert(task->varstore != NULL);

    task->conditionalLevel = 0;

    for(;;)
    {
        result = ReadHtmlFile(task->infile, task->outfile, &newPlaintext,
            &markupType);
        if(result == ERROR)
        {
            /* problem reading in the file */
            return FALSE;
        }
        else if(result == FALSE)
        {
            /* end-of-file */
            break;
        }

        /* use the new markup plain text to build an HTML_MARKUP structure */
        if(PlaintextToMarkup(newPlaintext, &htmlMarkup) == FALSE)
        {
            HtpMsg(MSG_ERROR, task->infile, "could not parse markup tag (out of memory?)");
            FreeMemory(newPlaintext);
            return FALSE;
        }

        /* destroy the ORIGINAL plain text, not needed again */
        FreeMemory(newPlaintext);
        newPlaintext = NULL;
        
        markupResult = ExpandAll(task, &htmlMarkup, &newPlaintext, markupType);
        
        /* destroy the structure, now only interested in the markup string */
        DestroyMarkupStruct(&htmlMarkup);

        switch(markupResult)
        {
            case MARKUP_OKAY:
            {
                /* add the markup to the output file as it should appear */
                StreamPrintF(task->outfile, "%c%s%c", 
                             MARKUP_OPEN_DELIM(markupType),
                             newPlaintext, MARKUP_CLOSE_DELIM(markupType));
            }
            break;

            case NEW_MARKUP:
            case MARKUP_REPLACED:
            {
                /* the markup has been replaced by a normal string */
                PutStreamString(task->outfile, newPlaintext);
            }
            break;

            case DISCARD_MARKUP:
            {
                /* markup will not be included in final output */
            }
            break;

            case MARKUP_ERROR:
            {
                /* (need to destroy plaintext buffer before exiting) */
                FreeMemory(newPlaintext);
                return FALSE;
            }

            default:
            {
                FreeMemory(newPlaintext);
                printf("%s: serious internal error\n", PROGRAM_NAME);
                exit(1);
            }
        }

        /* free the plain text buffer and continue with processing */
        if (newPlaintext)
        {
            FreeMemory(newPlaintext);
            newPlaintext = NULL;
        }
    }

    if (task->conditionalLevel > 0)
    {
        HtpMsg(MSG_ERROR, task->infile, "Missing </IF>");
    }

    return TRUE;
}


BOOL ProcessFileByName(const char *in, const char *out)
{
    STREAM infile;
    STREAM outfile;
    TASK task;
    BOOL result;
    STREAM project;
    const char *templateFile;

    assert(in != NULL);
    assert(out != NULL);

    /* before proceeding, find a local project default file */
    if(FileExists("htp.def"))
    {
        StringCopy(projectFilename, "htp.def", MAX_PATHNAME_LEN);
    }
    else
    {
        projectFilename[0] = NUL;
    }

    /* assume no processing required */
    result = TRUE;

    /* check the global and project default files first */
    if(*globalFilename != NUL)
    {
        if((result = FullyCheckDependencies(globalFilename, out)) == ERROR)
        {
            return FALSE;
        }
    }

    if((result == TRUE) && (*projectFilename != NUL))
    {
        if((result = FullyCheckDependencies(projectFilename, out)) == ERROR)
        {
            return FALSE;
        }
    }

    /* check the dependencies of the target file to see whether or not */
    /* to proceed ... the global and project default files are checked as well */
    if(result == TRUE)
    {
        if((result = FullyCheckDependencies(in, out)) == ERROR)
        {
            /* did not process the files */
            return FALSE;
        }
    }

    /* if TRUE, no need to go any further */
    if(result == TRUE)
    {
        /* explain why no processing required, and return as if processing */
        /* was completed */
        printf("%s: File \"%s\" is completely up to date.\n", PROGRAM_NAME,
            out);
        return TRUE;
    }

    /* continue, at least one file was found that requires out to be updated */

    /* initialize the project variable store and push it onto context */
    InitializeVariableStore(&projectVarStore);
    PushVariableStoreContext(&globalVarStore, &projectVarStore);
    projectVarStore.isGlobal = TRUE;

    /* initialize the ALT text store (used by the ALTTEXT tag) */
    InitializeVariableStore(&altTextVarStore);

    /* open the output file first, the project default file needs it */
    if(CreateFileWriter(&outfile, out, FALSE) == FALSE)
    {
        printf("%s: unable to open file \"%s\" for writing\n", PROGRAM_NAME, out);
        DestroyVariableStore(&altTextVarStore);
        DestroyVariableStore(&projectVarStore);
        return FALSE;
    }

    if(CONDENSE || SEMICONDENSE)
    {
        /* suppress all linefeeds for this file, makes the HTML output smaller */
        if (SEMICONDENSE) {
            SingleLinefeeds(&outfile);
        }
        else {
            SuppressLinefeeds(&outfile);
        }
    }

    /* clear the task struct, in case there is no project file */
    memset(&task, 0, sizeof(TASK));

    if(projectFilename[0] != NUL)
    {
        /* process the default project file */
        if(CreateFileReader(&project, projectFilename) == FALSE)
        {
            printf("%s: unable to open file \"%s\" for reading\n", PROGRAM_NAME,
                projectFilename);
            CloseStream(&outfile);
            DestroyVariableStore(&projectVarStore);
            DestroyVariableStore(&altTextVarStore);
            return FALSE;
        }

        /* build a task structure */
        task.infile = &project;
        task.outfile = &outfile;
        task.varstore = &projectVarStore;
        task.sourceFilename = in;

        printf("%s: Processing default project file \"%s\" ...\n", PROGRAM_NAME,
            projectFilename);

        result = ProcessTask(&task);

        CloseStream(&project);

        if(result != TRUE)
        {
            if(PRECIOUS == FALSE)
            {
                remove(out);
            }

            printf("%s: error during processing of default project file \"%s\"\n",
                PROGRAM_NAME, projectFilename);

            CloseStream(&outfile);
            DestroyVariableStore(&projectVarStore);
            DestroyVariableStore(&altTextVarStore);
            return result;
        }
    }

    if(CreateFileReader(&infile, in) == FALSE)
    {
        printf("%s: unable to open file \"%s\" for reading\n", PROGRAM_NAME, in);
        CloseStream(&outfile);
        DestroyVariableStore(&projectVarStore);
        DestroyVariableStore(&altTextVarStore);
        return FALSE;
    }

    if(InitializeLocalOption() == FALSE)
    {
        printf("%s: unable to initialize local option store\n", PROGRAM_NAME);
        CloseStream(&infile);
        CloseStream(&outfile);
        DestroyVariableStore(&projectVarStore);
        DestroyVariableStore(&altTextVarStore);
        return FALSE;
    }

    /* build a task structure */
    task.infile = &infile;
    task.outfile = &outfile;
    task.varstore = &projectVarStore;
    task.sourceFilename = in;
    StoreVariable(&projectVarStore, "_htpfile_in", (void *) in, 
                  VAR_TYPE_SET_MACRO, VAR_FLAG_QUOTED, NULL, NULL);
    StoreVariable(&projectVarStore, "_htpfile_out", (void *) out, 
                  VAR_TYPE_SET_MACRO, VAR_FLAG_QUOTED, NULL, NULL);
    printf("%s: Processing file \"%s\" to output file \"%s\" ...\n",
        PROGRAM_NAME, in, out);

    result = ProcessTask(&task);

    /* need to check for a template file */
    while((result == TRUE) && (VariableExists(&projectVarStore, VAR_TEMPLATE_NAME)))
    {
        /* go process it */

        /* done with this file, want to reuse struct */
        CloseStream(&infile);

        templateFile = GetVariableValue(&projectVarStore, VAR_TEMPLATE_NAME);

        if(CreateFileReader(&infile, templateFile) == FALSE)
        {
            printf("%s: unable to open template file \"%s\"\n",
                PROGRAM_NAME, templateFile);

            CloseStream(&outfile);
            DestroyLocalOption();
            DestroyVariableStore(&altTextVarStore);
            DestroyVariableStore(&projectVarStore);

            return FALSE;
        }

        task.infile = &infile;
        task.outfile = &outfile;
        task.varstore = &projectVarStore;
        task.sourceFilename = in;

        printf("%s: Processing template file \"%s\" ...\n", PROGRAM_NAME,
            templateFile);

        result = ProcessTask(&task);

        /* because this template file can, legally, reference another */
        /* template file, remove the current variable and let the while loop */
        /* continue until no more template files are specified */
        /* yes, this can lead to infinite loops and such, but the syntax */
        /* shouldnt bar this, and after all, the same problem could exist if */
        /* the user kept doing circular FILE INCLUDEs */
        /* Pilot error! */
        RemoveVariable(&projectVarStore, VAR_TEMPLATE_NAME);
    }

    if(result == TRUE)
    {
        printf("%s: final output file \"%s\" successfully created\n\n",
            PROGRAM_NAME, outfile.name);
    }
    else
    {
        printf("\n%s: error encountered, file \"%s\" not completed\n\n",
            PROGRAM_NAME, outfile.name);
    }

    CloseStream(&outfile);
    CloseStream(&infile);

    /* destroy incomplete file if not configured elsewise */
    if(result != TRUE)
    {
        if(PRECIOUS == FALSE)
        {
            assert(out != NULL);
            remove(out);
        }
    }

    /* destroy the local options for these files */
    DestroyLocalOption();

    /* destroy the project store as well */
    DestroyVariableStore(&projectVarStore);

    /* destroy the ALT text store */
    DestroyVariableStore(&altTextVarStore);

    return result;
}

BOOL ProcessResponseFile(const char *resp)
{
    char textline[128];
    char defResp[MAX_PATHNAME_LEN];
    char newDirectory[MAX_PATHNAME_LEN];
    char oldDirectory[MAX_PATHNAME_LEN];
    STREAM respfile;
    int result;
    char *in;
    char *out;
    char *ptr;
    BOOL useNewDir;
    BOOL respFileOpen;
    FIND_TOKEN findToken;
    uint numread;
    bitmap nl_bitmap;

    assert(resp != NULL);

    memset(nl_bitmap, 0, sizeof(nl_bitmap));
    BITMAP_SET(nl_bitmap, '\n');
    useNewDir = FALSE;

    if(strchr(ALL_FILESYSTEM_DELIMITERS, resp[strlen(resp) - 1]) != NULL)
    {
        /* some tests as done to ensure that (a) newDirectory does not trail */
        /* with a directory separator and that (b) the separator is present */
        /* before appending the filename ... requirement (a) is a DOS issue */

        /* the response file is actually a directory the response file is */
        /* possibly kept in ... copy it to the newDirectory variable for */
        /* later use, but remove the trailing delimiter (MS-DOS issue) */
        strcpy(newDirectory, resp);
        newDirectory[strlen(newDirectory) - 1] = NUL;

        /* now, see if default response file is present */
        strcpy(defResp, newDirectory);
        strcat(defResp, DIR_DELIMITER_STRING);
        strcat(defResp, DEFAULT_RESPONSE_FILE);

        useNewDir = TRUE;

        respFileOpen = CreateFileReader(&respfile, defResp);
    }
    else
    {
        respFileOpen = CreateFileReader(&respfile, resp);
    }

    if(respFileOpen == FALSE)
    {
        printf("%s: unable to open \"%s\" as a response file\n", PROGRAM_NAME,
            resp);
        return FALSE;
    }

    printf("%s: Processing response file \"%s\" ...\n", PROGRAM_NAME,
        respfile.name);

    /* processing a response file in another directory, change to that */
    /* directory before processing the files */
    if(useNewDir)
    {
        getcwd(oldDirectory, sizeof oldDirectory);
        chdir(newDirectory);
    }

    result = TRUE;
    do
    {
        numread = GetStreamBlock(&respfile, textline, sizeof(textline), 
                                 nl_bitmap);
        if(numread == 0)
        {
            /* EOF */
            break;
        }
        /* Kill the newline at the end of the line */
        if (textline[numread-1] == '\n')
            textline[numread-1] = NUL;

        in = NULL;
        out = NULL;

        /* walk tokens ... allow for tab character as token and ignore */
        /* multiple token characters between filenames */
        ptr = StringFirstToken(&findToken, textline, " \t");
        while(ptr != NULL)
        {
            /* is this just a repeated token? */
            if((*ptr == ' ') || (*ptr == '\t'))
            {
                ptr = StringNextToken(&findToken);
                continue;
            }

            /* found something ... like parsing the command-line, look for */
            /* options, then response files, then regular in and out filenames */
            if((*ptr == '-') || (*ptr == '/'))
            {
                /* option */
                ParseTextOption(ptr, OptionCallback, 0);
            }
            else if(*ptr == ';')
            {
                /* comment, ignore the rest of the line */
                break;
            }
            else if(in == NULL)
            {
                in = ptr;
            }
            else if(out == NULL)
            {
                out = ptr;
            }
            else
            {
                /* hmm ... extra information on line */
                HtpMsg(MSG_WARNING, &respfile, 
                       "extra option \"%s\" specified in response file, ignoring",
                       ptr);
            }

            ptr = StringNextToken(&findToken);
        }

        /* if in and out NULL, ignore the line entirely (all options or blank) */
        if((in == NULL) && (out == NULL))
        {
            continue;
        }

        if(out == NULL)
        {
            /* in is the response file ... recurse like theres no tomorrow */
            result = ProcessResponseFile(in);
            continue;
        }

        /* both in and out were specified, do it */
        result = ProcessFileByName(in, out);
    } while(result == TRUE);

    CloseStream(&respfile);

    /* restore the directory this all started in */
    if(useNewDir)
    {
        chdir(oldDirectory);
    }

    return result;
}   

BOOL ProcessDefaultFile(void)
{
    TASK defTask;
    STREAM infile;
    STREAM outfile;
    BOOL result;

    /* get the default filename */
    if(HtpDefaultFilename(globalFilename, MAX_PATHNAME_LEN) == FALSE)
    {
        /* nothing to do, but no error either */
        globalFilename[0] = NUL;
        return TRUE;
    }

    if(CreateFileReader(&infile, globalFilename) == FALSE)
    {
        printf("%s: unable to open default file \"%s\"\n", PROGRAM_NAME,
            globalFilename);
        return FALSE;
    }

    /* use a null outfile because there is no file to write to */
    CreateNullWriter(&outfile);

    /* build a task (just like any other) and process the file */
    /* use the global variable store to hold all the macros found */
    defTask.infile = &infile;
    defTask.outfile = &outfile;
    defTask.varstore = &globalVarStore;
    defTask.sourceFilename = globalFilename;

    printf("%s: Processing default file \"%s\" ... \n", PROGRAM_NAME,
        globalFilename);

    result = ProcessTask(&defTask);

    CloseStream(&infile);
    CloseStream(&outfile);

    return result;
}

int main(int argc, char *argv[])
{
    int result;
    uint ctr;
    char *in;
    char *out;
    char *resp;

    DisplayHeader();

    if(argc == 1)
    {
        usage();
        return 1;
    }

    /* initialize debugging */
#if DEBUG
    DebugInit("htpdeb.out");
    atexit(DebugTerminate);
#endif

#ifdef USE_SUBALLOC
    /* initialize the suballoc memory module */
    InitializeMemory();
    atexit(TerminateMemory);
#endif

    /* initialize global variable options */
    if(InitializeGlobalOption(OptionCallback, 0) == FALSE)
    {
        printf("%s: fatal error, unable to initialize internal options\n",
            PROGRAM_NAME);
        return 1;
    }

    in = NULL;
    out = NULL;
    resp = NULL;

    /* search command-line for options */
    for(ctr = 1; ctr < (uint) argc; ctr++)
    {
        if((*argv[ctr] == '-') || (*argv[ctr] == '/'))
        {
            /* command-line option specified */
            ParseTextOption(argv[ctr], OptionCallback, 0);
        }
        else if(*argv[ctr] == '@')
        {
            /* response file specified */
            resp = argv[ctr] + 1;
            if(*resp == NUL)
            {
                resp = (char *) DEFAULT_RESPONSE_FILE;
            }
        }
        else if(in == NULL)
        {
            /* input file was specified */
            in = argv[ctr];
        }
        else if(out == NULL)
        {
            /* output file was specified */
            out = argv[ctr];
        }
        else
        {
            printf("%s: unknown argument \"%s\" specified\n",
                PROGRAM_NAME, argv[ctr]);
            return 1;
        }
    }
    
    if(USAGE == TRUE)
    {
        usage();
        return 1;
    }

    if((in == NULL || out == NULL) && (resp == NULL))
    {
        usage();
        return 1;
    }

    /* initialize the global variable store before proceeding */
    if(InitializeVariableStore(&globalVarStore) != TRUE)
    {
        printf("%s: unable to initialize global variable store (out of memory?)\n",
            PROGRAM_NAME);
        return 1;
    }
    globalVarStore.isGlobal = TRUE;

    /* before reading in the response file or processing any files, handle */
    /* the default file, if there is one ... all of its macros are held in */
    /* the global variable store */
    ProcessDefaultFile();

    /* now, process the response file (if there is one) or the files */
    /* specified on the command-line */
    if(resp != NULL)
    {
        result = ProcessResponseFile(resp);
    }
    else
    {
        result = ProcessFileByName(in, out);
    }

    /* display varstore stats */
    DEBUG_PRINT(("Variable lookups=%u  string cmps=%u  missed string cmps=%u  cache hits=%u  hash calcs=%u\n",
        variableLookups, variableStringCompares, variableMissedStringCompares,
        variableCacheHits, variableHashCalcs));

    /* display macro expansion stats */
    DEBUG_PRINT(("Expansion skipped=%u  performed=%u\n", expandSkipped,
        expandPerformed));

    /* destroy the global variable store */
    DestroyVariableStore(&globalVarStore);

    /* destroy global option */
    DestroyGlobalOption();

#ifdef USE_SUBALLOC
    /* display suballoc stats */
    DEBUG_PRINT(("suballoc  total allocations=%u  free pool hits=%u  system heap allocs=%u\n",
        totalAllocations, freePoolHits, totalAllocations - freePoolHits));
#endif

    return (result == TRUE) ? 0 : 1;
}
