/*
 *	HT Editor
 *	srt_x86.cc
 *
 *	Copyright (C) 2001 Stefan Weyergraf (stefan@weyergraf.de)
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License version 2 as
 *	published by the Free Software Foundation.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <string.h>

#include "htexcept.h"
#include "htiobox.h"
#include "srt_x86.h"
#include "x86dis.h"

#define X86_FLAG_CARRY		0
#define X86_FLAG_PARITY		1
#define X86_FLAG_AUX		2
#define X86_FLAG_ZERO		3
#define X86_FLAG_SIGNED		4
#define X86_FLAG_OVERFLOW	5

#define X86_FLAGS			6

char *x86_flags[X86_FLAGS] = {
	"carry", "parity", "aux",
	"zero", "signed", "overflow"
};

char *x86_idx2reg(UINT idx)
{
	if (idx >= 8) {
		return x86_flags[(idx-8) % X86_FLAGS];
	}
	return x86_regs[2][idx & 7];
}

/*
 *	CLASS sym_int_reg_x86
 */

class sym_int_reg_x86: public sym_int_reg {
public:
	sym_int_reg_x86::sym_int_reg_x86(UINT r): sym_int_reg(r)
	{
	}

	object *duplicate()
	{
		return new sym_int_reg_x86(regidx);
	}

	virtual int nstrfy(char *buf, int n)
	{
		return sprintf(buf,"%s", x86_idx2reg(regidx));
	}
}; 

struct CPU {
	sym_int *regs[8];
	sym_bool *flags[X86_FLAGS];
};

/*
 *	srt_x86
 */

void x86_flags_std(ht_list *rm, x86dis_insn *insn, sym_int *cond)
{
	state_mod *zf = new state_mod();
	zf->ismem = false;
	zf->dest.regidx = 8 + X86_FLAG_ZERO;
	zf->isbool = true;

	state_mod *sf = new state_mod();
	sf->ismem = false;
	sf->dest.regidx = 8 + X86_FLAG_SIGNED;
	sf->isbool = true;

	sym_int *i;

	switch (insn->op[0].size) {
		case 1:
			i = (sym_int*)cond->duplicate();
			i->b_operate(b_and, new sym_int_const(0xff));
			zf->value.boolean = new sym_bool();
			zf->value.boolean->set(new sym_bool_intcmp(i, c_eq, new sym_int_const(0)));

			i = (sym_int*)cond->duplicate();
			i->b_operate(b_and, new sym_int_const(0x80));
			sf->value.boolean = new sym_bool();
			sf->value.boolean->set(new sym_bool_intcmp(i, c_ne, new sym_int_const(0)));
			break;
		case 2:
			i = (sym_int*)cond->duplicate();
			i->b_operate(b_and, new sym_int_const(0xffff));
			zf->value.boolean = new sym_bool();
			zf->value.boolean->set(new sym_bool_intcmp(i, c_eq, new sym_int_const(0)));

			i = (sym_int*)cond->duplicate();
			i->b_operate(b_and, new sym_int_const(0x8000));
			sf->value.boolean = new sym_bool();
			sf->value.boolean->set(new sym_bool_intcmp(i, c_ne, new sym_int_const(0)));
			break;
		case 4:
			i = (sym_int*)cond->duplicate();
			zf->value.boolean = new sym_bool();
			zf->value.boolean->set(new sym_bool_intcmp(i, c_eq, new sym_int_const(0)));

			i = (sym_int*)cond->duplicate();
			i->b_operate(b_and, new sym_int_const(0x80000000));
			sf->value.boolean = new sym_bool();
			sf->value.boolean->set(new sym_bool_intcmp(i, c_ne, new sym_int_const(0)));
			break;
	}
	rm->insert(zf);
	rm->insert(sf);
}

void x86_flags_carry(ht_list *rm, x86dis_insn *insn, sym_int *cond, sym_bool *carry)
{
	state_mod *cf = new state_mod();
	cf->ismem = false;
	cf->dest.regidx = 8 + X86_FLAG_CARRY;
	cf->isbool = true;

	sym_int *i;

	switch (insn->op[0].size) {
		case 1:
			i = (sym_int*)cond->duplicate();
			i->b_operate(b_and, new sym_int_const(0x100));
			cf->value.boolean = new sym_bool();
			cf->value.boolean->set(new sym_bool_intcmp(i, c_ne, new sym_int_const(0)));
		case 2:
			i = (sym_int*)cond->duplicate();
			i->b_operate(b_and, new sym_int_const(0x10000));
			cf->value.boolean = new sym_bool();
			cf->value.boolean->set(new sym_bool_intcmp(i, c_ne, new sym_int_const(0)));
		case 4:
			cf->value.boolean = (sym_bool*)carry->duplicate();
	}
	rm->insert(cf);
}

/*void x86_flags_overflow(cpu *CPU, x86dis_insn *insn, dword a, dword b, dword c)
{
	bool overflow_on = false;
	switch (insn->op[0].size) {
		case 1:
			if (((a & 0x80) == (b & 0x80)) && ((a & 0x80) != (c & 0x80))) {
				overflow_on = true;
				CPU->context.d.eflags |= FLAGS_OVERFLOW;
			} else {
				CPU->context.d.eflags &= ~FLAGS_OVERFLOW;
			}
			break;
		case 2:
			if (((a & 0x8000) == (b & 0x8000)) && ((a & 0x8000) != (c & 0x8000))) {
				CPU->context.d.eflags |= FLAGS_OVERFLOW;
			} else {
				CPU->context.d.eflags &= ~FLAGS_OVERFLOW;
			}
			break;
		case 4:
			if (((a & 0x80000000) == (b & 0x80000000)) && ((a & 0x80000000) != (c & 0x80000000))) {
				CPU->context.d.eflags |= FLAGS_OVERFLOW;
			} else {
				CPU->context.d.eflags &= ~FLAGS_OVERFLOW;
			}
			break;
	}
}*/

sym_int_token *srt_x86_mkreg(CPU *cpu, UINT regidx)
{
	return (sym_int_token*)cpu->regs[regidx]->duplicate();
}

sym_int *srt_x86_mkaddr(CPU *cpu, x86_insn_op *o)
{
	sym_int *a = new sym_int();
	bool first = true;
	if (o->mem.index != X86_REG_NO) {
		a->b_operate(b_invalid, srt_x86_mkreg(cpu, o->mem.index));
		if (o->mem.scale > 1)
			a->b_operate(b_mul, new sym_int_const(o->mem.scale));
		first = false;
	}
	if (o->mem.base != X86_REG_NO) {
		a->b_operate(first ? b_invalid : b_add, srt_x86_mkreg(cpu, o->mem.base));
		first = false;
	}
	if (o->mem.hasdisp && o->mem.disp) {
		a->b_operate(first ? b_invalid : b_add, new sym_int_const(o->mem.disp));
		first = false;
	}
	return a;
}

void srt_x86_mkdest(CPU *cpu, state_mod *m, x86_insn_op *d)
{
	switch (d->type) {
		case X86_OPTYPE_REG:
			m->ismem = false;
			m->dest.regidx = d->reg;
			break;
		case X86_OPTYPE_MEM:
			m->ismem = true;
			m->dest.mem.addr = srt_x86_mkaddr(cpu, d);
			m->dest.mem.size = d->size;
			m->dest.mem.endian = srte_be;
			break;
		default:
			throw new ht_io_exception("unknown dest type: %d", d->type);
	}
}

sym_int *srt_x86_mkvalue(CPU *cpu, x86_insn_op *o)
{
	sym_int *r = new sym_int();
	switch (o->type) {
		case X86_OPTYPE_IMM: {
			r->set(new sym_int_const(o->imm));
			break;
		}
		case X86_OPTYPE_REG: {
			r->set(srt_x86_mkreg(cpu, o->reg));
			break;
		}
		case X86_OPTYPE_MEM:
			r->set(new sym_int_mem(srt_x86_mkaddr(cpu, o), o->size, srte_le));
			break;
		default:
			throw new ht_io_exception("unknown op type: %d", o->type);
	}
	return r;
}

void srt_x86_mov(CPU *cpu, ht_list *rm, x86dis_insn *insn)
{
	state_mod *m = new state_mod();
	srt_x86_mkdest(cpu, m, &insn->op[0]);
	m->isbool = false;
	m->value.integer = srt_x86_mkvalue(cpu, &insn->op[1]);
	rm->insert(m);
}

void srt_x86_lea(CPU *cpu, ht_list *rm, x86dis_insn *insn)
{
	state_mod *m = new state_mod();
	srt_x86_mkdest(cpu, m, &insn->op[0]);
	if (insn->op[1].type != X86_OPTYPE_MEM) throw new ht_io_exception("internal error in %s at %d", __FILE__, __LINE__);
	m->isbool = false;
	m->value.integer = srt_x86_mkaddr(cpu, &insn->op[1]);
	rm->insert(m);
}

void srt_x86_add(CPU *cpu, ht_list *rm, x86dis_insn *insn)
{
	state_mod *m = new state_mod();
	srt_x86_mkdest(cpu, m, &insn->op[0]);
	m->isbool = false;
	m->value.integer = srt_x86_mkvalue(cpu, &insn->op[0]);
	m->value.integer->b_operate(b_add, srt_x86_mkvalue(cpu, &insn->op[1]));
	rm->insert(m);
	x86_flags_std(rm, insn, m->value.integer);
	sym_bool *carry = new sym_bool();
	carry->set(new sym_bool_intcmp(
		(sym_int*)m->value.integer->duplicate(), c_lt,
		srt_x86_mkvalue(cpu, &insn->op[1])));
	x86_flags_carry(rm, insn, m->value.integer, carry);
	delete carry;
}

void srt_x86_sub(CPU *cpu, ht_list *rm, x86dis_insn *insn)
{
	state_mod *m = new state_mod();
	srt_x86_mkdest(cpu, m, &insn->op[0]);
	m->isbool = false;
	m->value.integer = srt_x86_mkvalue(cpu, &insn->op[0]);
	m->value.integer->b_operate(b_sub, srt_x86_mkvalue(cpu, &insn->op[1]));
	rm->insert(m);
	x86_flags_std(rm, insn, m->value.integer);
	sym_bool *carry = new sym_bool();
	carry->set(new sym_bool_intcmp(
		srt_x86_mkvalue(cpu, &insn->op[0]), c_lt,
		srt_x86_mkvalue(cpu, &insn->op[1])));
	x86_flags_carry(rm, insn, m->value.integer, carry);
	delete carry;
}

ht_list *srt_x86_single(CPU *cpu, x86dis_insn *i)
{
	ht_clist *rm = new ht_clist();
	rm->init();

	if (strcmp(i->name, "mov") == 0) {
		srt_x86_mov(cpu, rm, i);
	} else if (strcmp(i->name, "lea") == 0) {
		srt_x86_lea(cpu, rm, i);
	} else if (strcmp(i->name, "add") == 0) {
		srt_x86_add(cpu, rm, i);
	} else if (strcmp(i->name, "sub") == 0) {
		srt_x86_sub(cpu, rm, i);
	} else {
		throw new ht_io_exception("unsupported cmd: %s", i->name);
	}

	return rm;
}

void create_cpu(CPU *cpu)
{
	for (UINT g = 0; g<8; g++) {
		char s[32];
		sprintf(s, "i%s", x86_idx2reg(g));
		cpu->regs[g] = new sym_int();
		cpu->regs[g]->init();
		cpu->regs[g]->set(new sym_int_symbol(s));
	}

	for (UINT g = 0; g<X86_FLAGS; g++) {
		char s[32];
		sprintf(s, "i%s", x86_idx2reg(8+g));
		cpu->flags[g] = new sym_bool();
		cpu->flags[g]->init();
		cpu->flags[g]->set(new sym_bool_symbol(s));
	}
}

void destroy_cpu(CPU *cpu)
{
	for (UINT g = 0; g<8; g++) {
		cpu->regs[g]->done();
		delete cpu->regs[g];
	}

	for (UINT g = 0; g<X86_FLAGS; g++) {
		cpu->flags[g]->done();
		delete cpu->flags[g];
	}
}

void srt_x86(analyser *analy, ADDR addr)
{
	x86dis *x = (x86dis*)analy->disasm;
	CPU_ADDR a;
	byte buf[15];

	a.addr32 = addr;
	CPU cpu;

	create_cpu(&cpu);
/**/
	analy->bufptr(addr, buf, sizeof buf);
	dis_insn *i = (x86dis_insn*)x->decode(buf, sizeof buf, a);
	x86dis_insn *xi = (x86dis_insn*)i;
	char *dname = x->str(i, X86DIS_STYLE_HEX_NOZEROPAD + X86DIS_STYLE_HEX_ASMSTYLE);
	ht_list *rm = NULL;

	try{
		rm = srt_x86_single(&cpu, xi);
	} catch (ht_exception *x) {
		errorbox("error: %s", x->what());
		return;
	}

	UINT c = rm->count();
	for (UINT i = 0; i<c; i++) {
		state_mod *r = (state_mod*)rm->get(i);
		char b[256];
		r->value.integer->nstrfy(b, sizeof b);
		if (r->ismem) {
			char c[256];
			r->dest.mem.addr->nstrfy(c, sizeof c);
			infobox("%s causes memmod:\n%s%d[%s] := '%s'", dname, srt_endian_to_str(r->dest.mem.endian), r->dest.mem.size, c, b);
		} else {
			infobox("%s causes regmod:\n%s := '%s'", dname, x86_idx2reg(r->dest.regidx), b);
		}
	}
/**/
	destroy_cpu(&cpu);
}
