module Crypto.CPoly
	(
	cpoly,
	cpoly2list,
	cpoly2revlist,
	) where
import Data.Bits
import Data.Array

-- Based on the data from "Table of Low-Weight Binary Irreducible Polynomials" of Gadiel Seroussi
-- Get the CMAC polynomial applicable to each block length from 2 to 10000
-- The values for 64 and 128 match the standard
-- Please note this takes 1 minute and some memory to compile on my I5 so may
-- take longer on other platforms
-- Generated by: cat polys.txt | sed -e 's/^.*$/(\0)/g' -e 's/^(\([0-9]*\),\([0-9]*\))$/cpolys \1 = T \2/g' -e 's/^(\([0-9]*\),\([0-9]*\),\([0-9]*\),\([0-9]*\))$/cpolys \1 = Q \2 \3 \4/g' > cpoly2.hs
-- Polys.txt generated by: cat /tmp/lalala.txt | egrep "[1-9][0-9]*,[1-9][0-9]*(,[1-9][0-9]*,[1-9][0-9]*)?" | tr ' ' '\n'  | sed '/^$/d' > polys.txt

data Poly = T Int
	  | Q Int Int Int

cpoly :: (Integral a, Bits b) => a -> b -> b
cpoly x = cpoly2 $ cpolys $ fromIntegral (x)
  where
	cpoly2 (T a) = (\x -> (x `complementBit` a `complementBit` 0))
	cpoly2 (Q a b c) = (\x -> (x `complementBit` a `complementBit` b `complementBit` c `complementBit` 0))

cpoly2list :: (Integral a, Integral b) => a -> [b]
cpoly2list x = cpoly2 $ cpolys $ fromIntegral (x)
  where
	cpoly2 (T a) = map fromIntegral [a,0]
	cpoly2 (Q a b c) = map fromIntegral [a,b,c,0]

cpoly2revlist :: (Integral a, Integral b) => a -> [b]
cpoly2revlist x = cpoly2 $ cpolys $ fromIntegral (x)
  where
	cpoly2 (T a) = map fromIntegral [0,a]
	cpoly2 (Q a b c) = map fromIntegral [0,c,b,a]

cpolys :: Int -> Poly
cpolys i
 | i < 2 || i > 10000 = error $ "The CMAC polynomial for the value" ++ show i ++ "hasn't been precomputed"
 | otherwise = cpolysArr ! i

cpolysArr :: Array Int Poly
cpolysArr = listArray (2,10000) [
  T 1,T 1,T 1,T 2,T 1,T 1,Q 4 3 1,T 1,T 3,T 2,T 3,Q 4 3 1,T 5,T 1,Q 5 3 1,
  T 3,T 3,Q 5 2 1,T 3,T 2,T 1,T 5,Q 4 3 1,T 3,Q 4 3 1,Q 5 2 1,T 1,T 2,T 1,
  T 3,Q 7 3 2,T 10,T 7,T 2,T 9,Q 6 4 1,Q 6 5 1,T 4,Q 5 4 3,T 3,T 7,Q 6 4 3,
  T 5,Q 4 3 1,T 1,T 5,Q 5 3 2,T 9,Q 4 3 2,Q 6 3 1,T 3,Q 6 2 1,T 9,T 7,Q 7 4 2,
  T 4,T 19,Q 7 4 2,T 1,Q 5 2 1,T 29,T 1,Q 4 3 1,T 18,T 3,Q 5 2 1,T 9,Q 6 5 2,
  Q 5 3 1,T 6,Q 10 9 3,T 25,T 35,Q 6 3 1,T 21,Q 6 5 2,Q 6 5 3,T 9,Q 9 4 2,
  T 4,Q 8 3 1,Q 7 4 2,T 5,Q 8 2 1,T 21,T 13,Q 7 6 2,T 38,T 27,Q 8 5 1,T 21,
  T 2,T 21,T 11,Q 10 9 6,T 6,T 11,Q 6 3 1,T 15,Q 7 6 1,T 29,T 9,Q 4 3 1,
  T 4,T 15,Q 9 7 4,T 17,Q 5 4 2,T 33,T 10,Q 5 4 3,T 9,Q 5 3 2,Q 8 7 5,Q 4 2 1,
  Q 5 2 1,T 33,T 8,Q 4 3 1,T 18,Q 6 2 1,T 2,T 19,Q 7 6 5,T 21,T 1,Q 7 2 1,
  T 5,T 3,Q 8 3 2,T 17,Q 9 8 2,T 57,T 11,Q 5 3 2,T 21,Q 8 7 1,Q 8 5 3,T 15,
  Q 10 4 1,T 21,Q 5 3 2,Q 7 4 2,T 52,T 71,T 14,T 27,Q 10 9 7,T 53,T 3,Q 6 3 2,
  T 1,T 15,T 62,T 9,Q 6 5 2,Q 8 6 5,T 31,Q 5 3 2,T 18,T 27,Q 7 6 3,Q 10 8 7,
  Q 9 8 3,T 37,T 6,Q 15 3 2,T 34,T 11,Q 6 5 2,T 1,Q 8 5 2,T 13,T 6,Q 11 3 2,
  T 8,T 31,Q 4 2 1,T 3,Q 7 6 1,T 81,T 56,Q 9 8 7,T 24,T 11,Q 7 6 5,Q 6 5 2,
  Q 6 5 2,Q 8 7 6,T 9,Q 7 2 1,T 15,T 87,Q 8 3 2,T 3,Q 9 4 2,T 9,T 34,Q 5 3 2,
  T 14,T 55,Q 8 7 1,T 27,Q 9 5 2,Q 10 9 5,T 43,Q 9 3 1,T 6,T 7,Q 11 10 8,
  T 105,Q 6 5 2,T 73,T 23,Q 7 3 1,T 45,T 11,Q 8 4 1,T 7,Q 8 6 2,Q 5 4 2,
  T 33,Q 9 8 3,T 32,Q 10 7 3,Q 10 9 4,T 113,Q 10 4 1,Q 8 7 6,T 26,Q 9 4 2,
  T 74,T 31,Q 9 6 1,T 5,Q 7 4 1,T 73,T 36,Q 8 5 3,T 70,T 95,Q 8 5 1,T 111,
  Q 6 4 1,Q 11 2 1,T 82,Q 15 14 10,T 35,T 103,Q 7 4 2,T 15,T 46,Q 7 2 1,
  T 52,Q 10 5 2,T 12,T 71,Q 10 6 2,T 15,Q 7 6 4,Q 9 8 4,T 93,Q 9 6 2,T 42,
  T 47,Q 8 6 3,T 25,Q 7 6 1,T 53,T 58,Q 9 3 2,T 23,T 67,Q 11 10 9,T 63,
  Q 12 6 3,T 5,T 5,Q 9 5 2,T 93,T 35,Q 12 7 5,T 53,Q 10 7 5,T 69,T 71,Q 11 10 1,
  T 21,Q 5 3 2,Q 12 11 5,T 37,Q 11 6 1,T 33,T 48,Q 7 3 2,T 5,Q 11 8 4,Q 11 6 4,
  T 5,Q 9 5 2,T 41,T 1,Q 11 2 1,T 102,Q 7 3 1,Q 8 4 2,T 15,Q 10 6 4,T 93,
  Q 7 5 3,Q 9 7 4,T 79,T 15,Q 10 9 1,T 63,Q 7 4 2,T 45,T 36,Q 4 3 1,T 31,
  T 67,Q 10 3 1,T 51,Q 10 5 2,Q 10 3 1,T 34,Q 8 3 1,T 50,T 99,Q 10 6 2,
  T 89,T 2,Q 5 2 1,Q 10 7 2,Q 7 4 1,T 55,Q 4 3 1,Q 16 10 7,T 45,Q 10 8 6,
  T 125,T 75,Q 7 2 1,T 22,T 63,Q 11 10 3,T 103,Q 6 5 2,T 53,T 34,Q 13 11 6,
  T 69,T 99,Q 6 5 1,Q 10 9 7,Q 11 10 2,T 57,T 68,Q 5 3 2,Q 7 4 1,T 63,Q 8 5 3,
  T 9,Q 9 6 5,T 29,T 21,Q 7 3 2,T 91,T 139,Q 8 3 2,T 111,Q 8 7 2,Q 8 6 5,
  T 16,Q 8 7 5,T 41,T 43,Q 10 8 5,T 47,Q 5 2 1,T 81,T 90,Q 12 3 2,T 6,T 83,
  Q 8 7 1,T 159,Q 10 9 5,T 9,T 28,Q 13 10 6,T 7,T 135,Q 11 6 5,T 25,Q 12 7 6,
  Q 7 6 2,T 26,Q 5 3 2,T 152,T 171,Q 9 8 5,T 65,Q 13 8 2,T 141,T 71,Q 5 3 2,
  T 87,Q 10 4 3,Q 12 10 3,T 147,Q 10 7 6,T 13,T 102,Q 9 5 2,T 107,T 199,
  Q 15 5 4,T 7,Q 5 4 2,T 149,T 25,Q 9 7 2,T 12,T 63,Q 11 6 5,T 105,Q 10 8 7,
  Q 14 6 1,T 120,Q 13 4 3,T 33,Q 12 11 5,Q 12 9 5,T 165,Q 6 2 1,T 65,T 49,
  Q 4 3 1,T 7,Q 7 5 2,Q 10 6 1,T 81,Q 7 6 4,T 105,T 73,Q 11 6 4,T 134,T 47,
  Q 16 10 1,Q 6 5 4,Q 15 6 4,Q 8 6 1,T 38,Q 18 9 6,T 16,T 203,Q 12 5 2,
  T 19,Q 7 6 1,T 73,T 93,Q 19 18 13,T 31,Q 14 11 6,Q 11 6 1,T 27,Q 9 5 2,
  T 9,T 1,Q 11 3 2,T 200,T 191,Q 9 8 4,T 9,Q 16 15 7,T 121,T 104,Q 15 9 6,
  T 138,Q 9 6 5,Q 9 6 4,T 105,Q 17 16 6,T 81,T 94,Q 4 3 1,T 83,T 219,Q 11 6 3,
  T 7,Q 10 5 3,T 17,T 76,Q 16 5 2,T 78,T 155,Q 11 6 5,T 27,Q 5 4 2,Q 8 5 4,
  T 3,Q 15 14 6,T 156,T 23,Q 13 6 3,T 9,Q 8 7 3,T 69,T 10,Q 8 5 2,T 26,
  T 67,Q 14 7 4,T 21,Q 12 10 2,T 33,T 79,Q 15 11 2,T 32,T 39,Q 13 6 2,T 167,
  Q 6 4 1,T 97,T 47,Q 11 6 2,T 42,Q 10 7 3,Q 10 5 4,T 1,Q 4 3 2,T 161,Q 8 6 2,
  Q 7 5 3,T 94,T 195,Q 10 5 4,T 9,Q 13 10 4,Q 8 6 1,T 16,Q 8 3 1,T 122,
  Q 8 2 1,Q 13 7 4,Q 10 5 3,Q 16 4 3,T 193,T 135,Q 19 16 9,T 39,Q 10 8 7,
  Q 10 9 4,T 153,Q 7 6 5,T 73,T 34,Q 11 9 6,T 71,Q 11 4 2,Q 14 7 3,T 163,
  Q 11 6 1,T 153,T 28,Q 15 7 6,T 77,T 67,Q 10 5 2,Q 12 8 1,Q 10 6 4,T 13,
  T 146,Q 13 4 3,T 25,Q 23 22 16,Q 12 9 7,T 237,Q 13 7 6,T 85,T 130,Q 14 13 3,
  T 88,Q 7 5 2,Q 11 6 1,T 35,Q 10 4 3,T 93,Q 9 6 4,Q 13 6 3,T 86,T 19,Q 9 2 1,
  T 273,Q 14 12 9,Q 7 6 1,T 30,Q 9 5 2,T 201,T 215,Q 6 4 3,T 105,Q 10 7 5,
  T 165,T 105,Q 19 13 6,T 31,T 127,Q 10 4 2,T 81,Q 19 10 4,T 45,T 211,Q 19 10 3,
  T 200,T 295,Q 9 8 5,T 9,Q 12 6 5,T 297,T 68,Q 11 6 5,T 133,T 251,Q 13 8 4,
  T 223,Q 6 5 2,Q 7 4 2,T 307,Q 9 2 1,T 101,T 39,Q 14 10 4,T 217,Q 14 9 1,
  Q 6 5 1,T 16,Q 14 3 2,T 11,T 119,Q 11 3 2,Q 11 6 5,Q 11 8 4,T 249,T 5,
  Q 13 3 1,T 37,T 3,T 14,T 93,Q 10 8 7,T 33,T 88,Q 7 5 4,T 38,T 55,Q 15 4 2,
  T 11,Q 12 11 4,T 21,T 107,Q 11 9 8,T 33,Q 10 7 2,Q 18 7 3,T 147,Q 5 4 2,
  T 153,T 15,Q 11 6 5,T 28,Q 11 7 4,Q 6 3 1,T 31,Q 8 4 3,Q 15 5 3,T 66,
  Q 23 16 9,Q 11 9 3,T 171,Q 11 6 1,T 209,Q 4 3 1,T 197,T 13,Q 19 14 6,
  T 14,T 79,Q 13 6 2,T 299,Q 15 8 2,T 169,T 177,Q 23 10 2,T 267,T 215,Q 15 10 1,
  T 75,Q 16 4 2,T 37,Q 12 7 1,Q 8 3 2,T 17,Q 12 11 8,Q 15 8 5,T 15,Q 4 3 1,
  Q 13 12 4,T 92,Q 5 4 3,T 41,T 23,Q 7 4 1,T 183,Q 16 7 1,T 165,T 150,Q 9 6 4,
  T 9,T 231,Q 16 10 4,T 207,Q 9 6 5,T 5,T 180,Q 4 3 2,T 58,T 147,Q 8 6 2,
  T 343,Q 8 7 2,Q 11 6 1,T 44,Q 13 8 6,T 5,T 347,Q 18 16 8,T 135,Q 9 8 3,
  T 85,T 90,Q 13 11 1,T 258,T 351,Q 10 6 4,T 19,Q 7 6 1,T 309,T 18,Q 13 10 3,
  T 158,T 19,Q 12 10 1,T 45,Q 7 6 1,T 233,T 98,Q 11 6 5,T 3,T 83,Q 16 14 9,
  Q 6 5 3,Q 9 7 4,Q 22 19 9,T 168,Q 19 17 4,T 120,Q 14 5 2,Q 17 15 6,T 7,
  Q 10 8 6,T 185,T 93,Q 15 14 7,T 29,T 375,Q 10 8 3,T 13,Q 17 16 2,T 329,
  T 68,Q 13 9 6,T 92,Q 12 10 3,Q 7 6 3,Q 17 10 3,Q 5 2 1,Q 9 6 1,T 30,Q 9 7 3,
  T 253,T 143,Q 7 4 1,Q 9 4 1,Q 12 10 4,T 53,T 25,Q 9 7 1,T 217,Q 15 13 9,
  Q 14 9 2,T 75,Q 8 7 2,T 21,T 7,Q 14 3 2,T 15,T 159,Q 12 10 8,T 29,Q 10 3 1,
  T 21,T 333,Q 11 8 2,T 52,T 119,Q 16 9 7,T 123,Q 15 11 2,T 17,T 9,Q 11 6 4,
  T 38,T 255,Q 12 10 7,T 189,Q 4 3 1,Q 17 10 7,T 49,Q 13 5 2,T 149,T 15,
  Q 14 7 5,Q 10 9 2,Q 8 6 5,T 61,T 54,Q 11 5 1,T 144,T 47,Q 11 10 7,T 105,
  T 2,T 105,T 136,Q 11 4 1,T 253,T 111,Q 13 10 5,T 159,Q 10 7 1,Q 7 5 3,
  T 29,Q 19 10 3,T 119,T 207,Q 17 15 4,T 35,T 14,T 349,Q 6 3 2,Q 21 10 6,
  T 1,T 75,Q 9 5 2,T 145,Q 11 7 6,T 301,T 378,Q 13 3 1,T 352,Q 12 7 4,Q 12 8 1,
  T 149,Q 6 5 4,Q 12 9 8,T 11,Q 15 7 5,T 78,T 99,Q 17 16 12,T 173,Q 8 7 1,
  Q 13 9 8,T 147,Q 19 18 10,T 127,T 183,Q 12 4 1,T 31,Q 11 8 6,T 173,T 12,
  Q 7 5 3,T 113,T 207,Q 18 15 5,T 1,Q 13 7 6,T 21,T 35,Q 12 7 2,T 117,T 123,
  Q 12 10 2,T 143,Q 14 4 1,Q 15 9 7,T 204,Q 7 5 1,T 91,Q 4 2 1,Q 8 6 3,
  T 183,Q 12 10 7,T 77,T 36,Q 14 9 6,T 221,Q 7 6 5,Q 16 14 13,T 31,Q 16 15 7,
  T 365,T 403,Q 10 3 2,Q 11 4 3,T 31,Q 10 9 4,T 177,Q 16 6 1,Q 22 6 5,T 417,
  Q 15 13 12,T 217,T 207,Q 7 5 4,Q 10 7 1,Q 11 6 1,T 45,T 24,Q 12 11 9,
  T 77,Q 21 20 13,Q 9 6 5,T 189,Q 8 3 2,Q 13 12 10,T 260,Q 16 9 7,T 168,
  T 131,Q 7 6 3,T 305,Q 10 9 6,Q 13 9 4,T 143,Q 12 9 3,T 18,Q 15 8 5,Q 20 9 6,
  T 103,Q 15 4 2,T 201,T 36,Q 9 5 2,T 31,Q 11 7 2,Q 6 2 1,T 7,Q 13 6 4,
  Q 9 8 7,T 19,Q 17 10 6,T 15,Q 9 3 1,T 178,Q 8 7 6,Q 12 6 5,T 177,T 230,
  Q 24 9 3,T 222,T 3,Q 16 13 12,T 121,Q 10 4 2,T 161,T 39,Q 17 15 13,T 62,
  T 223,Q 15 12 2,T 65,Q 12 6 3,T 101,T 59,Q 5 4 3,T 17,Q 5 3 2,Q 13 8 3,
  Q 10 9 7,Q 12 8 2,Q 5 4 3,T 75,Q 19 17 8,T 55,T 99,Q 10 7 4,T 115,Q 9 8 6,
  T 385,T 186,Q 15 6 3,Q 9 4 1,Q 12 10 5,Q 10 8 1,T 135,Q 5 2 1,T 317,T 7,
  Q 19 6 1,T 294,T 35,Q 13 12 6,T 119,T 98,T 93,T 68,Q 21 15 3,T 108,T 75,
  Q 12 6 5,T 411,Q 12 7 2,Q 13 7 2,T 21,Q 15 10 8,T 412,T 439,Q 10 7 6,
  T 41,Q 13 9 6,Q 8 5 2,T 10,Q 15 7 2,T 141,T 159,Q 13 12 10,T 291,Q 10 9 1,
  T 105,T 24,Q 11 2 1,T 198,T 27,Q 6 3 1,T 439,Q 10 3 1,T 49,T 168,Q 13 11 9,
  T 463,Q 10 9 3,Q 13 9 8,Q 15 8 3,Q 18 16 8,Q 15 14 11,T 7,Q 19 9 8,Q 12 6 3,
  Q 7 4 3,Q 15 14 5,Q 8 6 3,Q 10 9 7,T 361,T 230,Q 15 9 6,T 24,T 407,Q 16 7 2,
  T 189,T 62,T 189,T 112,Q 22 21 10,T 91,T 79,Q 12 10 5,T 23,Q 7 6 1,T 57,
  T 139,Q 24 15 6,T 14,T 83,Q 16 9 1,T 35,Q 9 7 4,T 117,T 65,Q 21 9 6,T 21,
  T 195,Q 23 11 10,T 327,Q 17 14 3,T 417,T 13,Q 15 8 6,T 107,Q 19 10 6,
  Q 18 15 3,T 59,Q 12 10 4,Q 9 7 5,T 283,Q 13 9 6,T 62,T 427,Q 14 7 3,Q 8 7 4,
  Q 15 8 3,T 105,T 27,Q 7 3 1,T 103,T 551,Q 10 6 1,Q 6 4 1,Q 11 6 4,T 129,
  T 9,Q 9 4 2,T 277,T 31,Q 13 12 5,T 141,Q 12 7 3,T 357,Q 7 2 1,Q 11 9 7,
  T 227,T 131,Q 7 6 3,T 23,Q 20 17 3,Q 13 4 1,T 90,Q 15 3 2,T 241,T 75,
  Q 13 6 1,T 307,Q 8 7 3,T 245,T 66,Q 15 11 2,T 365,Q 18 16 11,Q 11 10 1,
  T 19,Q 8 6 1,T 189,T 133,Q 12 7 2,T 114,T 27,Q 6 5 1,Q 15 5 2,Q 17 14 5,
  T 133,T 476,Q 11 9 3,T 16,T 375,Q 15 8 6,T 25,Q 17 11 6,T 77,T 87,Q 5 3 2,
  T 134,T 171,Q 13 8 4,T 75,Q 8 3 1,T 233,T 196,Q 9 8 7,T 173,Q 15 14 12,
  Q 13 6 5,T 281,Q 9 8 2,T 405,T 114,Q 15 9 6,T 171,T 287,Q 8 4 2,T 43,
  Q 4 2 1,T 513,T 273,Q 11 10 6,T 118,T 243,Q 14 7 1,T 203,Q 9 5 2,T 257,
  T 302,Q 27 25 9,T 393,T 91,Q 12 10 6,T 413,Q 15 14 9,Q 18 16 1,T 255,
  Q 12 9 7,T 234,T 167,Q 16 13 10,T 27,Q 15 6 2,T 433,T 105,Q 25 10 2,T 151,
  T 427,Q 13 9 8,T 49,Q 10 6 4,T 153,T 4,Q 17 7 5,T 54,T 203,Q 16 15 1,
  Q 16 14 7,Q 13 6 1,T 25,T 14,Q 15 5 3,T 187,Q 15 13 10,Q 13 10 5,T 97,
  Q 11 10 9,Q 19 10 4,T 589,Q 31 30 2,T 289,Q 9 6 4,Q 11 8 6,T 21,Q 7 4 1,
  Q 7 4 2,T 77,Q 5 3 2,T 119,T 7,Q 9 5 2,T 345,Q 17 10 8,T 333,T 17,Q 16 9 7,
  T 168,Q 15 13 4,Q 11 10 1,T 217,Q 18 11 10,T 189,T 216,Q 12 7 5,T 229,
  T 231,Q 12 9 3,T 223,Q 10 9 1,T 153,T 470,Q 23 16 6,T 99,Q 10 4 3,Q 9 8 4,
  Q 12 10 1,Q 14 9 6,T 201,T 38,Q 15 14 2,T 198,T 399,Q 14 11 5,T 75,Q 11 10 1,
  T 77,Q 16 12 8,Q 20 17 15,T 326,T 39,Q 14 12 9,T 495,Q 8 3 2,T 333,T 476,
  Q 15 14 2,T 164,T 19,Q 12 4 2,Q 8 6 3,Q 13 12 3,Q 12 11 5,T 129,Q 12 9 3,
  T 52,Q 10 8 3,Q 17 16 2,T 337,Q 12 9 3,T 397,T 277,Q 21 11 3,T 73,Q 11 6 1,
  Q 7 5 4,T 95,Q 11 3 2,T 617,T 392,Q 8 3 2,T 75,T 315,Q 15 6 4,T 125,Q 6 5 2,
  Q 15 9 7,T 348,Q 15 6 1,T 553,Q 6 3 2,Q 10 9 7,T 553,Q 14 10 4,T 237,
  T 39,Q 17 14 6,T 371,T 255,Q 8 4 1,T 131,Q 14 6 1,T 117,T 98,Q 5 3 2,
  T 56,T 655,Q 9 5 2,T 239,Q 11 8 4,T 1,T 134,Q 15 9 5,T 88,Q 10 5 3,Q 10 9 4,
  T 181,Q 15 11 2,T 609,T 52,Q 19 18 10,T 100,Q 7 6 3,Q 15 8 2,T 183,Q 18 7 6,
  Q 10 9 2,T 130,Q 11 5 1,T 12,T 219,Q 13 10 7,T 11,Q 19 9 4,T 129,T 3,
  Q 17 15 5,T 300,Q 17 13 9,Q 14 6 5,T 97,Q 13 8 3,T 601,T 55,Q 8 3 1,T 92,
  T 127,Q 12 11 2,T 81,Q 15 10 8,Q 13 2 1,T 47,Q 14 13 6,T 194,T 383,Q 25 14 11,
  T 125,Q 20 19 16,T 429,T 282,Q 10 9 6,T 342,Q 5 3 2,Q 15 9 4,T 33,Q 9 4 2,
  T 49,T 15,Q 11 6 2,T 28,T 103,Q 18 17 8,T 27,Q 11 6 5,T 33,T 17,Q 11 10 6,
  T 387,T 363,Q 15 10 9,T 83,Q 7 6 4,T 357,Q 13 12 4,Q 14 13 7,T 322,T 395,
  Q 16 5 1,T 595,Q 13 10 3,T 421,T 195,Q 11 3 2,T 13,Q 16 12 3,Q 14 3 1,
  T 315,Q 26 10 5,T 297,T 52,Q 9 4 2,T 314,T 243,Q 16 14 9,T 185,Q 12 5 3,
  Q 13 5 2,T 575,Q 12 9 3,T 39,T 311,Q 13 5 2,T 181,Q 20 18 14,T 49,T 25,
  Q 11 4 1,T 77,Q 17 11 10,Q 15 14 8,T 21,Q 17 10 5,T 69,T 49,Q 11 10 2,
  T 32,T 411,Q 21 16 3,Q 11 7 4,Q 22 10 3,T 85,T 140,Q 9 8 6,T 252,T 279,
  Q 9 5 2,T 307,Q 17 10 4,Q 13 12 9,T 94,Q 13 11 4,T 49,Q 17 11 10,Q 16 12 5,
  T 25,Q 6 5 2,Q 12 5 1,T 80,Q 8 3 2,T 246,Q 11 5 2,Q 11 10 2,T 599,Q 18 12 10,
  T 189,T 278,Q 10 9 3,T 399,T 299,Q 13 10 6,T 277,Q 13 10 6,T 69,T 220,
  Q 13 10 3,T 229,Q 18 11 10,Q 16 15 1,T 27,Q 18 9 3,T 473,T 373,Q 18 17 7,
  T 60,T 207,Q 13 9 8,Q 22 20 13,Q 25 18 7,T 225,T 404,Q 21 6 2,T 46,Q 6 2 1,
  Q 17 12 6,T 75,Q 4 2 1,T 365,T 445,Q 11 7 1,T 44,Q 10 8 5,Q 12 5 2,T 63,
  Q 17 4 2,T 189,T 557,Q 19 12 2,T 252,T 99,Q 10 8 5,T 65,Q 14 9 3,T 9,
  T 119,Q 8 5 2,T 339,T 95,Q 12 9 7,T 7,Q 13 10 2,T 77,T 127,Q 21 10 7,
  T 319,T 667,Q 17 10 3,T 501,Q 18 12 9,Q 9 8 5,T 17,Q 20 9 2,T 341,T 731,
  Q 7 6 5,T 647,Q 10 4 2,T 121,T 20,Q 21 19 13,T 574,T 399,Q 15 10 7,T 85,
  Q 16 8 3,T 169,T 15,Q 12 7 5,T 568,Q 10 7 1,Q 18 2 1,T 3,Q 14 3 2,Q 13 7 3,
  T 643,Q 14 11 1,T 548,T 783,Q 14 11 1,T 317,Q 7 6 4,T 153,T 87,Q 15 13 1,
  T 231,Q 11 5 3,Q 18 13 7,T 771,Q 30 20 11,Q 15 6 3,T 103,Q 13 4 3,T 182,
  T 211,Q 17 6 1,T 27,Q 13 12 10,Q 15 14 10,T 17,Q 13 11 5,T 69,Q 11 5 1,
  Q 18 6 1,T 603,Q 10 4 2,T 741,T 668,Q 17 15 3,T 147,T 227,Q 15 10 9,T 37,
  Q 16 6 1,T 173,T 427,Q 7 5 1,T 287,T 231,Q 20 15 10,Q 18 9 1,Q 14 12 5,
  Q 16 5 1,T 310,Q 18 13 1,T 434,T 579,Q 18 13 8,T 45,Q 12 8 3,Q 16 9 5,
  T 53,Q 19 15 10,T 16,Q 17 6 5,Q 17 10 1,T 37,Q 17 10 9,Q 21 13 7,T 99,
  Q 17 9 6,T 176,T 271,Q 18 17 13,T 459,Q 21 17 10,Q 6 5 2,T 202,Q 5 4 3,
  T 90,T 755,Q 15 7 2,T 363,Q 8 4 2,T 129,T 20,Q 11 6 2,T 135,Q 15 8 7,
  Q 14 13 2,Q 10 4 3,Q 24 13 10,Q 19 14 11,T 31,Q 15 8 6,T 758,Q 16 11 5,
  Q 16 5 1,T 359,Q 23 18 17,T 501,T 29,Q 15 6 3,T 201,T 459,Q 12 10 7,T 225,
  Q 22 17 13,Q 24 22 5,T 161,Q 14 11 3,T 52,Q 19 17 6,Q 21 14 12,T 93,Q 13 10 3,
  T 201,T 178,Q 15 12 5,T 250,Q 7 6 4,Q 17 13 6,T 221,Q 13 11 8,Q 17 14 9,
  T 113,Q 17 14 10,T 300,T 39,Q 18 13 3,T 261,Q 15 14 8,T 753,Q 8 4 3,Q 11 10 5,
  T 94,Q 15 13 1,Q 10 4 2,Q 14 11 10,Q 8 6 2,T 461,T 418,Q 19 14 6,T 403,
  T 267,Q 10 9 2,T 259,Q 20 4 3,T 869,T 173,Q 19 18 2,T 369,T 255,Q 22 12 9,
  T 567,Q 20 11 7,T 457,T 482,Q 6 3 2,T 775,Q 19 17 6,Q 6 4 3,T 99,Q 15 14 8,
  Q 6 5 2,T 165,Q 8 3 2,Q 13 12 10,Q 25 21 17,Q 17 14 9,T 105,Q 17 15 14,
  Q 10 3 2,T 250,Q 25 6 5,T 327,T 279,Q 13 6 5,T 371,Q 15 9 4,T 117,T 486,
  Q 10 9 3,T 217,T 635,Q 30 27 17,T 457,Q 16 6 2,T 57,T 439,Q 23 21 6,T 214,
  Q 20 13 6,Q 20 16 1,T 819,Q 15 11 8,T 593,T 190,Q 17 14 3,T 114,Q 21 18 3,
  Q 10 5 2,Q 12 9 5,Q 8 6 3,T 69,T 312,Q 22 5 2,T 502,T 843,Q 15 10 3,T 747,
  Q 6 5 2,T 101,T 123,Q 19 16 9,T 521,T 171,Q 16 7 2,Q 12 6 5,Q 22 21 20,
  T 545,T 163,Q 23 18 1,T 479,T 495,Q 13 6 5,T 11,Q 17 5 2,Q 18 8 1,T 684,
  Q 7 5 1,T 9,Q 18 11 3,Q 22 20 13,T 273,Q 4 3 2,T 381,T 51,Q 18 13 7,T 518,
  Q 9 5 1,Q 14 12 3,T 243,Q 21 17 2,T 53,T 836,Q 21 10 2,T 66,Q 12 10 7,
  Q 13 9 8,T 339,Q 16 11 5,T 901,T 180,Q 16 13 3,T 49,Q 6 3 2,Q 15 4 1,
  Q 16 13 6,Q 18 15 12,T 885,T 39,Q 11 9 4,T 688,Q 16 15 7,Q 13 10 6,T 13,
  Q 25 23 12,T 149,T 260,Q 11 9 1,T 53,T 11,Q 12 4 2,Q 9 7 5,Q 11 8 1,T 121,
  T 261,Q 10 5 2,T 199,Q 20 4 3,Q 17 9 2,Q 13 9 4,Q 12 8 7,T 253,T 174,
  Q 15 4 2,T 370,Q 9 6 1,Q 16 10 9,T 669,Q 20 10 9,T 833,T 353,Q 17 13 2,
  T 29,T 371,Q 9 8 5,Q 8 7 1,Q 19 8 7,Q 12 11 10,T 873,Q 26 11 2,Q 12 9 1,
  Q 10 7 2,Q 13 6 1,T 235,Q 26 24 19,T 733,T 778,Q 12 11 1,T 344,T 931,
  Q 16 6 4,T 945,Q 21 19 14,Q 18 13 11,T 67,Q 20 15 10,T 462,Q 14 5 1,Q 10 9 6,
  Q 18 11 10,Q 16 9 7,T 477,T 105,Q 11 3 2,T 468,Q 23 16 15,Q 16 15 6,T 327,
  Q 23 10 4,T 357,T 25,Q 17 16 7,T 31,Q 7 5 2,Q 16 7 6,T 277,Q 14 13 6,
  T 413,T 103,Q 15 10 1,T 231,T 747,Q 5 2 1,T 113,Q 20 10 7,Q 15 9 6,T 11,
  Q 27 22 18,T 91,T 51,Q 18 13 12,T 603,Q 10 7 3,T 9,T 121,Q 15 14 6,T 17,
  Q 16 11 2,Q 23 15 6,T 279,Q 16 12 6,T 89,T 371,Q 17 15 2,T 771,T 99,Q 7 6 3,
  T 21,Q 10 7 5,T 801,T 26,Q 25 19 14,T 175,Q 10 7 2,Q 20 5 4,Q 12 11 1,
  Q 22 5 1,T 165,T 841,Q 25 19 17,T 238,Q 11 8 6,Q 22 21 4,T 33,Q 8 7 6,
  Q 14 9 2,T 113,Q 13 11 5,T 311,T 891,Q 20 16 14,T 555,Q 23 14 8,T 133,
  T 546,Q 6 3 2,T 103,T 15,Q 10 7 3,T 307,Q 14 10 1,Q 15 12 2,T 367,Q 13 10 6,
  T 169,Q 22 21 11,Q 12 10 8,T 441,Q 17 12 7,T 917,T 205,Q 26 23 13,T 54,
  T 459,Q 17 15 4,Q 19 15 4,Q 5 4 2,Q 9 7 6,T 42,Q 21 15 7,T 330,Q 20 7 3,
  Q 20 7 2,T 81,Q 19 14 1,T 349,T 165,Q 40 35 9,T 274,T 475,Q 11 10 3,T 93,
  Q 12 7 4,Q 13 12 2,T 386,Q 7 6 2,T 881,T 143,Q 9 8 4,T 71,Q 19 18 3,Q 16 11 6,
  T 155,Q 7 2 1,T 735,Q 16 8 7,Q 9 7 4,T 45,Q 7 6 4,Q 12 11 3,T 3,Q 19 14 13,
  T 124,Q 15 13 8,Q 13 6 5,T 323,Q 21 13 6,T 201,T 11,Q 13 12 3,T 245,T 343,
  Q 14 12 10,T 387,Q 19 4 1,Q 16 3 2,T 48,Q 17 9 2,T 97,T 71,Q 17 13 8,
  Q 18 10 7,Q 18 9 8,T 237,Q 11 5 3,Q 13 10 3,T 253,T 231,Q 9 7 4,T 851,
  Q 15 14 4,Q 16 6 5,T 35,Q 4 3 1,T 467,T 523,Q 21 11 10,Q 4 2 1,Q 9 8 3,
  T 261,T 141,Q 18 11 5,T 150,Q 9 4 1,Q 12 9 5,Q 17 15 7,Q 16 15 7,T 645,
  T 256,Q 19 4 2,T 119,T 19,Q 15 12 9,T 35,Q 25 22 9,T 33,T 98,Q 19 15 9,
  T 153,T 111,Q 17 10 2,Q 21 5 3,Q 10 5 1,Q 12 9 6,T 249,Q 16 13 7,T 385,
  T 155,Q 11 10 1,T 25,Q 24 16 11,T 385,T 84,Q 17 14 6,T 304,T 91,Q 14 11 3,
  T 45,Q 24 17 14,T 881,T 539,Q 23 9 1,T 21,T 239,Q 13 6 5,T 213,Q 24 22 4,
  Q 23 13 2,T 47,Q 15 12 9,T 331,Q 13 9 2,Q 14 4 1,T 283,Q 16 3 1,T 69,
  T 345,Q 13 7 3,T 19,T 595,Q 8 3 2,T 549,Q 17 9 2,T 569,T 224,Q 24 13 7,
  T 582,Q 10 7 5,Q 10 9 8,T 405,Q 14 4 1,T 93,T 6,Q 31 25 14,T 766,T 47,
  Q 12 9 7,T 561,Q 10 4 2,T 693,T 840,Q 11 9 3,T 55,T 411,Q 7 6 4,Q 6 4 1,
  Q 15 8 4,T 225,T 128,Q 15 8 1,T 554,T 15,Q 8 7 2,T 111,Q 18 12 7,T 93,
  T 162,Q 11 10 5,T 51,T 51,Q 22 11 1,T 99,Q 19 8 7,T 441,T 111,Q 8 5 3,
  T 71,Q 15 13 9,Q 23 22 16,T 539,Q 6 5 2,T 893,T 49,Q 20 15 5,T 143,Q 15 3 2,
  Q 14 6 5,Q 11 7 1,Q 14 7 4,T 793,T 438,Q 21 16 6,T 142,T 539,Q 20 14 3,
  T 423,Q 20 19 4,T 1041,T 39,Q 24 7 2,T 455,T 603,Q 22 12 11,T 7,Q 17 16 6,
  T 333,Q 17 6 2,Q 21 19 5,T 47,Q 19 16 7,Q 14 9 8,T 425,Q 17 8 7,T 637,
  T 654,Q 19 17 4,T 249,Q 7 6 1,Q 20 17 11,T 63,Q 7 4 2,T 1053,T 120,Q 23 7 1,
  T 20,T 7,Q 27 15 2,T 399,Q 22 12 11,Q 23 15 6,T 217,Q 9 4 3,T 126,T 927,
  Q 19 16 13,T 75,Q 19 14 2,Q 10 9 2,T 729,Q 14 9 6,T 829,T 983,Q 16 10 6,
  Q 12 4 1,Q 14 12 7,T 57,T 273,Q 15 7 2,T 151,T 343,Q 18 17 8,T 115,Q 15 10 7,
  T 369,T 560,Q 21 10 9,T 630,T 239,Q 15 12 1,T 21,Q 10 4 2,Q 17 14 7,T 276,
  Q 13 4 2,T 715,T 975,Q 20 13 4,T 889,Q 8 6 2,T 249,T 651,Q 17 16 7,T 136,
  Q 23 6 5,Q 13 10 2,T 89,Q 10 8 3,Q 21 17 10,T 259,Q 15 10 1,T 405,Q 15 13 3,
  Q 16 6 1,T 95,Q 15 9 8,Q 15 8 1,T 80,Q 8 7 5,T 424,T 551,Q 11 7 2,T 31,
  Q 12 10 8,T 233,T 148,Q 19 6 4,T 221,T 879,Q 17 15 4,T 21,Q 17 4 2,T 245,
  T 161,Q 13 11 5,T 543,T 83,Q 16 3 2,T 717,Q 14 8 5,Q 13 10 7,T 32,Q 15 9 2,
  T 105,Q 15 5 1,T 14,T 349,Q 18 15 8,T 1125,T 553,Q 15 10 8,T 523,T 211,
  Q 10 3 2,T 39,Q 24 18 16,T 65,T 415,Q 27 26 14,T 29,T 987,Q 11 10 2,T 731,
  Q 31 16 9,Q 21 19 4,T 950,Q 23 20 2,T 328,Q 14 11 6,Q 12 11 6,T 183,Q 10 9 8,
  T 161,T 172,Q 19 10 8,T 646,Q 13 10 6,Q 9 7 4,T 643,Q 21 14 5,Q 16 13 6,
  T 610,Q 13 11 8,T 77,Q 12 11 6,Q 20 18 17,T 1139,Q 17 14 5,Q 24 16 13,
  T 198,Q 7 5 4,T 381,T 243,Q 22 9 3,T 1,Q 18 12 2,T 429,T 49,Q 21 19 1,
  T 607,Q 11 9 1,Q 8 7 6,T 11,Q 31 12 10,T 629,T 956,Q 31 13 3,T 59,T 423,
  Q 17 8 7,T 173,Q 22 17 4,Q 15 13 11,T 107,Q 20 19 17,T 61,T 251,Q 11 8 2,
  T 67,Q 17 14 5,Q 14 12 5,T 91,Q 23 6 4,T 1198,T 807,Q 12 2 1,T 25,Q 11 6 1,
  T 29,T 154,Q 23 6 5,T 225,T 311,Q 22 16 6,T 77,Q 11 8 4,T 1117,T 102,
  Q 21 16 6,T 678,Q 20 4 3,Q 8 6 5,T 301,Q 22 14 7,T 477,T 303,Q 29 22 19,
  T 305,T 507,Q 18 6 2,T 145,Q 9 4 3,T 929,T 404,Q 12 7 5,T 339,T 127,Q 15 13 4,
  T 1115,Q 23 20 10,Q 18 13 6,T 786,Q 21 10 4,T 621,T 191,Q 10 4 3,T 331,
  Q 21 14 11,T 357,T 313,Q 12 5 3,T 238,Q 23 20 18,Q 17 7 4,T 35,Q 19 18 10,
  Q 22 13 8,T 1172,Q 5 4 3,T 531,T 599,Q 18 14 2,T 99,Q 26 16 11,T 217,
  Q 15 6 3,Q 12 3 1,T 225,T 899,Q 12 11 9,Q 17 3 2,Q 19 17 6,T 765,T 72,
  Q 20 5 2,T 710,Q 11 7 6,Q 12 11 2,T 523,T 142,Q 19 14 9,T 155,Q 23 13 9,
  T 315,Q 8 7 5,Q 25 16 12,T 141,Q 18 15 7,Q 13 8 2,T 497,Q 12 3 1,T 1171,
  Q 8 7 4,Q 13 12 9,T 135,Q 22 21 5,T 45,T 316,Q 19 8 6,T 131,Q 17 11 3,
  Q 13 8 1,T 25,Q 14 13 3,T 1113,T 110,Q 29 21 7,T 99,T 183,Q 8 7 5,T 563,
  Q 14 4 1,Q 18 13 2,T 579,Q 31 15 13,T 426,Q 16 10 5,Q 23 17 14,Q 15 6 4,
  Q 7 6 5,T 141,T 640,Q 19 9 4,T 49,Q 14 5 3,Q 6 2 1,Q 26 22 13,Q 10 3 1,
  T 185,Q 24 19 16,Q 21 10 9,T 77,T 315,Q 10 9 3,T 209,Q 11 8 7,T 97,T 240,
  Q 21 20 6,T 982,T 891,Q 22 10 3,T 373,Q 10 9 5,T 333,T 103,Q 28 3 2,T 28,
  T 1123,Q 9 6 2,T 349,Q 18 17 7,Q 18 8 1,T 23,Q 9 3 1,T 201,T 203,Q 12 11 10,
  T 561,Q 25 16 14,T 37,T 122,Q 8 5 2,T 69,Q 18 15 14,Q 18 16 9,T 535,Q 12 11 3,
  T 5,T 867,Q 7 2 1,T 674,Q 15 7 3,Q 23 6 1,T 105,Q 26 14 12,Q 22 19 15,
  T 31,Q 25 19 12,T 263,T 1047,Q 23 12 10,Q 13 8 1,Q 29 11 10,T 1017,T 219,
  Q 15 12 5,T 297,T 863,Q 24 17 2,T 145,Q 16 8 7,T 225,T 289,Q 14 13 7,
  T 406,Q 11 6 1,Q 18 8 7,T 435,Q 19 14 5,T 1181,T 34,Q 15 11 2,T 425,T 427,
  Q 27 17 10,Q 21 14 6,Q 14 12 9,T 553,T 518,Q 17 8 7,T 462,T 71,Q 17 10 1,
  T 835,Q 8 7 1,Q 11 5 3,T 409,Q 15 10 4,T 112,T 43,Q 20 17 11,T 47,Q 13 9 6,
  T 177,T 139,Q 19 5 3,T 1241,Q 20 11 5,Q 25 21 14,Q 18 11 10,Q 9 6 4,Q 10 3 1,
  T 144,Q 23 11 9,T 736,T 551,Q 16 13 10,T 597,Q 18 11 10,T 297,T 513,Q 15 8 1,
  T 689,Q 17 13 5,Q 7 5 4,T 519,Q 17 4 2,Q 20 16 13,T 53,Q 19 11 5,T 242,
  Q 6 3 2,Q 20 18 16,T 5,Q 17 14 2,Q 14 12 7,T 458,Q 27 21 19,T 772,T 663,
  T 254,T 819,Q 18 4 2,T 229,T 46,Q 18 7 1,T 530,T 967,Q 13 10 9,T 93,Q 17 8 6,
  Q 15 6 5,T 286,Q 15 9 4,T 635,T 463,Q 11 6 1,Q 14 12 3,Q 8 2 1,T 789,
  T 225,Q 21 10 6,T 36,Q 12 9 3,Q 14 10 8,T 577,Q 10 5 3,T 621,T 123,Q 17 15 12,
  T 170,T 963,Q 32 30 29,T 3,Q 12 10 5,T 257,T 67,Q 12 9 7,Q 12 10 5,T 515,
  Q 9 6 4,T 423,Q 10 9 3,Q 7 3 1,T 690,Q 21 12 7,T 840,Q 12 8 7,Q 30 26 15,
  T 255,Q 14 8 3,T 369,T 102,Q 25 18 1,T 826,T 127,Q 9 6 5,T 121,Q 21 17 2,
  Q 10 6 1,T 430,Q 21 7 5,T 96,T 343,Q 15 11 2,T 845,Q 19 8 7,Q 9 5 4,T 933,
  Q 16 3 1,T 226,T 923,Q 12 9 5,T 109,Q 6 5 4,T 149,T 447,Q 19 18 10,T 484,
  Q 9 7 2,Q 15 11 6,T 25,Q 22 18 17,T 629,T 49,Q 15 4 2,T 716,T 231,Q 13 7 6,
  T 159,Q 24 23 12,Q 17 5 4,T 842,Q 29 26 7,T 108,T 1319,Q 12 10 6,T 687,
  Q 16 10 3,T 1285,T 102,Q 25 19 15,T 269,T 567,Q 13 12 5,T 135,Q 30 25 20,
  Q 28 3 2,T 802,Q 7 3 2,Q 22 21 17,T 1095,Q 20 17 9,T 51,Q 28 27 10,Q 22 10 9,
  T 168,Q 29 21 15,T 349,T 339,Q 19 18 3,Q 21 16 2,Q 14 12 8,T 837,T 490,
  Q 12 7 2,T 343,Q 11 9 4,Q 10 8 4,T 769,Q 19 6 1,Q 20 14 5,T 880,Q 17 14 6,
  T 279,Q 18 14 3,Q 18 16 13,T 609,Q 24 8 2,T 729,T 270,Q 15 13 1,T 1342,
  Q 23 10 9,Q 10 9 7,T 453,Q 13 7 6,T 621,T 84,Q 21 19 8,T 109,Q 15 9 1,
  Q 10 6 5,T 815,Q 16 6 4,Q 18 17 3,T 592,Q 15 14 10,T 288,T 135,Q 19 10 6,
  T 1103,Q 9 6 4,Q 17 15 13,T 186,Q 27 18 1,T 409,Q 15 13 7,Q 20 13 5,T 1113,
  Q 17 8 3,Q 20 4 1,T 1033,Q 20 15 9,T 370,T 1231,Q 7 3 2,T 25,Q 10 9 1,
  Q 23 15 4,T 329,Q 15 8 1,T 114,T 1411,Q 10 7 1,T 1145,Q 14 8 1,T 313,
  T 41,Q 15 13 3,T 756,Q 17 9 7,Q 29 20 11,T 603,Q 20 16 10,T 405,T 139,
  Q 21 17 15,T 212,Q 9 7 2,Q 15 13 10,T 915,Q 8 6 1,Q 12 11 1,T 272,Q 21 5 2,
  T 75,Q 13 6 3,Q 20 16 2,T 605,Q 10 7 4,T 781,T 149,Q 13 10 6,T 1201,T 1431,
  Q 16 13 12,T 529,Q 13 11 6,Q 20 14 9,T 469,Q 11 4 1,T 76,T 31,Q 16 15 10,
  T 309,Q 27 7 2,Q 16 14 9,T 358,Q 29 6 1,T 15,T 91,Q 19 10 1,T 303,Q 11 3 2,
  Q 14 10 9,T 279,Q 27 15 6,T 321,T 1155,Q 17 14 1,Q 19 13 10,Q 23 22 4,
  T 1301,T 685,Q 16 9 2,T 238,T 351,Q 18 7 5,T 21,Q 16 15 4,T 237,T 149,
  Q 19 9 5,T 480,T 559,Q 11 6 5,Q 12 4 1,Q 12 4 3,Q 20 14 1,T 974,Q 24 21 11,
  T 651,Q 9 4 1,Q 13 8 1,Q 14 7 6,Q 15 14 13,T 713,Q 13 12 7,Q 5 3 2,T 172,
  T 499,Q 30 17 5,T 49,Q 23 18 17,T 1425,T 320,Q 5 3 2,T 146,T 551,Q 22 20 11,
  Q 17 3 2,Q 17 7 4,T 397,T 872,Q 17 13 2,T 33,Q 9 6 5,Q 12 10 6,T 823,
  Q 19 14 3,Q 23 13 5,T 69,Q 12 3 2,T 86,T 319,Q 21 14 5,T 83,Q 25 22 15,
  T 861,T 1028,Q 29 27 4,T 561,T 583,Q 18 13 2,T 693,Q 18 10 4,Q 11 3 1,
  T 192,Q 21 10 3,T 126,T 375,Q 12 11 6,T 381,Q 13 2 1,T 669,T 330,Q 17 9 6,
  T 166,T 343,Q 8 3 2,T 313,Q 18 9 7,Q 26 22 9,T 292,Q 23 3 1,T 569,T 303,
  Q 9 6 4,T 345,Q 12 6 5,T 669,T 1011,Q 15 12 9,T 975,Q 22 21 10,Q 12 11 5,
  T 351,Q 14 12 5,Q 15 9 6,T 963,Q 15 13 1,T 1349,Q 25 12 10,Q 22 8 6,T 1327,
  Q 23 6 2,Q 17 15 5,T 308,Q 38 25 9,T 108,T 203,Q 16 6 1,T 413,Q 22 10 1,
  Q 14 12 1,T 734,Q 32 3 2,T 757,Q 19 18 13,Q 17 16 4,T 135,Q 11 6 4,Q 12 9 4,
  T 55,Q 17 15 4,T 238,T 399,Q 21 20 2,T 391,Q 7 6 3,T 633,T 436,Q 27 21 3,
  T 776,T 415,Q 18 16 15,T 69,Q 17 14 11,T 1021,Q 19 15 4,Q 18 3 2,T 765,
  T 651,Q 19 17 16,T 363,Q 22 20 15,Q 21 4 3,Q 13 7 1,Q 5 4 3,T 110,T 811,
  Q 15 10 1,T 405,Q 22 15 1,T 1053,T 32,Q 25 11 9,T 432,T 455,Q 18 16 13,
  T 215,Q 34 26 19,Q 20 13 8,T 65,Q 11 10 5,T 184,Q 17 9 3,Q 16 14 10,T 475,
  Q 12 10 8,T 105,T 174,Q 21 19 16,T 64,Q 9 6 1,Q 23 20 18,T 109,Q 25 14 12,
  T 1281,T 49,Q 20 13 11,T 261,T 279,Q 12 7 5,T 45,Q 14 11 8,T 769,T 419,
  Q 33 29 14,T 1162,Q 18 17 11,Q 14 13 11,T 45,Q 10 7 3,T 225,T 124,Q 23 9 5,
  T 833,Q 6 2 1,Q 14 12 11,T 61,Q 26 20 19,T 1421,T 199,Q 17 15 1,T 191,
  Q 19 15 4,Q 25 18 16,T 461,Q 19 8 4,T 525,T 315,Q 18 17 11,T 493,Q 22 7 6,
  Q 15 10 4,T 861,Q 24 21 18,T 449,T 139,Q 30 19 11,T 23,T 867,Q 22 8 7,
  T 123,Q 6 4 3,T 89,T 356,Q 15 12 10,T 587,Q 29 19 13,Q 14 11 10,T 1115,
  Q 23 18 12,T 981,T 8,Q 23 21 8,T 112,Q 18 11 6,Q 17 10 7,T 1171,Q 22 3 2,
  T 253,T 1254,Q 21 17 6,T 98,Q 19 17 6,Q 15 12 2,T 565,Q 24 14 10,Q 19 9 5,
  T 103,Q 7 6 2,T 858,T 315,Q 18 13 10,T 113,Q 17 13 10,Q 18 10 1,T 672,
  Q 33 31 18,T 1123,T 783,Q 19 14 13,T 301,Q 20 17 14,T 81,T 646,Q 13 10 5,
  T 484,T 915,Q 22 12 2,T 1085,Q 12 10 3,T 1205,T 1225,Q 11 10 2,T 204,
  T 891,Q 9 8 2,T 129,Q 19 18 12,Q 12 4 1,T 495,Q 25 8 7,T 211,T 1059,Q 19 14 1,
  T 175,Q 22 16 14,T 841,T 54,Q 11 6 4,T 674,Q 24 12 3,Q 14 7 3,T 31,Q 17 9 2,
  Q 15 8 6,T 704,Q 16 13 3,T 81,T 1303,Q 12 10 5,T 1559,Q 30 16 1,T 1197,
  T 614,Q 21 11 3,T 67,Q 10 9 8,Q 24 10 3,T 19,Q 11 6 5,T 145,T 784,Q 23 19 1,
  T 101,Q 9 7 5,Q 8 7 6,T 1225,Q 12 9 7,T 501,Q 15 9 8,Q 12 9 7,T 575,T 511,
  Q 21 11 8,T 887,Q 19 8 4,T 409,T 98,Q 12 3 2,T 127,Q 27 13 7,Q 22 13 5,
  T 1249,Q 11 10 4,T 1221,T 426,Q 15 8 1,T 149,Q 15 11 8,Q 9 6 5,T 567,
  Q 10 5 3,T 1485,T 124,Q 31 26 2,T 806,T 203,Q 22 4 1,T 237,Q 18 12 10,
  Q 15 13 7,T 939,Q 17 5 2,Q 18 16 7,Q 19 2 1,Q 20 19 10,T 73,Q 22 3 2,
  T 237,T 333,Q 23 10 1,T 1408,T 775,Q 24 13 10,T 69,Q 25 22 1,Q 22 12 1,
  T 446,Q 16 15 6,T 47,T 783,Q 30 28 21,Q 24 17 13,Q 18 4 1,T 397,T 717,
  Q 21 18 11,T 43,Q 11 7 3,Q 18 7 1,T 61,Q 20 18 15,T 249,T 594,Q 19 14 13,
  T 7,T 639,Q 18 17 14,T 55,Q 24 10 4,T 605,T 1336,Q 19 17 3,T 806,T 127,
  Q 15 10 2,T 717,Q 23 20 6,T 1,T 618,Q 14 9 3,T 436,T 1019,Q 12 8 2,T 1641,
  Q 22 17 7,T 585,T 58,Q 17 10 4,T 20,T 567,Q 28 14 10,T 173,Q 25 19 10,
  T 1145,T 875,Q 17 9 2,T 525,T 191,Q 18 17 11,T 587,Q 16 8 7,Q 6 4 1,T 636,
  Q 11 10 5,T 370,T 1155,Q 22 16 12,Q 11 7 5,Q 25 19 12,Q 9 6 5,T 73,Q 30 27 15,
  T 796,Q 15 6 1,Q 23 18 16,T 177,Q 20 19 17,T 1401,T 731,Q 21 20 19,T 389,
  Q 10 9 3,Q 10 6 4,T 339,Q 24 17 15,Q 19 8 6,T 99,Q 18 15 5,Q 12 10 4,
  Q 11 7 4,Q 14 10 2,T 85,Q 24 15 2,T 257,T 136,Q 7 5 1,T 1541,Q 15 10 1,
  Q 30 29 18,T 47,Q 14 6 4,T 417,T 49,Q 11 9 1,T 236,T 623,Q 25 20 9,T 659,
  Q 7 4 1,T 217,T 956,Q 21 9 3,T 603,Q 19 9 2,Q 26 25 16,T 169,Q 17 15 4,
  T 1381,T 465,Q 23 13 6,T 1615,Q 13 12 3,Q 22 10 6,Q 13 6 1,Q 19 4 1,T 245,
  T 416,Q 14 13 6,T 531,T 387,Q 15 12 6,T 173,Q 24 9 2,Q 22 13 12,T 507,
  Q 16 15 6,T 244,T 1023,Q 14 8 5,T 325,Q 14 9 6,T 93,T 1272,Q 28 27 1,
  T 32,T 15,Q 12 9 3,T 423,Q 19 14 5,T 1121,T 11,Q 22 15 6,T 189,T 1071,
  Q 16 12 1,Q 17 16 13,Q 16 12 6,T 153,T 153,Q 25 2 1,Q 28 25 12,Q 14 13 12,
  Q 15 14 5,T 159,Q 18 16 10,T 393,T 147,Q 27 16 1,T 394,Q 8 7 3,Q 26 19 3,
  T 69,Q 21 5 2,Q 21 17 8,T 404,Q 17 11 6,T 917,Q 11 8 3,Q 19 14 9,T 1145,
  Q 16 6 1,Q 25 23 21,T 21,Q 19 18 9,T 120,T 519,Q 19 18 12,T 1495,Q 20 10 7,
  T 225,T 289,Q 11 6 3,T 304,T 43,Q 28 26 6,T 921,Q 38 16 6,T 917,T 314,
  Q 17 14 7,T 720,T 735,Q 30 16 13,T 525,Q 16 15 12,T 465,T 155,Q 19 15 13,
  T 546,Q 15 5 4,Q 12 5 2,T 1329,Q 8 7 4,T 1085,T 120,Q 12 11 1,T 518,Q 16 12 3,
  Q 19 14 7,T 57,Q 19 18 1,Q 25 19 9,T 254,Q 35 21 4,T 1025,T 567,Q 29 24 4,
  T 375,Q 15 8 2,Q 15 13 6,T 993,Q 23 17 10,T 103,Q 13 5 3,Q 21 14 6,Q 10 7 6,
  Q 23 12 7,T 81,T 1141,Q 37 35 6,T 41,Q 11 9 4,Q 17 10 9,T 667,Q 22 14 12,
  Q 16 14 9,T 569,Q 32 29 3,T 129,T 399,Q 23 12 2,T 1439,Q 10 7 5,Q 12 11 10,
  T 476,Q 25 18 7,T 270,Q 10 9 5,Q 18 3 1,T 1561,Q 30 3 2,T 973,T 162,Q 12 7 5,
  T 218,Q 13 6 5,Q 16 2 1,T 75,Q 23 7 2,T 345,T 377,Q 21 14 2,T 998,T 151,
  Q 26 23 12,T 255,Q 14 6 3,T 1269,T 183,Q 15 9 6,Q 13 3 2,Q 24 23 17,Q 28 25 15,
  T 127,Q 14 8 5,T 397,T 69,Q 17 3 2,T 257,T 927,Q 18 15 6,T 225,Q 22 17 12,
  Q 8 6 1,Q 24 20 12,Q 21 12 10,T 1028,T 699,Q 30 13 3,T 1143,Q 13 8 2,
  T 889,T 339,Q 19 10 3,T 348,Q 17 9 5,Q 20 14 6,T 915,Q 22 15 2,T 713,
  T 747,Q 25 12 10,T 7,Q 19 14 8,Q 26 6 5,T 843,Q 30 28 8,T 1713,T 509,
  Q 38 33 14,T 72,T 59,Q 28 14 2,T 383,Q 22 9 3,Q 24 5 1,T 114,Q 9 5 2,
  T 669,Q 10 2 1,Q 23 11 6,T 637,Q 8 7 4,T 861,T 142,Q 15 14 10,T 1016,
  Q 12 5 2,Q 18 7 1,T 215,Q 17 7 6,T 29,T 47,Q 25 18 7,T 377,T 1539,Q 13 12 5,
  T 231,Q 22 21 16,T 481,Q 10 9 7,Q 29 27 12,T 279,Q 26 25 13,Q 7 6 4,T 957,
  Q 15 10 2,T 729,T 90,Q 26 17 5,T 553,T 651,Q 15 8 2,T 391,Q 7 6 5,Q 28 8 1,
  T 76,Q 20 15 10,T 1626,T 771,Q 14 13 8,T 1365,Q 21 14 6,Q 20 17 6,T 45,
  Q 23 7 2,T 394,T 1691,Q 15 13 6,T 721,Q 10 9 8,T 273,T 112,Q 17 12 11,
  T 928,T 1471,Q 18 13 2,T 61,Q 16 11 6,T 1365,T 130,Q 35 24 14,T 189,Q 30 20 11,
  Q 15 6 4,T 269,Q 22 7 4,Q 23 4 3,T 101,Q 19 17 8,T 544,Q 27 15 11,Q 30 10 9,
  T 609,Q 25 20 7,T 501,T 21,Q 14 13 7,T 115,T 471,Q 15 13 10,T 81,Q 9 4 3,
  T 81,T 889,Q 32 13 11,T 759,T 839,Q 26 9 2,Q 6 5 3,Q 26 20 18,T 1129,
  T 62,Q 36 33 22,T 91,T 1719,Q 24 21 5,T 675,Q 4 2 1,T 1281,T 429,Q 14 13 1,
  T 148,T 1195,Q 11 6 1,T 147,Q 16 14 6,T 797,T 1735,Q 13 12 7,T 413,T 459,
  Q 20 18 11,Q 24 11 4,Q 18 15 4,Q 23 18 10,T 488,Q 17 15 11,T 31,Q 15 7 5,
  Q 18 6 4,Q 10 9 8,Q 21 14 8,T 609,T 42,Q 9 4 2,T 184,T 1191,Q 26 20 5,
  T 1327,Q 8 7 3,T 1305,T 46,Q 33 22 18,T 287,T 75,Q 18 10 5,T 95,Q 16 15 4,
  Q 25 18 11,T 279,Q 27 14 2,T 684,Q 22 9 7,Q 32 22 11,Q 19 11 8,Q 15 4 1,
  T 1013,T 435,Q 9 4 2,T 407,T 1611,Q 15 13 8,T 291,Q 18 16 5,Q 21 20 9,
  T 208,Q 23 9 1,T 30,T 383,Q 23 10 2,T 1307,Q 28 19 12,Q 21 15 1,T 672,
  Q 14 7 2,T 300,T 107,Q 13 10 9,T 61,Q 10 9 4,Q 24 9 4,T 1416,Q 7 5 4,
  T 1414,Q 9 5 1,Q 23 8 2,T 63,Q 10 9 6,T 1785,T 272,Q 29 13 6,T 87,T 1027,
  Q 14 6 1,T 1173,Q 16 15 4,Q 22 21 17,T 45,Q 20 7 5,T 481,Q 17 4 3,Q 8 7 5,
  T 127,Q 16 8 6,T 1337,T 202,Q 24 23 21,T 112,Q 16 15 8,Q 18 15 6,T 349,
  Q 18 12 9,Q 9 7 5,T 68,Q 29 18 4,T 938,T 323,Q 9 8 4,T 1799,Q 11 8 7,
  Q 22 21 11,T 143,Q 19 13 9,T 252,Q 17 8 6,Q 16 6 3,Q 20 11 3,Q 8 7 6,
  T 29,T 609,Q 19 13 2,T 437,Q 23 8 1,Q 18 13 8,T 1217,Q 13 9 6,T 713,T 310,
  Q 35 13 2,T 35,T 567,Q 15 5 4,T 681,Q 22 18 3,T 273,T 503,Q 27 9 1,T 840,
  T 1331,Q 16 5 2,T 1063,Q 11 10 9,T 693,T 108,Q 29 18 13,T 71,T 583,Q 29 24 19,
  T 169,Q 12 7 5,T 765,T 1399,Q 39 25 3,T 50,T 459,Q 14 8 7,T 35,Q 31 10 2,
  Q 18 16 5,T 834,Q 19 15 9,T 289,T 315,Q 20 14 6,Q 13 12 9,Q 24 22 13,
  T 913,T 264,Q 10 3 2,T 32,Q 20 8 3,Q 11 10 4,T 157,Q 17 11 4,Q 19 9 2,
  T 121,Q 27 5 1,T 810,T 1775,Q 20 9 2,T 45,Q 15 8 3,T 273,T 915,Q 45 42 6,
  T 340,Q 20 19 10,Q 17 9 2,T 289,Q 16 13 2,T 1197,T 777,Q 15 7 5,T 310,
  Q 25 9 1,Q 21 20 12,T 65,Q 26 6 1,T 1845,T 350,Q 17 13 2,T 26,T 251,Q 15 4 1,
  T 855,Q 14 12 11,Q 28 22 13,T 1673,Q 24 11 2,T 393,T 531,Q 25 22 9,T 445,
  Q 16 12 11,T 117,T 285,Q 15 13 8,T 785,Q 26 21 1,Q 24 21 3,T 245,Q 18 16 5,
  Q 17 16 12,T 367,Q 8 7 5,T 1440,T 199,Q 23 9 4,T 1563,Q 30 19 3,Q 28 12 5,
  Q 20 15 8,Q 15 5 3,T 252,T 1835,Q 28 19 10,Q 21 5 2,Q 19 11 6,T 57,T 1125,
  Q 31 29 28,T 427,T 1155,Q 22 10 5,T 293,Q 28 22 3,T 873,T 752,Q 11 6 5,
  T 698,T 503,Q 24 8 5,T 429,Q 18 16 10,Q 27 4 2,T 891,Q 29 15 2,T 756,
  T 255,Q 13 8 1,T 735,Q 14 3 2,T 337,T 357,Q 25 18 14,T 196,T 163,Q 10 7 2,
  T 595,Q 13 11 8,T 861,T 322,Q 36 3 1,T 221,Q 19 9 7,Q 25 9 2,Q 16 9 4,
  Q 21 11 8,Q 21 13 8,T 11,Q 19 5 2,T 1038,Q 12 8 7,Q 11 4 2,T 1017,Q 6 5 2,
  T 469,T 168,Q 27 8 6,T 1468,Q 19 12 9,Q 12 9 8,T 19,Q 16 13 3,T 153,T 1250,
  Q 31 18 17,T 137,Q 24 11 6,Q 14 12 5,T 1479,Q 17 12 2,Q 17 6 1,T 705,
  Q 24 9 6,T 124,Q 18 15 2,Q 28 15 9,Q 21 20 8,Q 18 17 15,T 125,T 249,Q 33 32 23,
  T 22,T 1467,Q 7 6 1,T 375,Q 24 19 16,Q 7 4 2,T 985,Q 16 9 7,T 599,Q 23 20 10,
  Q 9 7 4,Q 22 12 7,Q 14 11 4,T 93,T 1805,Q 15 13 6,T 223,T 1163,Q 25 16 1,
  T 157,Q 21 14 10,T 953,T 1408,Q 29 20 15,T 410,Q 23 21 13,Q 6 5 1,T 1659,
  Q 22 12 10,T 981,T 158,Q 21 5 2,T 215,T 71,Q 24 14 7,T 17,Q 17 14 5,T 981,
  T 854,Q 21 17 6,T 871,T 419,Q 13 6 3,T 435,Q 28 20 10,T 765,T 118,Q 33 29 7,
  T 356,T 847,Q 24 8 5,T 825,Q 18 9 1,T 1529,T 661,Q 13 10 6,T 575,T 595,
  Q 21 10 6,Q 19 15 7,Q 24 18 2,Q 29 18 16,T 224,Q 15 9 6,T 78,Q 16 10 3,
  Q 19 10 4,T 1435,Q 12 11 6,T 1445,T 769,Q 15 7 2,T 463,T 79,Q 23 12 10,
  T 1491,Q 23 22 18,T 321,T 616,Q 27 15 1,T 1232,T 3,Q 19 18 2,T 615,Q 18 10 6,
  T 57,T 1278,Q 30 13 3,T 252,T 123,Q 27 6 2,T 1615,Q 19 12 1,Q 16 12 1,
  T 201,Q 23 12 1,T 913,T 219,Q 16 12 1,T 245,Q 19 18 13,Q 22 12 1,T 1015,
  Q 26 17 9,Q 12 6 3,T 595,Q 10 9 1,T 345,T 2,T 133,T 276,Q 31 2 1,T 1315,
  Q 21 7 5,Q 18 14 5,T 873,Q 19 14 12,T 1045,T 1554,Q 31 29 15,T 862,T 1899,
  Q 15 7 2,T 549,Q 28 23 2,T 525,T 421,Q 15 10 1,T 51,T 131,Q 14 10 5,T 1857,
  Q 28 8 2,T 573,T 186,Q 25 13 3,Q 20 13 5,T 231,Q 17 10 7,T 1165,Q 19 17 10,
  T 133,T 81,Q 27 18 12,T 1361,T 2047,Q 22 15 5,T 2045,Q 18 13 2,T 645,
  T 698,Q 17 10 6,T 105,T 387,Q 24 8 1,Q 13 9 5,Q 16 12 10,T 1297,T 1698,
  Q 26 15 5,T 805,Q 18 2 1,Q 14 6 3,T 273,Q 26 18 1,T 1985,T 1984,Q 29 15 7,
  T 341,T 1411,Q 15 14 6,T 947,Q 16 10 1,Q 11 7 6,T 454,Q 15 11 5,T 200,
  Q 18 5 2,Q 21 13 2,T 1565,Q 16 13 6,Q 19 14 6,T 225,Q 12 5 2,T 76,T 1031,
  Q 10 4 1,T 1327,Q 8 4 3,T 381,T 1846,Q 15 6 3,T 826,T 1491,Q 14 12 6,
  T 243,Q 18 15 10,T 473,T 34,Q 13 3 2,T 347,T 287,Q 24 6 4,T 525,Q 13 10 3,
  Q 18 10 1,T 144,Q 8 3 2,T 9,Q 30 23 10,Q 19 11 2,T 145,Q 7 6 1,T 533,
  T 756,Q 18 13 7,T 5,T 799,Q 15 3 2,T 459,Q 21 19 14,T 389,T 904,Q 31 6 1,
  T 273,T 503,Q 14 12 10,T 387,Q 26 20 15,T 273,T 783,Q 11 6 2,T 387,Q 7 6 4,
  Q 23 18 17,Q 28 10 7,Q 21 12 11,Q 11 5 3,T 754,Q 15 13 8,T 578,Q 9 7 5,
  Q 17 12 7,T 99,Q 23 20 18,T 1301,T 217,Q 15 14 5,T 362,T 727,Q 24 8 2,
  T 147,Q 19 17 16,Q 18 11 7,T 1775,Q 11 8 1,T 385,T 1859,Q 13 12 10,T 1471,
  Q 26 3 1,T 309,T 945,Q 17 14 5,T 226,T 1671,Q 30 27 15,T 81,Q 20 18 11,
  Q 12 5 4,T 679,Q 5 4 3,T 689,T 99,Q 18 14 10,Q 22 18 7,Q 20 19 17,Q 9 8 7,
  T 302,Q 14 5 2,Q 15 4 2,Q 30 24 11,Q 17 16 14,T 175,Q 12 11 2,T 569,T 84,
  Q 12 7 2,T 116,Q 9 5 1,Q 28 24 2,T 901,Q 23 15 10,Q 13 10 2,T 455,Q 21 5 2,
  T 1016,T 1071,Q 9 8 4,T 431,Q 20 18 11,T 973,T 1850,Q 21 20 14,T 390,
  T 111,Q 16 6 3,T 987,Q 22 16 12,T 485,T 540,Q 20 17 15,T 833,T 1351,Q 12 9 2,
  T 593,Q 21 17 10,Q 17 12 6,T 826,Q 18 11 5,T 1991,T 287,Q 25 22 16,T 725,
  Q 28 14 11,T 1477,T 678,Q 8 7 5,T 271,T 1911,Q 10 8 5,T 429,Q 13 11 10,
  T 301,T 900,Q 33 27 20,T 1042,Q 11 5 3,Q 24 23 14,T 667,Q 16 7 6,T 645,
  T 595,Q 21 4 2,T 272,T 539,Q 15 8 2,T 1523,Q 22 12 10,T 1773,T 197,Q 27 12 6,
  T 154,Q 26 15 11,Q 18 15 5,T 151,Q 8 7 1,T 729,T 253,Q 18 7 1,T 155,T 1939,
  Q 16 13 8,T 745,Q 22 19 12,T 1541,T 886,Q 27 19 17,Q 19 17 10,T 175,Q 23 18 4,
  Q 18 8 5,Q 15 10 3,T 1029,T 84,Q 20 19 5,Q 15 13 9,T 1535,Q 33 25 18,
  T 925,Q 25 18 11,T 933,T 585,Q 37 35 3,T 394,T 351,Q 15 14 4,T 543,Q 21 13 2,
  T 137,Q 21 3 2,Q 9 8 7,T 218,T 211,Q 22 6 3,T 1769,Q 23 22 20,Q 19 6 4,
  T 1269,Q 31 10 6,T 442,Q 23 19 7,Q 13 11 8,T 865,Q 32 22 18,T 1069,T 271,
  Q 21 19 13,T 533,Q 11 6 1,Q 27 22 16,T 63,Q 16 6 4,Q 14 4 1,T 47,Q 25 5 3,
  T 350,T 207,Q 12 5 2,T 273,Q 20 8 2,Q 8 6 1,T 869,Q 26 21 14,T 909,T 1187,
  Q 16 6 4,T 1839,T 254,T 589,T 21,Q 11 9 8,T 644,T 1603,Q 17 12 8,T 905,
  Q 26 15 2,T 621,T 1189,Q 24 9 7,T 329,T 255,Q 13 9 6,T 155,Q 15 10 6,
  Q 21 16 3,T 582,Q 13 3 1,T 351,T 639,Q 30 29 11,T 1621,Q 19 18 16,T 2117,
  T 471,Q 30 7 2,T 112,T 799,Q 16 5 3,T 595,Q 37 22 4,Q 25 21 8,T 1165,
  Q 28 21 15,T 273,T 919,Q 8 7 2,T 975,Q 8 2 1,T 1749,T 843,Q 38 35 13,
  T 193,T 1907,Q 18 7 3,T 895,Q 22 8 1,T 401,T 1,Q 17 10 7,T 299,Q 23 8 1,
  Q 27 8 5,T 5,Q 7 6 1,T 657,Q 7 5 1,Q 12 9 6,T 794,Q 13 9 3,Q 14 9 4,T 285,
  Q 14 5 4,Q 20 13 9,T 173,Q 12 3 1,T 196,T 323,Q 8 6 3,T 715,Q 21 16 11,
  T 577,T 1191,Q 39 25 23,T 650,T 1327,Q 17 12 7,T 1339,Q 16 11 8,T 849,
  T 674,Q 23 13 9,T 345,T 151,Q 8 5 1,Q 25 22 20,Q 15 4 1,T 381,T 993,Q 25 11 7,
  T 21,T 2223,Q 11 8 6,T 715,Q 12 11 2,T 897,T 358,Q 25 14 7,T 523,Q 27 23 16,
  Q 6 5 1,T 831,Q 23 12 7,T 297,T 730,Q 34 27 18,T 704,T 207,Q 25 22 12,
  Q 26 21 12,T 98,T 1869,T 1356,Q 14 13 7,T 835,T 87,Q 20 11 1,T 165,Q 18 6 2,
  T 905,T 126,Q 22 17 6,T 2110,T 2079,Q 10 6 1,T 81,Q 24 15 6,T 69,T 638,
  Q 26 19 9,Q 9 8 4,T 1419,Q 10 6 2,Q 21 12 5,Q 27 13 10,Q 14 12 1,T 1358,
  Q 19 15 9,T 223,T 483,Q 13 6 1,T 265,Q 26 18 16,T 497,T 1254,Q 21 16 11,
  Q 22 16 12,Q 30 11 4,Q 15 14 1,T 371,Q 34 13 12,Q 32 28 9,T 25,Q 17 14 1,
  T 68,T 675,Q 24 22 5,T 1907,Q 8 5 2,T 2013,T 1026,Q 23 20 13,T 780,T 239,
  Q 23 13 10,T 1557,Q 27 14 10,T 965,T 232,Q 27 23 5,T 287,T 975,Q 10 8 7,
  T 77,Q 28 15 4,Q 21 20 17,T 136,Q 26 23 10,T 42,T 367,Q 16 8 5,T 1779,
  Q 20 3 2,Q 14 3 2,Q 25 17 7,Q 19 16 2,T 1035,T 1535,Q 21 16 1,T 901,Q 18 17 3,
  Q 30 24 15,T 448,Q 25 8 7,T 149,T 1311,Q 15 13 10,T 189,Q 24 11 5,Q 16 13 9,
  T 959,Q 8 5 3,T 207,T 567,Q 29 24 8,T 95,Q 30 21 13,Q 21 20 1,T 474,Q 27 25 4,
  T 417,T 1943,Q 8 7 5,T 885,Q 26 21 2,T 297,T 841,Q 37 6 5,T 1663,Q 24 17 11,
  Q 36 25 20,T 917,Q 18 14 10,Q 29 12 2,T 392,Q 27 25 23,T 147,Q 10 5 4,
  Q 26 25 21,T 603,Q 19 11 10,T 2245,T 1146,Q 10 9 3,T 216,T 119,Q 35 28 14,
  T 819,Q 24 10 1,T 285,T 1330,Q 13 7 6,T 421,T 283,Q 26 21 3,T 21,Q 26 13 6,
  T 1725,T 1522,Q 18 17 11,T 386,T 351,Q 36 7 5,T 873,Q 23 21 18,Q 17 3 2,
  T 248,Q 11 8 1,T 1033,T 2151,Q 17 16 11,T 1543,Q 21 12 11,T 2257,T 426,
  Q 25 16 6,T 67,Q 19 17 6,Q 26 24 7,T 97,Q 30 17 16,Q 16 14 1,T 466,Q 24 19 9,
  Q 10 9 7,Q 24 19 18,Q 19 14 2,T 617,Q 13 10 8,T 1693,T 408,Q 26 21 14,
  T 73,Q 19 12 10,Q 14 6 5,T 157,Q 14 9 5,T 2193,T 412,Q 15 10 1,T 248,
  Q 12 7 5,Q 27 26 6,T 1941,Q 21 10 8,T 2201,T 857,Q 19 14 6,T 281,T 363,
  Q 25 24 8,T 1257,Q 29 19 16,T 541,T 1946,Q 29 5 2,T 2368,T 923,Q 19 6 4,
  T 279,Q 30 22 17,T 401,T 883,Q 28 11 9,T 170,Q 7 6 4,Q 17 16 7,T 477,
  Q 30 19 7,Q 25 22 21,T 178,Q 25 17 3,T 119,T 507,Q 24 5 4,T 213,Q 26 12 5,
  Q 16 13 6,T 2381,Q 27 23 6,T 348,Q 18 5 3,Q 14 8 1,T 321,Q 22 19 5,T 1397,
  T 369,Q 30 29 7,T 382,Q 20 3 1,Q 19 6 5,T 9,Q 14 7 2,T 1233,T 707,Q 29 18 4,
  T 413,Q 19 12 7,Q 23 7 2,Q 18 8 7,Q 15 14 3,T 117,T 158,Q 29 19 11,T 1146,
  T 291,Q 7 5 4,Q 18 15 4,T 1922,T 2349,T 12,Q 13 4 2,T 91,T 1171,Q 13 10 2,
  Q 22 15 13,Q 18 7 6,Q 22 10 9,T 638,Q 33 31 25,T 491,T 459,Q 28 25 7,
  T 515,Q 16 13 2,Q 33 17 13,T 1094,Q 25 16 3,T 558,Q 24 19 17,Q 29 20 14,
  T 913,Q 20 14 2,T 1493,T 337,Q 33 30 5,T 458,Q 19 17 2,Q 28 10 5,T 301,
  Q 14 9 1,Q 17 12 7,T 238,Q 25 21 2,Q 19 13 3,Q 33 29 18,Q 22 21 3,Q 16 15 1,
  Q 38 21 17,T 1477,T 1085,Q 28 27 6,T 184,T 243,Q 24 17 2,T 561,Q 17 7 4,
  T 633,T 172,Q 27 23 21,T 34,T 355,Q 25 22 14,T 81,Q 15 7 6,Q 15 8 1,T 166,
  Q 29 22 17,T 1551,Q 23 15 2,Q 18 11 1,T 593,Q 22 16 4,T 213,T 1251,Q 24 9 2,
  T 1798,T 555,Q 26 16 10,T 585,Q 23 20 6,T 1457,T 1342,Q 32 21 7,T 1276,
  Q 28 7 4,Q 26 8 2,T 91,Q 27 21 12,T 1989,T 139,Q 35 21 8,T 45,T 1967,
  Q 25 19 16,T 2381,Q 15 12 2,T 469,T 623,Q 21 19 12,T 210,Q 25 12 2,Q 33 28 6,
  T 697,Q 20 18 14,T 1253,T 774,Q 27 18 15,T 296,Q 22 21 6,Q 14 10 3,T 111,
  Q 30 16 11,T 377,T 1955,Q 30 29 7,T 693,T 259,Q 16 9 1,T 1793,Q 34 5 4,
  T 889,T 380,Q 11 9 5,T 277,T 2387,Q 10 3 1,T 1545,Q 19 6 2,T 693,T 21,
  Q 29 9 3,T 49,T 2143,Q 23 21 18,T 1039,Q 13 11 8,T 413,T 133,Q 27 22 15,
  T 63,Q 19 9 3,Q 24 8 1,T 383,Q 17 14 1,Q 13 12 9,T 494,Q 19 6 4,T 1011,
  Q 23 9 1,Q 24 6 4,T 601,Q 30 25 2,T 517,T 217,Q 29 11 5,T 938,Q 15 10 7,
  Q 27 12 5,T 559,Q 11 6 5,T 1709,T 920,Q 25 20 6,T 843,Q 23 11 6,Q 20 17 15,
  Q 13 10 5,Q 13 12 6,T 2205,T 1292,Q 26 25 17,T 1084,T 1131,Q 12 6 2,T 709,
  Q 26 22 16,T 1229,T 2301,Q 42 7 1,T 644,T 315,Q 29 11 8,T 111,Q 33 15 12,
  T 1701,T 1070,Q 23 16 9,T 384,T 239,Q 10 7 4,T 375,Q 15 8 2,Q 23 19 3,
  T 201,Q 15 8 6,T 39,T 1791,Q 28 25 19,Q 18 15 13,Q 31 11 6,T 161,Q 13 11 9,
  Q 17 15 7,T 637,Q 33 13 8,Q 18 16 8,T 671,Q 22 13 6,Q 12 3 2,T 464,Q 21 10 3,
  T 38,Q 17 8 6,Q 29 20 15,T 33,Q 38 35 15,T 1197,T 206,Q 15 12 9,T 385,
  T 731,Q 23 19 2,T 97,Q 14 11 2,T 329,T 777,Q 33 23 14,T 1379,Q 20 15 11,
  Q 28 9 5,T 1089,Q 22 3 2,T 2373,T 496,Q 25 6 2,T 1122,T 387,Q 24 21 10,
  T 1295,Q 22 15 3,T 45,T 489,Q 24 5 3,T 646,T 1607,Q 23 17 8,T 1165,Q 25 24 2,
  Q 17 11 4,T 415,Q 34 15 10,T 133,T 2467,Q 22 17 15,T 833,Q 26 18 11,T 1493,
  T 211,Q 22 9 6,T 126,T 1411,Q 20 19 14,T 161,Q 16 11 7,T 141,T 795,Q 21 12 11,
  T 126,T 783,Q 12 6 2,T 1635,Q 18 5 3,Q 16 13 7,T 258,Q 25 5 3,T 476,T 2119,
  Q 18 15 5,T 1467,Q 18 11 5,T 2201,T 35,Q 21 7 6,T 1562,T 35,Q 25 19 6,
  T 173,Q 20 14 13,Q 23 18 14,T 210,Q 22 21 3,T 88,T 147,Q 18 13 10,T 843,
  Q 15 6 2,T 1377,T 489,Q 19 18 13,T 454,T 243,Q 24 2 1,T 675,Q 18 16 15,
  Q 22 18 11,T 148,Q 30 13 2,T 72,T 1307,Q 17 2 1,T 191,Q 26 19 7,T 2209,
  T 279,Q 42 33 9,T 559,T 899,Q 14 8 5,T 1387,Q 19 14 6,T 2045,T 130,Q 33 27 5,
  T 457,T 199,Q 11 8 5,T 279,Q 23 18 4,Q 22 14 9,T 1519,Q 24 15 6,T 1514,
  T 603,Q 21 7 4,T 15,Q 15 11 8,T 1629,T 141,Q 27 25 4,T 1701,T 1571,Q 36 22 1,
  T 1015,Q 24 14 5,Q 12 11 3,T 1767,Q 15 11 5,T 308,T 1771,Q 12 10 2,T 103,
  Q 30 22 2,T 213,T 724,Q 15 8 1,T 708,Q 16 5 2,Q 15 14 1,Q 14 6 1,Q 8 6 3,
  Q 21 20 2,T 494,Q 35 31 13,T 315,T 419,Q 20 17 5,Q 12 7 3,Q 18 16 10,
  T 117,T 2193,Q 15 11 5,T 1196,T 1411,Q 18 3 2,T 2247,Q 23 22 2,T 189,
  T 347,Q 24 21 3,T 963,Q 15 13 10,Q 30 12 9,T 2055,Q 23 7 2,Q 16 13 6,
  T 1869,Q 20 11 5,T 373,T 1755,Q 16 8 5,T 2305,Q 28 22 2,T 677,T 330,Q 11 8 2,
  T 1006,Q 25 15 13,Q 27 11 6,T 617,Q 20 15 7,Q 27 10 8,T 1546,Q 33 27 19,
  T 125,T 1519,Q 14 4 3,T 917,Q 9 8 6,T 465,T 588,Q 23 22 2,T 156,T 2219,
  Q 20 19 1,T 487,Q 19 8 7,Q 12 9 8,T 354,Q 27 19 2,T 1507,T 2335,Q 23 3 2,
  T 205,Q 12 11 8,T 1637,T 1787,Q 29 15 1,T 503,T 2607,Q 24 10 7,T 327,
  Q 28 19 13,Q 13 12 11,T 1188,Q 21 11 2,T 205,T 275,Q 16 6 1,T 475,Q 12 10 7,
  T 765,T 549,Q 33 27 21,T 101,T 895,Q 10 9 1,T 2321,Q 28 23 3,T 2477,T 235,
  Q 27 18 1,T 197,T 1371,Q 35 25 14,T 627,Q 24 17 14,T 1305,T 167,Q 9 5 2,
  T 1537,T 1359,Q 10 8 4,T 1455,Q 22 10 9,Q 22 20 1,T 769,Q 16 11 9,T 41,
  T 483,Q 31 25 14,T 297,Q 23 6 1,T 341,T 1106,Q 29 18 10,T 1395,Q 17 11 10,
  Q 16 10 1,T 609,Q 9 7 6,T 2337,T 1299,Q 13 11 1,Q 29 9 4,Q 13 10 9,Q 25 17 2,
  Q 16 9 5,Q 25 24 23,Q 20 14 7,T 196,Q 11 10 1,T 1424,Q 17 9 4,Q 24 18 5,
  T 315,Q 25 9 2,Q 14 3 1,T 1133,Q 8 3 2,T 446,T 575,Q 17 12 1,T 2195,Q 14 8 1,
  T 2205,T 434,Q 27 6 5,T 1863,Q 15 4 1,Q 12 10 4,Q 18 14 11,Q 7 6 1,T 837,
  T 142,Q 22 3 2,T 598,T 7,Q 18 14 3,T 1067,Q 17 4 3,T 125,T 1573,Q 19 17 3,
  T 351,T 1199,Q 15 11 2,T 15,Q 18 14 11,T 1545,T 723,Q 30 27 9,T 1219,
  T 803,Q 18 5 3,T 171,Q 25 24 19,T 545,Q 16 15 11,Q 23 21 8,T 2279,T 1647,
  Q 17 16 15,T 731,Q 22 15 13,Q 27 9 3,T 485,Q 17 11 10,T 339,T 171,Q 24 22 11,
  T 2259,Q 16 13 10,Q 15 14 5,T 707,Q 23 20 1,T 1419,Q 22 7 6,Q 16 11 2,
  T 51,Q 29 24 10,T 2485,T 81,Q 19 11 1,T 35,Q 15 7 1,Q 20 7 1,T 863,Q 14 13 8,
  Q 20 3 2,T 194,Q 19 18 3,T 152,T 1919,Q 18 15 6,T 1151,Q 20 14 13,Q 24 19 14,
  T 188,Q 7 4 1,T 547,T 1059,Q 23 17 4,T 423,Q 13 11 10,Q 26 24 17,T 1509,
  Q 17 15 4,T 376,T 2055,Q 28 21 8,T 2261,Q 18 10 9,T 2409,T 434,Q 7 3 2,
  T 215,T 1523,Q 18 13 1,T 2255,Q 9 5 2,T 357,T 485,Q 33 17 3,T 420,Q 18 15 9,
  Q 14 11 6,T 387,Q 17 14 12,T 1205,T 955,Q 30 23 1,T 1720,T 2343,Q 28 10 3,
  T 917,Q 21 18 7,Q 25 2 1,T 331,Q 34 31 19,T 1256,T 303,Q 25 24 23,T 945,
  Q 30 29 6,T 445,T 311,Q 16 15 13,T 186,Q 18 11 8,Q 18 12 11,T 117,Q 40 22 5,
  T 513,T 2163,Q 37 34 23,T 791,Q 19 18 15,Q 24 18 14,T 1229,Q 21 8 2,T 1289,
  T 203,Q 24 15 10,T 1254,T 1635,Q 20 10 6,T 1235,Q 20 19 7,T 865,T 860,
  Q 43 33 15,T 103,T 479,Q 30 19 16,T 2691,Q 13 4 2,T 893,T 91,Q 21 14 10,
  T 767,T 1839,Q 36 20 2,T 29,Q 30 28 5,Q 15 11 1,T 167,Q 15 5 2,T 54,Q 32 23 10,
  Q 22 8 5,T 95,Q 32 23 2,Q 13 12 1,T 198,Q 21 15 3,T 589,T 1247,Q 18 16 13,
  Q 30 23 11,Q 23 18 10,T 305,T 370,Q 13 10 3,T 167,Q 20 17 9,Q 22 2 1,
  T 1029,Q 16 8 2,Q 14 9 8,T 1834,Q 21 19 9,T 1290,T 1143,Q 24 14 1,T 135,
  Q 15 12 7,T 537,T 279,Q 13 11 4,T 1729,Q 19 12 7,Q 29 21 18,T 175,Q 27 19 2,
  T 2537,T 237,Q 20 19 1,T 98,Q 23 20 12,Q 15 10 7,T 729,Q 11 9 4,Q 19 15 2,
  T 554,Q 13 11 6,T 1814,T 535,Q 19 3 2,T 1563,Q 40 13 11,Q 26 23 7,T 773,
  Q 10 5 2,T 1209,Q 26 15 2,Q 26 20 13,T 2113,Q 24 21 6,T 1505,T 318,Q 25 18 14,
  T 1756,T 1467,Q 20 15 1,T 1017,Q 19 16 10,T 1425,T 217,Q 20 5 3,T 492,
  Q 28 22 7,Q 14 9 7,T 1089,Q 15 9 4,T 957,T 248,Q 11 10 2,T 298,Q 23 12 6,
  Q 22 4 3,T 403,Q 33 11 10,T 1273,T 712,Q 15 9 8,T 116,Q 16 5 2,Q 24 20 5,
  T 269,Q 39 35 8,T 1301,T 938,Q 55 46 10,T 773,T 995,Q 22 19 13,T 705,
  Q 17 14 5,T 2185,T 239,Q 33 22 7,T 4,Q 17 16 3,Q 27 11 2,T 1053,Q 18 11 6,
  T 901,T 699,Q 27 23 6,T 328,Q 15 9 3,Q 25 20 12,T 81,Q 20 6 4,T 1697,
  T 1037,Q 13 9 7,T 1338,T 371,Q 21 16 7,T 39,Q 29 27 8,T 1381,T 870,Q 29 20 7,
  T 124,T 15,Q 21 12 2,T 853,Q 20 6 3,T 101,T 565,Q 8 5 3,T 487,T 963,Q 22 20 4,
  T 2251,Q 32 23 1,T 1905,T 43,Q 22 5 2,T 287,T 271,Q 20 9 7,T 135,Q 20 17 14,
  Q 6 3 1,T 461,Q 30 15 6,T 6,Q 17 15 2,Q 11 8 6,T 1251,Q 12 11 4,T 2465,
  T 709,Q 21 18 14,T 28,T 943,Q 13 9 8,T 863,Q 23 12 4,T 357,T 1502,Q 17 15 5,
  T 143,T 759,Q 17 10 3,T 555,Q 8 6 5,Q 20 17 13,T 1365,Q 29 27 23,T 568,
  Q 33 23 2,Q 10 3 1,T 2233,Q 19 16 2,Q 23 8 5,Q 17 16 9,Q 21 15 8,T 49,
  T 967,Q 10 7 2,T 929,Q 16 11 5,T 149,T 434,Q 27 12 9,T 165,Q 9 4 3,Q 12 8 2,
  T 39,Q 26 14 10,T 2637,T 90,Q 31 29 11,T 714,T 1215,Q 10 8 1,Q 34 27 3,
  Q 27 4 1,T 41,T 37,Q 26 25 10,T 82,Q 18 15 13,Q 17 15 4,T 1259,Q 27 26 2,
  Q 31 21 8,T 271,Q 40 17 2,T 326,Q 35 16 1,Q 22 18 11,T 87,Q 10 7 5,Q 27 7 3,
  T 2007,Q 41 20 11,T 1384,T 1623,Q 21 15 4,T 561,Q 24 4 2,T 1609,T 732,
  Q 26 21 14,Q 19 8 2,Q 14 9 5,Q 12 8 5,T 705,Q 16 6 5,Q 22 15 8,T 82,Q 27 25 14,
  T 248,T 2199,Q 24 21 16,T 1151,Q 22 9 1,T 409,T 441,Q 10 9 3,T 27,T 1775,
  Q 28 6 3,Q 14 13 8,Q 21 6 5,T 1361,T 375,Q 32 11 2,T 1562,Q 19 18 14,
  Q 28 19 6,T 103,Q 22 9 3,T 309,T 1310,Q 25 24 7,T 515,Q 25 18 10,Q 13 11 8,
  T 2267,Q 10 8 5,T 2629,T 744,Q 25 18 10,T 507,Q 30 17 4,Q 18 15 13,T 115,
  Q 16 6 4,T 1085,T 1621,Q 21 11 8,T 544,T 279,Q 13 4 2,Q 18 17 2,Q 26 19 18,
  T 377,T 127,Q 21 20 7,T 1721,T 1235,Q 8 7 2,T 371,Q 22 18 16,T 805,T 1431,
  Q 29 23 10,T 615,T 2759,Q 18 13 1,T 2019,Q 17 10 7,Q 13 10 4,T 124,Q 21 14 3,
  T 83,Q 9 7 2,Q 16 14 13,T 343,Q 12 11 8,Q 15 5 4,T 266,Q 19 12 1,T 134,
  T 1071,Q 21 7 4,T 1535,Q 24 14 6,T 2377,T 954,Q 25 22 6,T 1161,Q 20 19 7,
  Q 30 20 15,T 1585,Q 21 14 12,T 197,T 2101,Q 33 13 11,T 437,T 2383,Q 15 14 3,
  T 831,Q 33 26 23,T 1853,T 934,Q 25 7 6,T 129,T 1015,Q 16 15 10,Q 13 8 1,
  Q 10 7 3,T 369,T 731,Q 43 10 1,T 369,T 47,Q 21 17 2,T 295,Q 14 13 1,T 2233,
  T 1083,Q 17 15 7,T 343,T 763,T 1058,T 61,Q 18 11 4,T 2145,T 916,Q 23 17 10,
  T 323,T 2027,Q 28 16 10,Q 17 8 5,Q 24 6 5,Q 27 23 11,T 275,Q 21 18 13,
  T 264,T 935,Q 13 5 2,T 91,Q 13 9 2,T 153,T 169,Q 23 14 2,T 250,Q 21 18 7,
  Q 20 7 6,T 1809,Q 18 5 1,Q 20 17 3,T 1013,Q 19 8 6,T 105,T 27,Q 9 6 1,
  Q 15 13 10,Q 46 36 18,T 2217,T 1334,Q 23 15 6,T 516,T 2735,Q 28 25 16,
  T 1899,Q 16 2 1,Q 18 3 2,T 58,Q 27 11 10,T 1249,Q 30 23 2,Q 17 6 2,T 373,
  Q 26 23 4,Q 15 9 5,T 287,Q 19 15 10,T 113,T 267,Q 20 13 9,T 243,Q 37 36 30,
  Q 24 9 1,T 2556,Q 21 10 3,T 81,T 975,Q 19 6 4,Q 14 11 10,Q 38 29 14,T 297,
  T 142,Q 23 10 4,T 541,Q 13 8 3,Q 20 15 9,T 63,Q 28 3 1,Q 21 15 2,T 781,
  Q 30 23 1,T 390,T 247,Q 35 21 10,T 125,Q 23 15 12,T 1981,T 380,Q 32 19 5,
  T 172,Q 23 18 15,Q 25 17 2,T 2101,Q 34 24 14,T 449,T 2545,Q 33 22 13,
  T 292,Q 25 13 7,Q 30 29 28,Q 20 15 3,Q 21 12 6,Q 13 12 8,T 1843,Q 16 15 6,
  T 311,T 495,Q 18 10 5,T 1017,Q 24 7 3,T 325,T 1305,Q 16 7 2,Q 16 6 4,
  Q 17 13 10,Q 30 26 14,T 2673,Q 15 11 6,Q 28 25 4,T 84,Q 27 4 1,T 49,Q 23 7 3,
  Q 12 5 3,T 99,Q 20 18 11,T 245,T 1075,Q 23 21 14,Q 9 5 3,Q 33 20 3,Q 14 11 4,
  T 117,Q 23 21 18,Q 32 13 8,T 216,Q 25 23 2,T 1767,T 95,Q 28 8 5,T 385,
  Q 18 6 2,T 2297,T 1864,Q 37 35 25,T 2081,T 2599,Q 35 28 6,T 1475,Q 27 19 12,
  T 609,T 2359,Q 21 14 4,T 24,T 399,Q 11 3 2,T 315,Q 20 8 3,Q 23 19 3,T 284,
  Q 18 9 6,T 657,T 1151,Q 29 25 2,T 1951,Q 21 14 13,T 45,T 1960,Q 17 7 1,
  T 203,T 895,Q 23 18 4,T 1067,Q 36 20 2,T 2729,T 680,Q 29 9 2,T 1133,Q 17 9 5,
  Q 26 25 20,Q 20 18 3,Q 10 8 1,T 1765,T 362,Q 23 21 12,T 648,T 2895,Q 25 19 14,
  T 2287,Q 22 13 8,T 1025,T 435,Q 30 27 15,T 346,T 2907,Q 36 30 5,T 533,
  Q 31 24 16,Q 21 14 9,T 91,Q 35 34 2,T 261,T 1315,Q 19 18 11,T 645,Q 32 22 17,
  T 2217,T 143,Q 39 33 26,T 586,T 915,T 98,Q 18 8 7,Q 22 17 3,T 1377,T 613,
  Q 44 21 14,T 406,T 439,Q 28 14 13,T 1577,Q 10 7 1,Q 21 17 11,T 1556,Q 25 11 5,
  T 1476,Q 21 3 2,Q 15 12 6,T 2525,Q 19 16 10,Q 16 3 2,T 411,Q 17 15 8,
  T 988,T 1227,Q 34 17 1,T 127,Q 15 11 2,Q 24 22 21,T 1242,Q 7 6 1,T 2378,
  T 1155,Q 39 34 20,T 243,Q 31 25 6,Q 15 7 2,Q 28 24 9,Q 23 18 11,T 129,
  T 67,Q 30 18 10,T 1149,Q 21 5 2,T 37,T 789,Q 28 15 13,T 1753,T 35,Q 26 24 6,
  T 1387,Q 17 11 8,T 1241,T 382,Q 19 8 6,T 1081,Q 20 2 1,Q 33 26 6,T 279,
  Q 16 13 7,T 557,T 1948,Q 23 10 3,T 1062,T 11,Q 15 9 4,T 1475,Q 14 9 6,
  T 973,T 2186,Q 20 11 2,T 346,T 1395,Q 22 16 14,T 577,Q 24 20 11,Q 22 17 16,
  T 376,Q 13 11 6,T 823,T 2079,Q 23 12 10,T 213,Q 17 11 4,Q 26 19 13,T 467,
  Q 35 12 1,T 1283,Q 11 6 5,Q 22 19 10,T 873,Q 34 33 30,T 1765,T 287,Q 4 3 1,
  T 490,Q 32 24 11,Q 25 22 6,T 19,Q 22 8 6,Q 20 15 5,Q 31 9 4,Q 34 15 2,
  T 1421,Q 24 22 7,Q 36 34 7,T 135,Q 16 7 3,T 2445,T 572,Q 17 7 5,T 1682,
  Q 26 11 10,Q 14 13 3,T 135,Q 16 13 6,T 2245,T 1005,Q 26 7 1,T 348,T 71,
  Q 29 24 6,Q 25 18 16,Q 23 20 14,T 1297,T 538,Q 28 27 13,T 77,Q 23 12 10,
  Q 34 33 4,T 25,Q 17 16 14,Q 14 6 5,T 1165,Q 38 15 10,T 1556,T 603,Q 17 12 7,
  T 1179,Q 25 15 12,Q 5 4 2,T 33,Q 20 11 2,T 1204,T 1179,Q 22 10 8,T 3,
  Q 10 9 1,T 1029,T 823,Q 29 15 1,T 430,T 1719,Q 16 8 1,T 245,Q 22 13 6,
  T 1773,T 56,Q 39 13 12,T 308,Q 15 6 4,Q 26 25 18,T 1223,Q 26 8 5,T 877,
  T 1074,Q 20 5 2,T 919,Q 21 9 2,Q 30 24 23,T 1807,Q 25 14 4,T 213,T 109,
  Q 29 10 7,T 922,T 867,Q 20 15 10,T 691,Q 14 12 1,Q 22 7 5,T 1363,Q 25 23 14,
  T 935,T 711,Q 19 18 5,Q 27 16 14,Q 15 14 12,T 913,T 134,Q 39 30 9,T 711,
  T 383,Q 17 14 4,T 889,Q 21 12 3,T 693,T 1033,Q 13 4 2,T 317,T 1035,Q 18 9 8,
  T 2691,Q 12 7 6,T 1317,T 359,Q 17 3 1,T 777,Q 22 18 7,Q 23 18 10,T 1701,
  Q 32 14 9,T 2353,T 921,Q 11 10 2,T 592,Q 32 28 27,Q 17 10 5,T 475,Q 28 26 4,
  T 1565,T 586,Q 18 7 2,T 1000,T 363,Q 35 34 31,Q 15 10 6,Q 14 10 2,Q 30 29 4,
  T 2318,Q 11 10 5,T 852,T 647,Q 18 12 3,T 1601,Q 17 14 12,T 261,T 1466,
  Q 17 16 7,T 502,Q 10 7 5,Q 26 8 5,T 973,Q 31 8 3,T 737,T 54,Q 17 10 6,
  T 1106,T 999,Q 15 14 5,T 1787,Q 16 10 6,T 749,T 290,Q 9 7 5,T 2021,T 1895,
  Q 14 11 4,Q 16 13 8,Q 26 14 5,T 741,T 104,Q 34 25 5,T 237,T 1127,Q 29 6 5,
  T 2317,Q 30 20 18,T 57,T 82,Q 35 19 10,T 190,Q 15 10 1,Q 25 13 2,T 7,
  Q 17 8 7,T 2105,T 467,Q 13 3 1,T 1052,T 2079,Q 28 27 5,T 405,Q 23 13 10,
  Q 11 7 2,T 1155,Q 35 33 14,T 355,Q 16 11 7,Q 42 40 31,T 21,Q 11 6 1,T 333,
  T 1213,Q 29 28 10,T 1,T 387,Q 19 15 14,T 1273,Q 18 4 2,T 1029,Q 22 15 10,
  Q 15 6 1,Q 21 16 1,T 623,Q 21 12 1,T 2781,Q 22 14 2,T 1977,T 368,Q 22 15 9,
  T 2086,T 459,Q 17 10 7,T 1137,Q 16 14 9,T 1309,T 939,Q 21 15 2,T 2566,
  Q 31 22 16,Q 11 6 2,T 75,Q 8 6 1,Q 11 7 6,T 1234,Q 21 11 4,T 1025,T 1155,
  Q 30 27 16,T 1115,Q 37 27 12,T 229,T 305,Q 13 11 1,Q 31 17 3,T 1967,Q 32 23 14,
  T 67,Q 21 14 5,T 273,T 142,Q 31 9 1,Q 19 11 10,T 1147,Q 30 18 17,T 135,
  Q 21 18 9,Q 12 9 8,T 313,Q 28 27 5,T 275,T 2643,Q 30 10 5,T 1085,Q 33 26 3,
  T 2061,T 1013,Q 34 29 7,T 364,T 543,Q 28 14 4,T 2731,Q 19 17 4,T 581,
  T 324,Q 39 34 10,T 103,T 1447,Q 29 7 4,T 91,Q 17 6 4,Q 12 11 10,T 146,
  Q 37 12 3,T 1146,T 947,Q 15 10 4,T 1779,Q 18 10 7,T 613,T 231,Q 31 12 10,
  T 72,Q 18 15 2,Q 23 18 14,T 1131,Q 18 17 10,Q 34 15 9,T 663,Q 29 15 7,
  T 19,T 1095,Q 9 7 4,T 903,Q 24 16 14,T 621,Q 26 19 1,Q 29 18 5,T 231,
  T 207,Q 35 29 18,T 791,Q 18 16 13,T 1333,T 3,Q 26 13 7,T 810,T 651,Q 15 9 4,
  T 1141,Q 23 18 4,T 425,T 630,Q 21 18 14,T 41,Q 19 18 5,Q 25 17 16,T 1647,
  Q 30 10 4,T 2633,T 1547,Q 25 23 8,T 777,T 1707,Q 26 14 13,T 381,Q 29 24 14,
  T 825,T 732,Q 31 25 17,T 840,T 119,Q 25 17 2,T 2073,Q 36 15 6,Q 24 3 2,
  T 93,Q 25 22 6,T 224,Q 26 8 7,Q 7 6 2,T 217,Q 24 22 6,T 477,T 1910,Q 31 30 2,
  T 1211,T 563,Q 18 13 2,T 1535,Q 31 18 15,T 213,T 84,Q 11 10 1,T 538,Q 17 8 3,
  Q 10 5 1,T 43,Q 21 8 6,Q 23 11 1,T 880,Q 21 13 7,T 706,T 111,Q 17 16 6,
  T 1169,Q 38 11 4,Q 29 27 5,T 283,Q 21 5 2,T 1736,T 2079,Q 26 9 2,T 375,
  Q 24 21 14,T 465,T 330,Q 23 15 9,T 1911,Q 15 7 5,Q 25 12 9,Q 20 11 3,
  Q 14 12 7,T 61,T 2673,Q 29 27 13,T 682,T 2571,Q 27 26 12,Q 18 12 3,Q 24 11 2,
  T 513,T 725,Q 37 29 11,T 695,T 675,Q 12 6 3,T 1283,Q 16 12 10,T 2137,
  T 242,Q 16 7 2,T 765,Q 15 9 5,Q 17 14 12,Q 15 10 7,Q 18 11 9,T 45,T 31,
  Q 25 10 9,T 542,T 31,Q 12 8 2,T 333,Q 14 11 7,Q 33 29 9,Q 28 21 7,Q 19 15 1,
  T 1322,T 279,Q 12 11 10,T 945,Q 34 15 6,T 2401,T 941,Q 27 22 6,T 265,
  T 159,Q 28 24 22,T 885,Q 34 30 17,T 153,T 879,Q 19 13 11,T 1834,T 651,
  Q 35 30 21,T 525,Q 19 15 8,Q 10 3 1,T 1112,Q 19 15 1,T 1410,T 1323,Q 14 13 6,
  Q 20 15 11,Q 17 10 5,Q 29 15 14,T 884,Q 27 25 19,T 2341,T 1587,Q 32 15 6,
  T 2577,Q 10 8 6,Q 17 13 8,T 859,Q 37 23 7,T 476,Q 34 32 15,Q 27 4 2,T 1617,
  Q 30 25 7,T 957,T 1999,Q 45 42 1,T 279,T 2679,Q 33 8 5,T 665,Q 18 14 5,
  T 885,T 189,Q 21 19 16,T 457,Q 14 8 3,Q 30 20 8,T 1279,Q 25 15 14,T 2365,
  T 813,Q 9 4 2,T 286,Q 27 26 8,Q 13 10 3,T 289,Q 28 22 15,T 2105,T 136,
  Q 33 9 3,T 71,T 2379,Q 7 6 1,T 341,Q 21 19 6,T 645,T 491,Q 15 14 9,T 549,
  Q 20 3 1,Q 26 7 5,T 2259,Q 11 9 4,T 1617,T 823,Q 27 20 17,T 535,T 559,
  Q 14 13 9,T 1189,Q 29 26 21,T 3185,T 2135,Q 43 32 9,T 167,T 1063,Q 42 25 9,
  T 917,Q 23 18 3,Q 9 8 5,T 77,Q 25 19 16,T 202,T 1391,Q 22 19 14,T 1143,
  Q 8 7 3,T 1809,T 54,Q 19 15 1,T 839,Q 31 30 19,Q 20 16 1,T 475,Q 14 12 1,
  T 2901,T 416,Q 24 15 6,T 147,T 999,Q 13 8 2,T 221,Q 16 7 5,T 673,T 930,
  Q 26 21 5,T 256,T 983,Q 16 10 1,T 3141,Q 26 9 7,T 41,T 550,Q 29 7 3,T 124,
  Q 38 21 5,Q 12 5 3,T 1275,Q 18 11 1,T 617,T 872,Q 55 32 9,T 1764,T 1911,
  Q 15 12 1,T 2237,Q 10 6 4,T 2989,T 3072,Q 45 19 7,T 2613,Q 18 14 11,Q 14 12 3,
  T 625,Q 19 8 4,T 1145,T 675,Q 11 10 6,T 218,T 1683,Q 32 15 14,T 189,Q 18 12 7,
  T 1397,T 2929,Q 31 26 2,T 11,T 751,Q 17 6 1,T 1781,Q 16 11 7,Q 14 9 7,
  T 455,Q 12 9 7,T 210,T 2387,Q 21 18 4,T 1017,Q 23 3 2,T 21,T 397,Q 21 8 2,
  T 631,T 3,Q 32 27 6,T 965,Q 8 5 3,Q 21 14 9,T 1241,Q 27 14 2,T 1197,Q 35 24 5,
  Q 29 24 1,T 41,Q 18 15 11,Q 9 4 3,T 240,Q 29 21 15,T 136,T 1563,Q 22 21 1,
  T 1273,Q 18 16 4,T 2781,T 1104,Q 11 8 2,T 35,T 1483,Q 30 20 3,T 225,T 466,
  T 3105,T 29,Q 37 15 7,T 15,Q 28 4 3,Q 32 21 11,T 989,Q 23 3 2,Q 17 5 4,
  T 300,Q 18 15 10,T 97,Q 34 31 24,Q 36 13 12,T 321,Q 32 20 6,T 733,T 1293,
  Q 25 11 6,T 925,Q 34 33 1,Q 16 6 2,T 655,Q 16 15 4,T 3321,T 518,Q 16 15 1,
  T 452,T 171,Q 25 5 2,T 2345,Q 28 19 9,Q 22 8 5,T 107,Q 27 25 24,T 571,
  T 707,Q 17 2 1,T 895,Q 20 10 6,T 1029,T 1240,Q 26 25 1,T 140,Q 29 25 3,
  Q 32 21 16,T 49,Q 24 20 7,Q 25 23 16,T 1568,Q 37 19 13,Q 27 10 5,T 679,
  Q 21 20 8,T 1115,Q 24 19 16,Q 32 26 25,T 518,Q 22 5 2,T 682,T 1571,Q 21 11 8,
  T 747,Q 19 13 8,T 1425,Q 14 5 1,Q 21 14 10,T 133,T 435,Q 26 12 8,T 63,
  Q 20 16 6,T 537,T 115,Q 35 32 25,T 35,Q 23 14 1,Q 23 4 2,T 473,Q 21 17 14,
  T 357,T 96,Q 14 9 3,T 427,Q 18 17 7,Q 26 9 6,Q 20 15 5,Q 26 20 6,T 2497,
  T 2247,Q 29 22 18,T 221,Q 21 7 4,Q 31 30 4,T 839,Q 28 27 15,Q 21 14 1,
  T 254,Q 30 29 17,T 386,T 747,Q 16 12 9,T 531,Q 21 10 3,T 1405,T 956,Q 11 5 2,
  T 2494,T 647,Q 16 4 2,T 43,Q 17 10 1,Q 27 22 21,T 1719,Q 25 13 2,T 2056,
  T 2631,Q 22 21 4,T 287,Q 15 14 2,Q 21 15 5,Q 9 6 2,Q 31 30 19,T 152,T 1883,
  Q 32 26 25,T 383,Q 22 21 12,T 565,T 495,Q 24 11 9,T 652,Q 16 15 8,Q 30 15 10,
  T 421,Q 20 12 6,Q 12 9 8,T 256,Q 29 17 11,Q 27 18 8,T 439,Q 6 5 2,T 915,
  Q 6 4 1,T 645,T 2981,Q 36 13 11,T 2513,T 1667,Q 23 12 2,T 1613,Q 36 35 8,
  T 1953,T 1031,Q 25 15 12,T 414,Q 16 11 4,Q 18 15 7,Q 13 4 1,Q 22 13 8,
  T 1653,Q 15 14 2,Q 8 3 1,T 1046,Q 25 15 13,Q 17 9 2,T 93,Q 24 9 7,Q 29 9 3,
  T 653,Q 8 7 5,T 3216,T 699,Q 21 14 13,T 627,Q 35 26 4,Q 15 10 9,T 678,
  Q 37 31 30,T 582,T 963,Q 16 12 6,T 399,Q 27 12 11,T 1949,T 211,Q 23 21 8,
  Q 17 3 2,Q 23 16 14,Q 25 19 18,T 63,Q 16 5 2,Q 11 9 3,T 434,Q 32 21 19,
  T 1820,T 735,Q 13 7 4,T 1727,Q 24 10 3,T 2113,T 1070,Q 14 9 3,T 570,Q 16 11 1,
  Q 24 20 6,T 13,Q 18 8 7,T 249,T 2269,Q 35 32 17,T 1073,T 1867,Q 28 20 6,
  T 455,Q 36 34 14,T 369,T 992,Q 19 18 9,T 285,Q 23 18 3,Q 11 8 6,T 485,
  Q 17 16 11,Q 21 16 8,T 168,Q 8 3 2,T 439,T 579,Q 21 6 4,T 445,Q 15 12 2,
  T 1673,T 510,Q 27 11 9,T 166,Q 28 12 3,Q 25 12 4,T 1025,Q 27 17 4,T 1241,
  T 623,Q 27 26 11,T 1631,T 3375,Q 16 6 5,T 291,Q 14 11 4,T 981,T 914,Q 33 25 6,
  T 1932,Q 14 13 6,Q 5 2 1,T 273,Q 8 5 2,T 633,T 1483,Q 23 10 7,T 175,T 1695,
  Q 33 14 13,T 117,Q 23 6 4,T 1337,T 242,Q 22 15 10,T 1701,T 195,Q 20 9 2,
  Q 25 16 2,Q 29 23 16,T 793,T 1029,Q 19 13 9,T 16,T 2679,Q 29 9 8,T 1353,
  Q 26 24 20,T 2961,T 669,Q 19 18 7,T 587,T 2871,Q 15 12 6,T 23,Q 19 6 3,
  T 1961,T 238,Q 39 17 4,T 794,T 1379,Q 7 6 3,T 3471,Q 13 6 1,Q 17 11 1,
  T 803,Q 27 24 10,T 937,T 1103,Q 31 17 4,T 765,Q 24 23 1,T 2597,T 265,
  Q 11 5 2,T 599,Q 21 14 11,Q 19 11 10,T 549,Q 28 17 14,T 1217,T 137,Q 37 26 17,
  T 2202,T 259,Q 31 26 20,T 677,T 674,Q 11 10 2,T 351,Q 27 18 16,T 333,
  T 423,Q 26 22 2,T 345,Q 20 15 14,Q 22 6 3,T 1245,Q 32 5 2,T 2519,Q 21 15 1,
  Q 9 7 4,T 1929,Q 22 18 2,Q 23 13 12,T 67,Q 27 17 13,T 936,T 2695,Q 16 4 1,
  T 921,Q 8 7 1,Q 28 21 8,T 63,Q 15 10 4,T 1954,T 3123,Q 8 2 1,T 235,Q 26 21 8,
  T 41,T 1665,Q 35 5 2,T 1618,Q 13 12 3,Q 34 9 5,T 3417,Q 27 26 10,T 1113,
  T 935,Q 17 15 5,T 953,Q 35 13 2,Q 26 22 6,T 89,Q 11 10 5,T 2785,T 42,
  Q 19 13 2,T 130,Q 35 24 21,Q 21 10 7,T 273,Q 36 34 10,T 729,T 553,Q 16 3 2,
  T 337,Q 20 7 6,Q 26 21 11,T 225,Q 26 9 1,Q 27 20 15,T 2426,Q 33 13 3,
  T 1127,T 259,Q 20 17 9,T 935,Q 9 7 6,T 33,T 660,Q 37 23 16,T 301,Q 15 14 8,
  Q 34 21 18,T 1117,Q 20 18 16,Q 33 23 12,T 1332,Q 27 26 9,T 49,T 1647,
  Q 7 3 2,T 33,Q 21 17 2,T 3521,T 3529,Q 13 10 6,T 1110,T 1887,Q 22 18 3,
  T 2549,Q 28 26 19,Q 16 13 4,T 584,Q 33 31 21,T 918,Q 16 7 6,Q 30 20 15,
  T 415,Q 24 13 11,T 245,T 30,Q 33 27 2,T 1064,Q 19 10 1,Q 18 14 12,Q 23 19 7,
  Q 18 14 8,T 897,T 1201,Q 29 14 6,T 2412,T 1763,Q 32 30 7,T 1607,Q 17 14 12,
  Q 11 2 1,T 21,Q 29 23 3,T 153,T 1631,T 686,T 1695,Q 21 5 2,T 2349,T 246,
  Q 29 23 21,T 460,T 3111,Q 29 8 6,Q 29 26 4,Q 25 15 8,Q 28 7 2,T 19,Q 15 8 1,
  T 891,T 3263,Q 23 6 5,T 2975,Q 13 12 10,T 2733,T 245,Q 15 12 10,T 333,
  T 2807,Q 29 22 7,T 1165,Q 19 4 2,T 2421,T 2185,Q 35 12 9,T 1435,Q 29 22 18,
  Q 19 18 7,T 75,Q 21 10 2,Q 19 14 6,T 1610,Q 33 14 2,T 308,T 87,Q 25 10 1,
  T 497,Q 26 23 9,T 2505,T 726,Q 17 7 4,T 274,T 2427,Q 19 14 13,T 1885,
  Q 32 13 7,T 933,T 283,Q 22 21 7,T 3472,T 2731,Q 38 19 1,T 121,Q 28 3 2,
  Q 32 13 12,T 256,Q 41 36 19,T 776,T 139,Q 32 17 6,Q 14 4 3,Q 33 32 30,
  T 829,T 1325,Q 23 10 4,T 789,Q 38 31 18,Q 8 2 1,T 1621,Q 24 23 7,T 1741,
  T 3247,Q 37 6 4,T 179,Q 26 8 7,Q 34 22 12,T 625,Q 29 18 5,T 1793,T 1349,
  Q 25 23 17,T 1253,T 1431,Q 29 24 1,T 3033,Q 34 14 8,Q 22 13 10,T 1961,
  Q 37 7 2,T 570,Q 15 13 11,Q 26 11 3,T 925,Q 23 22 7,T 1521,T 3280,Q 37 30 17,
  T 602,T 7,Q 19 15 2,T 287,Q 23 15 2,T 3557,T 341,Q 25 24 3,T 45,T 2787,
  Q 25 10 4,T 1415,Q 20 19 4,T 2625,T 330,Q 35 28 10,T 202,T 1587,Q 26 11 2,
  T 523,Q 35 29 6,T 381,T 2947,Q 41 30 26,T 70,Q 11 10 9,Q 28 19 6,T 643,
  Q 26 21 5,T 237,T 1421,Q 34 21 5,T 828,T 1351,Q 14 5 1,T 311,Q 29 5 2,
  T 577,T 170,Q 33 28 3,T 1344,T 671,Q 27 26 2,T 2863,Q 12 6 2,T 573,T 601,
  Q 20 5 3,T 410,T 2443,Q 27 26 7,T 1179,Q 30 16 8,Q 20 15 6,T 2864,Q 23 18 2,
  T 155,T 355,Q 34 24 19,T 377,Q 26 20 5,T 397,T 213,Q 19 13 11,T 1041,
  T 1683,Q 28 16 9,T 1429,Q 11 8 6,T 829,T 172,Q 41 22 16,T 407,T 523,Q 22 8 4,
  T 105,Q 32 10 2,T 2537,T 20,Q 31 26 9,T 912,T 751,Q 23 20 10,T 629,Q 22 17 13,
  Q 13 12 10,T 35,Q 33 27 21,T 1407,T 1127,Q 8 4 2,Q 29 16 2,Q 28 26 21,
  T 145,T 2715,Q 19 9 5,T 1814,T 871,Q 30 20 14,T 297,Q 17 14 6,Q 27 20 1,
  Q 27 21 17,Q 13 3 1,T 854,T 2179,Q 24 18 16,T 1797,Q 17 6 2,Q 13 12 2,
  Q 10 9 5,Q 35 23 9,T 634,T 867,Q 32 29 2,Q 26 25 5,Q 26 14 2,T 2961,T 285,
  Q 18 13 7,T 371,T 1371,Q 30 17 1,T 343,Q 30 10 4,Q 11 9 6,T 65,Q 30 19 7,
  T 420,T 2295,Q 13 5 2,T 315,Q 31 20 19,T 3285,T 2505,Q 22 21 15,T 1053,
  Q 10 6 5,Q 23 12 6,T 679,Q 12 5 2,T 1217,T 1,Q 21 20 14,T 625,T 379,Q 16 14 9,
  T 1701,Q 30 29 7,Q 24 13 5,T 284,Q 27 8 1,Q 12 9 2,T 3127,Q 40 32 14,
  T 1535,Q 20 19 7,T 345,T 2939,Q 21 18 14,T 277,T 2087,Q 25 20 4,T 789,
  Q 30 26 22,Q 17 8 3,T 400,Q 23 13 7,T 457,Q 23 17 15,Q 24 10 8,T 423,
  Q 23 16 2,T 1797,T 1069,Q 32 15 2,T 887,Q 18 4 3,Q 25 18 2,T 3305,Q 26 8 1,
  Q 22 19 16,T 2940,Q 21 16 6,T 2644,T 2843,Q 10 9 7,T 1015,Q 20 10 3,Q 20 9 8,
  T 241,Q 23 21 12,T 49,Q 33 15 12,Q 27 14 9,T 125,Q 21 13 2,Q 20 17 8,
  T 628,Q 45 17 2,T 1451,T 135,Q 34 14 9,T 3503,Q 21 10 4,Q 26 7 2,T 51,
  Q 22 21 3,T 2697,Q 38 21 5,Q 18 17 11,Q 27 25 18,Q 20 19 10,T 3653,T 15,
  Q 19 9 7,T 1501,T 2331,Q 35 22 1,T 75,Q 17 11 4,T 533,T 1507,Q 18 13 1,
  T 953,T 307,Q 13 10 3,T 705,Q 11 8 7,T 2349,T 2726,Q 42 21 14,T 1377,
  T 735,Q 26 19 10,T 2127,Q 28 10 2,T 549,T 1173,Q 23 13 7,T 1679,Q 15 8 5,
  Q 24 22 11,T 1689,Q 22 16 12,Q 27 26 9,T 145,Q 17 8 3,T 492,Q 12 11 6,
  Q 26 13 2,T 2835,Q 18 16 12,T 1501,T 185,Q 41 21 3,T 846,T 1547,Q 29 8 7,
  T 1845,Q 24 22 13,T 209,T 151,Q 15 5 3,T 3409,T 3303,Q 16 14 7,T 2413,
  Q 12 7 6,T 197,T 2174,Q 41 40 11,T 846,T 423,Q 14 11 3,T 761,Q 23 21 14,
  Q 13 6 2,T 653,Q 51 46 10,T 268,T 3251,Q 15 9 4,T 655,Q 15 10 7,T 869,
  T 1236,Q 45 10 1,T 1210,T 3471,Q 34 32 13,T 55,Q 32 11 7,T 1469,T 2066,
  Q 19 18 2,T 2555,T 555,Q 6 5 2,T 1257,Q 28 15 3,Q 24 9 7,T 111,Q 39 25 10,
  T 495,Q 18 11 2,Q 32 27 6,T 1671,Q 25 8 6,T 945,Q 19 13 7,Q 31 21 14,
  Q 18 5 3,Q 22 17 6,Q 29 18 6,T 203,Q 32 29 19,T 581,T 637,Q 9 2 1,T 474,
  T 291,Q 40 15 2,T 3773,Q 35 10 7,T 2833,T 92,Q 54 3 2,Q 24 14 7,T 2963,
  Q 27 11 2,T 733,Q 13 10 8,T 2673,T 3729,Q 29 3 2,T 925,T 663,Q 16 5 1,
  T 161,Q 28 15 10,T 1233,T 1105,Q 37 33 10,T 2222,T 199,Q 24 16 13,T 2165,
  Q 21 14 11,Q 20 14 1,T 3629,Q 30 13 7,T 1069,T 3627,Q 17 13 10,T 2287,
  Q 30 20 4,T 2473,T 1203,Q 30 23 5,T 97,T 1351,Q 27 26 7,T 77,Q 30 6 3,
  Q 13 11 4,T 769,Q 39 5 1,T 120,T 1891,Q 9 6 1,Q 21 14 11,Q 38 25 11,T 2325,
  T 1497,Q 27 9 3,T 1500,T 479,Q 23 14 7,T 1419,Q 10 9 8,T 581,T 867,Q 35 16 6,
  T 728,Q 23 19 11,Q 36 29 2,T 1683,Q 30 9 4,T 2685,T 119,Q 17 16 15,T 282,
  T 111,Q 10 6 3,T 177,Q 20 14 8,T 541,T 2786,Q 15 12 10,T 1962,T 831,Q 19 18 15,
  T 13,Q 18 4 2,T 3853,T 3084,Q 27 22 5,T 1060,T 367,Q 22 16 14,T 1157,
  Q 7 6 1,Q 11 5 3,T 1993,Q 27 25 4,T 1382,T 15,Q 30 21 9,T 1967,Q 23 22 17,
  Q 30 23 7,T 1007,Q 21 6 3,T 594,T 1979,Q 11 10 1,T 49,Q 15 8 3,T 2089,
  T 717,Q 22 11 3,Q 13 5 1,T 3499,Q 48 33 29,T 47,Q 32 22 11,T 377,T 3007,
  Q 33 28 27,T 1364,Q 22 19 10,Q 28 26 12,T 1155,Q 28 25 2,T 721,T 273,
  Q 35 30 2,T 1459,Q 31 7 5,Q 16 7 2,Q 29 6 4,Q 17 10 7,T 1633,T 42,Q 32 23 21,
  T 2074,T 231,Q 14 10 6,T 499,Q 18 17 1,T 1341,T 410,Q 27 18 15,T 2933,
  T 1943,Q 38 35 27,T 2879,Q 30 19 18,T 3585,T 243,Q 41 39 36,T 150,Q 24 23 3,
  Q 40 22 14,T 325,Q 24 22 5,Q 25 17 1,T 118,Q 7 4 2,T 818,Q 16 15 9,Q 26 18 7,
  T 435,Q 17 13 2,T 1533,T 749,Q 47 31 29,T 233,T 675,Q 38 33 8,T 1841,
  Q 14 10 1,Q 29 22 3,T 137,Q 11 10 5,T 558,T 2103,Q 19 14 2,Q 9 8 2,Q 23 14 5,
  Q 17 12 6,T 1533,Q 25 24 10,T 2233,T 699,Q 34 23 18,T 413,Q 38 30 28,
  T 2121,T 994,Q 20 3 2,T 47,T 1291,Q 8 5 3,T 35,Q 14 10 8,T 2929,T 458,
  Q 39 36 14,T 966,T 3483,Q 31 28 6,T 163,Q 15 14 8,T 1281,T 825,Q 37 19 3,
  T 613,Q 16 15 9,Q 28 20 10,Q 23 20 7,Q 30 23 1,T 3789,T 374,Q 33 29 23,
  T 2829,Q 29 10 4,Q 16 9 6,Q 23 10 7,Q 32 31 7,Q 23 22 2,T 270,Q 34 31 21,
  T 967,T 3591,Q 28 13 5,T 2235,Q 30 28 26,T 213,T 1087,Q 12 3 2,T 215,
  T 3643,Q 22 3 2,T 2505,Q 20 16 2,Q 17 13 10,T 1568,Q 19 13 1,T 654,T 79,
  Q 40 35 21,T 1227,Q 17 6 2,T 981,T 2462,Q 27 22 18,T 154,T 855,Q 28 21 13,
  T 3165,Q 18 7 4,T 129,Q 24 19 8,Q 25 19 17,Q 26 9 6,Q 17 15 10,Q 17 16 6,
  T 215,Q 35 30 13,Q 25 15 13,T 664,Q 12 9 7,T 2400,T 531,Q 18 5 1,T 1113,
  Q 20 5 2,Q 21 16 13,T 197,Q 45 39 7,T 940,Q 29 8 2,Q 11 10 2,T 1681,Q 17 13 10,
  T 2729,T 612,Q 45 34 25,T 1612,Q 15 11 1,Q 20 10 6,T 1333,Q 20 2 1,T 221,
  T 197,Q 7 3 2,T 1466,Q 29 11 4,Q 32 22 6,Q 25 12 4,Q 27 26 23,Q 19 6 1,
  T 756,Q 20 19 5,T 2206,T 1335,Q 26 21 14,T 1089,Q 10 3 2,T 1653,T 321,
  Q 23 15 5,T 1178,Q 17 10 5,Q 27 24 17,T 515,Q 15 11 10,T 137,T 133,Q 40 23 21,
  T 617,T 891,Q 11 4 2,T 795,Q 15 14 2,T 1057,T 80,Q 14 7 1,T 2967,T 51,
  Q 18 13 10,T 3001,Q 28 17 7,Q 38 27 6,T 498,Q 19 13 2,T 413,Q 23 21 20,
  Q 16 11 1,T 1557,Q 26 14 1,T 1121,T 424,Q 19 13 12,T 1412,T 1463,Q 11 7 2,
  T 1253,Q 36 31 15,T 1153,T 483,Q 27 13 12,T 1054,Q 29 19 3,Q 18 16 15,
  T 1695,Q 21 14 3,Q 13 12 10,T 387,Q 44 5 3,T 1249,Q 14 4 3,Q 24 2 1,T 15,
  Q 20 15 14,Q 25 8 4,T 2678,Q 37 18 2,T 504,Q 28 10 3,Q 7 6 1,T 1679,Q 24 22 12,
  T 1041,T 2160,Q 26 25 17,T 2073,T 2423,Q 17 12 8,T 2581,Q 16 3 2,T 1989,
  T 952,Q 16 3 1,T 127,Q 12 4 3,Q 26 21 2,T 3087,Q 18 16 11,T 2201,T 554,
  Q 23 22 17,T 3159,T 687,Q 20 16 5,T 1541,Q 28 26 20,T 1753,T 776,Q 31 30 25,
  T 141,Q 18 15 9,Q 39 33 16,T 1713,Q 32 18 3,T 917,Q 16 11 9,Q 21 14 11,
  T 229,T 1843,Q 11 10 9,T 279,Q 33 19 16,T 53,Q 13 12 3,Q 19 15 13,T 2240,
  T 695,Q 30 24 4,T 131,Q 34 33 30,Q 13 9 8,T 98,Q 39 13 7,T 1305,Q 15 8 5,
  Q 25 17 16,T 3627,Q 18 8 3,T 1389,T 1444,Q 30 23 3,T 1009,Q 31 12 11,
  Q 24 15 5,Q 11 9 2,Q 20 19 12,T 1025,T 1069,Q 27 16 5,T 960,T 1115,Q 34 21 6,
  T 297,Q 10 7 2,T 2449,T 273,Q 27 23 9,T 253,Q 32 27 15,Q 42 29 17,T 1603,
  Q 15 14 4,T 413,T 358,Q 23 11 6,T 883,Q 28 26 3,Q 29 24 16,T 1731,Q 30 27 18,
  T 669,T 1523,Q 39 14 11,T 768,T 239,Q 31 17 6,T 2519,Q 31 22 13,T 1597,
  T 989,Q 25 10 8,T 967,T 3759,Q 16 9 1,T 351,Q 20 5 3,T 2889,T 907,Q 32 11 2,
  T 2513,T 2667,Q 28 25 23,T 9,Q 15 6 4,Q 31 2 1,T 2333,Q 30 23 17,T 1611,
  T 771,Q 25 16 1,T 627,Q 15 9 4,T 217,T 1130,Q 15 12 2,T 3058,T 1667,Q 20 17 5,
  T 1057,Q 32 15 14,Q 21 15 1,T 10,Q 8 3 2,T 3494,Q 19 12 4,Q 47 22 18,
  T 269,Q 26 18 10,Q 28 25 15,T 218,Q 25 24 19,T 548,T 719,Q 25 7 6,Q 9 5 4,
  Q 30 19 18,T 25,T 468,Q 11 3 2,T 739,T 1043,Q 24 12 10,T 1683,Q 32 8 6,
  Q 29 28 20,T 1890,Q 28 27 17,T 728,T 1911,Q 22 8 1,T 133,Q 18 8 4,T 89,
  T 3841,Q 29 25 14,T 1020,T 995,Q 18 10 5,T 3641,Q 19 16 14,Q 23 17 7,
  T 1026,Q 52 45 10,T 292,T 1391,Q 29 28 25,T 471,Q 29 26 9,T 381,T 2553,
  Q 25 16 6,T 481,Q 17 6 5,Q 21 17 8,T 539,Q 11 8 7,Q 21 9 8,T 661,Q 43 32 21,
  T 905,T 2611,Q 24 21 13,T 185,Q 9 6 4,T 1477,T 1139,Q 32 27 6,T 777,Q 10 7 3,
  Q 29 22 16,T 1155,Q 41 10 4,T 1869,T 714,Q 9 5 2,T 1055,T 3847,Q 15 14 4,
  T 2477,Q 21 8 2,T 657,T 2758,Q 18 15 3,T 365,Q 15 12 11,Q 33 28 23,T 1727,
  Q 21 18 13,Q 16 14 9,T 5,Q 19 13 11,T 943,T 3507,Q 34 18 13,T 1773,Q 24 10 4,
  Q 21 16 3,T 469,Q 27 24 17,T 94,T 1443,Q 29 27 10,T 327,Q 24 22 11,T 2345,
  T 308,Q 39 36 25,T 2588,T 447,Q 25 18 12,Q 23 18 6,Q 36 29 3,T 2149,T 3564,
  Q 22 15 9,T 1407,T 215,Q 38 36 21,T 351,Q 23 18 11,Q 15 5 3,T 1462,Q 14 13 7,
  T 1694,Q 36 35 2,Q 35 14 2,T 343,Q 18 16 14,T 173,T 2050,Q 25 15 13,Q 20 11 9,
  T 3399,Q 13 10 3,T 2861,Q 9 5 2,T 3157,T 1496,Q 32 21 14,T 1530,T 1407,
  Q 21 15 4,T 225,Q 20 17 14,T 693,T 2232,Q 15 6 3,T 161,Q 12 7 6,Q 21 16 2,
  T 477,Q 24 13 2,Q 21 17 4,T 44,Q 32 25 23,T 3500,T 419,Q 35 10 9,T 915,
  Q 30 8 3,T 193,T 2439,Q 30 19 9,T 226,T 351,Q 25 23 4,T 2197,Q 47 42 24,
  T 3957,T 1755,Q 27 5 4,T 98,Q 43 24 14,Q 28 26 19,T 637,Q 14 12 8,Q 21 12 11,
  T 938,Q 37 19 6,T 2846,Q 21 14 11,Q 29 28 8,T 185,Q 27 13 10,T 1305,T 3789,
  Q 32 17 15,T 417,Q 17 12 9,Q 19 17 10,Q 16 9 8,Q 28 26 10,T 3333,T 1470,
  Q 45 44 30,T 4117,Q 18 15 9,Q 20 15 2,T 405,Q 30 28 19,Q 21 16 10,T 458,
  Q 41 39 5,T 2985,Q 7 5 2,Q 35 12 10,T 1149,Q 31 14 10,Q 14 5 4,T 596,
  Q 37 23 14,T 775,T 1107,Q 37 34 10,T 651,Q 26 18 14,T 3285,T 129,Q 33 10 2,
  T 643,Q 16 15 7,Q 13 12 7,T 553,Q 50 48 19,T 2921,T 1667,Q 27 15 1,T 777,
  Q 26 25 10,Q 29 7 6,T 1737,Q 29 22 7,T 3177,T 170,Q 35 18 14,T 2548,T 1515,
  Q 18 6 5,Q 22 18 13,Q 22 12 5,T 677,T 1905,Q 15 14 5,Q 24 20 8,T 655,
  Q 20 3 2,T 2443,Q 22 12 4,T 2381,T 154,Q 35 17 4,Q 29 15 1,T 1643,Q 13 8 2,
  T 17,Q 14 5 2,T 4125,T 2316,Q 45 39 17,T 2035,T 3851,Q 26 15 8,T 1815,
  Q 22 15 5,Q 32 22 13,Q 23 13 8,Q 19 4 2,T 2369,T 907,Q 14 12 3,T 691,
  Q 39 16 13,T 3797,T 1336,Q 17 11 1,T 312,Q 30 10 3,Q 18 12 4,Q 17 14 11,
  Q 25 6 1,T 3481,T 4191,Q 29 28 11,T 1105,T 287,Q 22 6 1,T 805,Q 32 22 16,
  Q 30 9 1,T 1557,Q 20 9 2,T 2506,Q 33 17 5,Q 30 27 5,T 1049,Q 26 25 24,
  T 3197,T 1061,Q 40 33 22,T 2240,T 2187,Q 15 10 1,T 2751,Q 27 26 20,Q 33 22 9,
  T 168,Q 33 31 28,T 3,T 1539,Q 14 13 8,T 129,Q 35 14 6,T 361,T 97,Q 17 15 13,
  T 1255,T 1459,Q 33 28 26,T 401,Q 34 27 15,Q 17 10 7,T 43,Q 18 9 6,T 992,
  T 1863,Q 16 12 6,T 905,Q 38 6 5,T 3201,T 213,Q 21 17 6,T 25,Q 38 36 7,
  Q 13 5 2,Q 11 8 6,Q 14 11 5,Q 21 6 4,T 412,Q 15 13 8,T 230,Q 19 10 6,
  Q 10 9 7,T 127,Q 24 22 14,Q 33 32 31,T 397,Q 34 15 5,T 936,T 451,Q 31 24 21,
  Q 15 11 3,Q 26 14 13,T 1713,T 827,Q 30 21 5,T 2398,Q 37 6 1,Q 6 4 1,T 85,
  Q 46 34 26,T 405,T 1425,Q 21 6 3,T 641,T 331,Q 26 19 10,T 1421,Q 23 10 5,
  Q 22 21 19,T 4112,Q 13 6 3,T 3096,Q 26 16 13,Q 35 32 25,T 497,Q 33 26 23,
  Q 10 7 4,T 122,Q 25 19 3,T 540,Q 15 14 11,Q 28 18 15,T 753,Q 37 36 31,
  T 3157,T 91,Q 35 24 10,T 109,T 3555,Q 31 6 5,T 1207,Q 19 16 15,T 2709,
  T 4177,Q 33 27 4,T 3044,Q 19 14 8,Q 27 10 7,T 1221,Q 14 8 5,T 721,T 36,
  Q 28 15 13,T 2646,T 3603,Q 21 18 5,T 1575,Q 36 19 2,T 637,T 435,Q 15 4 2,
  T 2296,T 2019,Q 18 2 1,T 787,Q 23 20 4,Q 27 7 1,T 2996,Q 19 10 3,T 242,
  Q 20 11 4,Q 12 9 4,T 465,Q 26 9 5,Q 30 28 9,T 969,Q 31 5 2,T 1944,Q 19 11 10,
  Q 21 13 10,T 519,Q 41 23 6,T 893,T 639,Q 29 10 1,T 4171,Q 33 17 7,Q 14 13 7,
  T 2011,Q 30 23 17,T 4193,T 2161,Q 21 13 3,Q 15 5 1,T 939,Q 13 5 2,T 801,
  Q 33 14 11,T 2581,Q 24 16 11,Q 28 25 10,T 1429,Q 37 10 3,Q 12 8 2,T 1081,
  Q 22 14 4,Q 22 13 2,T 457,Q 19 9 1,T 439,Q 30 7 4,Q 13 11 6,T 143,Q 8 7 3,
  T 1817,T 3448,Q 32 15 13,T 137,T 407,Q 34 8 2,T 29,Q 15 12 10,Q 27 23 17,
  T 431,Q 23 5 4,T 94,Q 21 20 5,Q 17 13 10,Q 13 7 6,Q 33 27 26,Q 23 14 1,
  T 2385,Q 21 14 3,T 734,T 1431,Q 16 9 1,T 1381,Q 30 23 7,Q 19 17 7,T 203,
  Q 47 44 33,T 434,Q 21 17 14,Q 25 18 1,T 287,Q 24 15 14,T 1585,T 3713,
  Q 22 19 11,T 1150,T 1947,Q 29 28 5,T 1197,Q 22 10 2,Q 33 24 19,T 1483,
  Q 4 3 2,T 256,T 103,Q 38 15 11,Q 25 24 7,Q 33 16 11,Q 20 6 3,T 857,Q 41 32 3,
  T 3234,Q 19 14 8,Q 23 14 13,T 2345,Q 20 13 7,T 3961,T 441,Q 23 10 4,T 81,
  Q 15 11 6,Q 26 18 2,T 103,Q 17 14 12,T 4293,T 145,Q 31 17 6,T 280,T 783,
  Q 26 11 3,T 833,Q 26 18 10,T 2529,T 2863,Q 39 14 4,T 884,T 3323,Q 14 9 4,
  T 1157,Q 30 7 1,Q 31 22 18,T 2306,Q 29 19 11,T 2416,T 2111,Q 22 21 18,
  T 1839,Q 20 12 3,T 1061,T 1329,Q 29 23 12,T 28,T 3403,Q 18 17 10,T 927,
  Q 13 11 6,Q 25 19 15,T 1370,Q 32 7 2,T 1175,T 947,Q 19 18 11,T 3815,Q 31 9 6,
  T 2653,Q 21 18 3,Q 25 20 7,T 855,Q 24 14 11,Q 40 17 5,T 1095,Q 32 24 15,
  T 1809,T 2283,Q 25 20 6,T 164,T 2583,Q 26 20 3,T 1295,Q 33 5 2,Q 24 17 15,
  T 281,Q 39 20 2,T 59,Q 25 22 6,Q 8 6 1,T 417,Q 19 16 10,Q 26 25 18,T 360,
  Q 30 13 3,T 1785,T 231,Q 33 32 8,Q 36 22 21,Q 37 18 16,T 3465,T 7,Q 27 17 5,
  T 467,T 1551,Q 32 12 2,T 2279,Q 25 9 6,Q 21 19 8,T 748,Q 13 7 1,T 68,
  T 959,Q 30 26 16,T 525,Q 40 32 31,T 2817,T 1326,Q 35 33 24,T 3235,Q 28 18 11,
  Q 28 20 2,T 2835,Q 32 27 21,T 1053,T 90,Q 45 12 11,T 79,T 4183,Q 18 14 7,
  T 63,Q 29 22 6,T 53,T 1922,Q 30 29 11,T 693,T 2871,Q 25 24 14,T 1181,
  Q 35 26 12,Q 17 8 4,T 3675,Q 27 9 4,Q 27 25 15,T 2159,Q 18 17 5,Q 10 3 1,
  Q 14 9 3,T 1677,T 70,Q 27 10 8,T 188,T 603,Q 37 4 2,T 37,Q 20 5 3,Q 30 29 11,
  T 508,Q 26 19 9,T 257,Q 16 5 2,Q 45 5 2,T 495,Q 34 10 1,Q 33 10 4,T 860,
  Q 26 25 7,T 1597,Q 30 18 3,Q 49 45 22,Q 22 15 13,Q 27 7 2,Q 17 2 1,T 463,
  Q 24 9 2,T 272,Q 14 7 2,Q 12 5 3,T 1585,Q 26 18 7,T 1301,T 656,Q 25 14 6,
  T 309,T 2139,Q 31 21 4,T 549,Q 42 16 9,T 1773,T 3569,Q 15 9 6,T 1089,
  Q 24 17 7,Q 22 8 2,T 2295,Q 37 4 2,T 1497,T 1174,Q 17 8 6,T 1654,Q 29 11 9,
  Q 15 13 4,T 147,Q 28 24 23,Q 32 28 25,T 1028,Q 19 14 6,T 303,T 4275,Q 30 17 10,
  T 1133,Q 15 9 8,T 3889,T 891,Q 15 9 6,T 328,T 683,Q 20 16 5,T 1171,Q 20 10 6,
  T 1653,T 285,Q 33 22 16,T 539,T 4143,Q 33 20 17,T 355,Q 31 24 2,Q 21 20 12,
  T 1100,Q 33 4 3,T 2324,T 1191,Q 33 14 4,T 1697,Q 23 17 16,Q 12 5 1,T 138,
  Q 14 5 2,T 2493,Q 29 23 17,Q 35 30 27,T 3577,Q 18 16 10,T 49,T 1099,Q 37 12 7,
  T 2713,Q 29 16 5,Q 32 9 5,T 623,Q 34 19 7,T 1029,T 2422,Q 51 4 2,T 689,
  T 2239,Q 19 18 1,T 381,Q 21 13 10,T 3949,T 2369,Q 36 9 6,T 990,Q 17 11 6,
  Q 28 27 10,T 2121,Q 30 26 20,T 469,T 1107,Q 43 13 9,T 1216,T 2007,Q 10 9 4,
  T 249,Q 27 7 6,Q 17 6 5,T 958,Q 41 10 6,T 1439,T 287,Q 29 2 1,T 2285,
  Q 17 12 2,Q 10 9 2,T 222,Q 40 25 10,T 433,Q 35 16 13,Q 22 21 16,T 3031,
  Q 22 17 9,T 1749,T 963,Q 18 17 1,T 158,Q 16 4 3,Q 11 6 5,T 2259,Q 28 21 3,
  Q 16 10 9,T 2479,Q 33 32 31,T 6,T 787,Q 11 8 6,T 797,Q 22 11 7,T 853,
  T 1425,Q 35 29 2,T 358,T 83,Q 17 6 5,T 507,Q 24 10 4,T 1125,T 861,Q 17 3 1,
  T 3733,T 2295,Q 22 21 14,T 529,Q 36 35 31,Q 21 16 13,T 3635,Q 39 34 1,
  T 563,T 967,Q 28 20 6,T 2637,Q 29 22 13,Q 9 5 1,T 506,Q 27 14 2,T 25,
  Q 25 23 1,Q 18 13 1,T 4231,Q 16 10 3,T 1029,T 225,Q 12 7 5,T 176,T 691,
  Q 30 26 10,T 939,Q 30 7 5,T 1233,T 2744,Q 29 11 10,T 435,T 2191,Q 23 5 4,
  T 1619,Q 24 19 12,T 981,T 1128,Q 23 8 2,T 966,Q 18 17 5,Q 29 22 16,T 3093,
  Q 22 2 1,T 169,T 345,Q 9 8 6,T 2674,T 495,Q 16 12 10,T 1513,Q 15 7 2,
  T 333,T 2020,Q 35 6 2,T 2079,T 1051,Q 19 16 13,Q 25 16 12,Q 33 22 1,Q 23 10 2,
  T 1386,Q 28 19 17,T 1599,T 3995,Q 37 20 10,T 649,Q 19 14 4,T 1477,T 222,
  Q 31 17 7,T 29,Q 17 12 1,Q 31 14 10,T 1053,Q 32 9 7,T 1217,Q 12 3 1,Q 23 21 8,
  T 155,T 1327,Q 26 22 15,T 287,Q 28 22 10,T 861,T 4350,Q 32 29 11,T 744,
  T 35,Q 23 14 3,T 2935,Q 36 26 1,Q 32 8 1,T 714,Q 45 19 1,T 1615,T 2095,
  Q 30 22 5,T 449,Q 28 14 5,Q 21 9 8,T 1741,Q 9 6 4,T 813,T 103,Q 27 26 23,
  T 2819,Q 33 8 6,Q 36 30 15,T 287,Q 23 16 10,T 2490,T 1319,Q 17 10 7,T 2725,
  Q 21 15 12,T 609,T 946,Q 11 3 2,T 2345,T 235,Q 24 15 6,T 151,Q 26 18 7,
  T 3929,T 2320,Q 33 4 2,T 2022,Q 15 6 1,Q 9 2 1,T 2565,Q 36 31 12,Q 23 13 3,
  T 2276,Q 45 38 6,Q 20 16 5,T 207,Q 30 17 15,T 2395,Q 16 6 4,T 2681,T 2179,
  Q 17 10 7,T 190,Q 19 17 14,Q 31 18 5,T 1699,Q 26 13 5,T 2445,Q 23 19 7,
  Q 33 31 1,T 440,Q 18 7 3,Q 20 9 7,T 29,Q 11 8 4,T 2769,T 2546,Q 27 26 11,
  T 1260,Q 25 15 14,Q 16 9 1,T 55,Q 14 8 6,Q 15 14 2,T 4200,Q 47 33 24,
  T 1493,T 1723,Q 28 13 6,T 575,Q 9 5 2,Q 32 17 8,T 487,Q 9 7 5,T 3563,
  T 539,Q 30 9 7,T 1841,Q 18 16 7,Q 21 11 10,T 917,Q 35 25 5,T 249,Q 47 29 23,
  Q 15 13 10,Q 23 17 12,Q 26 12 10,T 181,T 1120,Q 39 25 22,T 23,T 2991,
  Q 7 5 4,T 207,Q 26 9 4,T 1545,T 944,Q 21 14 1,T 2163,T 2671,Q 10 3 1,
  T 1365,Q 36 7 1,T 4545,T 1268,Q 35 25 4,T 546,Q 12 2 1,Q 31 16 2,T 1921,
  Q 30 28 7,T 469,T 1330,Q 33 29 7,T 682,T 3087,Q 25 11 6,T 281,Q 30 10 7,
  T 809,T 3470,Q 22 15 3,T 1046,T 2807,Q 20 7 6,T 1913,Q 13 12 3,Q 19 10 3,
  Q 12 3 2,Q 15 10 4,T 229,T 1103,Q 38 5 4,Q 33 26 8,Q 31 17 12,Q 29 23 8,
  T 2806,Q 43 37 30,T 884,Q 19 17 8,Q 10 6 5,T 1153,Q 23 13 6,T 1769,Q 30 13 1,
  Q 33 26 14,T 207,T 3159,Q 20 18 4,T 4305,Q 26 14 8,T 2217,T 3936,Q 21 18 13,
  T 1216,Q 16 10 5,Q 14 13 1,Q 31 12 5,Q 14 12 5,T 1365,T 109,Q 31 24 2,
  T 146,T 211,Q 28 20 6,T 645,Q 14 11 3,T 4085,T 397,Q 35 10 9,T 2181,T 2603,
  Q 36 19 17,T 3839,Q 38 7 5,T 397,T 1217,Q 21 14 8,T 1956,T 2919,Q 30 6 4,
  T 1875,Q 33 26 5,Q 33 23 11,T 1023,Q 35 21 7,T 181,T 4315,Q 38 29 3,T 145,
  Q 20 19 17,Q 17 12 8,T 3899,Q 49 18 8,T 545,T 315,Q 23 22 2,Q 22 14 9,
  Q 18 16 12,T 49,T 1629,Q 17 11 7,T 909,T 239,Q 5 2 1,T 441,Q 43 35 18,
  Q 27 18 3,T 48,Q 37 18 13,T 1663,T 531,Q 8 3 2,T 327,Q 42 28 10,Q 24 17 13,
  T 298,Q 19 6 4,T 1364,T 3843,Q 22 16 6,T 2729,Q 30 18 10,T 1549,T 3173,
  Q 39 17 14,T 1123,Q 23 14 6,Q 23 8 5,T 3183,Q 42 18 14,T 837,T 1488,Q 13 4 3,
  T 502,T 1219,Q 10 6 2,T 135,Q 11 3 2,T 2145,T 1526,Q 23 7 2,T 2049,T 2307,
  Q 20 7 6,T 1883,Q 29 6 5,Q 25 20 14,T 2520,Q 49 43 18,T 358,T 1731,Q 8 5 1,
  T 3087,Q 20 18 3,T 3373,T 2149,Q 29 17 15,T 1838,T 2139,Q 22 21 1,T 135,
  Q 35 25 18,T 605,T 208,Q 27 13 9,T 3344,T 675,Q 31 25 4,T 1559,Q 19 16 10,
  T 3141,T 2657,Q 21 19 16,T 924,Q 27 23 12,Q 26 21 8,T 1233,Q 23 20 2,
  Q 18 6 3,T 477,Q 55 22 8,T 2627,T 4027,Q 18 4 1,T 665,Q 18 16 1,Q 35 23 20,
  T 4165,Q 21 14 8,T 2910,T 4263,Q 31 30 2,T 2081,Q 26 14 8,T 3141,T 1176,
  Q 17 5 3,T 546,Q 23 8 5,Q 23 14 1,T 1339,Q 19 18 14,Q 19 10 8,T 2700,
  Q 51 44 5,T 353,Q 18 10 3,Q 16 14 7,Q 19 9 3,Q 25 18 2,T 1221,T 56,Q 31 26 15,
  T 3221,Q 16 7 4,Q 22 16 4,T 1773,Q 35 34 2,T 453,T 1227,Q 28 23 1,T 369,
  Q 23 13 1,Q 26 21 17,T 3297,Q 21 20 7,T 2245,T 3172,Q 38 7 1,T 116,T 2311,
  Q 28 21 17,T 269,Q 28 10 4,Q 18 9 2,T 4459,Q 23 5 1,T 914,T 1347,Q 14 9 5,
  T 561,Q 19 9 6,T 3297,T 1124,Q 17 11 6,T 523,T 3863,Q 31 30 9,Q 24 7 3,
  Q 20 7 3,T 2601,T 1239,Q 29 24 11,T 623,T 3471,Q 17 15 4,T 369,Q 32 14 7,
  T 2769,T 1295,Q 18 13 1,T 429,T 267,Q 30 8 1,T 2009,Q 18 4 1,Q 28 27 7,
  T 260,Q 28 15 6,T 405,Q 32 11 8,Q 24 22 1,Q 11 9 2,Q 11 10 9,T 673,T 1678,
  Q 19 11 6,T 218,T 823,Q 30 19 13,T 2701,Q 14 9 5,Q 22 10 9,T 1280,Q 22 9 6,
  T 699,Q 21 6 5,Q 23 18 8,Q 21 17 7,Q 21 9 2,Q 37 18 5,T 3113,Q 52 43 25,
  T 42,Q 19 14 12,Q 7 5 4,T 225,Q 14 12 1,Q 15 13 6,T 105,Q 25 22 14,T 479,
  T 4611,Q 30 19 2,T 3513,Q 34 14 12,Q 12 5 3,T 458,Q 23 21 20,T 584,T 1431,
  Q 13 12 3,Q 18 14 13,Q 32 29 23,T 4189,T 2807,Q 35 19 5,T 4,Q 25 16 6,
  Q 36 18 13,Q 20 17 1,Q 32 28 6,Q 34 14 5,T 327,Q 37 18 13,T 197,Q 23 14 1,
  Q 27 14 7,T 1695,Q 28 22 18,T 2289,T 1459,Q 45 40 15,T 656,Q 23 20 7,
  Q 29 25 2,T 2583,Q 26 11 7,T 909,T 200,Q 18 9 2,T 4060,Q 19 12 4,Q 20 10 5,
  T 853,Q 23 22 9,T 945,T 211,Q 25 12 7,T 2089,T 999,Q 19 6 5,T 373,Q 38 22 3,
  T 477,Q 23 20 18,Q 29 17 15,T 2301,Q 41 17 10,Q 29 20 10,T 1175,Q 29 26 4,
  T 621,T 99,Q 19 15 6,Q 15 10 1,Q 12 7 5,Q 29 28 14,T 1141,Q 13 9 2,T 2349,
  T 399,Q 39 17 16,T 686,Q 28 21 9,Q 18 9 2,T 903,Q 32 18 11,T 1797,T 1573,
  Q 13 5 2,T 116,T 243,Q 8 4 2,T 3207,Q 30 27 2,Q 16 10 1,T 1650,Q 37 33 22,
  T 43,T 1623,Q 32 14 2,T 2103,Q 36 31 5,T 1193,T 1119,Q 31 26 6,T 565,
  Q 24 9 3,Q 38 27 13,T 3059,Q 31 21 14,T 1185,T 533,Q 43 32 9,T 539,T 47,
  Q 12 8 2,T 335,Q 37 34 32,Q 19 15 11,T 126,Q 11 4 1,T 1498,Q 29 28 15,
  Q 24 22 10,Q 21 9 1,Q 27 20 1,T 2041,T 702,Q 27 13 2,T 3059,T 687,Q 25 9 2,
  T 387,Q 6 5 4,T 1901,T 1883,Q 49 26 14,T 60,T 487,Q 24 16 5,Q 31 19 9,
  Q 8 6 5,Q 38 37 8,T 888,Q 29 25 7,T 1668,T 1503,Q 34 33 6,T 1771,Q 12 7 4,
  T 3177,T 790,Q 36 29 27,T 2524,T 1867,Q 36 29 22,T 525,Q 28 22 15,T 2061,
  T 841,Q 15 8 1,T 284,T 499,Q 28 18 15,Q 18 11 5,Q 28 25 22,T 4309,T 14,
  Q 19 6 4,T 963,Q 21 5 2,Q 19 10 4,Q 13 11 9,Q 37 26 12,Q 15 12 1,T 262,
  Q 14 13 1,T 1526,T 2883,Q 14 6 5,T 93,Q 32 16 3,T 4577,T 361,Q 25 8 2,
  T 1385,Q 16 3 1,Q 29 14 10,T 557,Q 29 20 11,T 2145,T 2429,Q 31 17 7,T 508,
  Q 7 6 1,Q 25 19 18,T 31,Q 16 15 10,T 37,T 295,Q 30 23 5,T 2333,T 3075,
  Q 26 21 10,T 515,Q 21 18 6,Q 21 19 13,T 700,Q 33 7 3,T 3287,T 3591,Q 18 14 11,
  T 1671,Q 9 7 6,Q 31 30 17,T 1227,Q 31 24 21,T 657,Q 32 18 11,Q 22 14 4,
  T 2733,Q 22 15 2,T 1969,T 3558,Q 19 16 2,T 1192,T 3691,Q 15 14 8,T 57,
  Q 22 14 6,Q 22 9 8,T 446,Q 35 21 1,T 2763,T 435,Q 34 22 6,T 225,Q 29 27 24,
  Q 35 34 3,T 1086,Q 17 9 3,T 3538,T 759,Q 44 25 23,T 3367,Q 30 27 12,T 4277,
  T 864,Q 25 6 2,T 910,T 3175,Q 46 37 21,T 2405,Q 17 14 4,T 2333,T 2054,
  Q 20 9 2,T 84,T 1503,Q 17 14 1,T 3381,Q 10 6 3,Q 50 18 17,T 207,Q 36 27 21,
  T 1201,Q 38 27 24,Q 30 6 1,T 565,Q 23 12 3,T 1533,Q 13 7 1,Q 7 3 2,T 1802,
  Q 22 15 11,Q 26 15 1,T 1753,Q 16 13 10,Q 30 15 9,T 917,Q 27 19 17,T 2933,
  T 1227,Q 27 10 6,T 263,Q 9 6 2,Q 46 29 11,T 3492,Q 37 4 2,T 171,Q 12 11 8,
  Q 32 29 25,Q 30 28 15,Q 14 8 5,Q 29 28 23,T 760,Q 30 5 2,T 938,T 663,
  Q 16 12 10,T 3747,Q 22 21 8,T 4125,T 2086,Q 15 7 6,T 275,T 3663,Q 15 14 2,
  T 3015,Q 11 8 1,Q 30 18 15,T 869,Q 36 9 2,T 2361,T 2559,Q 26 8 1,T 889,
  Q 20 7 6,T 833,T 1093,Q 42 17 3,T 1078,T 3891,Q 6 5 1,T 1701,Q 22 14 6,
  T 1557,T 706,Q 56 49 47,T 2588,T 4455,Q 30 18 1,Q 25 13 9,Q 28 17 11,
  T 585,T 1668,Q 41 23 22,T 1390,T 1091,Q 29 8 1,T 741,Q 32 10 6,T 401,
  T 537,Q 24 19 1,T 773,Q 24 18 15,Q 40 21 20,T 711,Q 40 28 15,T 161,T 88,
  Q 38 37 9,T 2036,T 4147,Q 20 13 9,T 2007,Q 17 8 2,T 177,T 390,Q 27 26 21,
  T 339,Q 24 15 6,Q 26 5 4,T 321,Q 35 32 2,T 2677,T 139,Q 42 41 10,T 284,
  Q 12 10 3,Q 18 10 2,T 1165,Q 22 21 18,Q 30 9 8,T 337,Q 43 32 21,Q 19 15 10,
  T 711,Q 41 28 27,Q 31 30 22,Q 26 17 12,T 1545,Q 27 26 8,Q 46 33 22,T 4420,
  Q 30 19 10,Q 18 4 3,T 105,Q 34 23 18,T 305,T 144,Q 13 10 3,T 2563,Q 22 10 7,
  Q 13 12 5,T 207,Q 4 3 1,T 1541,T 1159,Q 39 37 25,T 104,Q 32 25 7,Q 32 29 1,
  Q 18 13 10,Q 35 16 2,Q 25 19 12,T 389,Q 49 47 13,T 1788,Q 35 28 24,Q 18 12 2,
  T 565,Q 38 18 11,T 2349,T 3537,Q 25 23 10,T 329,Q 19 16 4,Q 11 6 5,T 3379,
  Q 18 11 10,T 317,T 803,Q 36 15 1,T 4386,T 4235,Q 34 28 1,T 2997,Q 33 18 16,
  Q 13 10 3,T 1809,Q 35 32 21,T 2604,Q 34 27 14,Q 26 8 2,T 339,Q 23 8 6,
  T 2681,T 903,Q 13 11 3,T 3842,T 4279,Q 19 18 2,T 183,Q 19 16 15,T 573,
  T 1181,Q 35 28 5,Q 27 7 5,T 747,Q 35 34 11,T 389,Q 38 15 3,T 3673,T 710,
  Q 33 6 4,T 2019,T 2583,Q 46 38 7,Q 31 4 1,Q 32 28 18,Q 27 7 1,T 787,Q 39 38 26,
  T 430,Q 23 13 5,Q 34 21 13,T 11,Q 10 4 3,Q 27 19 10,T 40,Q 21 10 2,T 219,
  T 2027,Q 10 7 1,T 2699,Q 11 10 7,Q 27 16 15,T 483,Q 42 35 15,T 1899,T 95,
  Q 29 17 8,T 4483,Q 32 9 6,T 381,T 1185,Q 49 18 14,T 901,T 2691,Q 37 33 26,
  Q 30 29 26,Q 12 9 7,T 1445,T 1987,Q 39 38 31,T 1382,T 331,Q 34 10 3,T 2397,
  Q 23 6 2,Q 34 7 3,T 2216,Q 22 21 1,T 451,Q 25 19 9,Q 32 26 17,T 2059,
  Q 29 12 10,T 133,T 3069,Q 15 14 6,T 1882,T 2355,Q 23 17 8,T 1535,Q 32 24 10,
  T 2453,T 1334,Q 31 30 11,T 539,T 343,Q 9 8 5,T 851,Q 25 11 4,Q 17 14 4,
  T 381,Q 30 15 10,T 2707,Q 20 14 3,Q 34 29 20,T 2691,Q 34 24 23,T 1701,
  T 4399,Q 36 3 2,T 295,T 2587,Q 11 8 5,T 519,Q 27 24 12,T 2045,T 124,Q 21 19 5,
  T 2954,T 1483,Q 26 10 2,T 707,Q 30 27 22,T 993,T 785,Q 27 10 7,T 1974,
  T 1143,Q 14 11 10,T 3129,Q 21 20 6,T 573,T 495,Q 7 4 2,T 121,Q 29 22 3,
  Q 41 40 31,T 1447,Q 26 10 6,T 4013,T 2951,Q 19 13 9]
