/*  -*-objc-*-
 *  Functions.m: Implementation of the Functions
 *  of the GNUstep GWorkspace application
 *
 *  Copyright (c) 2001 Enrico Sersale <enrico@imago.ro>
 *  
 *  Author: Enrico Sersale
 *  Date: August 2001
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <Foundation/Foundation.h>
#include <AppKit/AppKit.h>
#include "Functions.h"
#include "GNUstep.h"

#define ONE_KB 1024
#define ONE_MB (ONE_KB * ONE_KB)
#define ONE_GB (ONE_KB * ONE_MB)

typedef enum { 
	byname,
	bykind,
	bydate, 
	bysize,
	byowner
} SortType;

#define ATTRIBUTES_AT_PATH(a, p, l) \
a = [[NSFileManager defaultManager] fileAttributesAtPath: (NSString *)p traverseLink: l]

#define SORT_INDEX(i, p) { \
BOOL isdir; \
[[NSFileManager defaultManager] fileExistsAtPath: (NSString *)p isDirectory: &isdir]; \
if (isdir) { \
i = 2; \
} else { \
if ([[NSFileManager defaultManager] isExecutableFileAtPath: (NSString *)p] == YES) { \
i = 1; \
} else { \
i = 0; \
} \
} }

static inline int sort_Paths(id *p1, id *p2, void *context)
{
  int stype;
  int i1, i2;      
  NSDictionary *attributes; 
  NSDate *d1, *d2;
  float fs1, fs2;
  NSString *own1, *own2;
     
  stype = (int)context;

  switch(stype) {
    case byname:
			{
				NSString *n1 = [(NSString *)p1 lastPathComponent];
				NSString *n2 = [(NSString *)p2 lastPathComponent];

      	if ([n2 hasPrefix: @"."] || [n1 hasPrefix: @"."]) {
        	if ([n2 hasPrefix: @"."] && [n1 hasPrefix: @"."]) {
          	return [n1 caseInsensitiveCompare: n2];
        	} else {
          	return [n2 caseInsensitiveCompare: n1];
        	}
      	}
      	return [n1 caseInsensitiveCompare: n2];
      	break;
  		}
			
    case bykind:      
 			SORT_INDEX (i1, p1);
			SORT_INDEX (i2, p2);
			    
      if (i1 == i2) {			
        return [(NSString *)p1 compare: (NSString *)p2]; 
      }   
			   
      return (i1 < i2 ? 1 : -1);
      break;
  
    case bydate:
			ATTRIBUTES_AT_PATH(attributes, p1, NO);
      d1 = [attributes fileModificationDate];
			ATTRIBUTES_AT_PATH(attributes, p2, NO);
      d2 = [attributes fileModificationDate];
    
      return [d1 compare: d2]; 
      break;

    case bysize:
			ATTRIBUTES_AT_PATH(attributes, p1, NO);
      fs1 = [attributes fileSize];
			ATTRIBUTES_AT_PATH(attributes, p2, NO);
      fs2 = [attributes fileSize];
    
      return (fs1 < fs2 ? 1 : -1);    
      break;

    case byowner:
			ATTRIBUTES_AT_PATH(attributes, p1, NO);
      own1 = [attributes fileOwnerAccountName];
			ATTRIBUTES_AT_PATH(attributes, p2, NO);
      own2 = [attributes fileOwnerAccountName];
    
      return [own1 compare: own2];     
      break;
 
    default:
      break;
  }

  return 1;
}

int sortPaths(id *p1, id *p2, void *context)
{
  return sort_Paths(p1, p2, context);
}

BOOL isSubPath(NSString *p1, NSString *p2)
{
  int l1 = [p1 length];
  int l2 = [p2 length];  

  if ((l1 > l2) || ([p1 isEqualToString: p2])) {
    return NO;
  }

  return [[p2 substringToIndex: l1] isEqualToString: p1];
}

NSString *fileSizeDesc(unsigned long long size)
{
	NSString *sizeStr;
	char *sign = "";
    
	if(size == 1) {
		sizeStr = @"1 byte";
	} else if(size < 0) {
		sign = "-";
		size = -size;
	}
	if(size == 0) {
		sizeStr = @"0 bytes";
	} else if(size < (10 * ONE_KB)) {
		sizeStr = [NSString stringWithFormat:@"%s %d bytes", sign, size];
	} else if(size < (100 * ONE_KB)) {
 		sizeStr = [NSString stringWithFormat:@"%s %3.2fKB", sign,
                          					((double)size / (double)(ONE_KB))];
	} else if(size < (100 * ONE_MB)) {
		sizeStr = [NSString stringWithFormat:@"%s %3.2fMB", sign,
                          					((double)size / (double)(ONE_MB))];
	} else {
 		sizeStr = [NSString stringWithFormat:@"%s %3.2fGB", sign,
                          					((double)size / (double)(ONE_GB))];
	}

	return sizeStr;
}
