 /*  -*-objc-*-
 *  DesktopPreferences.m: Implementation of the DesktopPreferences Class 
 *  of the GNUstep GWorkspace application
 *
 *  Copyright (c) 2001 Enrico Sersale <enrico@imago.ro>
 *  
 *  Author: Enrico Sersale
 *  Date: August 2001
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#import <Foundation/Foundation.h>
#import <AppKit/AppKit.h>
#include "DesktopPreferences.h"
#include "GWorkspace.h"
#include "Functions.h"
#include "GNUstep.h"

NSString *GWDesktopViewColorChangedNotification = @"GWDesktopViewColorChangedNotification";
NSString *GWDesktopViewImageChangedNotification = @"GWDesktopViewImageChangedNotification";
NSString *GWDesktopViewUnsetImageNotification = @"GWDesktopViewUnsetImageNotification";

@implementation ColorsView

- (void)dealloc
{
  TEST_RELEASE (color);
  [super dealloc];
}

- (id)init
{
  self = [super init];
  color = nil;
  return self;
}

- (void)setColor:(NSColor *)c
{
  ASSIGN (color, c);
}

- (void)drawRect:(NSRect)rect
{
  NSRect bounds = [self bounds];  
  float x = rect.origin.x + 2;
  float y = rect.origin.y + 2;
  float w = rect.size.width - 4;
  float h = rect.size.height - 4;
  NSRect colorRect = NSMakeRect(x, y, w, h);
  
  [super drawRect: rect];
  
  NSDrawGrayBezel(bounds, rect);
  
  if (color != nil) {
    [color set];
    NSRectFill(colorRect);
  } 
}

@end

@implementation DesktopPrefsView

- (void)dealloc
{
  RELEASE (redSlider);
  RELEASE (greenSlider);
  RELEASE (blueSlider);
  RELEASE (colorsView);
  RELEASE (cell);
	RELEASE (redlabel);
	RELEASE (greenlabel);
	RELEASE (bluelabel);
	RELEASE (setColorButt);
	RELEASE (chooseButt);
	RELEASE (chooseLabel);
  RELEASE (setImageButt);
  TEST_RELEASE (color);
  [super dealloc];
}

- (id)init
{
  self = [super init];
  
  if (self) {  
    NSUserDefaults *defaults = [NSUserDefaults standardUserDefaults];	
    NSDictionary *desktopViewPrefs;
    NSString *imagePath;
		NSString *activeStr;
    id label;

	  [self setFrame: NSMakeRect(0, 0, 394, 270)];	     
		
		activeStr = [defaults objectForKey: @"desktop"];
		if (activeStr == nil) {
			active = NO;
		} else {
			active = [activeStr intValue];
		}
		
    desktopViewPrefs = [defaults dictionaryForKey: @"desktopviewprefs"];
		
    if (desktopViewPrefs != nil) { 
    	id dictEntry = [desktopViewPrefs objectForKey: @"backcolor"];
			imagePath = [desktopViewPrefs objectForKey: @"imagepath"];

    	if(dictEntry == nil) {
      	ASSIGN (color, [NSColor windowBackgroundColor]);
    	} else {
      	r = [[dictEntry objectForKey: @"red"] floatValue];
      	g = [[dictEntry objectForKey: @"green"] floatValue];
      	b = [[dictEntry objectForKey: @"blue"] floatValue]; 
				alpha = [[dictEntry objectForKey: @"alpha"] floatValue];
      	ASSIGN (color, [NSColor colorWithCalibratedRed: r green: g blue: b alpha: alpha]);
    	}
    } else {
			imagePath = nil;
    	ASSIGN (color, [NSColor windowBackgroundColor]);
		}
    
    cell = [[NSTextFieldCell alloc] initTextCell: NSLocalizedString(@"DeskTop Color", @"")];
    [cell setAlignment: NSCenterTextAlignment];
    [cell setBordered: NO];
    [cell setEditable: NO];
    [cell setDrawsBackground: YES];
       
    redSlider = [NSSlider new];
    [redSlider setFrame: NSMakeRect(74, 201, 120, 18)];   
    [redSlider setMinValue: 0.0];   
    [redSlider setMaxValue: 1.0];  
    [redSlider setTarget: self]; 
    [redSlider setAction: @selector(makeColor:)];   
    [redSlider setFloatValue: r];   
    [self addSubview: redSlider];

		MAKE_LOCALIZED_LABEL (redlabel, NSMakeRect(200, 201, 35, 18), @"red", @"", 'l', NO, self);		  

    greenSlider = [NSSlider new];
    [greenSlider setFrame: NSMakeRect(74, 176, 120, 18)];      
    [greenSlider setMinValue: 0.0];   
    [greenSlider setMaxValue: 1.0];  
    [greenSlider setTarget: self]; 
    [greenSlider setAction: @selector(makeColor:)];
    [greenSlider setFloatValue: g];          
    [self addSubview: greenSlider];

		MAKE_LOCALIZED_LABEL (greenlabel, NSMakeRect(200, 176, 35, 18), @"green", @"", 'l', NO, self);		  

    blueSlider = [NSSlider new];
    [blueSlider setFrame: NSMakeRect(74, 151, 120, 18)];      
    [blueSlider setMinValue: 0.0];   
    [blueSlider setMaxValue: 1.0]; 
    [blueSlider setTarget: self]; 
    [blueSlider setAction: @selector(makeColor:)];
    [blueSlider setFloatValue: b];          
    [self addSubview: blueSlider];

		MAKE_LOCALIZED_LABEL (bluelabel, NSMakeRect(200, 151, 35, 18), @"blue", @"", 'l', NO, self);		  
        
    colorsView = [ColorsView new];
    [colorsView setFrame: NSMakeRect(245, 151, 70, 68)];  
    [colorsView setColor: color];      
		[self addSubview: colorsView]; 

		setColorButt = [[NSButton alloc] initWithFrame: NSMakeRect(138, 109, 118, 24)];
		[setColorButt setButtonType: NSMomentaryLight];
		[setColorButt setTitle: NSLocalizedString(@"Set Color", @"")];	
    [setColorButt setTarget: self];
    [setColorButt setAction: @selector(setColor:)];	
		[self addSubview: setColorButt];    

		MAKE_LOCALIZED_LABEL (label, NSMakeRect(10, 42, 376, 60), @"Push the \"Set Image\" button\nto set your DeskTop image.\nThe image must have the same\nsize of your screen.", @"", 'c', YES, self);		  
    [label setTextColor: [NSColor grayColor]];
		
		chooseButt = [[NSButton alloc] initWithFrame: NSMakeRect(10, 10, 16, 16)];
		[chooseButt setButtonType: NSSwitchButton];
		[chooseButt setAllowsMixedState: NO];
		[chooseButt setImagePosition: NSImageOnly];	
		[chooseButt setImage: [NSImage imageNamed: @"common_SwitchOff.tiff"]];			
		[chooseButt setAlternateImage: [NSImage imageNamed: @"common_SwitchOn.tiff"]];
		[chooseButt setTarget: self];
		[chooseButt setAction: @selector(setState:)];
		[chooseButt setState: active];
		[self addSubview: chooseButt]; 

		chooseLabel = [[NSTextField alloc] initWithFrame: NSMakeRect(30, 10, 80, 16)];	
		[chooseLabel setFont: [NSFont systemFontOfSize: 12]];
		[chooseLabel setBackgroundColor: [NSColor windowBackgroundColor]];
		[chooseLabel setBezeled: NO];
		[chooseLabel setEditable: NO];
		[chooseLabel setSelectable: NO];
    [self addSubview: chooseLabel]; 

		setImageButt = [[NSButton alloc] initWithFrame: NSMakeRect(138, 10, 118, 24)];
		[setImageButt setButtonType: NSMomentaryLight];
    [setImageButt setTarget: self];
    if(imagePath != nil) {
      [setImageButt setTitle: NSLocalizedString(@"Unset Image", @"")];	
      [setImageButt setAction: @selector(unsetImage:)];	
    } else {
      [setImageButt setTitle: NSLocalizedString(@"Set Image", @"")];	
      [setImageButt setAction: @selector(chooseImage:)];	
    }
		[self addSubview: setImageButt]; 
		
		gw = [GWorkspace gworkspace];
		
		[self setState: chooseButt];
  }
  
  return self;
}

- (NSString *)prefsname
{
  return NSLocalizedString(@"DeskTop", @"");
}

- (void)setState:(id)sender
{
	if ([sender state] == NSOnState) {
		active = YES;
		[chooseLabel setStringValue: NSLocalizedString(@"Deactivate", @"")];	
		[cell setTextColor: [NSColor blackColor]];		
		[redlabel setTextColor: [NSColor blackColor]];
		[redlabel setNeedsDisplay: YES];	
		[greenlabel setTextColor: [NSColor blackColor]];
		[greenlabel setNeedsDisplay: YES];	
		[bluelabel setTextColor: [NSColor blackColor]];
		[bluelabel setNeedsDisplay: YES];	
		[redSlider setEnabled: YES];
		[greenSlider setEnabled: YES];
		[blueSlider setEnabled: YES];
		[colorsView setColor: color];
  	[colorsView setNeedsDisplay: YES];	
		[setColorButt setEnabled: YES];
		[setImageButt setEnabled: YES];
	} else {
		active = NO;
		[chooseLabel setStringValue: NSLocalizedString(@"Activate", @"")];
		[cell setTextColor: [NSColor grayColor]];
		[redlabel setTextColor: [NSColor grayColor]];
		[redlabel setNeedsDisplay: YES];	
		[greenlabel setTextColor: [NSColor grayColor]];
		[greenlabel setNeedsDisplay: YES];
		[bluelabel setTextColor: [NSColor grayColor]];		
		[bluelabel setNeedsDisplay: YES];
		[redSlider setEnabled: NO];
		[greenSlider setEnabled: NO];
		[blueSlider setEnabled: NO];
		[colorsView setColor: [NSColor windowBackgroundColor]];
		[colorsView setNeedsDisplay: YES];	
		[setColorButt setEnabled: NO];
		[setImageButt setEnabled: NO];
	}

	[gw showHideDesktop: active];	
}

- (void)makeColor:(id)sender
{
  if (sender == redSlider) {
    r = [sender floatValue];
  } else if (sender == greenSlider) {
    g = [sender floatValue];
  } else if (sender == blueSlider) {
    b = [sender floatValue];
  }

  ASSIGN (color, [NSColor colorWithCalibratedRed: r green: g blue: b alpha: 1]);  
  [colorsView setColor: color];
  [colorsView setNeedsDisplay: YES];
}

- (void)setColor:(id)sender
{
  NSUserDefaults *defaults;	
  NSMutableDictionary *desktopViewPrefs, *colorDict;
  id dictEntry;
   
  defaults = [NSUserDefaults standardUserDefaults];	
  
  dictEntry = [defaults dictionaryForKey: @"desktopviewprefs"];
  if (dictEntry == nil) {      
    desktopViewPrefs = [[NSMutableDictionary alloc] initWithCapacity: 1];
  } else {
 		desktopViewPrefs = [dictEntry mutableCopy];
  }

  colorDict = [NSMutableDictionary dictionaryWithCapacity: 1];
  [colorDict setObject: [NSString stringWithFormat: @"%.2f", r] forKey: @"red"];
  [colorDict setObject: [NSString stringWithFormat: @"%.2f", g] forKey: @"green"];
  [colorDict setObject: [NSString stringWithFormat: @"%.2f", b] forKey: @"blue"];
  [colorDict setObject: @"1.0" forKey: @"alpha"];
	
  [desktopViewPrefs setObject: colorDict forKey: @"backcolor"];
  [defaults setObject: desktopViewPrefs forKey: @"desktopviewprefs"];
  [defaults synchronize];  
	RELEASE (desktopViewPrefs);
	
	[[NSNotificationCenter defaultCenter]
 				postNotificationName: GWDesktopViewColorChangedNotification
	 								    object: colorDict];	
}

- (void)chooseImage:(id)sender
{
	NSOpenPanel *openPanel;
	NSArray *fileTypes;
	NSString *imagePath;
	int result;
  NSUserDefaults *defaults;	
  NSMutableDictionary *desktopViewPrefs;
  id dictEntry;
   
	fileTypes = [NSArray arrayWithObjects: @"tiff", @"tif", @"TIFF", @"TIFF", @"jpeg", @"jpg", @"JPEG", @"JPG", nil];
  
	openPanel = [NSOpenPanel openPanel];
	[openPanel setTitle: @"open"];	
  [openPanel setAllowsMultipleSelection: NO];
  [openPanel setCanChooseFiles: YES];
  [openPanel setCanChooseDirectories: NO];

  result = [openPanel runModalForDirectory: NSHomeDirectory() file: nil types: fileTypes];
	if(result != NSOKButton) {
		return;
  }
  
	imagePath = [NSString stringWithString: [openPanel filename]];
  [setImageButt setTitle: @"Unset Image"];	
  [setImageButt setAction: @selector(unsetImage:)];	
  [setImageButt setNeedsDisplay: YES];
                       
  defaults = [NSUserDefaults standardUserDefaults];	  
  dictEntry = [defaults dictionaryForKey: @"desktopviewprefs"];
  if (dictEntry == nil) {      
    desktopViewPrefs = [[NSMutableDictionary alloc] initWithCapacity: 1];
  } else {
 		desktopViewPrefs = [dictEntry mutableCopy];
  }  
  [desktopViewPrefs setObject: imagePath forKey: @"imagepath"];
  [defaults setObject: desktopViewPrefs forKey: @"desktopviewprefs"];
  [defaults synchronize];  
  RELEASE (desktopViewPrefs);      
  	
	[[NSNotificationCenter defaultCenter]
 				postNotificationName: GWDesktopViewImageChangedNotification
	 								    object: imagePath];
}

- (void)unsetImage:(id)sender
{
  NSUserDefaults *defaults;	
  NSMutableDictionary *desktopViewPrefs;
  NSString *imagePath;
  id dictEntry;

  defaults = [NSUserDefaults standardUserDefaults];	
  
  dictEntry = [defaults dictionaryForKey: @"desktopviewprefs"];
  if (dictEntry != nil) {      
 		desktopViewPrefs = [dictEntry mutableCopy];
    imagePath = [desktopViewPrefs objectForKey: @"imagepath"];
    if(imagePath != nil) {
      [desktopViewPrefs removeObjectForKey: @"imagepath"];  
      [defaults setObject: desktopViewPrefs forKey: @"desktopviewprefs"];
      [defaults synchronize];  
    }
    RELEASE (desktopViewPrefs);  
  }    
	
  [setImageButt setTitle: @"Set Image"];	
  [setImageButt setAction: @selector(chooseImage:)];	
  [setImageButt setNeedsDisplay: YES];

	[[NSNotificationCenter defaultCenter]
 				postNotificationName: GWDesktopViewUnsetImageNotification
	 								    object: nil];
}

- (void)drawRect:(NSRect)rect
{
  NSRect gr = NSMakeRect(67, 140, 259, 93);
  NSRect cellrect = NSMakeRect(150, 224, 93, 14);
  
  [super drawRect: rect];
  
  NSDrawGroove(gr, gr);
  
  [cell setBackgroundColor: [NSColor windowBackgroundColor]];
  [cell drawWithFrame: cellrect inView: self];
}

@end
