/*
 * Scan for the existance of a specific peer using fake ARP requests
 *
 * Copyright (C) 2003  Enrico Zini <enrico@debian.org>
 * Originally based on laptop-netconf.c by Matt Kern <matt@debian.org>
 * Which in turn was nased on divine.c by Felix von Leitner <felix@fefe.de>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "PeerScanner.h"
#include "PacketMaker.h"
#include "Environment.h"

#include <netinet/in.h> // ntohs, htons, ...

/*
extern "C" {
#include <libnet.h>
}
*/

using namespace std;
using namespace wibble::sys;

void PeerScanner::registerWith(NetWatcher& watcher) throw ()
{
	bool wantARP = false;
	bool wantEth = false;

	for (list<const PeerScan*>::const_iterator i = candidates.begin();
			i != candidates.end(); i++)
	{
		const PeerScan* scan = *i;
		if (scan->hasIP())
			wantARP = true;
		else
			wantEth = true;
	}

	if (wantARP)
	{
		debug("Listen ARP\n");
		watcher.addARPListener(this);
	}
	if (wantEth)
	{
		debug("Listen Ethernet\n");
		watcher.addEthernetListener(this);
	}
}


void PeerScanner::handleEthernet(const NetBuffer& pkt) throw ()
{
	MutexLock lock(candMutex);

	// Parse and check the ethernet header
	const libnet_ethernet_hdr* packet_header = pkt.cast<struct libnet_ethernet_hdr>();

	//debug("Got eth packet\n");

	// Just check that it comes from the MAC we're looking for
	for (list<const PeerScan*>::const_iterator i = candidates.begin();
			i != candidates.end(); i++)
	{
		// Just check that it comes from the MAC we're looking for
		const PeerScan* scan = *i;
		if (!scan->hasIP() && MAC_MATCHES(&scan->mac(), packet_header->ether_shost))
		{
			debug("Got reply from %s\n", fmt(scan->mac()).c_str());
			succeeded(*i);
		}
	}
}

void PeerScanner::handleARP(const NetBuffer& arp) throw ()
{
	MutexLock lock(candMutex);

	// Parse and check the arp header
	const libnet_arp_hdr* arp_header = arp.cast<libnet_arp_hdr>();
	if (ntohs (arp_header->ar_op) == ARPOP_REPLY)
	{
		//in_addr* ipv4_him = arp_get_tip(arp_header);
		in_addr* ipv4_him = arp_get_sip(arp_header);
		ether_addr* mac_him = arp_get_sha(arp_header);

		debug("Got ARP reply from %s %s\n", fmt(IPAddress(*ipv4_him)).c_str(), fmt(*mac_him).c_str());

		//IPv4_FROM_LIBNET(ipv4_me, arp_header->ar_tpa);
		//IPv4_FROM_ARP(ipv4_him, arp_header->ar_spa);

		for (list<const PeerScan*>::const_iterator i = candidates.begin();
				i != candidates.end(); i++)
		{
			const PeerScan* scan = *i;
			bool match = true;

			// Check if IP matches
			if (scan->hasIP() && ! IPv4_MATCHES(&scan->ip(), ipv4_him))
				match = false;

			// Check if MAC matches
			if (scan->hasMAC() && ! MAC_MATCHES(&scan->mac(), mac_him))
				match = false;

			if (match)
			{
				debug("ARP reply from %s %s matches\n",
						fmt(IPAddress(*ipv4_him)).c_str(), fmt(*mac_him).c_str());
				succeeded(*i);
			}
		}
	}
}


void PeerScanner::addCandidate(const PeerScan* scan)
	throw ()
{
	{
		MutexLock lock(candMutex);
		candidates.push_back(scan);
	}

	// Build and send the arp probe
	PacketMaker pm(sender);
	Buffer pkt;
	
	if (scan->hasIP())
	{
		pkt = pm.makeARPRequest(scan->ip(), scan->source());
		verbose("Sending 10 ARP probes, 1 every second...\n");
	}
	else
	{
		pkt = pm.makePingRequest(scan->mac(), scan->source());
		verbose("Sending 10 Ping probes, 1 every second...\n");
	}


	// Enqueue the packet for sending
	sender.post(pkt, 1000, 10000);
}

// vim:set ts=4 sw=4:
