#ifndef ENVIRONMENT_H
#define ENVIRONMENT_H

/*
 * Common environment for most program functions
 *
 * Copyright (C) 2003  Enrico Zini <enrico@debian.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <string>
#include <vector>

class Environment
{
protected:
	
	// Default iterface to use
	std::string _iface;

	// Default tag to print when nothing is found
	std::string _defprof;

	// True when operations should be verbose
	bool _verbose;

	// True when operations should be very verbose
	bool _debug;

	// Timeout after which we decide we haven't found anything
	int _timeout;

	// Time we wait after initializing an interface
	int _init_timeout;

	Environment() throw ();

	static void init(Environment* instance) throw ();

public:
	static Environment& get() throw ();

	// Accessor methods
	const std::string& iface() const throw () { return _iface; }
	std::string iface(const std::string& iface) throw () { return _iface = iface; }

	const std::string& defprof() const throw () { return _defprof; }
	std::string defprof(const std::string& defprof) throw () { return _defprof = defprof; }

	bool verbose() const throw () { return _verbose; }
	bool verbose(bool verbose) throw () { return _verbose = verbose; }

	bool debug() const throw () { return _debug; }
	bool debug(bool debug) throw ()
	{
		// Debug implies verbose
		if (debug)
			_verbose = true;
		return _debug = debug;
	}

	int timeout() const throw () { return _timeout; }
	int timeout(int timeout) throw () { return _timeout = timeout; }

	int initTimeout() const throw () { return _init_timeout; }
	int initTimeout(int initTimeout) throw () { return _init_timeout = initTimeout; }
};

// Commodity output functions

#ifndef ATTR_PRINTF
 #ifdef GCC
  #define ATTR_PRINTF(string, first) __attribute__((format (printf, string, first)))
 #else
  #define ATTR_PRINTF(string, first)
 #endif
#endif

void fatal_error(const char* fmt, ...) throw() ATTR_PRINTF(1, 2);
void error(const char* fmt, ...) throw() ATTR_PRINTF(1, 2);
void warning(const char* fmt, ...) throw() ATTR_PRINTF(1, 2);
void verbose(const char* fmt, ...) throw() ATTR_PRINTF(1, 2);
void debug(const char* fmt, ...) throw() ATTR_PRINTF(1, 2);

// vim:set ts=4 sw=4:
#endif
