/*Written with Emacs.*/
/*
  GtkDiskFree shows free space on your mounted partitions.  Copyright
  (C) 2001-2002 Dj-Death (Landwerlin Lionel)
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
  USA */

#include "main.h"
#include "utils.h"

gchar *
useutils_get_homedir (void)
{
	gchar *home = getenv("HOME");
	struct passwd *user;
	
	if (home != NULL)
		return home;
	
	user = getpwuid(getuid());
	if (user == NULL) {
		ERROR((_("Could not get home directory for UID %i, returning %s"),
		       getuid(), DIR_CHAR));
		
		return DIR_CHAR;
	}
	
	if (user->pw_dir == NULL) {
		ERROR((_("Could not get home directory for UID %i, returning %s"),
		       getuid(), DIR_CHAR));
		
		return DIR_CHAR;
	} else
		return user->pw_dir;
	
	return DIR_CHAR;
}

gint
filutils_get_type (const gchar file[])
{
	struct stat st;
	
	if (stat(file, &st) != 0)
		return NOEXIST_TYPE;
	if (S_IFDIR <= st.st_mode && S_IFBLK >= st.st_mode)
		return DIRECTORY_TYPE;
	else if (S_IFREG <= st.st_mode && S_IFLNK >= st.st_mode)
		return FILE_TYPE;
	
	return UNKNOW_TYPE;
}

gint
filutils_get_size (const gchar file[])
{
	struct stat st;
	
	if (stat(file, &st) != 0) {
		ERROR((_("Could not get size for file %s"), file)); 
		
		return -1;
	}
	
	return st.st_size;
}

gchar **
filutils_get_lines (const gchar file[])
{
	FILE *f;
	gchar *buffer, **lines = NULL;
	struct stat st;
	
	if (!strcmp(file, NONE_FILE))
		return lines;
	
	if (strstr(file, DIR_CHAR) == NULL) {
		ERROR((_("Could not open file %s"), file));
		
		return lines;
	}
	
	if ((f = fopen(file, "r")) == NULL) {
		ERROR((_("Could not open file %s"), file));
		
		return lines;
	}
	
	if (stat(file, &st) != 0) {
		ERROR((_("Could not get size for file %s"), file));
		fclose(f);
		
		return lines;
	}
	buffer = g_malloc(st.st_size + 1);
	fread(buffer, st.st_size, 1, f);
	fclose(f);
	buffer[st.st_size] = '\0';
	lines = g_strsplit(buffer, "\n", 0);
	g_free(buffer);
	
	return lines;
}

/* Utils for gchar strings. */

gchar *
strutils_uncomment_line (const gchar line[])
{
	gchar **escape, *ret = NULL;
	
	if (strstr(line, COMMENT_CHAR) == NULL) {
		ret = g_strdup(line);
		
		return ret;
	}
	escape = g_strsplit(line, COMMENT_CHAR, 1);
	ret = g_strdup(escape[0]);
	g_strfreev(escape);
	
	return ret;
}

gint
strutils_count_char (const gchar line[], gchar c)
{
	gint i, n;
	
	n = 0;
	i = 0;
	while (line[i] != '\0') {
		if (line[i] == c)
			n++;
		i++;
	}
	
	return n;
}

gchar *
strutils_get_arg (gchar *lines[], const gchar def[], const gchar del[])
{
	gint i;
	gchar **escape, *ret = NULL;
	
	/* Search line */
	i = 0;
	while (lines[i] != NULL &&
	       strstr(lines[i], def) == NULL)
		i++;
	
	if (lines[i] != NULL) {
		escape = g_strsplit(lines[i], del, 2);
		ret = g_strdup(g_strstrip(escape[1]));
		g_strfreev(escape);
	}
	
	return ret;
}
