/* $Id: grub.cpp,v 1.23 2002/03/19 20:08:37 lajesus Exp $ */

/***************************************************************************
 *                                                                         *
 *   Copyright (C) 2001 Grub, Inc.                                         *
 *   Author: Kord Campbell - kord@grub.org                                 *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2, or (at your option)   *
 *   any later version.                                                    *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the Free Software           *
 *   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA             *
 *                                                                         *
 ***************************************************************************/ 

#include <stdlib.h>
#include <unistd.h>
#include <signal.h>
#include <string.h>
#include <lockfile.h>
#include <pthread.h>
#include <Coordinator.h>
#include <StatusInterface.h>
#include <sys/types.h>
#include <sys/stat.h> 
#include <Gui.h>
#include <clog.h>

extern "C" {
  #include <rmfiles.h>
}

// path to the log and lock files
#ifndef DEFAULT_GRUB_LOG_LOCK_PATH
#define DEFAULT_GRUB_LOG_LOCK_PATH "./"
#endif

// log file name - must be greater than 4 characters long!
#ifndef GRUB_LOG_FILENAME
#define GRUB_LOG_FILENAME "grubclient.log"
#endif

// crash log file name - must be greater than 4 characters long!
#ifndef CRASH_LOGFILE
#define CRASH_LOGFILE "grubcrash.log"
#endif

#define KILL_SLEEP_COUNT 10  // timeout used while waiting for a process to exit

using namespace std;

// from util/grubconf.c, the path of the configure file
extern char *grub_config_file;

// function for shutting down the gui
int shutdown_gui()
{
	// test to see if we have a GUI to shutdown
	if (Crawler_Status_Info.console == false)
	{
		// indicate to the GUI that he should exit
		Crawler_Status_Info.gui_quit = true;

		// wait for the GUI to indicate he is exiting - he loops
		// on a 1 second delay, so we wait for 2 seconds
		sleep(2);

		// return value based on whether GUI indicated he exited - or not
		// GUI flips the gui_quit bit back to false when he exits
		if (Crawler_Status_Info.gui_quit)
		{
			// GUI didn't indicate that he is exiting
			return(1);
		}
		else
		{
			// GUI indicated that he is exiting
			return(0);
		}
	}
	return(2);
}

// do a hard shutdown if a killall -TERM is recieved
static void sigkillterm(int signum)
{
	setVerboseMode(0, 0); 
	int result = shutdown_gui();
	sleep(2);
	exit(1);	
}

// do some stuff if we get a SIGHUP
static void sighup(int signum)
{
	Crawler_Status_Info.hup_request = 1;
}

// do some stuff if we get a SIGPIPE
static void sigpipe(int signum)
{
	setVerboseMode(0, 0); 
	int result = shutdown_gui();
	sleep(2);
	printf("sigpipe recieved - it isn't a good thing.\n");	
	exit(1);	
}

// do a soft shutdown if a killall -USR1 is recieved. 
static void siguserone(int signum)
{
	setVerboseMode(0, 0); 
	Crawler_Status_Info.coordinator_quit = true;
}

static void sighandle(int signum)
{
	// turn off verbose mode to prevent URL status messages from printing to the screen
	setVerboseMode(0, 0); 

	// test to see if we have been here before i.e. we have an impatient user 
	if ( Crawler_Status_Info.signalhandle )
	{
		// clean up the gui, if we have one, before we go bye-bye
		// we do this because we may have a quick CTRL-C, CTRL-C
		int result = shutdown_gui();	

		fprintf( stderr, "\nEntering final shutdown of the client...\n\n");
		exit(1);
	}

	// let us remember that we have sighandled, should we come back
	Crawler_Status_Info.signalhandle = true;

	// start the studown of the gui, if we have one
	int result = shutdown_gui();	

	if (result == 0)
	{
		fprintf( stderr, "GUI successfully halted...\n");
	}
	else if (result == 1)
	{
		fprintf( stderr, "\n\nI'm having problems halting the GUI - sorry...\n\n");
	}

	fprintf( stderr, "Sending halt signal to Coordinator thread...\n\n");
	fprintf( stderr, "Note:  It may take upwards of 5-15 minutes to halt the coordinator\n");
	fprintf( stderr, "       so please be patient.  You can do another Ctrl-C to kill it \n");
	fprintf( stderr, "       outright, but it would be nicer if you waited for the crawler \n");
	fprintf( stderr, "       to return its data to the server.\n\n");
	
	// indicate to the coordinator that he should exit
	// we'll wait on him back in main, in our pthread_join
	// handling of a rowdy coordinator needs to be added
	Crawler_Status_Info.coordinator_quit = true;
}

// function used by calling the create_thread below
static void *startCoordinator(void *stuff)
{
	((Coordinator*)stuff)->start();
	return NULL;
}

static void usage()
{
	const char *usage_str =
	"Grub Distributed Crawling Client Version " VERSION "\n\n"
	"USAGE: grub [-q] [-v] [-vv] [-n <x>] [-N <x>] [-f file] [-l directory] [-h]\n\n"
	"OPTIONS:\n"
	"  -q            : forces quiet mode (no gui)\n"
	"  -v            : verbose (forces console mode)\n"
	"  -vv           : super verbose (forces console mode)\n"
	"  -d            : dumb mode, needs grubface running to configure settings\n"
	"  -n <num urls> : crawls no less than this number of URLs, and then exits\n"
	"  -N <bytes>    : crawls no less than this number of bytes, and then exits\n"
	"  -f file       : use alternate configure file (use absolute path)\n"
	"  -l directory  : use alternate log and lock file directory (use absolute path)\n"
	"  -h            : print this message\n"
	"  -V            : show version number\n\n"
	;

	fprintf( stderr, "%s", usage_str );
	exit(0);
}

// Check if config file exists.  First, checks if there is one in CWD.
// If -f option was used, it checks if that file exists.  If this check
// fails, check if config exists (probably) in INSTALL_PREFIX/etc/ --
// or whatever was set via configure.  grub_config_file = final file
static int exists_config_file()
{
	FILE *fp;

	// check if grub.conf is in CWD
	fp = fopen( grub_config_file, "r" );
	if ( fp ) 
	{
		fclose( fp );
		return 1;
	}

#ifdef DEFAULT_GRUB_CONF
	// check if grub.conf is in the default 
	fp = fopen( DEFAULT_GRUB_CONF, "r" );

	if ( ! fp )
	{
		return 0;
	}

	if ( grub_config_file )
	{
		free(grub_config_file);
	}

	grub_config_file = strdup(DEFAULT_GRUB_CONF);

	return 1;
#else
	return 0;
#endif
}

// Starts the lock file. If another instance of grub is running, we exit.
// If lock file established successfuly, will return otherwise will exit.
void set_lock_file( const char *lock_path )
{
	int ret;
	char buf[1024];

#ifdef HAVE_SNPRINTF
	snprintf( buf, 1024, "%s/%s", lock_path, "grub.lock" );
#else
	sprintf( buf, "%s/%s", lock_path, "grub.lock" );
#endif

	// check/create lock file 
	ret = check_lock_file( buf );

	switch (ret) {
	case 0:
		// new lock file created
	case 1:
		// lock file found, but no process running with that id
		return;
	case 2:
		fprintf(stderr, "grub: another instance is already running\n");
		break;
	default:
		fprintf(stderr, "grub: lockfile error detected\n");
		break;
	}

	exit(1);
}

static char *create_absolute_path( const char *relative_path )
{
	static char path[1024];

	if ( *relative_path == '~' || *relative_path == '/' ) 
	{
		strcpy( path, relative_path );
	}
	else 
	{
		if ( getcwd( path, 1024 ) == NULL ) 
		{
			fprintf( stderr, "grub: can't retrieve current working directory: %s\n", strerror( errno ));
			path[0] = '\0';
		}

		if ( path[0] != '\0' && path[strlen(path) - 1] != '/' )
		{
			strcat( path, "/" );
		}

		strcat( path, relative_path );
	}

	return path;
}

int main (int argc, char *argv[]) 
{
	int ch;

	char *log_lock_path = NULL;
	char full_logfile[1024];

	Coordinator *ceo;

	// define our functions for handling certain signals	
	signal(SIGTERM, sigkillterm );
	signal(SIGINT,  sighandle );
	signal(SIGPIPE, sigpipe );
	signal(SIGHUP,  sighup );
	signal(SIGUSR1, siguserone );

	// set permissions for files to be created
	umask(011);

	// set grub.conf in CWD to be default config file
	grub_config_file = strdup( create_absolute_path("grub.conf") );

	// set default log and lock path 
	log_lock_path = strdup( create_absolute_path(DEFAULT_GRUB_LOG_LOCK_PATH) );

	// disable getopt from printing to stderr
	opterr = 0;  

	// zero out byte_run, url_run variables and console variables
	Crawler_Status_Info.url_run = 0; 
	Crawler_Status_Info.byte_run = 0; 

	Crawler_Status_Info.dumb_mode = false;
	Crawler_Status_Info.verbose = false; 
	Crawler_Status_Info.console = false; 
	Crawler_Status_Info.superverbose = false; 
	Crawler_Status_Info.coordinator_pause = false;

	// command line options section

	while ( ( ch = getopt( argc, argv, "dv::qn:N:hf:l:V" ) ) != -1 ) 
	{
		switch ( ch ) 
		{
			case 'd':  // dumb mode
				Crawler_Status_Info.dumb_mode = true;
				Crawler_Status_Info.coordinator_pause = true;
				break;

			case 'v':  // turn verbose on 
				Crawler_Status_Info.console = true;
				Crawler_Status_Info.verbose = true;
				if(optarg)
				{
					Crawler_Status_Info.superverbose = true;
				}

			case 'q':  // turn output off 
				Crawler_Status_Info.console = true;
				break;

			case 'n':  // count # of urls 
				 sscanf(optarg,"%d",&Crawler_Status_Info.url_run);
				 if ( Crawler_Status_Info.url_run < 1 ) 
				 {
					fprintf(stderr, "A positive value after this option is required...try \'grub -h\' for help.\n");
					exit(1);
				 }
				 break;

			case 'N':  // for # of Bytes 
				 sscanf(optarg,"%d",&Crawler_Status_Info.byte_run);
				 if ( Crawler_Status_Info.byte_run < 1 )
				 {
					fprintf(stderr, "A positive value after this option is required...try \'grub -h\' for help.\n");
					exit(1);
				 }
				 break;

			case 'h':  // just print usage 
				usage();
				break;

			case 'f':  // use alternate configure file 
				if ( grub_config_file )
					free(grub_config_file);
				grub_config_file = strdup( optarg );
				break;

			case 'l':  // log and lock path 
				if ( log_lock_path )
					free(log_lock_path);
				log_lock_path = strdup( optarg );
				break;

			case 'V':  // show version number 
				printf("Version: ");
				printf("%s-%s\n",PACKAGE,VERSION);
				exit(0);

			default:
				fprintf(stderr, "Invalid option...try \'grub -h\' for help.\n");
				usage();
				break;
		} 
	}
  
	// this must happen before chdir() 
	if ( ! exists_config_file() ) {

	      fprintf(stderr, "Can't open grub configuration file\n");
	      exit(1);
	}

	// change the CWD to the one where log/lock files will be 
	// if it fails, will use the current working directory 
	if ( chdir( log_lock_path ) == -1 ) 
	{
	      fprintf( stderr, "Can't change to log and lock files directory: %s\n", strerror(errno));
	      fprintf( stderr, "Using the current working directory for log/lock files...\n");
	}

	// check/create lock file 
	// is it me, or does this need work?
	set_lock_file( "." );

	// Set the global logfile name
	Crawler_Status_Info.logpath = GRUB_LOG_FILENAME;

	// we pass the address of the Config_Info struct so that the config code can update it
	int ret = readGrubConfig(&Config_File_Info);

	if (ret==1) 
	{
		fprintf( stderr, "Cannot read config file...\n");
		exit(0);
	}

	// Initilizing the log process
	clog_init( GRUB_LOG_FILENAME );
	
	// set up the pthread paramaters for the crawler
	pthread_attr_t crawler_attr;
	pthread_attr_init(&crawler_attr);
	pthread_attr_setdetachstate(&crawler_attr, PTHREAD_CREATE_JOINABLE);
	int p_crawler_error;
	pthread_t crawler_pid;

	// thread off the parent coordinator
	ceo = new Coordinator();
	p_crawler_error = pthread_create(&crawler_pid, &crawler_attr, startCoordinator, (void *)ceo); 

	// wait for the coordinator to finish up
	pthread_join(crawler_pid, NULL);

	// Stop the login process
	clog_uninit();
	
	fprintf( stderr, "Shutdown complete...\n\n");
	return 0;
}
