/*
 * GQview
 * (C) 2001 John Ellis
 *
 * Author: John Ellis
 *
 * This software is released under the GNU General Public License (GNU GPL).
 * Please read the included file COPYING for more information.
 * This software comes with no warranty of any kind, use at your own risk!
 */


#include "gqview.h"
#include "img-main.h"

#include "collect.h"
#include "filelist.h"
#include "fullscreen.h"
#include "image.h"
#include "menu.h"
#include "slideshow.h"
#include "utilops.h"

#include "ui_fileops.h"

#include <gdk/gdkkeysyms.h> /* for keyboard values */


static FullScreenData *main_fs = NULL;
static SlideShowData *main_ss = NULL;


/*
 *-----------------------------------------------------------------------------
 * full screen keyboard
 *-----------------------------------------------------------------------------
 */

static gint main_image_full_screen_key_press_cb(GtkWidget *widget, GdkEventKey *event, gpointer data)
{
	ImageWindow *imd = data;
	gint stop_signal = FALSE;
	gint x = 0;
	gint y = 0;

	switch (event->keyval)
		{
		case GDK_Left: case GDK_KP_Left:
			x -= 1;
			stop_signal = TRUE;
			break;
		case GDK_Right: case GDK_KP_Right:
			x += 1;
			stop_signal = TRUE;
			break;
		case GDK_Up: case GDK_KP_Up:
			y -= 1;
			stop_signal = TRUE;
			break;
		case GDK_Down: case GDK_KP_Down:
			y += 1;
			stop_signal = TRUE;
			break;
		case '+': case '=': case GDK_KP_Add:
			image_zoom_adjust(imd, get_zoom_increment());
			break;
		case '-': case GDK_KP_Subtract:
			image_zoom_adjust(imd, -get_zoom_increment());
			break;
		case 'X': case 'x': case GDK_KP_Multiply:
			image_zoom_set(imd, 0.0);
			break;
		case 'Z': case 'z': case GDK_KP_Divide:
			image_zoom_set(imd, 1.0);
			break;
		case GDK_Page_Up: case GDK_KP_Page_Up:
		case GDK_BackSpace:
		case 'B': case 'b':
			main_image_prev();
			stop_signal = TRUE;
			break;
		case GDK_Page_Down: case GDK_KP_Page_Down:
		case GDK_space:
			main_image_next();
			stop_signal = TRUE;
			break;
		case GDK_Home: case GDK_KP_Home:
			main_image_first();
			stop_signal = TRUE;
			break;
		case GDK_End: case GDK_KP_End:
			main_image_last();
			stop_signal = TRUE;
			break;
		case ']':
			main_image_alter(ALTER_ROTATE_90);
			stop_signal = TRUE;
			break;
		case '[':
			main_image_alter(ALTER_ROTATE_90_CC);
			stop_signal = TRUE;
			break;
		case GDK_Delete: case GDK_KP_Delete:
			if (enable_delete_key)
				{
				if (confirm_delete) main_image_full_screen_stop();
				file_util_delete(main_image_get_path(), NULL);
				stop_signal = TRUE;
				}
			break;
		case GDK_Escape:
			main_image_full_screen_stop();
			stop_signal = TRUE;
			break;
		case 'Q': case 'q':
			exit_gqview();
			return FALSE;
                        break;
		case 'S': case 's':
			main_image_slideshow_toggle();
			break;
		case 'P': case 'p':
			main_image_slideshow_pause_toggle();
			break;
		case 'V': case 'v':
			main_image_full_screen_stop();
			break;
		}

	if (event->state & GDK_CONTROL_MASK)
		{
		gint n = -1;
		switch (event->keyval)
			{
			case '1':
				n = 0;
				break;
			case '2':
				n = 1;
				break;
			case '3':
				n = 2;
				break;
			case '4':
				n = 3;
				break;
			case '5':
				n = 4;
				break;
			case '6':
				n = 5;
				break;
			case '7':
				n = 6;
				break;
			case '8':
				n = 7;
				break;
			case 'C': case 'c':
				main_image_full_screen_stop();
				file_util_copy(main_image_get_path(), NULL, NULL);
				break;
			case 'M': case 'm':
				main_image_full_screen_stop();
				file_util_move(main_image_get_path(), NULL, NULL);
				break;
			case 'R': case 'r':
				main_image_full_screen_stop();
				file_util_rename(main_image_get_path(), NULL);
				break;
			case 'D': case 'd':
				if (confirm_delete) main_image_full_screen_stop();
				file_util_delete(main_image_get_path(), NULL);
				break;
			}
		if (n != -1)
			{
			main_image_full_screen_stop();
			start_editor_from_file(n, main_image_get_path());
			}
		}
	else if (event->state & GDK_SHIFT_MASK)
		{
		switch (event->keyval)
			{
			case 'R': case 'r':
				main_image_alter(ALTER_ROTATE_180);
				stop_signal = TRUE;
				break;
			case 'M': case 'm':
				main_image_alter(ALTER_MIRROR);
				stop_signal = TRUE;
				break;
			case 'F': case 'f':
				main_image_alter(ALTER_FLIP);
				stop_signal = TRUE;
				break;
			default:
				break;
			}
		x *= 3;
		y *= 3;
		}

	if (x != 0 || y!= 0)
		{
		keyboard_scroll_calc(&x, &y, event);
		image_scroll(imd, x, y);
		}

	return stop_signal;
}

/*
 *----------------------------------------------------------------------------
 * mouse button callbacks
 *----------------------------------------------------------------------------
 */

static void button1_cb(ImageWindow *imd, GdkEventButton *bevent, gpointer data)
{
	main_image_next();
}

static void button2_cb(ImageWindow *imd, GdkEventButton *bevent, gpointer data)
{
	main_image_prev();
}

static void main_image_full_screen_regrab_cb(GtkWidget *widget, gpointer data)
{
	/* full screen, re-grab keyboard after popup-menu close */
	if (full_screen_window)
		{
		gdk_keyboard_grab(full_screen_window->window, TRUE, GDK_CURRENT_TIME);
		}
}

static void button3_cb(ImageWindow *imd, GdkEventButton *bevent, gpointer data)
{
	GtkWidget *menu;

	menu = menu_popup_image();
	gtk_menu_popup (GTK_MENU(menu), NULL, NULL, NULL, NULL,
			bevent->button, bevent->time);
	if (full_screen_window != NULL)
		{
		/* full screen, re-grab keyboard after popup-menu close */
		gtk_signal_connect(GTK_OBJECT(menu), "selection_done",
				   GTK_SIGNAL_FUNC(main_image_full_screen_regrab_cb), NULL);
		}
}

static void button4_cb(ImageWindow *imd, GdkEventButton *bevent, gpointer data)
{
	if (bevent->state & GDK_CONTROL_MASK)
		{
		image_zoom_adjust(imd, get_zoom_increment());
		}
	else if ( (bevent->state & GDK_SHIFT_MASK) != (mousewheel_scrolls))
		{
		image_scroll(imd, 0, -MOUSEWHEEL_SCROLL_SIZE);
		}
	else
		{
		main_image_prev();
		}
}

static void button5_cb(ImageWindow *imd, GdkEventButton *bevent, gpointer data)
{
	if (bevent->state & GDK_CONTROL_MASK)
		{
		image_zoom_adjust(imd, -get_zoom_increment());
		}
	else if ( (bevent->state & GDK_SHIFT_MASK) != (mousewheel_scrolls))
		{
		image_scroll(imd, 0, MOUSEWHEEL_SCROLL_SIZE);
		}
	else
		{
		main_image_next();
		}
}

/*
 *----------------------------------------------------------------------------
 * main image full screen
 *----------------------------------------------------------------------------
 */

static void main_image_full_screen_stop_func(FullScreenData *fs, gpointer data)
{
	main_fs = NULL;
	main_image = normal_image;
	full_screen_image = NULL;
	full_screen_window = NULL;

	if (main_ss)
		{
		main_ss->imd = main_image;
		}
}

void main_image_full_screen_start(void)
{
	if (main_fs) return;

	main_fs = fullscreen_start(mainwindow, normal_image, main_image_full_screen_stop_func, NULL);
	full_screen_image = main_fs->imd;
	main_image = full_screen_image;
	full_screen_window = main_fs->window;

	if (main_ss)
		{
		main_ss->imd = main_image;
		}

	image_set_button_func(full_screen_image, 1, button1_cb, NULL);
	image_set_button_func(full_screen_image, 2, button2_cb, NULL);
	image_set_button_func(full_screen_image, 3, button3_cb, NULL);
	/* for wheel mice */
	image_set_button_func(full_screen_image, 4, button4_cb, NULL);
	image_set_button_func(full_screen_image, 5, button5_cb, NULL);

	gtk_signal_connect(GTK_OBJECT(main_fs->window), "key_press_event",
			   GTK_SIGNAL_FUNC(main_image_full_screen_key_press_cb), main_fs->imd);

}

void main_image_full_screen_stop(void)
{
	if (!main_fs) return;

	fullscreen_stop(main_fs);
}

void main_image_full_screen_toggle(void)
{
	if (main_fs)
		{
		main_image_full_screen_stop();
		}
	else
		{
		main_image_full_screen_start();
		}
}

gint main_image_full_screen_active(void)
{
	return (main_fs != NULL);
}

/*
 *----------------------------------------------------------------------------
 * main image slideshow
 *----------------------------------------------------------------------------
 */

void main_image_slideshow_next(void)
{
	if (main_ss) slideshow_next(main_ss);
}

void main_image_slideshow_prev(void)
{
	if (main_ss) slideshow_prev(main_ss);
}

static void main_image_slideshow_stop_func(SlideShowData *ss, gpointer data)
{
	main_ss = NULL;
	update_status_label(NULL);
}

void main_image_slideshow_start(void)
{
	CollectionData *cd;
	CollectInfo *info;

	if (main_ss) return;

	cd = image_get_collection(main_image, &info);

	if (cd && info)
		{
		main_ss = slideshow_start_from_collection(main_image, cd, main_image_slideshow_stop_func, NULL, info);
		}
	else
		{
		main_ss = slideshow_start(main_image, main_image_slideshow_stop_func, NULL,
			  find_file_in_list(main_image_get_path()) );
		}
	update_status_label(NULL);
}

/* note that slideshow will take ownership of the list, do not free it */
void main_image_slideshow_start_from_list(GList *list)
{
	if (main_ss || !list)
		{
		path_list_free(list);
		return;
		}

	main_ss = slideshow_start_from_path_list(main_image, list,
						 main_image_slideshow_stop_func, NULL);
	update_status_label(NULL);
}

void main_image_slideshow_stop(void)
{
	if (!main_ss) return;

	slideshow_free(main_ss);
}

void main_image_slideshow_toggle(void)
{
	if (main_ss)
		{
		main_image_slideshow_stop();
		}
	else
		{
		main_image_slideshow_start();
		}
}

gint main_image_slideshow_active(void)
{
	return (main_ss != NULL);
}

gint main_image_slideshow_pause_toggle(void)
{
	gint ret;
	ret = slideshow_pause_toggle(main_ss);
	update_status_label(NULL);

	return ret;
}

gint main_image_slideshow_paused(void)
{
	return (slideshow_paused(main_ss));
}

gint main_image_slideshow_continue_check(void)
{
	if (!main_ss) return FALSE;

	if (!slideshow_should_continue(main_ss))
		{
		main_image_slideshow_stop();
		return FALSE;
		}

	return TRUE;
}

/*
 *----------------------------------------------------------------------------
 * main image manipulation
 *----------------------------------------------------------------------------
 */

void main_image_scroll(gint x, gint y)
{
	image_scroll(main_image, x, y);
}

void main_image_adjust_zoom(gfloat increment)
{
	image_zoom_adjust(main_image, increment);
}

void main_image_set_zoom(gfloat zoom)
{
	image_zoom_set(main_image, zoom);
}

void main_image_alter(AlterType type)
{
	image_alter(main_image, type);
}

void main_image_set_path(const gchar *path)
{
	image_set_path(main_image, path);
}

const gchar *main_image_get_path(void)
{
	return image_get_path(main_image);
}

const gchar *main_image_get_name(void)
{
	return image_get_name(main_image);
}

CollectionData *main_image_get_collection(CollectInfo **info)
{
	return image_get_collection(main_image, info);
}

void main_image_change_to(const gchar *path, const gchar *read_ahead_path)
{
	image_change_path(main_image, path, image_zoom_get_default(main_image, zoom_mode));
	if (enable_read_ahead) image_prebuffer_set(main_image, read_ahead_path);
	main_image_slideshow_continue_check();
}

void main_image_change_to_from_collection(CollectionData *cd, CollectInfo *info)
{
	image_change_from_collection(main_image, cd, info, image_zoom_get_default(main_image, zoom_mode));
	if (enable_read_ahead)
		{
		CollectInfo *r_info = collection_next_by_info(cd, info);
		if (!r_info) r_info = collection_prev_by_info(cd, info);
		if (r_info) image_prebuffer_set(main_image, r_info->path);
		}
	main_image_slideshow_continue_check();
}

void main_image_reload(void)
{
	image_reload(main_image);
}

void main_image_next(void)
{
	gint current;
	gint total;
	CollectionData *cd;
	CollectInfo *info;

	if (main_image_slideshow_active())
		{
		main_image_slideshow_next();
		return;
		}

	cd = image_get_collection(main_image, &info);

	if (cd && info)
		{
		info = collection_next_by_info(cd, info);
		if (info)
			{
			image_change_from_collection(main_image, cd, info, image_zoom_get_default(main_image, zoom_mode));
			if (enable_read_ahead)
				{
				CollectInfo *r_info = collection_next_by_info(cd, info);
				if (!r_info) r_info = collection_prev_by_info(cd, info);
				if (r_info) image_prebuffer_set(main_image, r_info->path);
				}
			main_image_slideshow_continue_check();
			}
		return;
		}

	current = find_file_in_list(main_image_get_path());
	total = file_count();

	if (current >= 0)
		{
		if (current < file_count() - 1)
			{
			file_image_change_to(current + 1);
			}
		}
	else
		{
		file_image_change_to(0);
		}
}

void main_image_prev(void)
{
	gint current;
	CollectionData *cd;
	CollectInfo *info;

	if (main_image_slideshow_active())
		{
		main_image_slideshow_prev();
		return;
		}

	cd = image_get_collection(main_image, &info);

	if (cd && info)
		{
		info = collection_prev_by_info(cd, info);
		if (info)
			{
			image_change_from_collection(main_image, cd, info, image_zoom_get_default(main_image, zoom_mode));
			if (enable_read_ahead)
				{
				CollectInfo *r_info = collection_prev_by_info(cd, info);
				if (!r_info) r_info = collection_next_by_info(cd, info);
				if (r_info) image_prebuffer_set(main_image, r_info->path);
				}
			main_image_slideshow_continue_check();
			}
		return;
		}

	current = find_file_in_list(main_image_get_path());

	if (current >= 0)
		{
		if (current > 0)
			{
			file_image_change_to(current - 1);
			}
		}
	else
		{
		file_image_change_to(file_count() - 1);
		}
}

void main_image_first(void)
{
	gint current;
	CollectionData *cd;
	CollectInfo *info;

	cd = image_get_collection(main_image, &info);

	if (cd && info)
		{
		CollectInfo *new;
		new = collection_get_first(cd);
		if (new != info)
			{
			image_change_from_collection(main_image, cd, new, image_zoom_get_default(main_image, zoom_mode));
			if (enable_read_ahead)
				{
				CollectInfo *r_info = collection_next_by_info(cd, info);
				if (r_info) image_prebuffer_set(main_image, r_info->path);
				}
			main_image_slideshow_continue_check();
			}
		return;
		}

	current = find_file_in_list(main_image_get_path());
	if (current != 0 && file_count() > 0)
		{
		file_image_change_to(0);
		}
}

void main_image_last(void)
{
	gint current;
	gint count;
	CollectionData *cd;
	CollectInfo *info;

	cd = image_get_collection(main_image, &info);

	if (cd && info)
		{
		CollectInfo *new;
		new = collection_get_last(cd);
		if (new != info)
			{
			image_change_from_collection(main_image, cd, new, image_zoom_get_default(main_image, zoom_mode));
			if (enable_read_ahead)
				{
				CollectInfo *r_info = collection_prev_by_info(cd, info);
				if (r_info) image_prebuffer_set(main_image, r_info->path);
				}
			main_image_slideshow_continue_check();
			}
		return;
		}

	current = find_file_in_list(main_image_get_path());
	count = file_count();
	if (current != count - 1 && count > 0)
		{
		file_image_change_to(count - 1);
		}
}

void main_image_set_labels(GtkWidget *info, GtkWidget *zoom)
{
#if 0
	image_set_labels(normal_image, info, zoom);
#endif
}

GtkWidget *main_image_create(void)
{
	normal_image = image_new(TRUE);

	main_image = normal_image;

	image_set_button_func(main_image, 1, button1_cb, NULL);
	image_set_button_func(main_image, 2, button2_cb, NULL);
	image_set_button_func(main_image, 3, button3_cb, NULL);
	/* for wheel mice */
	image_set_button_func(main_image, 4, button4_cb, NULL);
	image_set_button_func(main_image, 5, button5_cb, NULL);

	return main_image->widget;
}

void main_image_to_root(void)
{
	image_to_root_window(main_image, (image_zoom_get(main_image) == 0));
}




