/* -*- Mode: C; indent-tabs-mode: nil; c-basic-offset: 4 c-style: "K&R" -*- */

/*---------------------------------------------------------------------------

  vorstra - calculates differential quantities (like vorticity and
  strain) from PIV data


   Copyright (C) 2002, 2003, 2004 Gerber van der Graaf

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation,
   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  

----------------------------------------------------------------------------

  Some formulea:
     vorty_z = dV/dx - dU/dy 
     sh_strain = dU/dy + dV/dx 
      n_strain = dU/dx + dV/dy 


-----------------------------------------------------------------------------*/

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <glib.h>
#include <gpiv.h>

/* #define PARFILE "vorstra.par" */	/* Parameter file name */
#define PARFILE "gpivrc"	/* Parameter file name */

#define GNUPLOT_DISPLAY_COLOR "DarkBlue"
#define GNUPLOT_DISPLAY_SIZE 250

#define USAGE "\
Usage: vorstra | vorty | nstrain | sstrain \n\
               [-d int][-f filename][-g] [-no_g] [-h | --help] [-o] [-s] [-n] \n\
               [-p | --print] [-v | --version] < stdin > stdout \n\
\n\
keys: \n\
-d N:                  differential type; central (0), least squares \n\
                       (1), richardson (2), circulation method (3)  \n\
-f filename:           files (without .piv extension) instead of stdin and \n\
                       stdout \n\
-g:                    graphical visualization with gnuplot (needs -f) \n\
-no_g:                 suppresses graphical visualization \n\
-h | --help:           this on-line help \n\
-n:                    normal strain analyses \n\
-o:                    vorticity analyses \n\
-p | --print:          print parameters to stdout  \n\
-s:                    shear strain analyses \n\
-v | --version;        version number"

#ifdef DEBUG
#define USAGE_DEBUG "\
Developers version also contains: \n\
               [-p_main][-p_differential] \n\
\n\
keys: \n\
-p_'function' N: prints data to be generated in the function; the \n\
                 higher N, the more detailed the output. \n\
                 For N = 10, err_vec will exit at the end of the function"
#endif


#define HELP  "\
vorstra calculates the differential quantities vorticity, shear strain and \n\
normal strain from PIV data."

#define RCSID "$Id: vorstra.c,v 2.9 2006/01/31 14:18:04 gerber Exp $"


gboolean fname_logic = FALSE;
gboolean print_par = FALSE;
GpivPivData in_data;
GpivScalarData out_data;

int gnuplot = 0;
gboolean gnuplot_logic = FALSE;

#ifdef DEBUG
/*
 * Parameters for development version
 */
int print_main = 0, print_differential = 0;
#endif



void 
command_args(int argc, 
             char *argv[], 
             char fname[GPIV_MAX_CHARS],
             GpivPostPar * piv_post_par
             )
/* ----------------------------------------------------------------------------
 * Command line argument handling
 */
{
    char c;
    int argc_next;


    while (--argc > 0 && (*++argv)[0] == '-') {
	argc_next = 0;

/*
 * argc_next is set to 1 if the next cmd line argument has to be
 * searched for; in case that the command line argument concerns more
 * than one char or cmd line argument needs a parameter
 */
	while (argc_next == 0 && (c = *++argv[0])) {
	    switch (c) {
/*
 * Use Revision Control System (RCS) for version
 */
	    case 'v':
		printf("\n%s\n", RCSID);
		exit(0);

/*
 * differential operator type
 */
	    case 'd':
		piv_post_par->diff_type_logic = TRUE;
		piv_post_par->diff_type = atoi(*++argv);
		argc_next = 1;
		--argc;
		break;

/* 
 * use -f instead of stdin stdout
 */
	    case 'f':
		strcpy(fname, *++argv);
		fname_logic = TRUE;
		argc_next = 1;
		--argc;
		break;

/*
 * graphic output with gnuplot
 */
	    case 'g':
	      gnuplot = 1;
	      gnuplot_logic = TRUE;
	      break;
	    case 'h':
		printf("\n%s", RCSID);
		printf("\n%s", HELP);
		printf("\n%s\n", USAGE);
		exit(0);
	    case 'n':
/*
 * do not plot with gnuplot
 */
		if (strcmp("no_g", *argv) == 0) {
		  gnuplot = 0;
		  gnuplot_logic = TRUE;
                  argc_next = 1;

/*
 * normal strain analysis
 */		} else {
                    piv_post_par->operator_vorstra_logic = TRUE;
                    piv_post_par->operator_vorstra = GPIV_N_STRAIN;
                }
		break;

/*
 * vorticity analysis
 */
	    case 'o':
		piv_post_par->operator_vorstra_logic = TRUE;
		piv_post_par->operator_vorstra = GPIV_VORTICITY;
		break;
	    case 'p':
#ifdef DEBUG
		if (strcmp(*argv, "p_main") != 0) {
#endif
		    print_par = TRUE;
#ifdef DEBUG
		} else if (strcmp("p_main", *argv) == 0) {
		    print_main = atoi(*++argv);
		    --argc;
		    argc_next = 1;
		}
#endif
		break;
/*
 * shear strain analysis
 */
	    case 's':
		piv_post_par->operator_vorstra_logic = TRUE;
		piv_post_par->operator_vorstra = GPIV_S_STRAIN;
		break;

/*
 * long option keys
 */
	    case '-':
		if (strcmp("-help", *argv) == 0) {
                    printf("\n%s", RCSID);
                    printf("\n%s", HELP);
                    printf("\n%s", USAGE);
                    exit(0);
                } else if (strcmp("-print", *argv) == 0) {
		    print_par = TRUE;
                } else if (strcmp("-version", *argv) == 0) {
                    printf("%s\n", RCSID);
                    exit(0);
                } else {
		    gpiv_error("%s: unknown option: %s", RCSID, *argv);
		}
		argc_next = 1;
		break;

	    default:
		gpiv_error("\n%s error : unknown option: %s\n",
			RCSID, *argv);
		break;
	    }
	}
    }

    if (argc != 0) {
	gpiv_error("\n%s error: unknown argument: %s\n", RCSID,
		*argv);
    }

}



void 
make_fname(char *fname, 
           char *fname_parameter, 
           char *fname_in,
           char *fname_out, 
           GpivPostPar piv_post_par
           )
/* ----------------------------------------------------------------------------
 * generate filenames
 */
{

    if (fname_logic == FALSE) {
	gpiv_error("\n%s error: Filename has to be set\n", RCSID);
    }



    gpiv_io_make_fname(fname, GPIV_EXT_PAR, fname_parameter);
    if (print_par)
	printf("# Parameter file: %s\n", fname_parameter);

    gpiv_io_make_fname(fname, GPIV_EXT_PIV, fname_in);
    if (print_par)
	printf("# Input data file: %s\n", fname_in);

    if (piv_post_par.operator_vorstra == GPIV_VORTICITY) {
        gpiv_io_make_fname(fname, GPIV_EXT_VOR, fname_out);
    } else if (piv_post_par.operator_vorstra == GPIV_S_STRAIN) {
        gpiv_io_make_fname(fname, GPIV_EXT_SSTR, fname_out);
    } else if (piv_post_par.operator_vorstra == GPIV_N_STRAIN) {
        gpiv_io_make_fname(fname, GPIV_EXT_NSTR, fname_out);
    } else {
        gpiv_error("%s: non valid operation", RCSID);
    }
    if (print_par)
	printf("# output data file: %s\n", fname_out);

}



int 
main(int argc, 
     char *argv[]
     )
/*-----------------------------------------------------------------------------
*/
{
    char *err_msg = NULL, *c = NULL;
    FILE *fp_par_dat;
    char fname[GPIV_MAX_CHARS], fname_out[GPIV_MAX_CHARS],
	fname_parameter[GPIV_MAX_CHARS], fname_in[GPIV_MAX_CHARS];
    char d_line[GPIV_MAX_LINES][GPIV_MAX_CHARS], 
        c_line[GPIV_MAX_LINES_C][GPIV_MAX_CHARS];
    int nd_lines = 0, nc_lines = 0, scale = 0;
    char title[GPIV_MAX_CHARS];

    GpivPostPar piv_post_par, piv_post_par_default;

    print_par = FALSE;
    fname_logic = FALSE;
    piv_post_par.operator_vorstra_logic = FALSE;
    piv_post_par.diff_type_logic = FALSE;
    gpiv_post_default_parameters(&piv_post_par_default, TRUE);

    if ((c = strstr(argv[0], "vorstra")) != NULL) {
/*
 * no action
 */
    } else if ((c = strstr(argv[0], "vorty")) != NULL) {
	piv_post_par.operator_vorstra = GPIV_VORTICITY;
	piv_post_par.operator_vorstra_logic = TRUE;
    } else if ((c = strstr(argv[0], "nstrain")) != NULL) {
	piv_post_par.operator_vorstra = GPIV_N_STRAIN;
	piv_post_par.operator_vorstra_logic = TRUE;
    } else if ((c = strstr(argv[0], "sstrain")) != NULL) {
	piv_post_par.operator_vorstra = GPIV_S_STRAIN;
	piv_post_par.operator_vorstra_logic = TRUE;
    } else {
        gpiv_error("vorstra: unvalid program name or symlink");
    }

    command_args(argc, argv, fname, &piv_post_par);
    if (print_par) {
      printf("# %s\n# Command line options:\n", RCSID);
      gpiv_post_print_parameters(piv_post_par);
    }


    if (fname_logic == TRUE) {
	make_fname(fname, fname_parameter, fname_in, fname_out, piv_post_par);

/*
 * Prints command line parameters to par-file
 */
	if ((fp_par_dat = fopen(fname_parameter, "a")) == NULL)
	     gpiv_error("\n%s: failure opening %s for input",
			RCSID, fname_parameter);
      fprintf(fp_par_dat, "# %s\n# Command line options:\n", RCSID);
      gpiv_post_fprint_parameters(fp_par_dat, piv_post_par);

/*
 * Reading parametes from PARFILE (and writing to data par-file)
 */
        gpiv_scan_parameter(GPIV_POST_PAR_KEY, PARFILE, &piv_post_par, print_par);
        if ((err_msg =
             gpiv_scan_resourcefiles(GPIV_POST_PAR_KEY, &piv_post_par, print_par))
            != NULL)  gpiv_error ("%s: %s", RCSID, err_msg);
        gpiv_post_fprint_parameters(fp_par_dat, piv_post_par);
	fclose(fp_par_dat);


    } else {
        gpiv_scan_parameter(GPIV_POST_PAR_KEY, PARFILE, &piv_post_par, print_par);
        if ((err_msg =
             gpiv_scan_resourcefiles(GPIV_POST_PAR_KEY, &piv_post_par, print_par))
            != NULL)  gpiv_error ("%s: %s", RCSID, err_msg);
    }

    gpiv_post_check_parameters_read(&piv_post_par, piv_post_par_default);


/*
 * Check parameters on correct values and adjust belonging variables
 */
    if (piv_post_par.diff_type_logic) {
	 if (piv_post_par.diff_type != 0 && piv_post_par.diff_type != 1 && 
	     piv_post_par.diff_type != 2
	     && piv_post_par.diff_type != 3)
	      gpiv_error("%s error: no value differential differentiator type",
			 RCSID);
    }

    if (fname_logic == FALSE && gnuplot == 1)
		gpiv_error("%s: -f filename has to be used in combination "
"with 'gnuplot'\n",
		RCSID);

    if ((piv_post_par.operator_vorstra == GPIV_S_STRAIN || 
	 piv_post_par.operator_vorstra == GPIV_N_STRAIN) &&
	piv_post_par.diff_type == GPIV_CIRCULATION) {
	fprintf(stderr,
		"\n%s error: strain can not be calculated by circulation method",
		RCSID);
	return 1;
    }


/*
 * As in_data.nx or in_data.ny are not known, the input data file will first 
 * be read
 */

    if (fname_logic == TRUE) {
	if ((scale = gpiv_fcount_pivdata(fname_in, &in_data)) == -1) {
	    fprintf(stderr,
		    "\n%s error: Failure calling gpiv_count_pivdata\n",
		    RCSID);
	    return 1;
	}
    } else {
	if ((scale =
	     gpiv_count_pivdata(&in_data, d_line, &nd_lines)) == -1) {
	    fprintf(stderr,
		    "\n%s error: Failure calling gpiv_count_pivdata\n",
		    RCSID);
	    return 1;
	}
    }


/*
 * Now the parameters are known, data  memory can be allocated
 */
     gpiv_alloc_pivdata(&in_data);
     out_data.nx = in_data.nx;
     out_data.ny = in_data.ny;
     gpiv_alloc_scdata(&out_data);

/*
 * Reading input file of piv data
 */
    if (fname_logic == TRUE) {
	if ((err_msg = gpiv_fread_pivdata(fname_in, &in_data, c_line,
					     &nc_lines))
            != NULL)  gpiv_error ("%s: %s", RCSID, err_msg);
    } else {
	gpiv_read_pivdata(&in_data, d_line, nd_lines, c_line, &nc_lines);
    }


/*
 * here the function calls of post-processing; calculating vorticity, strain
 */

    if ((err_msg = 
         gpiv_post_vorstra(in_data, &out_data, piv_post_par))
        != NULL) gpiv_error ("%s: %s", RCSID, err_msg);

    snprintf(c_line[nc_lines], GPIV_MAX_CHARS, "\n# scalar data: vorticity [1/s]");
    nc_lines++;
    if (fname_logic == TRUE) {
	if (gnuplot == 1) {
	    if ((err_msg =
		 gpiv_fwrite_sc_griddata(fname_out, &out_data,
					 c_line, nc_lines, scale,
					 RCSID))
                != NULL)  gpiv_error ("%s: %s", RCSID, err_msg);
	} else {
            if ((err_msg =
                 gpiv_fwrite_scdata(fname_out, &out_data, c_line,
                                    nc_lines, scale, RCSID)) 
                != NULL) gpiv_error ("%s: %s", RCSID, err_msg);
	}
        
    } else {
        gpiv_write_scdata(&out_data, c_line, nc_lines, scale, RCSID);
    }


/*
 * Freeing allocated memory of matrices
 */
    gpiv_free_pivdata(&in_data);
    gpiv_free_scdata(&out_data);


/* 
 * Graphical output with gnuplot
 */
    if (gnuplot) {
        snprintf(title, GPIV_MAX_CHARS, "vorticity of %s", fname);
	gpiv_scalar_gnuplot(fname_out, title, GNUPLOT_DISPLAY_COLOR, 
			    GNUPLOT_DISPLAY_SIZE);
    }
    
    exit(0);
}



