# gozerbot/irc.py
#
#

""" an Irc object handles the connection to the irc server .. receiving,
    sending, connect and reconnect code """

__copyright__ = 'this file is in the public domain'

from gozerbot.generic import rlog, handle_exception, getrandomnick, toenc, \
fix_format, splittxt, waitforqueue, strippedtxt, fromenc, uniqlist
from gozerbot.wait import Wait
from gozerbot.config import config
from gozerbot.ircevent import Ircevent
from gozerbot.monitor import saymonitor
from gozerbot.less import Less
from gozerbot.ignore import shouldignore
from gozerbot.pdod import Pdod
from gozerbot.datadir import datadir
from gozerbot.eventhandler import Outputhandler
from gozerbot.fleet import fleet
from gozerbot.botbase import BotBase
import gozerbot.thr as thr
import time, thread, socket, threading, os, Queue

class AlreadyConnected(Exception):

    """ already connected exception """

    pass

class AlreadyConnecting(Exception):

    """ bot is already connecting exception """

    pass

class Irc(BotBase):

    """ the irc class, provides interface to irc related stuff """

    def __init__(self, name='main', owner=[]):
        BotBase.__init__(self, name, owner)
        self.type = 'irc'
        self.wait = Wait()
        self.outputlock = thread.allocate_lock()
        self.fsock = None
        self.oldsock = None
        self.sock = None
        self.nolimiter = config['nolimiter']
        self.reconnectcount = 0
        self.pongcheck = 0
        self.nickchanged = 0
        self.noauto433 = 0
        if not self.state.has_key('alternick'):
            self.state['alternick'] = config['alternick']
        if not self.state.has_key('no-op'):
            self.state['no-op'] = []
        if not self.state.has_key('nick'):
            self.state['nick'] = config['nick'] or 'gb1'
        self.nick = self.state['nick']
        self.nrevents = 0
        self.gcevents = 0
        self.outqueues = [Queue.Queue() for i in range(10)]
        self.nicks401 = []

    def _raw(self, txt):
        """ send raw text to the server """
        if not txt:
            return
        rlog(2, self.name + '.sending', txt)
        try:
            if self.ssl:
                self.sock.write(txt + '\n')
            else:
                self.sock.send(txt[:500] + '\n')
        except Exception, ex:
            # check for broken pipe error .. if so ignore 
            # used for nonblocking sockets
            try:
                (errno, errstr) = ex
                if errno != 32:
                    raise
                else:
                    rlog(10, self.name, 'broken pipe error .. ignoring')
            except:
                rlog(10, self.name, "ERROR: can't send %s" % str(ex))

    def _connect(self, nick, server, port, password, ipv6, ssl=0):
        """ connect to server/port using nick """
        self.connecting = True
        self.connectok.clear()
        self.nick = str(nick)
        self.orignick = self.nick
        self.server = str(server)
        self.port = int(port)
        self.password = str(password)
        self.ipv6 = int(ipv6)
        self.ssl = int(ssl)
        # create socket
        if self.ipv6:
            rlog(10, self.name, 'creating ipv6 socket')
            self.oldsock = socket.socket(socket.AF_INET6, socket.SOCK_STREAM)
            self.ipv6 = 1
        else:
            rlog(10, self.name, 'creating ipv4 socket')
            self.oldsock = socket.socket(socket.AF_INET, socket.SOCK_STREAM)
        # optional bind
        elite = config['bindhost']
        if elite:
            try:
                self.oldsock.bind((elite, 0))
            except socket.gaierror:
                rlog(10, self.name, "can't bind to %s" % elite)
        # do the connect .. set timeout to 30 sec upon connecting
        rlog(10, self.name, 'connecting to ' + self.server)
        self.oldsock.settimeout(30)
        self.oldsock.connect((self.server, int(self.port)))
        # we are connected
        rlog(10, self.name, 'connection ok')
        self.stopped = 0
        # make file socket
        self.fsock = self.oldsock.makefile("r")
        # set blocking
        self.oldsock.setblocking(self.blocking)
        self.fsock._sock.setblocking(self.blocking)
        # set socket time out
        if self.blocking:
            socktimeout = config['socktimeout']
            if not socktimeout:
                socktimeout = 301.0
            else:
                socktimeout = float(socktimeout)
            self.oldsock.settimeout(socktimeout)
            self.fsock._sock.settimeout(socktimeout)
        # start readloop
        if self.ssl:
            rlog(10, self.name, 'ssl enabled')
            self.sock = socket.ssl(self.oldsock) 
        else:
            self.sock = self.oldsock
        thr.start_new_thread(self._readloop, ())
        thr.start_new_thread(self._outloop, ())
        # execute onconnect file .. code that can be executed after login
        self._logon()
        self._onconnect()
        # init 
        self.reconnectcount = 0
        self.nickchanged = 0
        self.connecting = False
        saymonitor.start()
        return 1

    def _readloop(self):
        """ loop on the socketfile """
        self.stopped = 0
        doreconnect = 0
        timeout = 1
        rlog(5, self.name, 'starting readloop')
        prevtxt = ""
        while not self.stopped:
            try:
                time.sleep(0.001)
                if self.ssl:
                    intxt = self.sock.read().split('\n')
                else:
                    intxt = self.fsock.readline().split('\n')
                # if intxt == "" the other side has disconnected
                if not intxt and not self.stopped:
                    doreconnect = 1
                    break
                if self.stopped:
                    break 
                if prevtxt:
                    intxt[0] = prevtxt + intxt[0]
                    prevtxt = ""
                if intxt[-1] != '':
                    prevtxt = intxt[-1]
                    intxt = intxt[:-1]
                for r in intxt:
                    r = r.rstrip()
                    rr = fromenc(r)
                    if not rr:
                        continue
                    res = strippedtxt(rr)
                    res = rr
                    rlog(2, self.name, res)
                    # parse txt read into an ircevent
                    try:
                        ievent = Ircevent().parse(self, res)
                    except Exception, ex:
                        handle_exception()
                        continue
                    # call handle_ievent 
                    if ievent:
                        self.handle_ievent(ievent)
                    timeout = 1
            except UnicodeError:
                handle_exception()
                continue
            except socket.timeout:
                # timeout occured .. first time send ping .. reconnect if
                # second timeout follows
                if self.stopped:
                    break
                timeout += 1
                if timeout > 2:
                    doreconnect = 1
                    rlog(10, self.name, 'no pong received')
                    break
                rlog(1, self.name, "socket timeout")
                pingsend = self.ping()
                if not pingsend:
                    doreconnect = 1
                    break
                continue
            except socket.sslerror, ex:
                # timeout occured .. first time send ping .. reconnect if
                # second timeout follows
                if self.stopped:
                    break
                if not 'timed out' in str(ex):
                    handle_exception()
                    doreconnect = 1
                    break
                timeout += 1
                if timeout > 2:
                    doreconnect = 1
                    rlog(10, self.name, 'no pong received')
                    break
                rlog(1, self.name, "socket timeout")
                pingsend = self.ping()
                if not pingsend:
                    doreconnect = 1
                    break
                continue
            except IOError, ex:
                if 'temporarily' in str(ex):
                    continue
            except Exception, ex:
                if self.stopped:
                    break
                err = ex
                try:
                    (errno, msg) = ex
                except:
                    errno = -1
                    msg = err
                # check for temp. unavailable error .. raised when using
                # nonblocking socket .. 35 is FreeBSD 11 is Linux
                if errno == 35 or errno == 11:
                    time.sleep(0.5)
                    continue
                rlog(10, self.name, "error in readloop: %s" % msg)
                doreconnect = 1
                break
        rlog(5, self.name, 'readloop stopped')
        # see if we need to reconnect
        self.connectok.clear()
        self.connecting = False
        self.connected = False
        if doreconnect:
            self.reconnect()

    def _getqueues(self):
        res = []
        time.sleep(0.1)
        for i in range(10):
            q = self.outqueues[i]
            if q.qsize():
                res.append(q)
        return res

    def _outloop(self):
        rlog(5, self.name, 'starting output loop')
        while not self.stopped:
            for queue in self._getqueues():
                res = queue.get()
                (printto, what, who, how, fromm, speed) = res
                if not self.stopped and printto not in self.nicks401:
                    self.out(printto, what, who, how, fromm, speed)
        rlog(5, self.name, 'stopping output loop')

    def _logon(self):
        """ log on to the network """
        # if password is provided send it
        if self.password:
            rlog(10, self.name ,'sending password')
            self._raw("PASS %s" % self.password)
        # register with irc server
        rlog(10, self.name, 'registering with %s using nick %s' % \
(self.server, self.nick))
        rlog(10, self.name, 'this may take a while')
        # check for username and realname
        username = self.nick or config['username']
        realname = config['realname'] or username
        # first send nick
        self._raw("NICK %s" % self.nick)
        # send USER
        self._raw("USER %s localhost localhost :%s" % (username, \
realname))
        # wait on login
        self.connectok.wait()
        # registration ok
        rlog(10, self.name, 'logged on !')

    def _onconnect(self):
        """ run the onconnect-botname file after connect """
        onconnectfile = 'onconnect-%s' % self.name
        try:
            execfile(onconnectfile, globals(), locals())
            rlog(5, self.name, '%s done' % onconnectfile)
        except Exception, ex:
            try:
                (errnr, error) = ex
                if errnr == 2:
                    pass
                else:
                    rlog(5, self.name, 'error excecuting %s: %s' % \
(onconnectfile, str(error)))
            except ValueError:
                rlog(5, self.name, 'error excecuting %s: %s' % \
(onconnectfile, str(ex)))

    def _resume(self, data, reto=None):
        """ resume to server/port using nick """
        try:
            if data['ssl']:
                return 0
        except KeyError:
            pass
        self.connecting = False # we're already connected
        self.nick = data['nick']
        self.orignick = self.nick
        self.server = str(data['server'])
        self.port = int(data['port'])
        self.password = data['password']
        self.ipv6 = data['ipv6']
        self.ssl = data['ssl']
        # create socket
        if self.ipv6:
            rlog(1, self.name, 'resuming ipv6 socket')
            self.sock = socket.fromfd(data['fd'], socket.AF_INET6, socket.SOCK_STREAM)
            self.ipv6 = 1
        else:
            rlog(1, self.name, 'resuming ipv4 socket')
            self.sock = socket.fromfd(data['fd'], socket.AF_INET, socket.SOCK_STREAM)
        # do the connect .. set timeout to 30 sec upon connecting
        rlog(10, self.name, 'resuming to ' + self.server)
        self.sock.settimeout(30)
        # we are connected
        rlog(10, self.name, 'connection ok')
        self.stopped = 0
        # make file socket
        self.fsock = self.sock.makefile("r")
        # set blocking
        self.sock.setblocking(self.blocking)
        # set socket time out
        if self.blocking:
            socktimeout = config['socktimeout']
            if not socktimeout:
                socktimeout = 301.0
            else:
                socktimeout = float(socktimeout)
            self.sock.settimeout(socktimeout)
        # start readloop
        rlog(0, self.name, 'resuming readloop')
        thr.start_new_thread(self._readloop, ())
        thr.start_new_thread(self._outloop, ())
        # init 
        self.reconnectcount = 0
        self.nickchanged = 0
        self.connecting = False
        # still there server?
        self._raw('PING :RESUME %s' % str(time.time()))
        self.connectok.set()
        self.connected = True
        if reto:
            self.say(reto, 'rebooting done')
        return 1

    def _resumedata(self):
        try:
            fd = self.sock.fileno()
        except:
            fd = None
            self.exit()
        return {self.name: {
            'nick': self.nick,
            'server': self.server,
            'port': self.port,
            'password': self.password,
            'ipv6': self.ipv6,
            'ssl': self.ssl,
            'fd': fd
            }}

    def broadcast(self, txt):
        for i in self.state['joinedchannels']:
            self.say(i, txt, speed=1)

    def save(self):
        self.state.save()

    def connect(self, nick, server, port=6667, password="", ipv6=0, \
ssl=0, reconnect=True):
        """ connect to server/port using nick .. connect can timeout so catch
            exception .. reconnect if enabled """
        if self.connecting:
            rlog(10, self.name, 'already connecting')
            raise AlreadyConnecting()
        if self.connectok.isSet():
            rlog(10, self.name, 'already connected')
            raise AlreadyConnected()
        try:
            return self._connect(nick, server, port, password, ipv6, ssl)
        except Exception, ex:
            if self.stopped:
                return 0
            rlog(10, self.name, 'connecting: %s' % str(ex))
            if reconnect:
                return self.reconnect()
            raise
        if not fleet.byname(self.name):
            fleet.addbot(self)

    def shutdown(self):
        """ shutdown sockets """
        rlog(10, self.name, 'shutdown')
        self.stopped = 1
        time.sleep(1)
        self.outqueues[0].put_nowait(('mekker', 'mekker', 'mekker', 'mekker', \
'mekker', 6))
        try:
            if self.ssl:
                self.oldsock.shutdown(2)
            else:
                self.sock.shutdown(2)
        except:
            pass
        try:
            if self.ssl:
                self.oldsock.close()
            else:
                self.sock.close()
            self.fsock.close()
        except:
            pass
        self.connecting = False
        self.connected = False
        self.connectok.clear()

    def exit(self):
        """ exit the bot """
        self.stopped = 1
        self.connected = False
        self.shutdown()

    def reconnect(self):
        """ reconnect to the irc server """
        if self.stopped:
            return 0
        time.sleep(2)
        self.shutdown()
        rlog(10, self.name, 'reconnecting')
        time.sleep(5)
        self.stopped = 0
        # determine how many seconds to sleep
        if self.reconnectcount > 0:
            reconsleep = self.reconnectcount*60
            rlog(10, self.name, 'sleeping %s seconds' % reconsleep)
            time.sleep(reconsleep)
        if self.stopped:
            return 0
        self.reconnectcount += 1
        return self.connect(self.nick, self.server, self.port, \
self.password, self.ipv6, self.ssl)

    def handle_pong(self, ievent):
        """ set pongcheck on received pong """
        rlog(1, self.name, 'received server pong')
        self.pongcheck = 1

    def sendraw(self, txt):
        """ send raw text to the server """
        if not txt or self.stopped or not self.connected:
            return
        rlog(2, self.name + '.sending', txt)
        try:
            if self.ssl:
                self.sock.write(txt + '\n')
            else:
                self.sock.send(txt[:500] + '\n')
        except Exception, ex:
            # check for broken pipe error .. if so ignore 
            # used for nonblocking sockets
            if not self.blocking and 'broken pipe' in str(ex):
                rlog(10, self.name, 'broken pipe error .. ignoring')
            else:
                rlog(10, self.name, 'send error: %s' % str(ex))
                self.reconnect()

    def fakein(self, txt):
        """ do a fake ircevent """
        if not txt:
            return
        rlog(10, self.name + '.fakein', txt)
        self.handle_ievent(Ircevent().parse(self, txt))

    def say(self, printto, what, who=None, how='msg', fromm=None, speed=5):
        """ say what to printto """
        if not printto or not what or printto in self.nicks401:
            return
        # check if printto is a socket
        if 'socket' in repr(printto):
            try:
                printto.send(what + '\n')
            except Exception, ex :
                if "Broken pipe" in str(ex):
                    return
                handle_exception()
            return
        # if who is set add "who: " to txt
        if who:
            what = "%s: %s" % (who, what)
        if speed > 10:
            speed = 10
        self.outqueues[10-speed].put((printto, what, who, how, fromm, speed))

    def out(self, printto, what, who=None, how='msg', fromm=None, speed=5):
        """ output the first 375 chars .. put the rest into cache """
        # first do mombo to get txt converted to ascii
        try:
            what = toenc(what.rstrip())
        except Exception, ex:
            rlog(10, self.name, "can't output: %s" % str(ex))
            return
        if not what:
            return
        # split up in parts of 375 chars overflowing on word boundaries
        txtlist = splittxt(what)
        size = 0
        # see if we need to store output in less cache
        if len(txtlist) > 1:
            if not fromm:
                self.less.add(printto, txtlist)
            else:
                self.less.add(fromm, txtlist)
            size = len(txtlist) - 1
            result = txtlist[:1][0]
            if size:
                result += " (+%s)" % size
        else:
            result = txtlist[0]
        self.output(printto, result, how, who, fromm)

    def output(self, printto, what, how='msg' , who=None, fromm=None):
        """ first output .. then call saymonitor """
        self.outputnolog(printto, what, how, who, fromm)
        saymonitor.put(self.name, printto, what, who, how, fromm)
        
    def outputnolog(self, printto, what, how, who=None, fromm=None):
        """ do output to irc server .. rate limit to 3 sec """
        if fromm and shouldignore(fromm):
            return
        self.outputlock.acquire()
        try:
            what = fix_format(what)
            now = time.time()
            timetosleep = 3 - (now - self.lastoutput)
            if timetosleep > 0 and not self.nolimiter:
                rlog(1, self.name, 'flood protect')
                time.sleep(timetosleep)
            self.lastoutput = time.time()
            if what:
                if how == 'msg':
                    self.privmsg(printto, what)
                elif how == 'notice':
                    self.notice(printto, what)
                elif how == 'ctcp':
                    self.ctcp(printto, what)
        except Exception, ex:
            handle_exception()
        self.outputlock.release()

    def donick(self, nick, setorig=0, save=0):
        """ change nick .. do not change orignick """
        if not nick:
            return
        self.noauto433 = 1
        queue = Queue.Queue()
        nick = nick[:16]
        self.wait.register('NICK', self.nick[:16], queue, 12)
        self._raw('NICK %s\n' % nick)
        result = waitforqueue(queue)
        self.noauto433 = 0
        if not result:
            return 0
        self.nick = nick
        self.whois(nick)
        if setorig:
            self.orignick = nick
        if save:
            self.state['nick'] = nick
            self.state.save()
            if self.name == 'jabbermain' or self.name == 'main':
                config.set('nick', nick)
                config.save()
        return 1

    def join(self, channel, password=None):
        """ join channel with optional password """
        if not channel:
            return
        q = Queue.Queue()
        self.wait.register('JOIN', channel, q, 5)
        if password:
            self.send('JOIN %s %s' % (channel, password))
        else:
            self.send('JOIN %s' % channel)
        result = waitforqueue(q, 5)
        if not result:
            return 'failed to join %s' % channel
        else:
            return 1

    def part(self, channel):
        """ leave channel """
        if not channel:
            return
        q = Queue.Queue()
        self.wait.register('PART', channel, q, 5)
        self.send('PART %s' % channel)
        result = waitforqueue(q, 5)
        if not result:
            return 0
        else:
            return 1

    def who(self, who):
        """ send who query """
        if not who:
            return
        self.send('WHO %s' % who.strip())

    def names(self, channel):
        """ send names query """
        if not channel:
            return
        self.send('NAMES %s' % channel)

    def whois(self, who):
        """ send whois query """
        if not who:
            return
        self.send('WHOIS %s' % who)

    def privmsg(self, printto, what):
        """ send privmsg to irc server """
        if not printto or not what:
            return
        self.send('PRIVMSG %s :%s' % (printto, what))

    def send(self, txt):
        """ send text to irc server """
        if not txt:
            return
        if self.stopped or not self.connected:
            return
        txt = toenc(strippedtxt(txt))
        txt = txt.rstrip()
        rlog(2, self.name + '.send', txt)
        try:
            if self.ssl:
                self.sock.write(txt + '\n')
            else:
                self.sock.send(txt[:500] + '\n')
        except Exception, ex:
            # check for broken pipe error .. if so ignore 
            # used for nonblocking sockets
            if not self.blocking and 'broken pipe' in str(ex):
                rlog(10, self.name, 'broken pipe error .. ignoring')
            else:
                rlog(10, self.name, 'send error: %s' % str(ex))
                self.reconnect()

    def voice(self, channel, who):
        """ give voice """
        if not channel or not who:
            return
        self.send('MODE %s +v %s' % (channel, who))
 
    def doop(self, channel, who):
        """ give ops """
        if not channel or not who:
            return
        self.send('MODE %s +o %s' % (channel, who))

    def delop(self, channel, who):
        """ de-op user """
        if not channel or not who:
            return
        self.send('MODE %s -o %s' % (channel, who))

    def quit(self, reason='http://gozerbot.org'):
        """ send quit message """
        rlog(10, self.name, 'sending quit')
        try:
            self._raw('QUIT :%s' % reason)
        except IOError:
            pass

    def notice(self, printto, what):
        """ send notice """
        if not printto or not what:
            return
        self.send('NOTICE %s :%s' % (printto, what))
 
    def ctcp(self, printto, what):
        """ send ctcp privmsg """
        if not printto or not what:
            return
        self.send("PRIVMSG %s :\001%s\001" % (printto, what))

    def ctcpreply(self, printto, what):
        """ send ctcp notice """
        if not printto or not what:
            return
        self.send("NOTICE %s :\001%s\001" % (printto, what))

    def action(self, printto, what):
        """ do action """
        if not printto or not what:
            return
        self.send("PRIVMSG %s :\001ACTION %s\001" % (printto, what))

    def handle_ievent(self, ievent):
        """ handle ircevent .. dispatch to 'handle_command' method """ 
        try:
            if ievent.cmnd == 'JOIN' or ievent.msg:
                if ievent.nick.lower() in self.nicks401:
                    self.nicks401.remove(ievent.nick)
                    rlog(10, self.name, '%s joined .. unignoring')
            # see if the irc object has a method to handle the ievent
            method = getattr(self,'handle_' + ievent.cmnd.lower())
            # try to call method
            try:
                method(ievent)
            except:
                handle_exception()
        except AttributeError:
            # no command method to handle event
            pass
        try:
            # see if there are wait callbacks
            self.wait.check(ievent)
        except:
            handle_exception()

    def handle_432(self, ievent):
        """ erroneous nick """
        self.handle_433(ievent)

    def handle_433(self, ievent):
        """ handle nick already taken """
        if self.noauto433:
            return
        nick = ievent.arguments[1]
        # check for alternick
        alternick = self.state['alternick']
        if alternick and not self.nickchanged:
            rlog(10, self.name, 'using alternick %s' % alternick)
            if self.donick(alternick):
                self.nickchanged = 1
                return
        # use random nick
        randomnick = getrandomnick()
        self.donick(randomnick)
        self.nick = randomnick
        rlog(100, self.name, 'ALERT: nick %s already in use/unavailable .. \
using randomnick %s' % (nick, randomnick))
        self.nickchanged = 1

    def handle_ping(self, ievent):
        """ send pong response """
        if not ievent.txt:
            return
        self._raw('PONG :%s' % ievent.txt)

    def handle_001(self, ievent):
        """ we are connected  """
        self.connectok.set()
        self.connected = True
        self.whois(self.nick)

    def handle_privmsg(self, ievent):
        """ check if msg is ctcp or not .. return 1 on handling """
        if ievent.txt and ievent.txt[0] == '\001':
            self.handle_ctcp(ievent)
            return 1

    def handle_notice(self, ievent):
        """ handle notice event .. check for version request """
        if ievent.txt and ievent.txt.find('VERSION') != -1:
            self.say(ievent.nick, config['version'], None, 'notice')
            return 1

    def handle_ctcp(self, ievent):
        """ handle client to client request .. version and ping """
        if ievent.txt.find('VERSION') != -1:
            self.ctcpreply(ievent.nick, 'VERSION %s' % config['version'])
        if ievent.txt.find('PING') != -1:
            try:
                pingtime = ievent.txt.split()[1]
                pingtime2 = ievent.txt.split()[2]
                if pingtime:
                    self.ctcpreply(ievent.nick, 'PING ' + pingtime + ' ' + \
pingtime2)
            except IndexError:
                pass

    def handle_error(self, ievent):
        """ show error """
        if ievent.txt.startswith('Closing'):
            rlog(10, self.name, ievent.txt)
        else:
            rlog(10, self.name + '.ERROR', ievent.txt)

    def ping(self):
        """ ping the irc server """
        rlog(1, self.name, 'sending ping')
        try:
            self._raw('PING :%s' % self.server)
            return 1
        except Exception, ex:
            rlog(10, self.name, "can't send ping: %s" % str(ex))
            return 0

    def handle_401(self, ievent):
        try:
            nick = ievent.arguments[1]
            if nick not in self.nicks401:
                rlog(10, self.name, '401 on %s .. ignoring' % nick)
                self.nicks401.append(nick)
        except:
            pass
