/* Josh Pieper, (c) 2000 */

/* This file is distributed under the GPL, see file COPYING for details */

#ifndef QUERY_H
#define QUERY_H

#include "lib.h"
#include "list.h"

#define MAX_OUTSTANDING_PUSHES 100

/* The query.resp struct is used in several different ways. The original
   use of query.resp was for the list of responses that is accumulated by
   gnut by the #find# command. query.resp items then get copied out of
   this list and passed to download threads; they are also copied and placed
   on a special "push queue" if the download is requested via the #push#
   command or if a push is generated by the retry algorithm of the #get#
   command. In addition, query.resp structs are created by the recieve
   packet reader for downloading files to the file cache. 
     In the future, the query.resp structures will be used in the lru_hash
   the implements #mreply# and #mpush# and the forthcoming instant-search
   feature. */
typedef struct {
  uint8  qr_ip[4];    /* from QHD packet: address of        */
  uint8  qr_port_le[2];  /*             host serving this file */
  uint32 qr_ref;      /* from QHD packet: file refnum       */
  uint32 qr_speed;    /* from QHD packet: server's speed    */
  uint32 qr_size;     /* from QHD packet: file's size       */
  uint8  qr_guid[16]; /* from QHD packet: GUID for PUSH     */
  int16  qr_htype;    /* From EQHD: Host type (bearshare, limewire, etc.) */
  int16  qr_rating;   /* From EQHD: Rating from 0 to 4 stars */
  int32  qr_uindex;   /* number printed in 'resp' command's output */
  char   qr_flag;     /* flag in 'resp' command output */
  uint32 qr_crc;      /* used to optimize process of weeding out multiple
                         hits with same host address and name */
  int8   qr_dcache;   /* set if we're downloading this to cache */
  int8   qr_retry;    /* set if this download should auto-retry */
  uint16 qr_nlen;     /* Length of entity name or path       */
  char   qr_ndata[1]; /* Placeholder for the ending null byte of the entity
                         name or path. The actual allocate adds the actual
                         string length to this before allocating */
} query_resp;

#define QR_NAME_MAXLEN 1024

#ifdef __cplusplus
extern "C" {
#endif

  void fre_qrpl(query_resp ***x, int bugnum);
  void fre_qrp(query_resp **x, int bugnum);

  void query_kname(query_resp *a_qrp);
  void query_kill(query_resp **p_qrp, int tn1);
  query_resp *query_copy(query_resp *qr, uint32 tn1);
  query_resp *query_create(int tn1,
			   uint8 ip[4], uint8 port[2], uint32 ref,
			   uint32 speed, uint32 size, uchar guid[16],
			   int16 htype, int16 rating,
			   int32 uindex, char flag,
			   int dcache, int retry, uint32 crc, char *name);

  int query_init();
  int query_unique();

  int query_add(uint8 ip[4], uint8 port[2], uint32 ref, uint32 speed,
		uint32 size, char *name, uint8 guid[16], int16 htype,
		int16 rating);
  int query_add_cache(uint8 ip[4], uint8 port[2], uint32 ref, uint32 speed,
		      uint32 size, char *name, uint8 guid[16], int16 htype,
		      int16 rating);

  int query_clear();

  query_resp * query_index(int32 uindex, char flag, int make_copy); /* 0.4.28.c21 */

  query_resp * query_find(uchar guid[16], uint32 ref);

  uint32 query_num();

  int query_lock();
  Gnut_List * query_retrieve();
  int query_unlock();

  int query_push_add(query_resp *qr);
  query_resp * query_push_find(uchar guid[16], uint32 ref);
  void query_tag_filter(char flag, int (*testfunc)(query_resp *));

#ifdef __cplusplus
}
#endif

#endif
