/*
 * Copyright (C) 2002,2003 Pascal Haakmat.
 * Licensed under the GNU GPL.
 * Absolutely no warranty.
 */

#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <unistd.h>
#include <stdio.h>
#include <audiofile.h>
#include <gnome.h>
#include <config.h>
#include "lib/benchmark.h"
#include "pref.h"
#include "mem.h"
#include "shell.h"
#include "gui.h"
#include "file.h"
#include "action.h"
#include "marker.h"
#include "mixer.h"

extern int quit_requested;
extern int buffers_being_saved;
extern int draw_prio;
extern int Emergency;

static char untitled_name[] = "Untitled\0        ";
static int untitled_counter = 1;

int
file_load_settings(shell *shl,
                   const char *filename) {
    char usxpath[512], key[512];
    int i;
    struct stat stats;

    snprintf(usxpath, 512, "%s.usx", filename);
    if(!strcmp(usxpath, filename)) {
        FAIL("%s == %s, did not load settings\n", usxpath, filename);
        return -1;
    }

    if(stat(usxpath, &stats) == -1)
        return -1;

    mixer_load(shl->mixer, usxpath);
    grid_load(&shl->grid, usxpath);
    for(i = 0; i < shl->sr->channels; i++)
        marker_list_load(shl->sr->markers[i], usxpath, i);
    snprintf(key, 512, "=%s=/Selection/Selection Start=0", usxpath);
    shl->select_start = gnome_config_get_int(key);
    snprintf(key, 512, "=%s=/Selection/Selection End=0", usxpath);
    shl->select_end = gnome_config_get_int(key);
    snprintf(key, 512, "=%s=/Selection/Selection Channel Map", usxpath);
    shl->select_channel_map = gnome_config_get_int(key);
    snprintf(key, 512, "=%s=/Selection/Loop Start=0", usxpath);
    shl->loop_start = gnome_config_get_int(key);
    snprintf(key, 512, "=%s=/Selection/Loop End=0", usxpath);
    shl->loop_end = gnome_config_get_int(key);
    snprintf(key, 512, "=%s=/Selection/Loop Enabled=0", usxpath);
    shl->loop = gnome_config_get_int(key);

    return 0;
}

int
file_load_internal(shell *shl,
                   const char *filename) {
    char path[512];
    int fail, i, input_map;
    size_t fb_sz, fb_dmx_sz;
    frame_bits_t fb = NULL, fb_dmx = NULL;
    AFfilehandle fh = AF_NULL_FILEHANDLE;
    AFframecount frame_offset = 0, 
        frame_count = 0,
        total = 0, 
        r;
    float perc, seconds;
    struct timeval tv_start, tv_stop;
    snd *sr;
 
    sr = (snd *)snd_new();
    if(RESULT_IS_ERROR(sr)) {
        gui_alert("Out of memory trying to create sound object: %s.", 
                  snd_error_get(sr));
        snd_destroy(sr);
        return -1;
    }

    snd_name_set(sr, filename);

    /* Open file. */

    fh = afOpenFile(sr->name, "r", AF_NULL_FILESETUP);
    if(fh == AF_NULL_FILEHANDLE) {
        gui_alert("Error loading \n%s. \n" \
                  "Perhaps this is not an audio file or the \n" \
                  "format is not supported.", 
                  sr->name);
        goto failed;
    }
    
    afGetSampleFormat(fh, 
                      AF_DEFAULT_TRACK, 
                      &sr->fmt, 
                      &sr->frame_width);

    sr->frame_width = sr->frame_width + (sr->frame_width % 8);
    if(sr->frame_width == 24)
        sr->frame_width = 32;
    sr->frame_width /= 8;
    sr->endian = afGetByteOrder(fh, AF_DEFAULT_TRACK);
    sr->channels = afGetChannels(fh, AF_DEFAULT_TRACK);
    sr->rate = afGetRate(fh, AF_DEFAULT_TRACK);

    if(sr->channels < 1 || sr->channels > pref_get_as_int("invariant_max_tracks")) {
        gui_alert("Sorry, %d channel sound format not supported (%s).",
                  sr->channels, sr->name);
        goto failed;
    }
    
    if(sr->frame_width < 1 || sr->frame_width > 4) {
        gui_alert("Sorry, %d bits sample format not supported (%s).",
                  sr->frame_width * 8, sr->name);
        goto failed;
    }

    input_map = 0;
    for(i = 0; i < sr->channels; i++)
        input_map += (1 << i);

    shl->select_channel_map = input_map;

    /* Reserve & setup memory. */

    fail = 0;
    for(i = 0; i < sr->channels; i++)
        sr->tracks[i] = NULL;
    for(i = 0; i < sr->channels; i++) {
        sr->tracks[i] = track_new(sr->frame_width, 0);
        if(sr->tracks[i] == NULL) {
            fail = 1;
            for(i--; i >= 0; i--)
                track_destroy(sr->tracks[i]);
            break;
        }
    }

    if(fail) {
        gui_alert("Out of memory allocating %d tracks.", 
                  sr->channels);
        goto failed;
    }

    fb_sz = LOAD_BUF_SIZE * sr->frame_width * sr->channels;
    fb_dmx_sz = LOAD_BUF_SIZE * sr->frame_width;

    fb = mem_alloc(fb_sz);
    if(!fb) {
        gui_alert("Out of memory (need %d bytes).", fb_sz);
        goto failed;
    }
    fb_dmx = mem_alloc(fb_dmx_sz);
    if(!fb_dmx) {
        gui_alert("Out of memory (need %d bytes).", fb_dmx_sz);
        goto failed;
    }

    /* Read audio file.  

       afSetVirtualSampleFormat() causes libaudiofile to return signed
       twos-complement samples regardless of the file's real sample
       format. */

    afSetVirtualSampleFormat(fh,
                             AF_DEFAULT_TRACK,
                             AF_SAMPFMT_TWOSCOMP,
                             sr->frame_width * 8);

    r = afSeekFrame(fh, AF_DEFAULT_TRACK, frame_offset);
    if(r != frame_offset) {
        gui_alert("Seek error in %s: requested: %ld, returned: %ld.", 
                  sr->name, frame_offset, r);
        goto failed;
    }

    /* Point of no return: substitute sound object in shell for this
       sound object. */
    
    if(shl->sr)
        snd_destroy(shl->sr);
    shl->sr = sr;
    mixer_configure(shl->mixer, 
                    pref_get_as_int("playback_channels"),
                    shl->sr->channels);


    /* Try to load settings. Try to load old-style .umix if .usx file
       does not exist. */

    if(file_load_settings(shl, filename)) {
        snprintf(path, 512, "%s.umix", filename);
        if(strcmp(path, filename)) 
            mixer_load_compat(shl->mixer, path);
        else 
            FAIL("%s == %s, did not load mixer settings\n", path, filename);
    }
            
    grid_rate_set(&shl->grid, sr->rate);
    shell_grid_bpm_set(shl, shl->grid.bpm);
    shell_grid_units_set(shl, shl->grid.units);
    shell_grid_measurement_set(shl, shl->grid.measurement);

    /* Load sound file proper. */

    gettimeofday(&tv_start, NULL);
    frame_count = afGetFrameCount(fh, AF_DEFAULT_TRACK);

    rwlock_rlock(&sr->rwl);

    for(frame_offset = 0; 
        frame_offset < frame_count; 
        frame_offset += LOAD_BUF_SIZE) {

        r = afReadFrames(fh, 
                         AF_DEFAULT_TRACK,
                         fb,
                         MIN(frame_count - total, LOAD_BUF_SIZE));
        if(r <= 0)
            break;

        snd_demux(sr,
                  fb_dmx,
                  fb,
                  input_map,
                  sr->channels,
                  frame_offset,
                  r);
        
        if(RESULT_IS_ERROR(sr)) {
            gui_alert("Error loading %s: %s", 
                      sr->name, snd_error_get(sr));
            break;
        }
        
        /* Progress meter behaves strange for values around
           0.01. */
        
        perc = (float) ((float) total / 
                        (float) (frame_count));
        if(perc > 0.02)
            gtk_progress_bar_set_fraction(shl->progress, perc);
        
        shell_redraw(shl);
        total += r;

        /* Process GUI events, exit if that kills us. */
        
        if(gui_yield() || shl->file_load_cancel_requested) {
            DEBUG("exiting per cancel request\n");
            break;
        }
    }

    rwlock_runlock(&sr->rwl);

    DEBUG("loaded %ld frames, expected %ld frames, calculated %ld frames\n",
          total, frame_count, snd_frame_count(sr));

    if(total != frame_count && !shl->file_load_cancel_requested) 
        gui_alert("File might be damaged!\nExpected %ld frames but read only %ld.\n",
                  frame_count, total);

    gettimeofday(&tv_stop, NULL);
    seconds = TIMEVAL_DIFF(tv_start, tv_stop);

    INFO("read %ld frames (%ld bytes) in %f secs (%.3f KB/sec)\n", 
         snd_frame_count(sr), 
         snd_frame_count(sr) * sr->frame_width * sr->channels, 
         seconds,
         (snd_frame_count(sr) * sr->frame_width * sr->channels / seconds) / 1024);

    mem_free(fb);
    mem_free(fb_dmx);
    afCloseFile(fh);
    return 0;
    
 failed:
    if(sr)
        snd_destroy(sr);
    if(fb)
        mem_free(fb);
    if(fb_dmx)
        mem_free(fb_dmx);
    if(fh != AF_NULL_FILEHANDLE)
        afCloseFile(fh);
    return -1;
}

void
file_load(shell *shl,
          const char *filename) {
    int r;
    
    gtk_widget_show(GTK_WIDGET(shl->appwindow));
    shell_status_push(shl, "Loading %s ...", filename);
    shell_cursor_set(shl, GDK_WATCH);
    gtk_progress_bar_set_fraction(shl->progress, 0);

    /* Higher draw priority means less chance of blocking the playback
       thread. */
    
    draw_prio += 64;
    set_flag(shl->action_state, ACTION_LOAD_IN_PROGRESS);
    r = file_load_internal(shl, filename);
    draw_prio -= 64;
    clear_flag(shl->action_state, ACTION_LOAD_IN_PROGRESS);
    
    if(r) {
        shl->has_name = 0;
        if(!shl->sr) 
            action_do(ACTION_FILE_INIT_NEW(shl));
    } else
        shl->has_name = 1;

    if(shl->file_load_cancel_requested)
        shl->has_name = 0;
    
    /* Configure mixer for the number of tracks that we have. */
    
    mixer_configure(shl->mixer, 
                    pref_get_as_int("playback_channels"),
                    shl->sr->channels);
    gtk_progress_bar_set_fraction(shl->progress, 0);
    shell_cursor_set(shl, GDK_LEFT_PTR);
    shell_status_default_set(shl);
    shell_redraw(shl);
}


void
file_save_settings(shell *shl,
                   const char *filename) {
    char usxpath[512], gcpath[512], key[512];
    int i;

    snprintf(usxpath, 512, "%s.usx", filename);
    if(!strcmp(usxpath, filename)) {
        FAIL("%s == %s, did not save settings\n", usxpath, filename);
        return;
    }
    snprintf(gcpath, 512, "=%s=", usxpath);

    mixer_save(shl->mixer, usxpath);
    grid_save(&shl->grid, usxpath);
    for(i = 0; i < shl->sr->channels; i++)
        marker_list_save(shl->sr->markers[i], usxpath, i);
    snprintf(key, 512, "=%s=/Selection/Selection Start", usxpath);
    gnome_config_set_int(key, shl->select_start);
    snprintf(key, 512, "=%s=/Selection/Selection End", usxpath);
    gnome_config_set_int(key, shl->select_end);
    snprintf(key, 512, "=%s=/Selection/Selection Channel Map", usxpath);
    gnome_config_set_int(key, shl->select_channel_map);
    snprintf(key, 512, "=%s=/Selection/Loop Start", usxpath);
    gnome_config_set_int(key, shl->loop_start);
    snprintf(key, 512, "=%s=/Selection/Loop End", usxpath);
    gnome_config_set_int(key, shl->loop_end);
    snprintf(key, 512, "=%s=/Selection/Loop Enabled", usxpath);
    gnome_config_set_int(key, shl->loop);

    gnome_config_sync_file(gcpath);
}

int
file_save_internal(const char *filename,
                   shell *shl,
                   int src_channels,
                   int dst_channels,
                   mixer *output_mixer) {
    AFfilesetup file_setup;
    AFfilehandle fh = AF_NULL_FILEHANDLE;
    AFframecount off, ct, r, total_written = 0;
    int written = 0;
    frame_bits_t fb_muxbuf, fb_srcbufs[pref_get_as_int("invariant_max_tracks")];
    snd *sr = shl->sr;
    struct timeval tv_start, tv_stop;
    float perc, seconds;

    if(!filename) {
        gui_alert("Internal error: filename == NULL.");
        return -1;
    }

    /* Try to save settings. */
    
    file_save_settings(shl, filename);
    
    /* Setup local stuff. */

    file_setup = afNewFileSetup();

    if(!file_setup) {
        gui_alert("Not enough memory to create AFfilesetup. " \
                  "Very low memory.");
        return -1;
    }
    
    fb_muxbuf = mixer_buffers_alloc(sr->frame_width,
                                    MAX(src_channels, dst_channels), 
                                    &fb_muxbuf,
                                    fb_srcbufs,
                                    SAVE_BUF_SIZE);
    
    if(!fb_muxbuf) {
        gui_alert("Cannot get temporary memory for output buffer.");
        afFreeFileSetup(file_setup);
        return -1;
    }

    /* Setup the output file. */

    afInitFileFormat(file_setup,
                     AF_FILE_WAVE);
    afInitSampleFormat(file_setup,
                       AF_DEFAULT_TRACK,
                       sr->fmt,
                       sr->frame_width * 8);
    afInitByteOrder(file_setup,
                    AF_DEFAULT_TRACK,
                    sr->endian);
    afInitChannels(file_setup,
                   AF_DEFAULT_TRACK,
                   dst_channels);
    afInitRate(file_setup,
               AF_DEFAULT_TRACK,
               sr->rate);
    fh = afOpenFile(filename, "w", file_setup);

    if(fh == AF_NULL_FILEHANDLE) {
        gui_alert("Could not open %s for writing. Check filename " \
                  "and permissions.", filename);
        goto recover;
    }

    afSetVirtualSampleFormat(fh,
                             AF_DEFAULT_TRACK,
                             AF_SAMPFMT_TWOSCOMP,
                             sr->frame_width * 8);

    gettimeofday(&tv_start, NULL);

    DEBUG("starting mux-to-disk: name: %s, src_channels: %d, dst_channels: %d, frame_count: %ld\n",
          filename, src_channels, dst_channels, snd_frame_count(sr));

    if(!Emergency)
        gtk_progress_bar_set_fraction(shl->progress, 0);
    ct = snd_frame_count(shl->sr);
    off = 0;
    r = 1;
    while(ct && r) {
        gui_yield();

        memset(fb_muxbuf, 
               '\0', 
               SAVE_BUF_SIZE * sr->frame_width * output_mixer->target_channels);
        r = snd_mux(sr,
                    fb_muxbuf,
                    fb_srcbufs, 
                    output_mixer, 
                    MAP_ALL, 
                    off, 
                    SAVE_BUF_SIZE);
        written = afWriteFrames(fh, 
                                AF_DEFAULT_TRACK,
                                fb_muxbuf, 
                                r);

        if(written == -1) 
            break;
        
        total_written += written;
        ct -= r;
        off += r;

        /* Progress meter behaves strange for values around
           0.01. */
        
        perc = (float) ((float) off / 
                        (float) (snd_frame_count(shl->sr)));
        if(perc > 0.02) 
            if(!Emergency)
                gtk_progress_bar_set_fraction(shl->progress, perc);
    }
    if(!Emergency)
        gtk_progress_bar_set_fraction(shl->progress, 0);
    
    if(ct || total_written != snd_frame_count(shl->sr)) 
        gui_alert("Save %s: not all frames were written (%ld of %ld). " \
                  "Some data is not on disk. Check disk space (%s).",
                  filename, 
                  total_written,
                  snd_frame_count(shl->sr),
                  written == -1 ? strerror(errno) : "Unknown error");

    if(afCloseFile(fh)) 
        gui_alert("An error occurred trying to close %s. Some data " \
                  "may not have been written to disk. Check disk space (%s).",
                  filename,
                  strerror(errno));

    gettimeofday(&tv_stop, NULL);
    seconds = TIMEVAL_DIFF(tv_start, tv_stop);

    INFO("wrote %ld frames (%ld bytes) in %f secs (%.3f KB/sec)\n",
         total_written,
         total_written * sr->frame_width * dst_channels,
         seconds,
         (total_written * sr->frame_width * dst_channels / seconds) / 1024);

    mixer_buffers_free(MAX(src_channels, dst_channels), fb_muxbuf, fb_srcbufs);
    afFreeFileSetup(file_setup);
    return 0;

 recover:
    mixer_buffers_free(MAX(src_channels, dst_channels), fb_muxbuf, fb_srcbufs);
    afFreeFileSetup(file_setup);
    return -1;
}

void
file_save_or_mixdown(shell *shl, 
                     const char *name,
                     gboolean keep_backup,
                     gboolean is_mixdown) {
    char *newname;
    mixer *output_mixer;
    int in_chans, out_chans;
    if(!name) {
        FAIL("file has no name\n");
        return;
    }

    /* Rename old file. */

    if(keep_backup) {
        newname = mem_alloc(strlen(name) + 5);
        if(!newname) {
            gui_alert("Cannot %s, very low memory.\n",
                      is_mixdown ? "mixdown" : "save");
            return;
        }

        strcpy(newname, name);
        strcat(newname, ".bak");

        if(rename(name, newname)) {
            gui_alert("Cannot %s, unable to rename %s to %s.bak (%s).\n",
                      is_mixdown ? "mixdown" : "save", name, name,
                      strerror(errno));
            mem_free(newname);
            return;
        }

        DEBUG("renamed '%s' to '%s'\n", name, newname);
        mem_free(newname);
    }

    /* Setup output mixer and channels. */

    if(is_mixdown) {
        output_mixer = shl->mixer;
        in_chans = shl->sr->channels;
        out_chans = output_mixer->target_channels;
    } else {
        output_mixer = mixer_new(shl->sr->channels, shl->sr->channels);        
        if(!output_mixer) {
            gui_alert("Cannot create output mixer for %d tracks. " \
                      "Very low memory.", shl->sr->channels);
            return;
        }
        in_chans = shl->sr->channels;
        out_chans = shl->sr->channels;
        output_mixer->is_unity = 1;
    }

    /* Do output mix. */

    buffers_being_saved++;
    set_flag(shl->action_state, ACTION_SAVE_IN_PROGRESS);
    shell_status_push(shl, "%s %s ...", is_mixdown ? "Mixdown" : "Saving", name);
    shell_cursor_set(shl, GDK_WATCH);
    shl->has_name = 1;
    if(file_save_internal(name, shl, in_chans, out_chans, output_mixer)) {
        shl->has_changed = 1;
    } else {
        shl->has_changed = 0;
    }
    if(!is_mixdown)
        mixer_destroy(output_mixer);
    shell_cursor_set(shl, GDK_LEFT_PTR);
    shell_status_default_set(shl);
    shell_grid_bpm_set(shl, shl->grid.bpm);
    shell_grid_units_set(shl, shl->grid.units);
    shell_redraw(shl);
    buffers_being_saved--;
    clear_flag(shl->action_state, ACTION_SAVE_IN_PROGRESS);

}

void
file_mixdown(shell *shl,
             const char *filename,
             gboolean keep_backup) {
    file_save_or_mixdown(shl,
                         filename,
                         keep_backup,
                         TRUE);
}

void
file_save(shell *shl, 
          const char *filename,
          gboolean keep_backup) {

    snd_name_set(shl->sr, filename);
    file_save_or_mixdown(shl,
                         shl->sr->name,
                         keep_backup,
                         FALSE);
}

int
file_init(shell *shl) {
    snd *sr;
    sr = (snd *)snd_new();    
    if(RESULT_IS_ERROR(sr)) {
        gui_alert("Error creating new sound object, you are probably out of memory: %s.", 
                  snd_error_get(sr));
        snd_destroy(sr);
        return -1;
    }
    snprintf(untitled_name, 12, "Untitled%d", untitled_counter++);
    snd_name_set(sr, untitled_name);
    snd_tracks_insert(sr, NULL, 1);
    if(shl->sr)
        snd_destroy(shl->sr);
    shl->sr = sr;
    shl->has_name = 0;
    mixer_configure(shl->mixer, pref_get_as_int("playback_channels"), sr->channels);
    grid_rate_set(&shl->grid, sr->rate);
    shell_status_default_set(shl);
    shell_redraw(shl);
    return 0;
}

