/*
    GNOME Commander - A GNOME based file manager 
    Copyright (C) 2001-2002 Marcus Bjurman

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/ 
#include <config.h>
#include "gnome-cmd-includes.h"


static GnomeDialogClass *parent_class = NULL;


struct _GnomeCmdFtpDialogPrivate {
	GnomeCmdFtpServer *selected_server;
	GnomeCmdFtpServer *server_to_add;
	
	GtkWidget         *server_list;
	GtkWidget         *anonymous_pw_entry;
};



/******************************************************
	The main ftp dialog
******************************************************/

static void
load_servers (GnomeCmdFtpDialog *dialog)
{
	GList *tmp = gnome_cmd_data_get_ftp_servers ();
	GtkCList *server_list = GTK_CLIST (dialog->priv->server_list);

	gtk_clist_clear (server_list);

	if (tmp)
		dialog->priv->selected_server = (GnomeCmdFtpServer*)tmp->data;
	
	while (tmp)
	{
		GnomeCmdFtpServer *server = (GnomeCmdFtpServer*)tmp->data;
		if (server)
		{
			int row;
			gchar *text[4];
			
			text[0] = NULL;
			text[1] = (gchar*)gnome_cmd_ftp_server_get_alias (server);
			text[2] = (gchar*)gnome_cmd_ftp_server_get_host_name (server);
			text[3] = NULL;
			row = gtk_clist_append (server_list, text);
			if (!row)
				gtk_clist_select_row (server_list, 0, 0);
			
			gtk_clist_set_row_data (server_list, row, server);
			gtk_clist_set_pixmap (GTK_CLIST (server_list), row, 0,
								  IMAGE_get_pixmap (PIXMAP_SERVER_SMALL),
								  IMAGE_get_mask (PIXMAP_SERVER_SMALL));
		}
		else
		{
			g_warning ("Null entry in the ftp-server list\n");				
		}
		tmp = tmp->next;
 	}
}


static void
do_connect (GnomeCmdFtpDialog *dialog, const gchar *password)
{
	GnomeCmdConnection *con;
	GnomeCmdFileSelector *fs, *fs2;
	GnomeCmdFtpServer *server = dialog->priv->selected_server;

	if (!server)
		return;

	gtk_widget_hide (GTK_WIDGET (dialog));
	
	con = gnome_cmd_connection_new_ftp (
		gnome_cmd_ftp_server_get_alias (server),
		gnome_cmd_ftp_server_get_host_name (server),
		gnome_cmd_ftp_server_get_host_port (server),
		gnome_cmd_ftp_server_get_user_name (server),
		password);

	fs = gnome_cmd_main_win_get_active_fs (main_win);
	fs2 = gnome_cmd_main_win_get_inactive_fs (main_win);

	gnome_cmd_file_selector_update_connections (fs);
	gnome_cmd_file_selector_update_connections (fs2);
	
	gnome_cmd_file_selector_set_connection (fs, con);

}


static void
on_password_ok (gchar *password, GnomeCmdFtpDialog *dialog)
{
	do_connect (dialog, password);
}


static void
on_password_cancel (GtkButton *button, GnomeCmdFtpDialog *dialog)
{
}

	
static void
on_connect_btn_clicked                 (GtkButton       *button,
                                        GnomeCmdFtpDialog *ftp_dialog)
{
	GnomeCmdFtpServer *server = ftp_dialog->priv->selected_server;
	
	if (server)
	{
		gchar *anonymous_pw = gtk_entry_get_text (GTK_ENTRY (ftp_dialog->priv->anonymous_pw_entry));
		
		/* store the anonymous password as the user might have changed it */
		gnome_cmd_data_set_ftp_anonymous_password (anonymous_pw);

		if (strcmp (gnome_cmd_ftp_server_get_user_name (server), "anonymous") != 0)
		{
			GtkWidget *dialog = gnome_cmd_string_dialog_new_with_cancel (
				TRUE,
				_("Password"),
				_("Enter your password"),
				PIXMAP_LOCK,
				(GnomeStringCallback)on_password_ok,
				(GtkSignalFunc)on_password_cancel,
				ftp_dialog);

			gtk_widget_ref (dialog);
			gtk_object_set_data_full (GTK_OBJECT (ftp_dialog),
									  "ftp-password-dialog", dialog,
									  (GtkDestroyNotify)gtk_widget_unref);
			gtk_widget_show (dialog);			
		}
		else
			do_connect (ftp_dialog, anonymous_pw);
	}    
}


static void
on_cancel_btn_clicked                  (GtkButton       *button,
                                        GnomeCmdFtpDialog *dialog)
{
	gtk_widget_hide (GTK_WIDGET (dialog));
}


static void
on_new_ftp_server_dialog_ok_btn_clicked (GtkButton       *button,
										 gpointer         user_data)
{
	GnomeCmdFtpDialog *dialog = GNOME_CMD_FTP_DIALOG (user_data);
	gnome_cmd_data_add_ftp_server (dialog->priv->server_to_add);
	dialog->priv->server_to_add = NULL;
	load_servers (dialog);
}


static void
on_edit_ftp_server_dialog_ok_btn_clicked (GtkButton       *button,
										 gpointer         user_data)
{
	GnomeCmdFtpDialog *dialog = GNOME_CMD_FTP_DIALOG (user_data);
	load_servers (dialog);
}


static void
on_new_btn_clicked                     (GtkButton       *button,
                                        gpointer         user_data)
{
	GnomeCmdFtpServer *server = gnome_cmd_ftp_server_new ();
	GnomeCmdFtpDialog *dialog = GNOME_CMD_FTP_DIALOG (user_data);
	GnomeCmdFtpServerDialog *ftp_server_dialog =
		GNOME_CMD_FTP_SERVER_DIALOG (gnome_cmd_ftp_server_dialog_new (server));

	gtk_signal_connect (GTK_OBJECT (ftp_server_dialog->ok_button), "clicked",
						GTK_SIGNAL_FUNC (on_new_ftp_server_dialog_ok_btn_clicked),
						dialog);

	dialog->priv->server_to_add = server;

	gtk_widget_show (GTK_WIDGET (ftp_server_dialog));
}


static void
on_edit_btn_clicked                    (GtkButton       *button,
                                        gpointer         user_data)
{
	GnomeCmdFtpDialog *dialog = GNOME_CMD_FTP_DIALOG (user_data);
	GnomeCmdFtpServer *server = dialog->priv->selected_server;

	if (server)
	{
		GnomeCmdFtpServerDialog *ftp_server_dialog =
			GNOME_CMD_FTP_SERVER_DIALOG (gnome_cmd_ftp_server_dialog_new (server));

		gtk_signal_connect (GTK_OBJECT (ftp_server_dialog->ok_button), "clicked",
							GTK_SIGNAL_FUNC (on_edit_ftp_server_dialog_ok_btn_clicked),
							dialog);
		
		gtk_widget_show (GTK_WIDGET (ftp_server_dialog));
	}
	else
		g_printerr (_("No server selected"));
}


static void
on_remove_btn_clicked                  (GtkButton       *button,
                                        GnomeCmdFtpDialog *dialog)
{
	GnomeCmdFtpServer *server = dialog->priv->selected_server;
	
    if (server)
	{
		gnome_cmd_data_remove_ftp_server (server);
		load_servers (dialog);
		dialog->priv->selected_server = NULL;
	}
	else
		g_printerr (_("No server selected"));
}


static void
on_server_list_select_row                (GtkCList *clist, gint row, gint column,
										  GdkEvent *event, gpointer user_data)
{
	GnomeCmdFtpDialog *dialog = GNOME_CMD_FTP_DIALOG (user_data);

	if (!event)
		return;
	
	if (event->type == GDK_2BUTTON_PRESS)
	{
		on_connect_btn_clicked (NULL, dialog);
	}
	else
	{
		dialog->priv->selected_server =
			(GnomeCmdFtpServer*)gtk_clist_get_row_data (clist, row);
	}
}


static void
on_server_list_row_move                  (GtkCList        *clist,
										  gint             arg1,
										  gint             arg2,
										  gpointer         user_data)
{
    gpointer s;
	GList *servers = gnome_cmd_data_get_ftp_servers ();
    
    s = g_list_nth_data (servers, arg1);
	g_return_if_fail (s != NULL);
    
    servers = g_list_remove (servers, s);    
    servers = g_list_insert (servers, s, arg2);

	gnome_cmd_data_set_ftp_servers (servers);
}


static void
on_server_list_scroll_vertical                  (GtkCList          *clist,
												 GtkScrollType      scroll_type,
												 gfloat             position,
												 GnomeCmdFtpDialog *dialog)
{
	dialog->priv->selected_server =
		(GnomeCmdFtpServer*)gtk_clist_get_row_data (clist, clist->focus_row);
	
    gtk_clist_select_row (clist, clist->focus_row, 0);
}
							   

/*******************************
 * Gtk class implementation
 *******************************/

static void
destroy (GtkObject *object)
{
	GnomeCmdFtpDialog *dialog = GNOME_CMD_FTP_DIALOG (object);

	if (!dialog->priv)
		g_warning ("GnomeCmdFtpDialog: dialog->priv != NULL test failed\n");
	else
		g_free (dialog->priv);

	if (GTK_OBJECT_CLASS (parent_class)->destroy)
		(*GTK_OBJECT_CLASS (parent_class)->destroy) (object);
}

static void
map (GtkWidget *widget)
{
	if (GTK_WIDGET_CLASS (parent_class)->map != NULL)
		GTK_WIDGET_CLASS (parent_class)->map (widget);
}


static void
class_init (GnomeCmdFtpDialogClass *class)
{
	GtkObjectClass *object_class;
	GtkWidgetClass *widget_class;

	object_class = GTK_OBJECT_CLASS (class);
	widget_class = GTK_WIDGET_CLASS (class);

	parent_class = gtk_type_class (gnome_dialog_get_type ());

	object_class->destroy = destroy;

	widget_class->map = map;
}

static void
init (GnomeCmdFtpDialog *ftp_dialog)
{
	GtkWidget *dialog_vbox;
	GtkWidget *vbox;
	GtkWidget *server_frame;
	GtkWidget *server_list_vbox;
	GtkWidget *scrolledwindow;
	GtkWidget *viewport;
	GtkWidget *label119;
	GtkWidget *label120;
	GtkWidget *buttonbox;
	GtkWidget *new_button;
	GtkWidget *edit_button;
	GtkWidget *remove_button;
	GtkWidget *options_frame;
	GtkWidget *hbox1;
	GtkWidget *label143;
	GtkWidget *connect_button;
	GtkWidget *cancel_button;
	
	GtkWidget *dialog = GTK_WIDGET (ftp_dialog);

	ftp_dialog->priv = g_new (GnomeCmdFtpDialogPrivate, 1);
	

	gtk_window_set_position (GTK_WINDOW (dialog), GTK_WIN_POS_CENTER);
	gtk_window_set_title (GTK_WINDOW (dialog), _("FTP Connect"));

	dialog_vbox = GNOME_DIALOG (dialog)->vbox;
	gtk_object_set_data (GTK_OBJECT (dialog), "dialog_vbox", dialog_vbox);
	gtk_widget_show (dialog_vbox);

	vbox = gtk_vbox_new (FALSE, 0);
	gtk_widget_ref (vbox);
	gtk_object_set_data_full (GTK_OBJECT (dialog), "vbox", vbox,
							  (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (vbox);
	gtk_box_pack_start (GTK_BOX (dialog_vbox), vbox, TRUE, TRUE, 0);

	server_frame = gtk_frame_new (_("servers"));
	gtk_widget_ref (server_frame);
	gtk_object_set_data_full (GTK_OBJECT (dialog), "server_frame", server_frame,
							  (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (server_frame);
	gtk_box_pack_start (GTK_BOX (vbox), server_frame, TRUE, TRUE, 2);
	gtk_widget_set_usize (server_frame, 250, 250);

	server_list_vbox = gtk_vbox_new (FALSE, 0);
	gtk_widget_ref (server_list_vbox);
	gtk_object_set_data_full (GTK_OBJECT (dialog), "server_list_vbox", server_list_vbox,
							  (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (server_list_vbox);
	gtk_container_add (GTK_CONTAINER (server_frame), server_list_vbox);

	scrolledwindow = gtk_scrolled_window_new (NULL, NULL);
	gtk_widget_ref (scrolledwindow);
	gtk_object_set_data_full (GTK_OBJECT (dialog), "scrolledwindow", scrolledwindow,
							  (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (scrolledwindow);
	gtk_box_pack_start (GTK_BOX (server_list_vbox), scrolledwindow, TRUE, TRUE, 0);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolledwindow), GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);

	viewport = gtk_viewport_new (NULL, NULL);
	gtk_widget_ref (viewport);
	gtk_object_set_data_full (GTK_OBJECT (dialog), "viewport", viewport,
							  (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (viewport);
	gtk_container_add (GTK_CONTAINER (scrolledwindow), viewport);

	ftp_dialog->priv->server_list = gtk_clist_new (2);
	gtk_widget_ref (ftp_dialog->priv->server_list);
	gtk_object_set_data_full (GTK_OBJECT (dialog), "server_list", ftp_dialog->priv->server_list,
							  (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (ftp_dialog->priv->server_list);
	gtk_container_add (GTK_CONTAINER (viewport), ftp_dialog->priv->server_list);
	gtk_widget_set_usize (ftp_dialog->priv->server_list, 170, -2);
	gtk_clist_set_column_width (GTK_CLIST (ftp_dialog->priv->server_list), 0, 16);
	gtk_clist_set_column_width (GTK_CLIST (ftp_dialog->priv->server_list), 1, 80);
	gtk_clist_column_titles_show (GTK_CLIST (ftp_dialog->priv->server_list));

	label119 = gtk_label_new ("");
	gtk_widget_ref (label119);
	gtk_object_set_data_full (GTK_OBJECT (dialog), "label119", label119,
							  (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (label119);
	gtk_clist_set_column_widget (GTK_CLIST (ftp_dialog->priv->server_list), 0, label119);

	label120 = gtk_label_new (_("alias"));
	gtk_widget_ref (label120);
	gtk_object_set_data_full (GTK_OBJECT (dialog), "label120", label120,
							  (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (label120);
	gtk_clist_set_column_widget (GTK_CLIST (ftp_dialog->priv->server_list), 1, label120);

	buttonbox = gtk_hbox_new (TRUE, 0);
	gtk_widget_ref (buttonbox);
	gtk_object_set_data_full (GTK_OBJECT (dialog), "buttonbox", buttonbox,
							  (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (buttonbox);
	gtk_box_pack_start (GTK_BOX (server_list_vbox), buttonbox, FALSE, FALSE, 0);

	new_button = gtk_button_new_with_label (_("New..."));
	gtk_widget_ref (new_button);
	gtk_object_set_data_full (GTK_OBJECT (dialog), "new_button", new_button,
							  (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (new_button);
	gtk_box_pack_start (GTK_BOX (buttonbox), new_button, TRUE, TRUE, 1);

	edit_button = gtk_button_new_with_label (_("Edit..."));
	gtk_widget_ref (edit_button);
	gtk_object_set_data_full (GTK_OBJECT (dialog), "edit_button", edit_button,
							  (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (edit_button);
	gtk_box_pack_start (GTK_BOX (buttonbox), edit_button, TRUE, TRUE, 1);

	remove_button = gtk_button_new_with_label (_("Remove"));
	gtk_widget_ref (remove_button);
	gtk_object_set_data_full (GTK_OBJECT (dialog), "remove_button", remove_button,
							  (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (remove_button);
	gtk_box_pack_start (GTK_BOX (buttonbox), remove_button, TRUE, TRUE, 1);

	options_frame = gtk_frame_new (_("options"));
	gtk_widget_ref (options_frame);
	gtk_object_set_data_full (GTK_OBJECT (dialog), "options_frame", options_frame,
							  (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (options_frame);
	gtk_box_pack_start (GTK_BOX (vbox), options_frame, TRUE, TRUE, 0);

	hbox1 = gtk_hbox_new (FALSE, 0);
	gtk_widget_ref (hbox1);
	gtk_object_set_data_full (GTK_OBJECT (dialog), "hbox1", hbox1,
							  (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (hbox1);
	gtk_container_add (GTK_CONTAINER (options_frame), hbox1);

	label143 = gtk_label_new (_("anonymous password:"));
	gtk_widget_ref (label143);
	gtk_object_set_data_full (GTK_OBJECT (dialog), "label143", label143,
							  (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (label143);
	gtk_box_pack_start (GTK_BOX (hbox1), label143, FALSE, FALSE, 2);

	ftp_dialog->priv->anonymous_pw_entry = gtk_entry_new ();
	gtk_widget_ref (ftp_dialog->priv->anonymous_pw_entry);
	gtk_object_set_data_full (GTK_OBJECT (dialog),
							  "anonymous_pw_entry", ftp_dialog->priv->anonymous_pw_entry,
							  (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (ftp_dialog->priv->anonymous_pw_entry);
	gtk_box_pack_start (GTK_BOX (hbox1), ftp_dialog->priv->anonymous_pw_entry,
						TRUE, TRUE, 2);
	gtk_entry_set_text (GTK_ENTRY (ftp_dialog->priv->anonymous_pw_entry),
						_(gnome_cmd_data_get_ftp_anonymous_password ()));
 
	gnome_dialog_append_button_with_pixmap (GNOME_DIALOG (dialog),
											_("Connect"), GNOME_STOCK_PIXMAP_REFRESH);
	connect_button = g_list_last (GNOME_DIALOG (dialog)->buttons)->data;
	gtk_widget_ref (connect_button);
	gtk_object_set_data_full (GTK_OBJECT (dialog), "connect_button", connect_button,
							  (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (connect_button);
	GTK_WIDGET_SET_FLAGS (connect_button, GTK_CAN_DEFAULT);

	gnome_dialog_append_button (GNOME_DIALOG (dialog), GNOME_STOCK_BUTTON_CANCEL);
	cancel_button = g_list_last (GNOME_DIALOG (dialog)->buttons)->data;
	gtk_widget_ref (cancel_button);
	gtk_object_set_data_full (GTK_OBJECT (dialog), "cancel_button", cancel_button,
							  (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (cancel_button);
	GTK_WIDGET_SET_FLAGS (cancel_button, GTK_CAN_DEFAULT);

	gtk_signal_connect (GTK_OBJECT (ftp_dialog->priv->server_list), "select_row",
						GTK_SIGNAL_FUNC (on_server_list_select_row), dialog);
	gtk_signal_connect (GTK_OBJECT (ftp_dialog->priv->server_list), "row_move",
						GTK_SIGNAL_FUNC (on_server_list_row_move), dialog);
	gtk_signal_connect_after (GTK_OBJECT (ftp_dialog->priv->server_list),
							  "scroll-vertical",
							  GTK_SIGNAL_FUNC (on_server_list_scroll_vertical), dialog);
	gtk_signal_connect (GTK_OBJECT (new_button), "clicked",
						GTK_SIGNAL_FUNC (on_new_btn_clicked), dialog);
	gtk_signal_connect (GTK_OBJECT (edit_button), "clicked",
						GTK_SIGNAL_FUNC (on_edit_btn_clicked), dialog);
	gtk_signal_connect (GTK_OBJECT (remove_button), "clicked",
						GTK_SIGNAL_FUNC (on_remove_btn_clicked), dialog);
	gtk_signal_connect (GTK_OBJECT (cancel_button), "clicked",
						GTK_SIGNAL_FUNC (on_cancel_btn_clicked), dialog);
	gtk_signal_connect (GTK_OBJECT (connect_button), "clicked",
						GTK_SIGNAL_FUNC (on_connect_btn_clicked), dialog);


	gtk_widget_grab_focus (ftp_dialog->priv->server_list);
	gnome_dialog_set_default (GNOME_DIALOG (ftp_dialog), 0);
}






/***********************************
 * Public functions
 ***********************************/

GtkType
gnome_cmd_ftp_dialog_get_type         (void)
{
	static GtkType dlg_type = 0;

	if (dlg_type == 0)
	{
		GtkTypeInfo dlg_info =
		{
			"GnomeCmdFtpDialog",
			sizeof (GnomeCmdFtpDialog),
			sizeof (GnomeCmdFtpDialogClass),
			(GtkClassInitFunc) class_init,
			(GtkObjectInitFunc) init,
			/* reserved_1 */ NULL,
			/* reserved_2 */ NULL,
			(GtkClassInitFunc) NULL
		};

		dlg_type = gtk_type_unique (gnome_dialog_get_type (), &dlg_info);
	}
	return dlg_type;
}


GtkWidget*
gnome_cmd_ftp_dialog_new (void)
{
	GnomeCmdFtpDialog *dialog;

	dialog = gtk_type_new (gnome_cmd_ftp_dialog_get_type ());

	dialog->priv->selected_server = NULL;
	dialog->priv->server_to_add = NULL;
	load_servers (dialog);

	return GTK_WIDGET (dialog);
}
