/*
    GNOME Commander - A GNOME based file manager 
    Copyright (C) 2001-2002 Marcus Bjurman

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/ 
#include <config.h>
#include "gnome-cmd-includes.h"
#include "gnome-cmd-file-selector-dnd.h"


#define MAX_TYPE_LENGTH 2
#define MAX_NAME_LENGTH 128
#define MAX_PERM_LENGTH 10
#define MAX_DATE_LENGTH 64
#define MAX_SIZE_LENGTH 32
#define MAX_LENGTH 128 // must be as long as the longest of the above!



#define LOCAL_FILE "file://"
#define FTP_FILE "ftp://"

typedef gint (* GnomeVFSListCompareFunc) (gconstpointer a, gconstpointer b,
                                         gpointer data);

GList *gnome_vfs_list_sort (GList *list,
                           GnomeVFSListCompareFunc compare_func,
                           gpointer data);


#define LIST_NUM_COLUMNS 7
static gchar *list_column_titles[LIST_NUM_COLUMNS] = {
	"",
	N_("name"),
	N_("size"),
	N_("perm"),
	N_("date"),
	N_("uid"),
	N_("gid")
};

static gushort list_column_titles_width[LIST_NUM_COLUMNS] = {
	16,
	140,
	70,
	70,
	150,
	50,
	50	
};

static GtkVBoxClass *parent_class = NULL;


struct _GnomeCmdFileSelectorPrivate {
	GnomeVFSListCompareFunc sort_func;
	gint current_col;
	gboolean sort_raising[LIST_NUM_COLUMNS];
	GtkWidget *column_pixmaps[LIST_NUM_COLUMNS];
	GtkWidget *column_labels[LIST_NUM_COLUMNS];
	GtkWidget *popup_menu;

	gboolean active;
	GList *shown_files;
	GList *selected_files; /* contains GnomeCmdFile pointers */
	gint cur_file;
	gint shift_cnt;
	gint shift_down_row;
	gboolean realized;
	gboolean selection_lock;
	gboolean sel_first_file;
	GnomeCmdConnection *con;
	GnomeCmdDir *cwd;
	gboolean right_mouse_sel_state;
};


static void
on_list_scroll_vertical                  (GtkCList             *clist,
										  GtkScrollType         scroll_type,
										  gfloat                position,
										  GnomeCmdFileSelector *fs);

static void
on_file_updated (GnomeCmdFile *finfo, GnomeCmdFileSelector *fs);


/*******************************
 * Utility functions
 *******************************/

/******************************************************************************
*
*   Function: add_file_to_list
*
*   Purpose:  Add a file to the list
*
*   Params:   @fs: The FileSelector to add the file to
*             @finfo: The file to add
*             @in_row: The row to add the file at. Set to -1 to append the file at the end.
*
*   Returns: 
*
*   Statuses: 
*
******************************************************************************/
static void
add_file_to_list (GnomeCmdFileSelector *fs, GnomeCmdFile *finfo, gint in_row)
{
	gint row;
	gchar *text[8];
	GtkCList *clist;
	GnomeCmdLayoutColors *colors = gnome_cmd_data_get_layout_colors ();

	clist = GTK_CLIST(fs->list);

	/* If the user wants a character insted of icon for filetype set it now
	 *
	 */
	if (gnome_cmd_data_get_layout () == GNOME_CMD_LAYOUT_TEXT)
		text[0] = (gchar*)gnome_cmd_file_get_type_string (finfo);
	else
		text[0] = NULL;

	/* Set other file information
	 *
	 */
	text[1] = (gchar*)gnome_cmd_file_get_name (finfo);
	text[2] = (gchar*)gnome_cmd_file_get_size (finfo);
	text[3] = (gchar*)gnome_cmd_file_get_perm (finfo);
	text[4] = (gchar*)gnome_cmd_file_get_date (finfo);
	text[5] = (gchar*)gnome_cmd_file_get_owner (finfo);
	text[6] = (gchar*)gnome_cmd_file_get_group (finfo);
	text[7] = NULL;

	/* Add the file to the list
	 *
	 */
	gnome_cmd_file_ref (finfo);
	fs->priv->shown_files = g_list_append (fs->priv->shown_files, finfo);	
	if (in_row == -1)
		row = gtk_clist_append (clist, text);	
	else
		row = gtk_clist_insert (clist, in_row, text);	
	
	/* Setup row data and color
     *
	 */   
	gtk_clist_set_foreground (GTK_CLIST (fs->list), row, colors->norm_fg);
	gtk_clist_set_background (GTK_CLIST (fs->list), row, colors->norm_bg);
	gtk_clist_set_row_data (GTK_CLIST (fs->list), row, finfo);
	
	/* If the use wants icons to show file types set it now
	 *
	 */
	if (gnome_cmd_data_get_layout () == GNOME_CMD_LAYOUT_ICONS) {
		gtk_clist_set_pixmap (GTK_CLIST (fs->list), row, 0,
							  gnome_cmd_file_get_type_pixmap_small (finfo),
							  gnome_cmd_file_get_type_mask_small (finfo));
	}
}


static int
get_num_files (GnomeCmdFileSelector *fs)
{
	g_return_val_if_fail (fs != NULL, -1);
	g_return_val_if_fail (fs->list != NULL, -1);

	/* compensate for the ".." row */
	return GTK_CLIST (fs->list)->rows-1;
}

static void
create_column_titles (GnomeCmdFileSelector *fs)
{
	int i;

	g_return_if_fail (fs != NULL);
	
	gtk_clist_set_column_width (GTK_CLIST (fs->list), 0, list_column_titles_width[0]);
	
	for ( i=1 ; i<LIST_NUM_COLUMNS ; i++ )
	{
		GtkWidget *hbox,*pixmap;

		GdkPixmap *pm = IMAGE_get_pixmap (PIXMAP_FLIST_ARROW_BLANK);
		GdkBitmap *bm = IMAGE_get_mask (PIXMAP_FLIST_ARROW_BLANK);
		
		hbox = gtk_hbox_new (FALSE, 1);
		gtk_widget_ref (hbox);
		gtk_object_set_data_full (GTK_OBJECT (fs), "column-hbox", hbox,
								  (GtkDestroyNotify) gtk_widget_unref);
		gtk_widget_show (hbox);		

		fs->priv->column_labels[i] = gtk_label_new (list_column_titles[i]);
		gtk_widget_ref (fs->priv->column_labels[i]);
		gtk_object_set_data_full (GTK_OBJECT (fs), "column-label", fs->priv->column_labels[i],
								  (GtkDestroyNotify) gtk_widget_unref);
		gtk_widget_show (fs->priv->column_labels[i]);
		gtk_box_pack_start (GTK_BOX (hbox), fs->priv->column_labels[i], TRUE, TRUE, 0);
		gtk_widget_set_style (fs->priv->column_labels[i], main_style);

		pixmap = gtk_pixmap_new (pm, bm);
		gtk_widget_ref (pixmap);
		gtk_object_set_data_full (GTK_OBJECT (fs), "column-pixmap", pixmap,
								  (GtkDestroyNotify) gtk_widget_unref);
		gtk_widget_show (pixmap);
		gtk_box_pack_start (GTK_BOX (hbox), pixmap, FALSE, FALSE, 0);		
		
		fs->priv->column_pixmaps[i] = pixmap;
		gtk_clist_set_column_widget (GTK_CLIST (fs->list), i, hbox);
		gtk_clist_set_column_width (GTK_CLIST (fs->list), i, list_column_titles_width[i]);	
	}

	gtk_clist_column_titles_show (GTK_CLIST (fs->list));
}


static void
update_selected_files_label (GnomeCmdFileSelector *fs)
{
	gint num_files;
	GList *files;
	
	g_return_if_fail (fs != NULL);
	g_return_if_fail (fs->priv != NULL);

	files = gnome_cmd_dir_list_files (fs->priv->cwd);
	g_return_if_fail (files != NULL);
	
	num_files = get_num_files (fs);

	if (num_files > -1)
	{
		GList *tmp;
		gchar *info_str;
		gint sel_bytes = 0;
		gint total_bytes = 0;

		tmp = files;
		while (tmp)
		{
			GnomeCmdFile *finfo = (GnomeCmdFile*)tmp->data;
			total_bytes += finfo->info->size;
			tmp = tmp->next;
		}

		tmp = fs->priv->selected_files;
		while (tmp)
		{
			GnomeCmdFile *finfo = (GnomeCmdFile*)tmp->data;
			sel_bytes += finfo->info->size;
			tmp = tmp->next;
		}
		
		info_str = g_strdup_printf (_("%d of %d bytes in %d of %d files selected"),
									sel_bytes,
									total_bytes,
									g_list_length (fs->priv->selected_files),
									num_files);	
		gtk_label_set_text (GTK_LABEL (fs->info_label), info_str);	
		g_free (info_str);
	}
}


static void
do_select_file (GnomeCmdFileSelector *fs,
				GnomeCmdFile *finfo)
{
	gint row;
	GnomeCmdLayoutColors *colors = gnome_cmd_data_get_layout_colors ();
	
	g_return_if_fail (fs != NULL);
	g_return_if_fail (fs->priv != NULL);
	g_return_if_fail (finfo != NULL);
	g_return_if_fail (finfo->info != NULL);

	if (strcmp (finfo->info->name, "..") == 0)
		return;

	row = gtk_clist_find_row_from_data (GTK_CLIST(fs->list), finfo);
	if (row == -1)
		return;

	if (g_list_index (fs->priv->selected_files, finfo) != -1)
		return;

	gnome_cmd_file_ref (finfo);
	fs->priv->selected_files = g_list_append (
		fs->priv->selected_files, finfo);
		
	gtk_clist_set_foreground (GTK_CLIST (fs->list), row, colors->sel_fg);
	gtk_clist_set_background (GTK_CLIST (fs->list), row, colors->sel_bg);

	update_selected_files_label (fs);
}


static void
do_unselect_file (GnomeCmdFileSelector *fs,
				  GnomeCmdFile *finfo)
{
	gint row;
	GnomeCmdLayoutColors *colors = gnome_cmd_data_get_layout_colors ();
	
	g_return_if_fail (fs != NULL);
	g_return_if_fail (fs->priv != NULL);
	g_return_if_fail (finfo != NULL);

	row = gtk_clist_find_row_from_data (GTK_CLIST(fs->list), finfo);
	if (row == -1)
		return;
	
	if (g_list_index (fs->priv->selected_files, finfo) == -1)
		return;

	gnome_cmd_file_unref (finfo);
	fs->priv->selected_files = g_list_remove (
		fs->priv->selected_files, finfo);

	gtk_clist_set_foreground (GTK_CLIST (fs->list), row, colors->norm_fg);
	gtk_clist_set_background (GTK_CLIST (fs->list), row, colors->norm_bg);

	update_selected_files_label (fs);
}


static void
select_file (GnomeCmdFileSelector *fs,
			 GnomeCmdFile *finfo)
{
	gint row;
	GList *files;
	
	g_return_if_fail (fs != NULL);
	g_return_if_fail (fs->priv != NULL);
	g_return_if_fail (finfo != NULL);

	row = gtk_clist_find_row_from_data (GTK_CLIST(fs->list), finfo);
	if (row == -1)
		return;
	
	files = gnome_cmd_dir_list_files (gnome_cmd_file_selector_get_directory (fs));
	
	if (row < g_list_length (files))
	{
		if (g_list_index (fs->priv->selected_files, finfo) == -1)
			do_select_file (fs, finfo);
		else
			do_unselect_file (fs, finfo);
	}
}


static void
select_file_at_row (GnomeCmdFileSelector *fs,
					gint row)
{
	GnomeCmdFile *finfo;
	GnomeCmdDir *dir;
	GList *files;	

	g_return_if_fail (fs != NULL);
	g_return_if_fail (fs->priv != NULL);
	g_return_if_fail (fs->list != NULL);

	fs->priv->cur_file = row;

	dir = gnome_cmd_file_selector_get_directory (fs);
	files = gnome_cmd_dir_list_files (dir);
	
	finfo = gtk_clist_get_row_data (GTK_CLIST (fs->list), row);
	g_return_if_fail (finfo != NULL);

	select_file (fs, finfo);
}


static void
select_file_range (GnomeCmdFileSelector *fs,
				   gint start_row, gint end_row)
{
	gint i;
	gint num_files;

	num_files = get_num_files (fs);	
	
	if (start_row > end_row)
	{
		i = start_row;
		start_row = end_row;
		end_row = i;
	}
        
	for ( i= start_row ; i<end_row ; i++ )
		select_file_at_row (fs, i);

	if (end_row == num_files)
		select_file_at_row (fs, end_row);
}


static void focus_file_at_row (GnomeCmdFileSelector *fs,
							   gint row)
{
	g_return_if_fail (fs != NULL);
	g_return_if_fail (fs->list != NULL);
	
	GTK_CLIST (fs->list)->focus_row=row;
	gtk_clist_select_row (GTK_CLIST (fs->list), row, 0);
}


static void
focus_first_file_that_starts_with (GnomeCmdFileSelector *fs, gchar keyval)
{
	GList *files;
	gchar s[2];
	g_return_if_fail (fs != NULL);
	g_return_if_fail (fs->priv != NULL);

	s[0] = keyval;
	s[1] = '\0';
	
	files = gnome_cmd_file_selector_get_all_files (fs);
	while (files) {
		GnomeCmdFile *finfo = (GnomeCmdFile*)files->data;
		if (strncasecmp (finfo->info->name, s, 1) == 0) {
			gint row = gtk_clist_find_row_from_data (GTK_CLIST (fs->list), finfo);
			if (row > 0)
				focus_file_at_row (fs, row);
			return;
		}
		files = files->next;
	}
}


static gboolean
file_is_wanted (GnomeCmdFileSelector *fs, GnomeVFSFileInfo *info)
{
		// check that the file is "wanted"
		if (info->type == GNOME_VFS_FILE_TYPE_UNKNOWN
			&& gnome_cmd_data_get_type_filter (GNOME_VFS_FILE_TYPE_UNKNOWN))
			return FALSE;
		if (info->type == GNOME_VFS_FILE_TYPE_REGULAR
			&& gnome_cmd_data_get_type_filter (GNOME_VFS_FILE_TYPE_REGULAR))
			return FALSE;
		if (info->type == GNOME_VFS_FILE_TYPE_DIRECTORY
			&& gnome_cmd_data_get_type_filter (GNOME_VFS_FILE_TYPE_DIRECTORY))
			return FALSE;
		if (info->type == GNOME_VFS_FILE_TYPE_FIFO
			&& gnome_cmd_data_get_type_filter (GNOME_VFS_FILE_TYPE_FIFO))
			return FALSE;
		if (info->type == GNOME_VFS_FILE_TYPE_SOCKET
			&& gnome_cmd_data_get_type_filter (GNOME_VFS_FILE_TYPE_SOCKET))
			return FALSE;
		if (info->type == GNOME_VFS_FILE_TYPE_CHARACTER_DEVICE
			&& gnome_cmd_data_get_type_filter (GNOME_VFS_FILE_TYPE_CHARACTER_DEVICE))
			return FALSE;
		if (info->type == GNOME_VFS_FILE_TYPE_BLOCK_DEVICE
			&& gnome_cmd_data_get_type_filter (GNOME_VFS_FILE_TYPE_BLOCK_DEVICE))
			return FALSE;
		if ((info->flags == GNOME_VFS_FILE_FLAGS_SYMLINK
			 || info->symlink_name != NULL)
			&& gnome_cmd_data_get_type_filter (GNOME_VFS_FILE_TYPE_SYMBOLIC_LINK))
			return FALSE;
		if (strcmp (info->name, ".") == 0)
			return FALSE;
		if (info->name[0] == '.' && strcmp (info->name, "..") != 0
			&& gnome_cmd_data_get_hidden_filter ())
			return FALSE;

		return TRUE;
}


static void
update_files (GnomeCmdFileSelector *fs)
{
	GnomeVFSFileInfo *info;
	GList *list;
	GnomeCmdDir *dir;

	g_return_if_fail (fs != NULL);
	g_return_if_fail (fs->list != NULL);

	dir = gnome_cmd_file_selector_get_directory (fs);
	g_return_if_fail (dir != NULL);

	gtk_clist_freeze (GTK_CLIST (fs->list));
	gtk_clist_clear (GTK_CLIST (fs->list));
	gnome_cmd_file_list_free (fs->priv->shown_files);
	gnome_cmd_file_list_free (fs->priv->selected_files);
	fs->priv->shown_files = NULL;
	fs->priv->selected_files = NULL;

	list = g_list_copy (gnome_cmd_dir_list_files (dir));
	list = gnome_vfs_list_sort (list, (GnomeVFSListCompareFunc)fs->priv->sort_func, fs);
	
	while (list != NULL)
	{
		GnomeCmdFile *finfo;

		finfo = GNOME_CMD_FILE (list->data);
		gnome_cmd_file_add_observer (finfo, (GnomeCmdFileUpdateCallbackFunc)on_file_updated, fs);
		info = finfo->info;

		if (file_is_wanted (fs, info))
			add_file_to_list (fs, finfo, -1);
		
		list = list->next;
	}
	
	gtk_clist_thaw (GTK_CLIST (fs->list));
	update_selected_files_label (fs);
	g_list_free (list);
}


static void update_direntry (GnomeCmdFileSelector *fs)
{
	const gchar *tmp;
	
	g_return_if_fail (GNOME_CMD_IS_FILE_SELECTOR (fs));
	g_return_if_fail (fs->priv != NULL);	
	g_return_if_fail (GTK_IS_ENTRY (fs->dir_entry));

	tmp = gnome_cmd_dir_get_path (fs->priv->cwd);
	gtk_entry_set_text (GTK_ENTRY (fs->dir_entry), tmp);
}


/******************************************************
 * Filesorting functions
 **/
static gint
my_strcmp (gchar *s1, gchar *s2, gboolean raising)
{	
	int ret;

	if (gnome_cmd_data_get_case_sens_sort ())
		ret = strcmp (s1,s2);
	else
		ret = g_strcasecmp (s1,s2);
	
	if (ret > 0)
		return (raising?-1:1);
	else if (ret < 0)
		return (raising?1:-1);

	return 0;
}


static gint
my_intcmp (gint i1, gint i2, gboolean raising)
{
	if (i1 > i2)
		return (raising?-1:1);
	else if (i2 > i1)
		return (raising?1:-1);
	
	return 0;
}


static gint
sort_by_name (GnomeCmdFile *f1, GnomeCmdFile *f2, GnomeCmdFileSelector *fs)
{
	gboolean raising = fs->priv->sort_raising[fs->priv->current_col];

	if (strcmp (f1->info->name, "..") == 0)
		return -1;
	else if (strcmp (f2->info->name, "..") == 0)
		return 1;
	else if (f1->info->type > f2->info->type)
		return -1;
	else if (f1->info->type < f2->info->type)
		return 1;

	return my_strcmp (f1->info->name, f2->info->name, raising);
}


static gint
sort_by_size (GnomeCmdFile *f1, GnomeCmdFile *f2, GnomeCmdFileSelector *fs)
{
	int ret;
	gboolean raising = fs->priv->sort_raising[fs->priv->current_col];
	gboolean file_raising = fs->priv->sort_raising[1];

	if (strcmp (f1->info->name, "..") == 0)
		return -1;
	else if (strcmp (f2->info->name, "..") == 0)
		return 1;
	
	ret = my_intcmp (f1->info->type, f2->info->type, TRUE);
	if (!ret)
	{
		ret = my_intcmp (f1->info->size, f2->info->size, raising);
		if (!ret)
		{
			ret = my_strcmp (f1->info->name, f2->info->name, file_raising);
		}
	}
	return ret;
}


static gint
sort_by_perm (GnomeCmdFile *f1, GnomeCmdFile *f2, GnomeCmdFileSelector *fs)
{
	int ret;
	gboolean raising = fs->priv->sort_raising[fs->priv->current_col];
	gboolean file_raising = fs->priv->sort_raising[1];

	if (strcmp (f1->info->name, "..") == 0)
		return -1;
	else if (strcmp (f2->info->name, "..") == 0)
		return 1;
	
	ret = my_intcmp (f1->info->type, f2->info->type, TRUE);
	if (!ret)
	{
		ret = my_intcmp (f1->info->permissions, f2->info->permissions, raising);
		if (!ret)
		{
			ret = my_strcmp (f1->info->name, f2->info->name, file_raising);
		}
	}
	return ret;
}


static gint
sort_by_date (GnomeCmdFile *f1, GnomeCmdFile *f2, GnomeCmdFileSelector *fs)
{
	int ret;
	gboolean raising = fs->priv->sort_raising[fs->priv->current_col];
	gboolean file_raising = fs->priv->sort_raising[1];

	if (strcmp (f1->info->name, "..") == 0)
		return -1;
	else if (strcmp (f2->info->name, "..") == 0)
		return 1;
	
	ret = my_intcmp (f1->info->type, f2->info->type, TRUE);
	if (!ret)
	{
		ret = my_intcmp (f1->info->mtime, f2->info->mtime, raising);
		if (!ret)
		{
			ret = my_strcmp (f1->info->name, f2->info->name, file_raising);
		}
	}
	return ret;
}


static gint
sort_by_owner (GnomeCmdFile *f1, GnomeCmdFile *f2, GnomeCmdFileSelector *fs)
{
	int ret;
	gboolean raising = fs->priv->sort_raising[fs->priv->current_col];
	gboolean file_raising = fs->priv->sort_raising[1];

	if (strcmp (f1->info->name, "..") == 0)
		return -1;
	else if (strcmp (f2->info->name, "..") == 0)
		return 1;
	
	ret = my_intcmp (f1->info->type, f2->info->type, TRUE);
	if (!ret)
	{
		ret = my_intcmp (f1->info->uid, f2->info->uid, raising);
		if (!ret)
		{
			ret = my_strcmp (f1->info->name, f2->info->name, file_raising);
		}
	}
	return ret;
}


static gint
sort_by_group (GnomeCmdFile *f1, GnomeCmdFile *f2, GnomeCmdFileSelector *fs)
{
	int ret;
	gboolean raising = fs->priv->sort_raising[fs->priv->current_col];
	gboolean file_raising = fs->priv->sort_raising[1];

	if (strcmp (f1->info->name, "..") == 0)
		return -1;
	else if (strcmp (f2->info->name, "..") == 0)
		return 1;
	
	ret = my_intcmp (f1->info->type, f2->info->type, TRUE);
	if (!ret)
	{
		ret = my_intcmp (f1->info->gid, f2->info->gid, raising);
		if (!ret)
		{
			ret = my_strcmp (f1->info->name, f2->info->name, file_raising);
		}
	}
	return ret;
}




static void
goto_directory (GnomeCmdFileSelector *fs,
				const gchar *dir)
{
	GnomeCmdDir *cur_dir, *new_dir;
	GnomeVFSURI *cur_uri, *new_uri;
	gchar *new_uri_str;
	gchar *focus_dir = NULL;

	g_return_if_fail (GNOME_CMD_IS_FILE_SELECTOR (fs));
	g_return_if_fail (fs->priv != NULL);
	g_return_if_fail (dir != NULL);

	cur_dir = gnome_cmd_file_selector_get_directory (fs);
	g_return_if_fail (cur_dir);

	cur_uri = gnome_cmd_dir_get_uri (cur_dir);
	
	if (strcmp (dir, "..") == 0) {
		/* lets get the uri of the parent directory */
		const gchar *cur_path = gnome_vfs_uri_get_path (cur_uri);
		if (strcmp (cur_path, "/") == 0)
			return;
		new_uri = gnome_vfs_uri_get_parent (cur_uri);
		focus_dir = gnome_vfs_unescape_string (gnome_vfs_uri_get_basename (cur_uri), 0);
	}
	else {
		/* check if it's an absolute address or not */
		if (dir[0] == '/')
			new_uri = gnome_vfs_uri_append_path (
				gnome_cmd_connection_get_baseuri (fs->priv->con), dir);
		else
			new_uri = gnome_vfs_uri_append_file_name (cur_uri, dir);
	}

	new_uri_str = gnome_vfs_uri_to_string (new_uri, 0);
	gnome_vfs_uri_unref (new_uri);
	
	new_dir = dir_pool_get (new_uri_str);

	gnome_cmd_file_selector_set_directory (fs, new_dir);
	
	/* focus the current dir when going back to the parent dir */
	if (focus_dir) {
		gnome_cmd_file_selector_focus_file (fs, focus_dir);
		g_free (focus_dir);
	}
	
	g_free (new_uri_str);
}


static void
do_file_specific_action                  (GnomeCmdFileSelector *fs,
										  GnomeCmdFile *finfo)
{
	g_return_if_fail (fs != NULL);
	g_return_if_fail (finfo != NULL);
	g_return_if_fail (finfo->info != NULL);
	
	if (finfo->info->type == GNOME_VFS_FILE_TYPE_DIRECTORY) {
		goto_directory (fs, finfo->info->name);
	}
	else if (finfo->info->type == GNOME_VFS_FILE_TYPE_REGULAR) {
		mime_exec_single (finfo);
	}
}


static void
update_connection (GnomeCmdFileSelector *fs)
{
	GList *connections;
	gchar *text;

	g_return_if_fail (fs != NULL);
	g_return_if_fail (fs->priv != NULL);
	g_return_if_fail (GNOME_CMD_IS_COMBO (fs->con_combo));

	text = gtk_entry_get_text (
		GTK_ENTRY (GNOME_CMD_COMBO (fs->con_combo)->entry));
	
	connections = gnome_cmd_connection_get_list ();
	while (connections)
	{
		GnomeCmdConnection *tmp_con = (GnomeCmdConnection*)connections->data;

		g_return_if_fail (tmp_con != NULL);
		
		if (tmp_con->type == CON_TYPE_LOCAL && strcmp ("local", text)==0)
		{
			if (fs->priv->con->type != CON_TYPE_LOCAL)
			{
				fs->priv->selection_lock = TRUE;
				gnome_cmd_file_selector_set_connection (fs, tmp_con);
				fs->priv->selection_lock = FALSE;
			}
			return;
		}
		else if (tmp_con->type == CON_TYPE_FTP && strcmp (tmp_con->alias, text)==0)
		{
			fs->priv->selection_lock = TRUE;
			gnome_cmd_file_selector_set_connection (fs, tmp_con);
			fs->priv->selection_lock = FALSE;
			return;
		}
			
		connections = connections->next;
	}		
}



/*******************************
 * Callbacks
 *******************************/

static void
on_file_updated (GnomeCmdFile *finfo, GnomeCmdFileSelector *fs)
{
	gchar *text[8];
	gint i,row;

	row = gtk_clist_find_row_from_data (GTK_CLIST(fs->list), finfo);
	if (row == -1)
		return;
	
	text[1] = (gchar*)gnome_cmd_file_get_name (finfo);
	text[2] = (gchar*)gnome_cmd_file_get_size (finfo);
	text[3] = (gchar*)gnome_cmd_file_get_perm (finfo);
	text[4] = (gchar*)gnome_cmd_file_get_date (finfo);
	text[5] = (gchar*)gnome_cmd_file_get_owner (finfo);
	text[6] = (gchar*)gnome_cmd_file_get_group (finfo);
	text[7] = NULL;

	for ( i=1 ; i<8; i++ )
		gtk_clist_set_text (GTK_CLIST (fs->list), row, i, text[i]);
}


static void
on_dir_file_created (GnomeCmdDir *dir, GnomeCmdFile *finfo, GnomeCmdFileSelector *fs)
{
	gint i;
	GtkCList *clist;

	g_return_if_fail (dir != NULL);
	g_return_if_fail (fs != NULL);
	g_return_if_fail (fs->priv != NULL);

	clist = GTK_CLIST(fs->list);
	
	if (!file_is_wanted (fs, finfo->info))
		return;
		
	for ( i=0 ; i<clist->rows ; i++ ) {
		GnomeCmdFile *finfo2 = (GnomeCmdFile*)gtk_clist_get_row_data (clist, i);
		if (fs->priv->sort_func (finfo2, finfo, fs) == 1) {
			add_file_to_list (fs, finfo, i);
			return;
		}
	}

	/* Insert the file at the end of the list
	 *
	 */
	add_file_to_list (fs, finfo, -1);
}


static void
on_dir_file_deleted (GnomeCmdDir *dir, GnomeCmdFile *finfo, GnomeCmdFileSelector *fs)
{
	g_return_if_fail (dir != NULL);
	g_return_if_fail (fs != NULL);
	g_return_if_fail (fs->priv != NULL);
	
	if (fs->priv->cwd == dir && file_is_wanted (fs, finfo->info)) {
		gint row = gtk_clist_find_row_from_data (GTK_CLIST (fs->list), finfo);
		gtk_clist_remove (GTK_CLIST (fs->list), row);
	}
}


static void
on_dir_file_changed (GnomeCmdDir *dir, GnomeCmdFile *finfo, GnomeCmdFileSelector *fs)
{
	g_return_if_fail (dir != NULL);
	g_return_if_fail (fs != NULL);
	g_return_if_fail (fs->priv != NULL);

	if (file_is_wanted (fs, finfo->info))
		on_file_updated (finfo, fs);
}


static void on_con_selected_button       (GtkCList *list,
										  gint row,
										  gint col,
										  GdkEventButton *event,
										  GnomeCmdFileSelector *fs)
{
	g_return_if_fail (GTK_IS_CLIST (list));
	g_return_if_fail (event != NULL);
	g_return_if_fail (GNOME_CMD_IS_FILE_SELECTOR (fs));
	g_return_if_fail (fs->priv != NULL);
	
	if (fs->priv->selection_lock)
		return;

	if (!event)
		return;
	
	if (event->type != GDK_BUTTON_RELEASE)
		return;

	if (!fs->priv->realized)
		return;
	
	update_connection (fs);
}


static int
on_con_selected_key (GtkWidget *widget,
					 GdkEventKey *event,
					 GnomeCmdFileSelector *fs)
{
	g_return_val_if_fail (event != NULL, 0);
	
	if (event->keyval == GDK_KP_Enter || event->keyval == GDK_Return)
	{
		update_connection (fs);
		return TRUE;
	}
	
	return FALSE;
}


static void
on_realize                               (GnomeCmdFileSelector *fs,
										  gpointer user_data)
{
	g_return_if_fail (GNOME_CMD_IS_FILE_SELECTOR (fs));
	g_return_if_fail (fs->priv != NULL);
	
	fs->priv->realized = TRUE;
	
	update_files (fs);
	update_direntry (fs);
	gnome_cmd_file_selector_update_connections (fs);

	create_column_titles (fs);
}


static void
on_dir_entry_key_pressed                 (GtkWidget *entry,
										  GdkEventKey *event,
										  GnomeCmdFileSelector *fs)
{
	g_return_if_fail (event != NULL);
	g_return_if_fail (GTK_IS_ENTRY (entry));
	
	if (event->keyval == GDK_Return)	
		goto_directory (fs, gtk_entry_get_text (GTK_ENTRY (entry)));	
}


static gint
on_list_button_press (GtkWidget *widget, GdkEventButton *event, GnomeCmdFileSelector *fs)
{
	g_return_val_if_fail (widget != NULL, FALSE);
	g_return_val_if_fail (event != NULL, FALSE);
	
	if (event->type == GDK_BUTTON_PRESS)
	{
		if (event->button == 3)
		{
			gint row;
			GtkCList *clist = GTK_CLIST (fs->list);			
			GnomeCmdFile *finfo;
				
			row = gnome_cmd_file_selector_get_row (fs, event->x, event->y);
			if (row < 0)
				return FALSE;
			
			finfo = (GnomeCmdFile*)gtk_clist_get_row_data (clist, row+1);
			
			if (gnome_cmd_data_get_right_mouse_button_mode() == RIGHT_BUTTON_SELECTS) {
				
				if (g_list_index (fs->priv->selected_files, finfo) == -1) {
					do_select_file (fs, finfo);
					fs->priv->right_mouse_sel_state = 1;
				}
				else {
					do_unselect_file (fs, finfo);
					fs->priv->right_mouse_sel_state = 0;
				}				
			}
			else {
				/* create the popup menu */
				GList *files = gnome_cmd_file_selector_get_selected_files (fs);
				GtkWidget *menu = gnome_cmd_file_popmenu_new (files);
				gtk_widget_ref (menu);
				gtk_object_set_data_full (GTK_OBJECT (fs),
										  "file_popup_menu", menu,
										  (GtkDestroyNotify)gtk_widget_unref);

				focus_file_at_row (fs, row+1);
				gtk_clist_sort(GTK_CLIST(fs->list));
				gnome_cmd_file_selector_set_active (fs, TRUE);
				
				gtk_menu_popup (GTK_MENU(menu), NULL, NULL, NULL, NULL, 
								event->button, event->time);
			}
			return TRUE;
		}
		else if (event->button == 2) {
			goto_directory (fs, "..");			
		}
	}

	return FALSE;
}


static void
on_list_motion_notify                     (GtkCList *clist,
										   GdkEventMotion *event,
										   GnomeCmdFileSelector *fs)
{
	g_return_if_fail (event != NULL);

	if (event->state & GDK_BUTTON3_MASK) {
		gint row;
				
		row = gnome_cmd_file_selector_get_row (fs, event->x, event->y);
		
		if (row != -1) {
			GnomeCmdFile *finfo = (GnomeCmdFile*)gtk_clist_get_row_data (clist, row+1);
			if (finfo) {
				gnome_cmd_file_selector_select_row (fs, row+1);
				if (fs->priv->right_mouse_sel_state)
					do_select_file (fs, finfo);
				else
					do_unselect_file (fs, finfo);
			}
		}		
	}
}

static void
on_list_row_selected                     (GtkCList *clist,
										  gint row,
										  gint column,
										  GdkEventButton *event,
										  GnomeCmdFileSelector *fs)
{
	GnomeCmdFile *finfo;

	// not using g_return_if_fail (event != NULL); since NULL is normal
	// when switching focus with TAB
	if (!event)
		return; 

	fs->priv->cur_file = row;
	
	if (event->type == GDK_2BUTTON_PRESS)
	{
		finfo = (GnomeCmdFile*)gtk_clist_get_row_data (clist, row);
		do_file_specific_action (fs, finfo);
	}
	else
	{
		if (event->state & GDK_SHIFT_MASK)
			select_file_range (fs, fs->priv->shift_down_row, row);
	}
}

/* This function should set x and y to the position of the current focus_row
 * 
 */
static void
cb_determinate_popmenu_pos (GtkMenu *menu, gint *x, gint *y, GnomeCmdFileSelector *fs) {
	
}


static void
on_list_key_pressed                      (GtkCList *clist,
										  GdkEventKey *event,
										  GnomeCmdFileSelector *fs)
{
	gint num_files;
	
	g_return_if_fail (GTK_IS_CLIST (clist));
	g_return_if_fail (event != NULL);
	g_return_if_fail (GNOME_CMD_IS_FILE_SELECTOR (fs));
	g_return_if_fail (fs->priv != NULL);

	num_files = get_num_files (fs);

	if (event->state & GDK_SHIFT_MASK) {
		if (event->keyval == GDK_Left
			|| event->keyval == GDK_Right) {
			event->state -= GDK_SHIFT_MASK;
			return;
		}
	}
	
	switch (event->keyval)
	{
		case GDK_Left:
		case GDK_BackSpace:
			goto_directory (fs, "..");
			break;
			
		case GDK_Home:
			GTK_CLIST (fs->list)->focus_row = 0;
			on_list_scroll_vertical (GTK_CLIST (fs->list), GTK_SCROLL_JUMP, 0, fs);
			gtk_clist_moveto (GTK_CLIST (fs->list), 0, 0, 0, 0);
			break;

		case GDK_End:
			GTK_CLIST (fs->list)->focus_row = num_files;
			on_list_scroll_vertical (GTK_CLIST (fs->list), GTK_SCROLL_JUMP, 0, fs);
			gtk_clist_moveto (GTK_CLIST (fs->list), num_files, 0, 1, 0);
			break;			
		
		case GDK_Right:
			/* prevent the list from scrolling right by changing the keyval */
			event->keyval = GDK_Return;
		case GDK_Return:
		{
			gpointer row_data = gtk_clist_get_row_data (
				clist, clist->focus_row);
			if (row_data)
			{
				GnomeCmdFile *finfo = (GnomeCmdFile*)row_data;			
				do_file_specific_action (fs, finfo);
			}
		}		
		break;

		case GDK_Shift_L:
		case GDK_Shift_R:
			fs->priv->shift_cnt++;
			fs->priv->shift_down_row = clist->focus_row;
			break;

		case GDK_Menu:
		{
			GList *files = gnome_cmd_file_selector_get_selected_files (fs);
			GtkWidget *menu = gnome_cmd_file_popmenu_new (files);
			gtk_menu_popup (GTK_MENU(menu), NULL, NULL, NULL, NULL, 
							3, event->time);
		}
		break;
	}

	if (event->state == 0) {
		if ((event->keyval >= GDK_A && event->keyval <= GDK_Z)
			|| (event->keyval >= GDK_a && event->keyval <= GDK_z)) {
			focus_first_file_that_starts_with (fs, (gchar)event->keyval);
		}
	}
	
	/* brutal hack to prevent the list from locking up when shift is pressed */
	if (event->state & GDK_SHIFT_MASK)
		event->state -= GDK_SHIFT_MASK;
}


static gboolean
on_list_key_release                  (GtkCList      *clist,
									  GdkEventKey   *event,
									  GnomeCmdFileSelector *fs)
{
	g_return_val_if_fail (event != NULL, FALSE);
	g_return_val_if_fail (GTK_IS_CLIST (clist), FALSE);
	
    switch (event->keyval)
    {
		case GDK_Shift_R:
        case GDK_Shift_L:
			fs->priv->shift_cnt--;
            gtk_clist_select_row (clist, clist->focus_row, 0);
            break;
    }
    
    return FALSE;
}


static void
on_list_scroll_vertical                  (GtkCList        *clist,
										  GtkScrollType    scroll_type,
										  gfloat           position,
										  GnomeCmdFileSelector *fs)
{
	g_return_if_fail (GTK_IS_CLIST (clist));
	g_return_if_fail (GNOME_CMD_IS_FILE_SELECTOR (fs));
	g_return_if_fail (fs->priv != NULL);
	
    if (fs->priv->shift_cnt > 0)
    {    	
        int start_row = fs->priv->cur_file;
        int end_row = clist->focus_row;

		if (start_row < 0 || end_row < 0)
			return;
        
        if (scroll_type == GTK_SCROLL_STEP_BACKWARD || scroll_type == GTK_SCROLL_STEP_FORWARD)
			select_file_at_row (fs, start_row);
        else
			select_file_range (fs, start_row, end_row);
    }

	fs->priv->cur_file = clist->focus_row;
    gtk_clist_select_row (clist, clist->focus_row, 0);
}
							   

static void
on_root_btn_clicked                      (GtkButton *button,
										  GnomeCmdFileSelector *fs)
{
	gnome_cmd_file_selector_set_connection (fs, gnome_cmd_connection_get_local ());
	goto_directory (fs, "/");
}


static void
on_home_btn_clicked                      (GtkButton *button,
										  GnomeCmdFileSelector *fs)
{
	gnome_cmd_file_selector_set_connection (fs, gnome_cmd_connection_get_local ());
	goto_directory (fs, g_get_home_dir ());
}


static void
on_parent_btn_clicked                    (GtkButton *button,
										  GnomeCmdFileSelector *fs)
{
	goto_directory (fs, "..");	
}


static void
on_column_clicked                        (GtkCList *list,
										  gint col,
										  GnomeCmdFileSelector *fs)
{
	g_return_if_fail (GTK_IS_CLIST (list));
	g_return_if_fail (GNOME_CMD_IS_FILE_SELECTOR (fs));
	g_return_if_fail (fs->priv != NULL);
	
	switch (col)
	{
		case 1:
			fs->priv->sort_func = (GnomeVFSListCompareFunc)sort_by_name;
			break;
		case 2:
			fs->priv->sort_func = (GnomeVFSListCompareFunc)sort_by_size;
			break;
		case 3:
			fs->priv->sort_func = (GnomeVFSListCompareFunc)sort_by_perm;
			break;
		case 4:
			fs->priv->sort_func = (GnomeVFSListCompareFunc)sort_by_date;
			break;
		case 5:
			fs->priv->sort_func = (GnomeVFSListCompareFunc)sort_by_owner;
			break;
		case 6:
			fs->priv->sort_func = (GnomeVFSListCompareFunc)sort_by_group;
			break;
		default:
			debug_print ("This column does not have a sort_func\n");
			return;
	}

	if (fs->priv->current_col == col)
		fs->priv->sort_raising[col] = !fs->priv->sort_raising[col];
	else
		gtk_pixmap_set (GTK_PIXMAP (fs->priv->column_pixmaps[fs->priv->current_col]),
						IMAGE_get_pixmap (PIXMAP_FLIST_ARROW_BLANK),
						IMAGE_get_mask (PIXMAP_FLIST_ARROW_BLANK));

	if (fs->priv->sort_raising[col])
	{
		gtk_pixmap_set (GTK_PIXMAP (fs->priv->column_pixmaps[col]),
						IMAGE_get_pixmap (PIXMAP_FLIST_ARROW_UP),
						IMAGE_get_mask (PIXMAP_FLIST_ARROW_UP));
	}
	else
	{
		gtk_pixmap_set (GTK_PIXMAP (fs->priv->column_pixmaps[col]),
						IMAGE_get_pixmap (PIXMAP_FLIST_ARROW_DOWN),
						IMAGE_get_mask (PIXMAP_FLIST_ARROW_DOWN));
	}

	fs->priv->current_col = col;
	gnome_cmd_file_selector_reload (fs);
}


/*******************************
 * Gtk class implementation
 *******************************/

static void
destroy (GtkObject *object)
{
	GnomeCmdFileSelector *fs;

	fs = GNOME_CMD_FILE_SELECTOR (object);

	gnome_cmd_file_list_free (fs->priv->shown_files);
	gnome_cmd_file_list_free (fs->priv->selected_files);
	
	gnome_cmd_dir_unref (fs->priv->cwd);	
	g_free (fs->priv);

	if (GTK_OBJECT_CLASS (parent_class)->destroy)
		(*GTK_OBJECT_CLASS (parent_class)->destroy) (object);
}

static void
map (GtkWidget *widget)
{
	if (GTK_WIDGET_CLASS (parent_class)->map != NULL)
		GTK_WIDGET_CLASS (parent_class)->map (widget);
}


static void
class_init (GnomeCmdFileSelectorClass *class)
{
	GtkObjectClass *object_class;
	GtkWidgetClass *widget_class;

	object_class = GTK_OBJECT_CLASS (class);
	widget_class = GTK_WIDGET_CLASS (class);

	parent_class = gtk_type_class (gtk_vbox_get_type ());

	object_class->destroy = destroy;

	widget_class->map = map;
}

static void
init (GnomeCmdFileSelector *fs)
{
	int i;
	GtkVBox *vbox;

	fs->priv = g_new (GnomeCmdFileSelectorPrivate, 1);
	fs->priv->shown_files = NULL;
	fs->priv->selected_files = NULL;
	fs->priv->shift_cnt = 0;
	fs->priv->realized = FALSE;
	fs->priv->selection_lock = FALSE;
	fs->priv->sel_first_file = TRUE;
	fs->priv->sort_func = (GnomeVFSListCompareFunc)sort_by_name;
	fs->priv->current_col = 1;
	fs->priv->cwd = NULL;

	for ( i=0 ; i<LIST_NUM_COLUMNS ; i++ )
		fs->priv->sort_raising[i] = FALSE;

	
	vbox = GTK_VBOX (fs);

	/* create the box used for packing the dir_entry and buttons */
	fs->hbox = gtk_hbox_new (FALSE, 2);
	gtk_widget_ref (fs->hbox);
	gtk_object_set_data_full (GTK_OBJECT (fs),
							  "hbox", fs->hbox,
							  (GtkDestroyNotify) gtk_widget_unref);
	
	/* create the root button */
	fs->root_btn = gtk_button_new_with_label ("/");
	gtk_widget_ref (fs->root_btn);
	gtk_object_set_data_full (GTK_OBJECT (fs),
							  "root_btn", fs->root_btn,
							  (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_set_usize (fs->root_btn, 16, 16);

	/* create the home button */
	fs->home_btn = gtk_button_new_with_label ("~");
	gtk_widget_ref (fs->home_btn);
	gtk_object_set_data_full (GTK_OBJECT (fs),
							  "home_btn", fs->home_btn,
							  (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_set_usize (fs->home_btn, 16, 16);

	/* create the parent dir button */
	fs->parent_btn = gtk_button_new_with_label ("..");
	gtk_widget_ref (fs->parent_btn);
	gtk_object_set_data_full (GTK_OBJECT (fs),
							  "parent_btn", fs->parent_btn,
							  (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_set_usize (fs->parent_btn, 16, 16);

	/* create the connection combo */
	fs->con_combo = gnome_cmd_combo_new ();
	gtk_widget_ref (fs->con_combo);
	gtk_object_set_data_full (GTK_OBJECT (fs),
							  "con_combo", fs->con_combo,
							  (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_set_usize (fs->con_combo, 70, -2);

	gtk_widget_ref (GNOME_CMD_COMBO (fs->con_combo)->entry);
	gtk_widget_show (GNOME_CMD_COMBO (fs->con_combo)->entry);
	gtk_entry_set_editable (GTK_ENTRY (GNOME_CMD_COMBO (fs->con_combo)->entry), FALSE);
	

	/* create the directory entry */
	fs->dir_entry = gtk_entry_new ();
	gtk_widget_ref (fs->dir_entry);
	gtk_object_set_data_full (GTK_OBJECT (fs),
							  "dir_entry", fs->dir_entry,
							  (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_set_style (fs->dir_entry, list_style);

	/* create the scrollwindow that we'll place the list in */
	fs->scrolledwindow = gtk_scrolled_window_new (NULL, NULL);
	gtk_widget_ref (fs->scrolledwindow);
	gtk_object_set_data_full (GTK_OBJECT (fs),
							  "scrolledwindow", fs->scrolledwindow,
							  (GtkDestroyNotify) gtk_widget_unref);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (fs->scrolledwindow),
									GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
  
	/* create the list */
	fs->list = gtk_clist_new (LIST_NUM_COLUMNS);
	gtk_widget_ref (fs->list);
	gtk_object_set_data_full (GTK_OBJECT (fs), "list", fs->list,
							  (GtkDestroyNotify) gtk_widget_unref);
	gtk_clist_set_column_justification (GTK_CLIST (fs->list), 2, GTK_JUSTIFY_RIGHT);
	gtk_widget_set_style (fs->list, list_style);

	/* create the info label */
	fs->info_label = gtk_label_new ("empty");
	gtk_widget_ref (fs->info_label);
	gtk_object_set_data_full (GTK_OBJECT (fs), "info_label", fs->info_label,
							  (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_set_style (fs->info_label, main_style);
	

	/* pack the widgets */
	gtk_box_pack_start (GTK_BOX (vbox), fs->hbox, FALSE, FALSE, 0);
	gtk_container_add (GTK_CONTAINER (fs->scrolledwindow), fs->list);
	gtk_box_pack_start (GTK_BOX (vbox), fs->scrolledwindow, TRUE, TRUE, 0);
	gtk_box_pack_start (GTK_BOX (vbox), fs->info_label, FALSE, TRUE, 0);

	gtk_box_pack_start (GTK_BOX (fs->hbox), fs->root_btn, FALSE, FALSE, 0);
	gtk_box_pack_start (GTK_BOX (fs->hbox), fs->home_btn, FALSE, FALSE, 0);
	gtk_box_pack_start (GTK_BOX (fs->hbox), fs->parent_btn, FALSE, FALSE, 0);
	gtk_box_pack_start (GTK_BOX (fs->hbox), fs->con_combo, FALSE, FALSE, 0);
	gtk_box_pack_start (GTK_BOX (fs->hbox), fs->dir_entry, TRUE, TRUE, 0);


	/* initialize dnd */
	gnome_cmd_file_selector_init_dnd (fs);
	
	/* connect signals */
	gtk_signal_connect (GTK_OBJECT (GNOME_CMD_COMBO (fs->con_combo)->list),
						"key-press-event",
						GTK_SIGNAL_FUNC (on_con_selected_key), fs);
	gtk_signal_connect (GTK_OBJECT (GNOME_CMD_COMBO (fs->con_combo)->list),
						"select-row",
						GTK_SIGNAL_FUNC (on_con_selected_button), fs);
	gtk_signal_connect (GTK_OBJECT (fs), "realize",
						GTK_SIGNAL_FUNC (on_realize), fs);
	gtk_signal_connect (GTK_OBJECT (fs->dir_entry), "key-press-event",
						GTK_SIGNAL_FUNC (on_dir_entry_key_pressed), fs);
	
	gtk_signal_connect (GTK_OBJECT (fs->list), "select-row",
						GTK_SIGNAL_FUNC (on_list_row_selected), fs);
	gtk_signal_connect (GTK_OBJECT (fs->list), "button-press-event",
						GTK_SIGNAL_FUNC (on_list_button_press), fs);
	gtk_signal_connect (GTK_OBJECT (fs->list), "motion-notify-event",
						GTK_SIGNAL_FUNC (on_list_motion_notify), fs);	
	gtk_signal_connect_after (GTK_OBJECT (fs->list), "scroll_vertical",
							  GTK_SIGNAL_FUNC (on_list_scroll_vertical), fs);
	
	gtk_signal_connect (GTK_OBJECT (fs->list), "key-press-event",
						GTK_SIGNAL_FUNC (on_list_key_pressed), fs);
	gtk_signal_connect (GTK_OBJECT (fs->list), "key-release-event",
						GTK_SIGNAL_FUNC (on_list_key_release), fs);
	
	gtk_signal_connect (GTK_OBJECT (fs->home_btn), "clicked",
						GTK_SIGNAL_FUNC (on_home_btn_clicked), fs);
	gtk_signal_connect (GTK_OBJECT (fs->root_btn), "clicked",
						GTK_SIGNAL_FUNC (on_root_btn_clicked), fs);
	gtk_signal_connect (GTK_OBJECT (fs->parent_btn), "clicked",
						GTK_SIGNAL_FUNC (on_parent_btn_clicked), fs);
	gtk_signal_connect (GTK_OBJECT (fs->list), "click-column",
						GTK_SIGNAL_FUNC (on_column_clicked), fs);

	 
	/* show the widgets */
	gtk_widget_show (GTK_WIDGET (vbox));
	gtk_widget_show (fs->hbox);
	gtk_widget_show (fs->root_btn);
	gtk_widget_show (fs->home_btn);
	gtk_widget_show (fs->parent_btn);
	gtk_widget_show (fs->dir_entry);
	gtk_widget_show (fs->scrolledwindow);
	gtk_widget_show (fs->con_combo);
	gtk_widget_show (fs->list);
	gtk_widget_show (fs->info_label);

	gtk_widget_grab_focus (fs->list);
}






/***********************************
 * Public functions
 ***********************************/


GtkType
gnome_cmd_file_selector_get_type         (void)
{
	static GtkType fs_type = 0;

	if (fs_type == 0)
	{
		GtkTypeInfo fs_info =
		{
			"GnomeCmdFileSelector",
			sizeof (GnomeCmdFileSelector),
			sizeof (GnomeCmdFileSelectorClass),
			(GtkClassInitFunc) class_init,
			(GtkObjectInitFunc) init,
			/* reserved_1 */ NULL,
			/* reserved_2 */ NULL,
			(GtkClassInitFunc) NULL
		};

		fs_type = gtk_type_unique (gtk_vbox_get_type (), &fs_info);
	}
	return fs_type;
}


GtkWidget *
gnome_cmd_file_selector_new              (void)
{
	GnomeCmdFileSelector *fs;

	fs = gtk_type_new (gnome_cmd_file_selector_get_type ());

	return GTK_WIDGET (fs);
}


GnomeCmdDir*
gnome_cmd_file_selector_get_directory    (GnomeCmdFileSelector *fs)
{
	g_return_val_if_fail (GNOME_CMD_IS_FILE_SELECTOR (fs), NULL);
	g_return_val_if_fail (fs->priv != NULL, NULL);
	
	return fs->priv->cwd;
}


void
gnome_cmd_file_selector_reload           (GnomeCmdFileSelector *fs)
{
	GnomeCmdDir *dir;

	g_return_if_fail (GNOME_CMD_IS_FILE_SELECTOR (fs));
	g_return_if_fail (fs->priv != NULL);

	gnome_cmd_file_selector_unselect_all (fs);
	
	dir = gnome_cmd_file_selector_get_directory (fs);
	g_return_if_fail (dir != NULL);

	gnome_cmd_dir_relist_files (dir);
	
	fs->priv->sel_first_file = FALSE;
	update_files (fs);
	fs->priv->sel_first_file = TRUE;

	update_selected_files_label (fs);
}


GnomeVFSURI*
gnome_cmd_file_selector_get_directory_uri (GnomeCmdFileSelector *fs)
{
	return gnome_cmd_dir_get_uri (gnome_cmd_file_selector_get_directory (fs));
}


void
gnome_cmd_file_selector_set_directory (GnomeCmdFileSelector *fs, GnomeCmdDir *dir)
{
	GnomeCmdConnection *con;
	
	g_return_if_fail (GNOME_CMD_IS_FILE_SELECTOR (fs));
	g_return_if_fail (fs->priv != NULL);
	g_return_if_fail (dir != NULL);

	if (fs->priv->cwd == dir)
		return;
	
	if (fs->priv->cwd)
	{
		gnome_cmd_dir_remove_observer (
			fs->priv->cwd,
			(gpointer)fs);
		gnome_cmd_dir_unref (fs->priv->cwd);
	}

	fs->priv->cwd = dir;
	gnome_cmd_dir_add_observer (
		dir,
		(GnomeCmdDirFileCreatedFunc)on_dir_file_created,
		(GnomeCmdDirFileDeletedFunc)on_dir_file_deleted,
		(GnomeCmdDirFileChangedFunc)on_dir_file_changed,
		(gpointer)fs);
	gnome_cmd_dir_ref (dir);

	con = gnome_cmd_file_selector_get_connection (fs);
	gnome_cmd_connection_set_cwd (con, dir);

	if (!fs->priv->realized)
		return;
	
	update_files (fs);
	update_direntry (fs);

	if (fs->priv->sel_first_file && fs->priv->active)
		gtk_clist_select_row (GTK_CLIST (fs->list), 0, 0);
}



void
gnome_cmd_file_selector_set_active       (GnomeCmdFileSelector *fs,
										  gboolean value)
{
	g_return_if_fail (GNOME_CMD_IS_FILE_SELECTOR (fs));
	g_return_if_fail (fs->priv != NULL);
	
	fs->priv->active = value;	
	
	if (value)
	{
		gtk_widget_grab_focus (GTK_WIDGET (fs->list));
		gtk_clist_select_row (GTK_CLIST (fs->list), GTK_CLIST (fs->list)->focus_row, 0);
	}
	else
		gtk_clist_unselect_all (GTK_CLIST (fs->list));
}



GnomeCmdFileSelectorModeType
gnome_cmd_file_selector_get_mode         (GnomeCmdFileSelector *fs)
{
	g_return_val_if_fail (GNOME_CMD_IS_FILE_SELECTOR (fs), 0);

	return fs->mode;
}


void
gnome_cmd_file_selector_set_mode         (GnomeCmdFileSelector *fs,
										  GnomeCmdFileSelectorModeType mode)
{
	g_return_if_fail (GNOME_CMD_IS_FILE_SELECTOR (fs));
	
	fs->mode = mode;
}


/******************************************************************************
*
*   Function: gnome_cmd_file_selector_get_selected_files
*
*   Purpose: Returns a list with all selected files. The list returned is
*            a copy and should be freed when no longer needed. The files
*            in the list is however not refed before returning.
*
*   Params: 
*
*   Returns: 
*
*   Statuses: 
*
******************************************************************************/

GList*
gnome_cmd_file_selector_get_selected_files (GnomeCmdFileSelector *fs)
{
	GnomeCmdFile *file;
	
	g_return_val_if_fail (GNOME_CMD_IS_FILE_SELECTOR (fs), NULL);
	g_return_val_if_fail (fs->priv != NULL, NULL);
	
	if (fs->priv->selected_files)
		return g_list_copy (fs->priv->selected_files);

	file = gnome_cmd_file_selector_get_selected_file (fs);
	if (file)
		return g_list_append (NULL, file);
		
	return NULL;
}


/******************************************************************************
*
*   Function: gnome_cmd_file_selector_get_all_files
*
*   Purpose: Returns a list with all files shown in the file-selector. The list
*            is the same as that in the file-selector it self so make a copy and ref the files
*            if needed.
*
*   Params: 
*
*   Returns: 
*
*   Statuses: 
*
******************************************************************************/

GList*
gnome_cmd_file_selector_get_all_files (GnomeCmdFileSelector *fs)
{
	g_return_val_if_fail (fs != NULL, NULL);
	g_return_val_if_fail (fs->priv != NULL, NULL);
	g_return_val_if_fail (fs->priv->shown_files != NULL, NULL);
	
	return fs->priv->shown_files;
}


/******************************************************************************
*
*   Function: gnome_cmd_file_selector_get_selected_file
*
*   Purpose: Returns the currently focused file if any. The returned file is not
*            refed
*
*   Params: 
*
*   Returns: 
*
*   Statuses: 
*
******************************************************************************/

GnomeCmdFile*
gnome_cmd_file_selector_get_selected_file (GnomeCmdFileSelector *fs)
{
	GnomeCmdFile *finfo;
	
	g_return_val_if_fail (fs != NULL, NULL);
	g_return_val_if_fail (GTK_IS_CLIST (fs->list), NULL);
	
	if (GTK_CLIST (fs->list)->focus_row < 0)
		return NULL;

	finfo = (GnomeCmdFile*)gtk_clist_get_row_data (
		GTK_CLIST (fs->list),
		GTK_CLIST (fs->list)->focus_row);
	
	return finfo;
}


void
gnome_cmd_file_selector_set_connection (GnomeCmdFileSelector *fs, GnomeCmdConnection *con)
{
	g_return_if_fail (fs != NULL);
	g_return_if_fail (fs->priv != NULL);
	g_return_if_fail (con != NULL);

	if (fs->priv->con == con)
		return;
	
	fs->priv->con = con;
	gnome_cmd_file_selector_set_directory (fs, gnome_cmd_connection_get_cwd (con));

	update_selected_files_label (fs);
	
	if (!fs->priv->realized)
		return;
	
	update_files (fs);
	
	gnome_cmd_combo_set_selection (
		GNOME_CMD_COMBO (fs->con_combo),
		gnome_cmd_connection_get_alias (con));
}


void
gnome_cmd_file_selector_update_connections (GnomeCmdFileSelector *fs)
{
	GList *connections;
	gboolean found_my_con = FALSE;

	g_return_if_fail (fs != NULL);
	g_return_if_fail (fs->priv != NULL);
	
	if (!fs->priv->realized)
		return;
	
	fs->priv->selection_lock = TRUE;

	gnome_cmd_combo_clear (GNOME_CMD_COMBO (fs->con_combo));
	
	connections = gnome_cmd_connection_get_list ();
	while (connections)
	{
		gchar *text[3];
		GnomeCmdConnection *con = (GnomeCmdConnection*)connections->data;

		if (con == fs->priv->con)
			found_my_con = TRUE;
		
		text[0] = con->alias;
		text[1] = con->desc;
		text[2] = NULL;		

		gnome_cmd_combo_append (GNOME_CMD_COMBO (fs->con_combo), text);
		
		connections = connections->next;
	}

	fs->priv->selection_lock = FALSE;

	/* If the connection of this fileselector is no longer available use the local connection */
	if (!found_my_con)
		gnome_cmd_file_selector_set_connection (fs, gnome_cmd_connection_get_local ());
	else
		gnome_cmd_combo_set_selection (GNOME_CMD_COMBO (fs->con_combo),
									   gnome_cmd_connection_get_alias (fs->priv->con));
}


GnomeCmdConnection*
gnome_cmd_file_selector_get_connection (GnomeCmdFileSelector *fs)
{
	g_return_val_if_fail (fs != NULL, NULL);
	g_return_val_if_fail (fs->priv != NULL, NULL);
	
	return fs->priv->con;
}


void
gnome_cmd_file_selector_select_all (GnomeCmdFileSelector *fs)
{
	GList *tmp;

	g_return_if_fail (fs != NULL);
	g_return_if_fail (fs->priv != NULL);

	gnome_cmd_file_list_free (fs->priv->selected_files);
	fs->priv->selected_files = NULL;

	tmp = gnome_cmd_file_selector_get_all_files (fs);
	while (tmp) {
		GnomeCmdFile *finfo = (GnomeCmdFile*)tmp->data;
		do_select_file (fs, finfo);
		tmp = tmp->next;
	}
	update_selected_files_label (fs);
}


void
gnome_cmd_file_selector_unselect_all (GnomeCmdFileSelector *fs)
{
	GList *tmp;

	g_return_if_fail (fs != NULL);
	g_return_if_fail (fs->priv != NULL);
	
	tmp = g_list_copy (fs->priv->selected_files);
	while (tmp)	{
		GnomeCmdFile *finfo = (GnomeCmdFile*)tmp->data;
		do_unselect_file (fs, finfo);
		tmp = tmp->next;
	}

	gnome_cmd_file_list_free (fs->priv->selected_files);
	fs->priv->selected_files = NULL;
	g_list_free (tmp);
	update_selected_files_label (fs);
}


void
gnome_cmd_file_selector_toggle (GnomeCmdFileSelector *fs)
{
	GtkCList *clist;
	GnomeCmdFile *finfo;
	
	g_return_if_fail (fs != NULL);
	g_return_if_fail (fs->list != NULL);

	clist = GTK_CLIST (fs->list);
	finfo = (GnomeCmdFile*)gtk_clist_get_row_data (clist, clist->focus_row);
	if (finfo)
		select_file (fs, finfo);
}


void
gnome_cmd_file_selector_toggle_and_step (GnomeCmdFileSelector *fs)
{
	GtkCList *clist;
	GnomeCmdFile *finfo;
	gint num_files;
	
	g_return_if_fail (fs != NULL);
	g_return_if_fail (fs->list != NULL);

	clist = GTK_CLIST (fs->list);
	num_files = get_num_files (fs);
	finfo = (GnomeCmdFile*)gtk_clist_get_row_data (clist, clist->focus_row);
	if (finfo)
		select_file (fs, finfo);
	if (clist->focus_row < num_files)
		focus_file_at_row (fs, clist->focus_row+1);
}


void
gnome_cmd_file_selector_focus_file (GnomeCmdFileSelector *fs, const gchar *focus_file)
{
	GList *tmp;

	g_return_if_fail (fs != NULL);
	g_return_if_fail (fs->priv != NULL);
	g_return_if_fail (fs->list != NULL);

	tmp = gnome_cmd_file_selector_get_all_files (fs);
	while (tmp)
	{
		gint row;
		
		GnomeCmdFile *finfo = (GnomeCmdFile*)tmp->data;
		g_return_if_fail (finfo != NULL);
		g_return_if_fail (finfo->info != NULL);
		
		row = gtk_clist_find_row_from_data (GTK_CLIST(fs->list), finfo);
		if (row == -1)
			return;
		
		if (strcmp (finfo->info->name, focus_file) == 0)
		{
			fs->priv->cur_file = row;
			gnome_cmd_file_selector_select_row (fs, row);
			gtk_clist_moveto (GTK_CLIST (fs->list), row, 0, 0, 0);
			return;
		}
		tmp = tmp->next;
	}
}


void
gnome_cmd_file_selector_update_style (GnomeCmdFileSelector *fs)
{
	gint i;
	
	g_return_if_fail (fs != NULL);
	g_return_if_fail (fs->priv != NULL);

	for ( i=1 ; i<LIST_NUM_COLUMNS ; i++ )
		gtk_widget_set_style (fs->priv->column_labels[i], main_style);
	
	gtk_widget_set_style (fs->dir_entry, list_style);
	gtk_widget_set_style (fs->list, list_style);
	gtk_widget_set_style (fs->info_label, main_style);
	gnome_cmd_combo_update_style (GNOME_CMD_COMBO (fs->con_combo));
	
	update_files (fs);
}


static GList *
gnome_vfs_list_sort_merge (GList *l1,
                          GList *l2,
                          GnomeVFSListCompareFunc compare_func,
                          gpointer data)
{
	GList list, *l, *lprev;

	l = &list;
	lprev = NULL;

	while (l1 && l2) {
		if (compare_func (l1->data, l2->data, data) < 0) {
			l->next = l1;
			l = l->next;
			l->prev = lprev;
			lprev = l;
			l1 = l1->next;
		} else {
			l->next = l2;
			l = l->next;
			l->prev = lprev;
			lprev = l;
			l2 = l2->next;
		}
	}

	l->next = l1 ? l1 : l2;
	l->next->prev = l;

	return list.next;
}


GList *
gnome_vfs_list_sort (GList *list,
					 GnomeVFSListCompareFunc compare_func,
					 gpointer data)
{
	GList *l1, *l2;

	if (!list)
		return NULL;
	if (!list->next)
		return list;

	l1 = list;
	l2 = list->next;

	while ((l2 = l2->next) != NULL) {
		if ((l2 = l2->next) == NULL)
			break;
		l1 = l1->next;
	}

	l2 = l1->next;
	l1->next = NULL;

	return gnome_vfs_list_sort_merge
		(gnome_vfs_list_sort (list, compare_func, data),
		 gnome_vfs_list_sort (l2, compare_func, data),
		 compare_func,
		 data);
}


gint
gnome_cmd_file_selector_get_row (GnomeCmdFileSelector *fs, gint x, gint y)
{
	gint row;
	GtkCList *clist;

	g_return_val_if_fail (fs != NULL, -1);
	
	clist = GTK_CLIST (fs->list);

	/* Find the row that was clicked
	 *
	 */				
	y -= (GTK_CONTAINER (clist)->border_width
		  - clist->column_title_area.y
		  + clist->column_title_area.height);
				
	if (gtk_clist_get_selection_info (clist, x, y, &row, NULL) == 0)
		row = -1;

	return row;
}


void
gnome_cmd_file_selector_select_row (GnomeCmdFileSelector *fs, gint row)
{
	GtkCList *clist;

	g_return_if_fail (fs != NULL);
	
	clist = GTK_CLIST (fs->list);

	clist->focus_row = row;
	gtk_clist_select_row (clist, row, 0);
}

