/*
    GNOME Commander - A GNOME based file manager 
    Copyright (C) 2001-2002 Marcus Bjurman

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/ 
#include <config.h>
#include "gnome-cmd-includes.h"


GList *con_list = NULL;
GnomeCmdConnection *local = NULL;


static void
update_connections (void)
{
	GnomeCmdFileSelector *fs, *fs2;
	
	fs = gnome_cmd_main_win_get_active_fs (main_win);
	fs2 = gnome_cmd_main_win_get_inactive_fs (main_win);

	gnome_cmd_file_selector_update_connections (fs);
	gnome_cmd_file_selector_update_connections (fs2);
}

static void
add_connection (GnomeCmdConnection *connection)
{	
	g_return_if_fail (connection != NULL);
	
	con_list = g_list_append (con_list, connection);
	update_connections ();
}


static void
remove_connection (GnomeCmdConnection *connection)
{
	g_return_if_fail (connection != NULL);
	g_return_if_fail (con_list != NULL);
	
	con_list = g_list_remove (con_list, connection);
	update_connections ();
}


GList*
gnome_cmd_connection_get_list (void)
{
	return con_list;
}


void
gnome_cmd_connection_create_local (void)
{
	gchar *home_dir;
	
	local = g_new (GnomeCmdConnection, 1);

	local->type = CON_TYPE_LOCAL;
	local->alias = g_strdup (_("local"));
	local->base_uri = gnome_vfs_uri_new ("file://");
	home_dir = g_get_home_dir ();
	local->cwd = gnome_cmd_dir_new_with_values (
		gnome_cmd_connection_get_baseuri (local),
		home_dir);
	gnome_cmd_dir_ref (local->cwd);
	local->desc = g_strdup (_("local filesystem"));

	add_connection (local);
}


GnomeCmdConnection*
gnome_cmd_connection_get_local (void)
{
	return local;
}


GnomeCmdConnection*
gnome_cmd_connection_new_ftp (const gchar *alias,
							  const gchar *host,
							  gshort port,
							  const gchar *user,
							  const gchar *pw_in)
{
	GnomeCmdConnection *con;
	gchar *pw;
	
	g_return_val_if_fail (host != NULL, NULL);
	g_return_val_if_fail (user != NULL, NULL);

	con = g_new (GnomeCmdConnection, 1);
	g_return_val_if_fail (con != NULL, NULL);
	
	con->type = CON_TYPE_FTP;
	con->base_uri = gnome_vfs_uri_new ("ftp://");
	con->alias = g_strdup (alias);
	con->host = g_strdup (host);
	con->desc = g_strdup ("ftp connection");

	gnome_vfs_uri_set_host_name (con->base_uri, host);
	gnome_vfs_uri_set_host_port (con->base_uri, port);
	gnome_vfs_uri_set_user_name (con->base_uri, user);

	if (strcmp (user, "anonymous") == 0)	
		pw = gnome_vfs_escape_string (gnome_cmd_data_get_ftp_anonymous_password ());
	else
		pw = gnome_vfs_escape_string (pw_in);
	
	gnome_vfs_uri_set_password (con->base_uri, pw);
	
	con->cwd = gnome_cmd_dir_new_with_values (
		gnome_cmd_connection_get_baseuri (con),
		"/");
	gnome_cmd_dir_ref (con->cwd);
	
	add_connection (con);
	g_free (pw);

	return con;
}


static void
do_free_connection (GnomeCmdConnection *con)
{
	g_return_if_fail (con != NULL);
	g_return_if_fail (con->base_uri != NULL);
	g_return_if_fail (con->cwd != NULL);
	
	gnome_cmd_dir_unref (con->cwd);
	gnome_vfs_uri_unref (con->base_uri);
	g_free (con->alias);
	g_free (con->desc);
	g_free (con);
}


void
gnome_cmd_connection_free (GnomeCmdConnection *con)
{	
	remove_connection (con);
	do_free_connection (con);
}


void
gnome_cmd_connection_free_all (void)
{
	g_list_foreach (con_list, (GFunc)do_free_connection, NULL);
	g_list_free (con_list);
	con_list = NULL;
}


GnomeCmdDir *
gnome_cmd_connection_get_cwd (GnomeCmdConnection *con)
{
	g_return_val_if_fail (con != NULL, NULL);

	return con->cwd;
}


GnomeVFSURI *
gnome_cmd_connection_get_baseuri (GnomeCmdConnection *con)
{
	g_return_val_if_fail (con != NULL, NULL);

	return con->base_uri;
}


void
gnome_cmd_connection_set_cwd (GnomeCmdConnection *con,
							  GnomeCmdDir *dir)
{
	g_return_if_fail (con != NULL);
	g_return_if_fail (dir != NULL);

	if (con->cwd == dir)
		return;

	if (con->cwd != NULL)
		gnome_cmd_dir_unref (con->cwd);
	
	gnome_cmd_dir_ref (dir);
	con->cwd = dir;
	
	gnome_cmd_dir_list_files (con->cwd); //FIXME: Is this needed?
}


const gchar *
gnome_cmd_connection_get_alias (GnomeCmdConnection *con)
{
	g_return_val_if_fail (con != NULL, NULL);

	return con->alias;
}


const gchar *
gnome_cmd_connection_get_desc (GnomeCmdConnection *con)
{
	g_return_val_if_fail (con != NULL, NULL);

	return con->desc;
}


