/*
    GNOME Commander - A GNOME based file manager 
    Copyright (C) 2001-2002 Marcus Bjurman

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/ 
#include <config.h>
#include <sys/types.h>
#include <unistd.h>
#include <errno.h>
#include "gnome-cmd-includes.h"


struct _GnomeCmdChownDialogPrivate
{
	GList *files;
	GtkWidget *user_combo, *group_combo;
	GtkWidget *user_label, *group_label;
	GtkWidget *recursive_check;
	GList *user_strings;
	GList *group_strings;
};

static GnomeDialogClass *parent_class = NULL;


static void do_chown (GnomeCmdFile *in_finfo, uid_t uid, gid_t gid, gboolean recursive)
{
	GnomeVFSResult ret;

	g_return_if_fail (in_finfo != NULL);
	g_return_if_fail (in_finfo->info != NULL);

	ret = gnome_cmd_file_chown (in_finfo, uid, gid);

	if (ret != GNOME_VFS_OK) {
		gchar *msg = g_strdup_printf (
			_("Could not chown %s\n%s"),
			gnome_cmd_file_get_path (in_finfo),
			gnome_vfs_result_to_string (ret));
		gnome_error_dialog (msg);
		g_free (msg);
	}
	else if (!recursive) {
		return;
	}
	
	if (in_finfo->info->type == GNOME_VFS_FILE_TYPE_DIRECTORY) {
		const gchar *dir_uri = gnome_cmd_file_get_uri_str (in_finfo);
		GnomeCmdDir *dir = dir_pool_get (dir_uri);
		GList *files, *tmp;

		gnome_cmd_dir_ref (dir);
		files = tmp = gnome_cmd_dir_list_files (dir);
			
		while (tmp) {
			GnomeCmdFile *finfo = (GnomeCmdFile*)tmp->data;
			if (strcmp (finfo->info->name, ".") != 0
				&& strcmp (finfo->info->name, "..") != 0
				&& finfo->info->flags != GNOME_VFS_FILE_FLAGS_SYMLINK) {
				do_chown (finfo, uid, gid, TRUE);
			}
			
			tmp = tmp->next;
		}
		gnome_cmd_dir_unref (dir);
	}
}


static void on_ok (GtkButton *button, GnomeCmdChownDialog *dialog)
{
	uid_t uid = -1;
	gid_t gid = -1;	
	gchar *owner;
	gchar *group;
	GList *tmp;
	user_t *user;
	gboolean recursive;
	GnomeCmdChownDialogPrivate *data = dialog->priv;

	user = OWNER_get_program_user();
	
	if (user && user->uid == 0) {		
		owner = gtk_entry_get_text (GTK_ENTRY (GTK_COMBO (data->user_combo)->entry));
		g_return_if_fail (owner != NULL);		
		uid = OWNER_get_uid_by_name (owner);
	}

	group = gtk_entry_get_text (GTK_ENTRY (GTK_COMBO (data->group_combo)->entry));
	g_return_if_fail (group != NULL);	
	gid = OWNER_get_gid_by_name (group);
	
	g_return_if_fail (gid != -1);

	recursive = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->recursive_check));
	
	tmp = data->files;
	while (tmp) {
		GnomeCmdFile *finfo = (GnomeCmdFile*)tmp->data;
		g_return_if_fail (finfo != NULL);
		
		if (GNOME_VFS_FILE_INFO_LOCAL (finfo->info))
			do_chown (finfo, uid, gid, recursive);
		
		tmp = tmp->next;
	}

	gnome_cmd_file_list_free (dialog->priv->files);
	gtk_widget_hide (GTK_WIDGET (dialog));
}


static void on_cancel (GtkButton *button, GnomeCmdChownDialog *dialog)
{
	gnome_cmd_file_list_free (dialog->priv->files);
	gtk_widget_hide (GTK_WIDGET (dialog));
}




/*******************************
 * Gtk class implementation
 *******************************/

static void
destroy (GtkObject *object)
{
	GnomeCmdChownDialog *dialog = GNOME_CMD_CHOWN_DIALOG (object);

	g_free (dialog->priv);
	
	if (GTK_OBJECT_CLASS (parent_class)->destroy)
		(*GTK_OBJECT_CLASS (parent_class)->destroy) (object);
}


static void
map (GtkWidget *widget)
{
	if (GTK_WIDGET_CLASS (parent_class)->map != NULL)
		GTK_WIDGET_CLASS (parent_class)->map (widget);
}


static void
class_init (GnomeCmdChownDialogClass *class)
{
	GtkObjectClass *object_class;
	GtkWidgetClass *widget_class;

	object_class = GTK_OBJECT_CLASS (class);
	widget_class = GTK_WIDGET_CLASS (class);

	parent_class = gtk_type_class (gnome_dialog_get_type ());
	object_class->destroy = destroy;
	widget_class->map = map;
}


static void
init (GnomeCmdChownDialog *chown_dialog)
{
	GtkWidget *dialog_vbox1;
	GtkWidget *vbox1;
	GtkWidget *table1;
	GtkWidget *label1;
	GtkWidget *label2;
	GtkWidget *combo1;
	GtkWidget *entry1;
	GtkWidget *combo2;
	GtkWidget *entry2;
	GtkWidget *recurs_check;
	GtkWidget *button1;
	GtkWidget *button3;
	GnomeCmdChownDialogPrivate *data = g_new (GnomeCmdChownDialogPrivate, 1);
	chown_dialog->priv = data;

	
	gtk_window_set_policy (GTK_WINDOW (chown_dialog), FALSE, FALSE, FALSE);
	gtk_window_set_position (GTK_WINDOW (chown_dialog), GTK_WIN_POS_CENTER);
	gtk_window_set_title (GTK_WINDOW (chown_dialog), _("Chown"));

	dialog_vbox1 = GNOME_DIALOG (chown_dialog)->vbox;
	gtk_object_set_data (GTK_OBJECT (chown_dialog), "dialog_vbox1", dialog_vbox1);
	gtk_widget_show (dialog_vbox1);

	vbox1 = gtk_vbox_new (FALSE, 0);
	gtk_widget_ref (vbox1);
	gtk_object_set_data_full (GTK_OBJECT (chown_dialog), "vbox1", vbox1,
							  (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (vbox1);
	gtk_box_pack_start (GTK_BOX (dialog_vbox1), vbox1, TRUE, TRUE, 0);

	table1 = gtk_table_new (2, 2, FALSE);
	gtk_widget_ref (table1);
	gtk_object_set_data_full (GTK_OBJECT (chown_dialog), "table1", table1,
							  (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (table1);
	gtk_box_pack_start (GTK_BOX (vbox1), table1, TRUE, TRUE, 0);
	gtk_table_set_row_spacings (GTK_TABLE (table1), 2);
	gtk_table_set_col_spacings (GTK_TABLE (table1), 2);

	label1 = gtk_label_new (_("Owner"));
	gtk_widget_ref (label1);
	gtk_object_set_data_full (GTK_OBJECT (chown_dialog), "label1", label1,
							  (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (label1);
	gtk_table_attach (GTK_TABLE (table1), label1, 0, 1, 0, 1,
					  (GtkAttachOptions) (GTK_FILL),
					  (GtkAttachOptions) (0), 0, 0);
	gtk_misc_set_alignment (GTK_MISC (label1), 0, 0.5);

	label2 = gtk_label_new (_("Group"));
	gtk_widget_ref (label2);
	gtk_object_set_data_full (GTK_OBJECT (chown_dialog), "label2", label2,
							  (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (label2);
	gtk_table_attach (GTK_TABLE (table1), label2, 0, 1, 1, 2,
					  (GtkAttachOptions) (GTK_FILL),
					  (GtkAttachOptions) (0), 0, 0);
	gtk_misc_set_alignment (GTK_MISC (label2), 0, 0.5);

	combo1 = gtk_combo_new ();
	gtk_widget_ref (combo1);
	gtk_object_set_data_full (GTK_OBJECT (chown_dialog), "combo1", combo1,
							  (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (combo1);
	gtk_table_attach (GTK_TABLE (table1), combo1, 1, 2, 0, 1,
					  (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
					  (GtkAttachOptions) (0), 0, 0);

	entry1 = GTK_COMBO (combo1)->entry;
	gtk_widget_ref (entry1);
	gtk_object_set_data_full (GTK_OBJECT (chown_dialog), "entry1", entry1,
							  (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (entry1);

	combo2 = gtk_combo_new ();
	gtk_widget_ref (combo2);
	gtk_object_set_data_full (GTK_OBJECT (chown_dialog), "combo2", combo2,
							  (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (combo2);
	gtk_table_attach (GTK_TABLE (table1), combo2, 1, 2, 1, 2,
					  (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
					  (GtkAttachOptions) (0), 0, 0);

	entry2 = GTK_COMBO (combo2)->entry;
	gtk_widget_ref (entry2);
	gtk_object_set_data_full (GTK_OBJECT (chown_dialog), "entry2", entry2,
							  (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (entry2);

	recurs_check = gtk_check_button_new_with_label (_("Apply recursivly"));
	gtk_widget_ref (recurs_check);
	gtk_object_set_data_full (GTK_OBJECT (chown_dialog), "recurs_check", recurs_check,
							  (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (recurs_check);
	gtk_box_pack_start (GTK_BOX (vbox1), recurs_check, FALSE, FALSE, 0);

	gnome_dialog_append_button (GNOME_DIALOG (chown_dialog), GNOME_STOCK_BUTTON_OK);
	button1 = GTK_WIDGET (g_list_last (GNOME_DIALOG (chown_dialog)->buttons)->data);
	gtk_widget_ref (button1);
	gtk_object_set_data_full (GTK_OBJECT (chown_dialog), "button1", button1,
							  (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (button1);
	GTK_WIDGET_SET_FLAGS (button1, GTK_CAN_DEFAULT);

	gnome_dialog_append_button (GNOME_DIALOG (chown_dialog), GNOME_STOCK_BUTTON_CANCEL);
	button3 = GTK_WIDGET (g_list_last (GNOME_DIALOG (chown_dialog)->buttons)->data);
	gtk_widget_ref (button3);
	gtk_object_set_data_full (GTK_OBJECT (chown_dialog), "button3", button3,
							  (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (button3);
	GTK_WIDGET_SET_FLAGS (button3, GTK_CAN_DEFAULT);

	
	gtk_signal_connect (GTK_OBJECT (button1), "clicked",
						GTK_SIGNAL_FUNC (on_ok), chown_dialog);
	gtk_signal_connect (GTK_OBJECT (button3), "clicked",
						GTK_SIGNAL_FUNC (on_cancel), chown_dialog);

	data->user_combo = combo1;
	data->group_combo = combo2;
	data->recursive_check = recurs_check;
	data->user_label = label1;
	data->group_label = label2;
}




/***********************************
 * Public functions
 ***********************************/

GtkWidget*
gnome_cmd_chown_dialog_new (GList *files)
{	
	user_t *prog_user;
	GnomeCmdChownDialog *dialog = gtk_type_new (gnome_cmd_chown_dialog_get_type ());
	GnomeCmdChownDialogPrivate *data = dialog->priv;

	prog_user = OWNER_get_program_user ();
	if (!prog_user)
		return NULL;
	
	/* disable chown gui if user is not root,
	 else fill the combo with all users in the system */
	if (prog_user->uid != 0)
	{
		gtk_widget_set_sensitive (data->user_label, FALSE);
		gtk_widget_set_sensitive (data->user_combo, FALSE);
	}
	else
	{
		GList *tmp = OWNER_get_all_users ();
		GList *strings = NULL;
		
		while (tmp)
		{
			user_t *user = (user_t*)tmp->data;
			strings = g_list_append (strings, g_strdup (user->name));
			tmp = tmp->next;
		}

		gtk_combo_set_popdown_strings (GTK_COMBO (data->user_combo), strings);
		data->user_strings = strings;
	}


	/* fill the groups combo with all groups that the user is part of if ordinary user
	 or all groups if root */
	{
		GList *tmp;
		GList *strings = NULL;

		if (prog_user->uid != 0)
			tmp = prog_user->groups;
		else
			tmp = OWNER_get_all_groups ();
		
		
		while (tmp)
		{
			group_t *group = (group_t*)tmp->data;
			strings = g_list_append (strings, g_strdup (group->name));
			tmp = tmp->next;
		}

		gtk_combo_set_popdown_strings (GTK_COMBO (data->group_combo), strings);
		data->group_strings = strings;
	}

	/* set the current owner and group as default */
	{
		GnomeCmdFile *finfo = (GnomeCmdFile*)files->data;
		const gchar *owner = OWNER_get_name_by_uid (finfo->info->uid);
		const gchar *group = OWNER_get_name_by_gid (finfo->info->gid);
	
		gtk_entry_set_text (GTK_ENTRY (GTK_COMBO (data->user_combo)->entry), owner);
		gtk_entry_set_text (GTK_ENTRY (GTK_COMBO (data->group_combo)->entry), group);
	}
							
	
	data->files = gnome_cmd_file_list_copy (files);

	
	return GTK_WIDGET (dialog);
}



GtkType
gnome_cmd_chown_dialog_get_type         (void)
{
	static GtkType dlg_type = 0;

	if (dlg_type == 0)
	{
		GtkTypeInfo dlg_info =
		{
			"GnomeCmdChownDialog",
			sizeof (GnomeCmdChownDialog),
			sizeof (GnomeCmdChownDialogClass),
			(GtkClassInitFunc) class_init,
			(GtkObjectInitFunc) init,
			/* reserved_1 */ NULL,
			/* reserved_2 */ NULL,
			(GtkClassInitFunc) NULL
		};

		dlg_type = gtk_type_unique (gnome_dialog_get_type (), &dlg_info);
	}
	return dlg_type;
}



