/*
    GNOME Commander - A GNOME based file manager 
    Copyright (C) 2001-2002 Marcus Bjurman

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/ 
#include <config.h>
#include <sys/types.h>
#include <unistd.h>
#include <errno.h>
#include "gnome-cmd-includes.h"


struct _GnomeCmdChmodDialogPrivate
{
	GList *files;
	GnomeCmdFile *finfo;

	GtkWidget *or_check;
	GtkWidget *gr_check;
	GtkWidget *wr_check;
	GtkWidget *ow_check;
	GtkWidget *gw_check;
	GtkWidget *ww_check;
	GtkWidget *oe_check;
	GtkWidget *ge_check;
	GtkWidget *we_check;

	GtkWidget *textview_label;
	GnomeVFSFilePermissions perm;
};


static GnomeDialogClass *parent_class = NULL;



static void do_chmod_files (GnomeCmdChmodDialog *dialog)
{
	GnomeCmdChmodDialogPrivate *data = dialog->priv;
	GList *tmp = data->files;
	
	while (tmp)
	{
		GnomeCmdFile *finfo = (GnomeCmdFile*)tmp->data;
		gnome_cmd_file_chmod (finfo, data->perm);
		tmp = tmp->next;
	}	
}


static mode_t vfsperm_to_mode (GnomeVFSFilePermissions p)
{
	mode_t mode = 0;

	if (p & GNOME_VFS_PERM_USER_READ)
		mode += GNOME_VFS_PERM_USER_READ;
	if (p & GNOME_VFS_PERM_USER_WRITE)
		mode += GNOME_VFS_PERM_USER_WRITE;
	if (p & GNOME_VFS_PERM_USER_EXEC)
		mode += GNOME_VFS_PERM_USER_EXEC;
	if (p & GNOME_VFS_PERM_GROUP_READ)
		mode += GNOME_VFS_PERM_GROUP_READ;
	if (p & GNOME_VFS_PERM_GROUP_WRITE)
		mode += GNOME_VFS_PERM_GROUP_WRITE;
	if (p & GNOME_VFS_PERM_GROUP_EXEC)
		mode += GNOME_VFS_PERM_GROUP_EXEC;
	if (p & GNOME_VFS_PERM_OTHER_READ)
		mode += GNOME_VFS_PERM_OTHER_READ;
	if (p & GNOME_VFS_PERM_OTHER_WRITE)
		mode += GNOME_VFS_PERM_OTHER_WRITE;
	if (p & GNOME_VFS_PERM_OTHER_EXEC)
		mode += GNOME_VFS_PERM_OTHER_EXEC;

	return mode;
}


static void update_textview_perms (GnomeCmdChmodDialog *dialog)
{
	GnomeCmdChmodDialogPrivate *data = dialog->priv;
	gchar *text;

	text = g_strdup_printf ("%s%s%s%s%s%s%s%s%s", 
							(data->perm & GNOME_VFS_PERM_USER_READ)   ? "r" : "-",
							(data->perm & GNOME_VFS_PERM_USER_WRITE)  ? "w" : "-",
							(data->perm & GNOME_VFS_PERM_USER_EXEC)   ? "x" : "-",
							(data->perm & GNOME_VFS_PERM_GROUP_READ)  ? "r" : "-",
							(data->perm & GNOME_VFS_PERM_GROUP_WRITE) ? "w" : "-",
							(data->perm & GNOME_VFS_PERM_GROUP_EXEC)  ? "x" : "-",
							(data->perm & GNOME_VFS_PERM_OTHER_READ)  ? "r" : "-",
							(data->perm & GNOME_VFS_PERM_OTHER_WRITE) ? "w" : "-",
							(data->perm & GNOME_VFS_PERM_OTHER_EXEC)  ? "x" : "-");
	gtk_label_set_text (GTK_LABEL (data->textview_label), text);
	g_free (text);
}


static void show_perms (GnomeCmdChmodDialog *dialog)
{
	GnomeCmdChmodDialogPrivate *data = dialog->priv;
	mode_t p = data->perm;

	update_textview_perms (dialog);
	
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (data->or_check), (p & GNOME_VFS_PERM_USER_READ));
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (data->ow_check), (p & GNOME_VFS_PERM_USER_WRITE));
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (data->oe_check), (p & GNOME_VFS_PERM_USER_EXEC));
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (data->gr_check), (p & GNOME_VFS_PERM_GROUP_READ));
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (data->gw_check), (p & GNOME_VFS_PERM_GROUP_WRITE));
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (data->ge_check), (p & GNOME_VFS_PERM_GROUP_EXEC));
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (data->wr_check), (p & GNOME_VFS_PERM_OTHER_READ));
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (data->ww_check), (p & GNOME_VFS_PERM_OTHER_WRITE));
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (data->we_check), (p & GNOME_VFS_PERM_OTHER_EXEC));
}


static void get_perms (GnomeCmdChmodDialog *dialog)
{
	GnomeCmdChmodDialogPrivate *data = dialog->priv;
	data->perm = 0;
	
	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->or_check)))
		data->perm += GNOME_VFS_PERM_USER_READ;
	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->ow_check)))
		data->perm += GNOME_VFS_PERM_USER_WRITE;
	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->oe_check)))
		data->perm += GNOME_VFS_PERM_USER_EXEC;
	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->gr_check)))
		data->perm += GNOME_VFS_PERM_GROUP_READ;
	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->gw_check)))
		data->perm += GNOME_VFS_PERM_GROUP_WRITE;
	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->ge_check)))
		data->perm += GNOME_VFS_PERM_GROUP_EXEC;
	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->wr_check)))
		data->perm += GNOME_VFS_PERM_OTHER_READ;
	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->ww_check)))
		data->perm += GNOME_VFS_PERM_OTHER_WRITE;
	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->we_check)))
		data->perm += GNOME_VFS_PERM_OTHER_EXEC;

	update_textview_perms (dialog);
}


static void on_ok (GtkButton *button, GnomeCmdChmodDialog *dialog)
{
	do_chmod_files (dialog);
	gnome_cmd_file_list_free (dialog->priv->files);
	gtk_widget_hide (GTK_WIDGET (dialog));
}


static void on_cancel (GtkButton *button, GnomeCmdChmodDialog *dialog)
{
	gnome_cmd_file_list_free (dialog->priv->files);
	gtk_widget_hide (GTK_WIDGET (dialog));
}

static void on_toggle (GtkToggleButton *togglebutton, GnomeCmdChmodDialog *dialog)
{
	get_perms (dialog);
}





/*******************************
 * Gtk class implementation
 *******************************/

static void
destroy (GtkObject *object)
{
	GnomeCmdChmodDialog *dialog = GNOME_CMD_CHMOD_DIALOG (object);

	g_free (dialog->priv);
	
	if (GTK_OBJECT_CLASS (parent_class)->destroy)
		(*GTK_OBJECT_CLASS (parent_class)->destroy) (object);
}


static void
map (GtkWidget *widget)
{
	if (GTK_WIDGET_CLASS (parent_class)->map != NULL)
		GTK_WIDGET_CLASS (parent_class)->map (widget);
}


static void
class_init (GnomeCmdChmodDialogClass *class)
{
	GtkObjectClass *object_class;
	GtkWidgetClass *widget_class;

	object_class = GTK_OBJECT_CLASS (class);
	widget_class = GTK_WIDGET_CLASS (class);

	parent_class = gtk_type_class (gnome_dialog_get_type ());
	object_class->destroy = destroy;
	widget_class->map = map;
}


static void
init (GnomeCmdChmodDialog *dialog)
{
	GtkWidget *chmod_dialog = GTK_WIDGET (dialog);
	GnomeCmdChmodDialogPrivate *data = g_new (GnomeCmdChmodDialogPrivate, 1);
	GtkWidget *dialog_vbox2;
	GtkWidget *table2;
	GtkWidget *label3;
	GtkWidget *label4;
	GtkWidget *label5;
	GtkWidget *label6;
	GtkWidget *label7;
	GtkWidget *label8;
	GtkWidget *hseparator1;
	GtkWidget *table3;
	GtkWidget *label9;
	GtkWidget *button4;
	GtkWidget *button6;

	dialog->priv = data;

	
	gtk_window_set_policy (GTK_WINDOW (chmod_dialog), FALSE, FALSE, FALSE);
	gtk_window_set_position (GTK_WINDOW (chmod_dialog), GTK_WIN_POS_CENTER);
	gtk_window_set_title (GTK_WINDOW (chmod_dialog), _("Chmod"));

	dialog_vbox2 = GNOME_DIALOG (chmod_dialog)->vbox;
	gtk_object_set_data (GTK_OBJECT (chmod_dialog), "dialog_vbox2", dialog_vbox2);
	gtk_widget_show (dialog_vbox2);

	table2 = gtk_table_new (4, 4, FALSE);
	gtk_widget_ref (table2);
	gtk_object_set_data_full (GTK_OBJECT (chmod_dialog), "table2", table2,
							  (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (table2);
	gtk_box_pack_start (GTK_BOX (dialog_vbox2), table2, TRUE, TRUE, 0);
	gtk_table_set_row_spacings (GTK_TABLE (table2), 4);
	gtk_table_set_col_spacings (GTK_TABLE (table2), 10);

	label3 = gtk_label_new (_("Read"));
	gtk_widget_ref (label3);
	gtk_object_set_data_full (GTK_OBJECT (chmod_dialog), "label3", label3,
							  (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (label3);
	gtk_table_attach (GTK_TABLE (table2), label3, 1, 2, 0, 1,
					  (GtkAttachOptions) (GTK_FILL),
					  (GtkAttachOptions) (0), 0, 0);
	gtk_misc_set_alignment (GTK_MISC (label3), 0, 0.5);

	label4 = gtk_label_new (_("Write"));
	gtk_widget_ref (label4);
	gtk_object_set_data_full (GTK_OBJECT (chmod_dialog), "label4", label4,
							  (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (label4);
	gtk_table_attach (GTK_TABLE (table2), label4, 2, 3, 0, 1,
					  (GtkAttachOptions) (GTK_FILL),
					  (GtkAttachOptions) (0), 0, 0);
	gtk_misc_set_alignment (GTK_MISC (label4), 0, 0.5);

	label5 = gtk_label_new (_("Execute"));
	gtk_widget_ref (label5);
	gtk_object_set_data_full (GTK_OBJECT (chmod_dialog), "label5", label5,
							  (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (label5);
	gtk_table_attach (GTK_TABLE (table2), label5, 3, 4, 0, 1,
					  (GtkAttachOptions) (GTK_FILL),
					  (GtkAttachOptions) (0), 0, 0);
	gtk_misc_set_alignment (GTK_MISC (label5), 0, 0.5);

	label6 = gtk_label_new (_("Owner"));
	gtk_widget_ref (label6);
	gtk_object_set_data_full (GTK_OBJECT (chmod_dialog), "label6", label6,
							  (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (label6);
	gtk_table_attach (GTK_TABLE (table2), label6, 0, 1, 1, 2,
					  (GtkAttachOptions) (GTK_FILL),
					  (GtkAttachOptions) (0), 0, 0);
	gtk_misc_set_alignment (GTK_MISC (label6), 0, 0.5);

	label7 = gtk_label_new (_("Group"));
	gtk_widget_ref (label7);
	gtk_object_set_data_full (GTK_OBJECT (chmod_dialog), "label7", label7,
							  (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (label7);
	gtk_table_attach (GTK_TABLE (table2), label7, 0, 1, 2, 3,
					  (GtkAttachOptions) (GTK_FILL),
					  (GtkAttachOptions) (0), 0, 0);
	gtk_misc_set_alignment (GTK_MISC (label7), 0, 0.5);

	label8 = gtk_label_new (_("Others"));
	gtk_widget_ref (label8);
	gtk_object_set_data_full (GTK_OBJECT (chmod_dialog), "label8", label8,
							  (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (label8);
	gtk_table_attach (GTK_TABLE (table2), label8, 0, 1, 3, 4,
					  (GtkAttachOptions) (GTK_FILL),
					  (GtkAttachOptions) (0), 0, 0);
	gtk_misc_set_alignment (GTK_MISC (label8), 0, 0.5);

	data->or_check = gtk_check_button_new_with_label ("");
	gtk_widget_ref (data->or_check);
	gtk_object_set_data_full (GTK_OBJECT (chmod_dialog), "data->or_check", data->or_check,
							  (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (data->or_check);
	gtk_table_attach (GTK_TABLE (table2), data->or_check, 1, 2, 1, 2,
					  (GtkAttachOptions) (GTK_FILL),
					  (GtkAttachOptions) (0), 0, 0);

	data->gr_check = gtk_check_button_new_with_label ("");
	gtk_widget_ref (data->gr_check);
	gtk_object_set_data_full (GTK_OBJECT (chmod_dialog), "data->gr_check", data->gr_check,
							  (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (data->gr_check);
	gtk_table_attach (GTK_TABLE (table2), data->gr_check, 1, 2, 2, 3,
					  (GtkAttachOptions) (GTK_FILL),
					  (GtkAttachOptions) (0), 0, 0);

	data->wr_check = gtk_check_button_new_with_label ("");
	gtk_widget_ref (data->wr_check);
	gtk_object_set_data_full (GTK_OBJECT (chmod_dialog), "data->wr_check", data->wr_check,
							  (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (data->wr_check);
	gtk_table_attach (GTK_TABLE (table2), data->wr_check, 1, 2, 3, 4,
					  (GtkAttachOptions) (GTK_FILL),
					  (GtkAttachOptions) (0), 0, 0);

	data->ow_check = gtk_check_button_new_with_label ("");
	gtk_widget_ref (data->ow_check);
	gtk_object_set_data_full (GTK_OBJECT (chmod_dialog), "data->ow_check", data->ow_check,
							  (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (data->ow_check);
	gtk_table_attach (GTK_TABLE (table2), data->ow_check, 2, 3, 1, 2,
					  (GtkAttachOptions) (GTK_FILL),
					  (GtkAttachOptions) (0), 0, 0);

	data->gw_check = gtk_check_button_new_with_label ("");
	gtk_widget_ref (data->gw_check);
	gtk_object_set_data_full (GTK_OBJECT (chmod_dialog), "data->gw_check", data->gw_check,
							  (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (data->gw_check);
	gtk_table_attach (GTK_TABLE (table2), data->gw_check, 2, 3, 2, 3,
					  (GtkAttachOptions) (GTK_FILL),
					  (GtkAttachOptions) (0), 0, 0);

	data->ww_check = gtk_check_button_new_with_label ("");
	gtk_widget_ref (data->ww_check);
	gtk_object_set_data_full (GTK_OBJECT (chmod_dialog), "data->ww_check", data->ww_check,
							  (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (data->ww_check);
	gtk_table_attach (GTK_TABLE (table2), data->ww_check, 2, 3, 3, 4,
					  (GtkAttachOptions) (GTK_FILL),
					  (GtkAttachOptions) (0), 0, 0);

	data->oe_check = gtk_check_button_new_with_label ("");
	gtk_widget_ref (data->oe_check);
	gtk_object_set_data_full (GTK_OBJECT (chmod_dialog), "data->oe_check", data->oe_check,
							  (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (data->oe_check);
	gtk_table_attach (GTK_TABLE (table2), data->oe_check, 3, 4, 1, 2,
					  (GtkAttachOptions) (GTK_FILL),
					  (GtkAttachOptions) (0), 0, 0);

	data->ge_check = gtk_check_button_new_with_label ("");
	gtk_widget_ref (data->ge_check);
	gtk_object_set_data_full (GTK_OBJECT (chmod_dialog), "data->ge_check", data->ge_check,
							  (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (data->ge_check);
	gtk_table_attach (GTK_TABLE (table2), data->ge_check, 3, 4, 2, 3,
					  (GtkAttachOptions) (GTK_FILL),
					  (GtkAttachOptions) (0), 0, 0);

	data->we_check = gtk_check_button_new_with_label ("");
	gtk_widget_ref (data->we_check);
	gtk_object_set_data_full (GTK_OBJECT (chmod_dialog), "data->we_check", data->we_check,
							  (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (data->we_check);
	gtk_table_attach (GTK_TABLE (table2), data->we_check, 3, 4, 3, 4,
					  (GtkAttachOptions) (GTK_FILL),
					  (GtkAttachOptions) (0), 0, 0);

	hseparator1 = gtk_hseparator_new ();
	gtk_widget_ref (hseparator1);
	gtk_object_set_data_full (GTK_OBJECT (chmod_dialog), "hseparator1", hseparator1,
							  (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (hseparator1);
	gtk_box_pack_start (GTK_BOX (dialog_vbox2), hseparator1, TRUE, TRUE, 0);

	table3 = gtk_table_new (1, 2, FALSE);
	gtk_widget_ref (table3);
	gtk_object_set_data_full (GTK_OBJECT (chmod_dialog), "table3", table3,
							  (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (table3);
	gtk_box_pack_start (GTK_BOX (dialog_vbox2), table3, TRUE, TRUE, 0);
	gtk_table_set_row_spacings (GTK_TABLE (table3), 10);
	gtk_table_set_col_spacings (GTK_TABLE (table3), 10);

	label9 = gtk_label_new (_("Text view:"));
	gtk_widget_ref (label9);
	gtk_object_set_data_full (GTK_OBJECT (chmod_dialog), "label9", label9,
							  (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (label9);
	gtk_table_attach (GTK_TABLE (table3), label9, 0, 1, 0, 1,
					  (GtkAttachOptions) (GTK_FILL),
					  (GtkAttachOptions) (0), 0, 0);
	gtk_misc_set_alignment (GTK_MISC (label9), 0, 0.5);

	data->textview_label = gtk_label_new ("label10");
	gtk_widget_ref (data->textview_label);
	gtk_object_set_data_full (GTK_OBJECT (chmod_dialog), "data->textview_label", data->textview_label,
							  (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (data->textview_label);
	gtk_table_attach (GTK_TABLE (table3), data->textview_label, 1, 2, 0, 1,
					  (GtkAttachOptions) (GTK_FILL),
					  (GtkAttachOptions) (0), 0, 0);
	gtk_misc_set_alignment (GTK_MISC (data->textview_label), 0, 0.5);

	gnome_dialog_append_button (GNOME_DIALOG (chmod_dialog), GNOME_STOCK_BUTTON_OK);
	button4 = GTK_WIDGET (g_list_last (GNOME_DIALOG (chmod_dialog)->buttons)->data);
	gtk_widget_ref (button4);
	gtk_object_set_data_full (GTK_OBJECT (chmod_dialog), "button4", button4,
							  (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (button4);
	GTK_WIDGET_SET_FLAGS (button4, GTK_CAN_DEFAULT);

	gnome_dialog_append_button (GNOME_DIALOG (chmod_dialog), GNOME_STOCK_BUTTON_CANCEL);
	button6 = GTK_WIDGET (g_list_last (GNOME_DIALOG (chmod_dialog)->buttons)->data);
	gtk_widget_ref (button6);
	gtk_object_set_data_full (GTK_OBJECT (chmod_dialog), "button6", button6,
							  (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (button6);
	GTK_WIDGET_SET_FLAGS (button6, GTK_CAN_DEFAULT);

	gtk_signal_connect (GTK_OBJECT (button4), "clicked",
						GTK_SIGNAL_FUNC (on_ok), chmod_dialog);
	gtk_signal_connect (GTK_OBJECT (button6), "clicked",
						GTK_SIGNAL_FUNC (on_cancel), chmod_dialog);	
	gtk_signal_connect (GTK_OBJECT (data->or_check), "toggled",
						GTK_SIGNAL_FUNC (on_toggle), chmod_dialog);	
	gtk_signal_connect (GTK_OBJECT (data->ow_check), "toggled",
						GTK_SIGNAL_FUNC (on_toggle), chmod_dialog);	
	gtk_signal_connect (GTK_OBJECT (data->oe_check), "toggled",
						GTK_SIGNAL_FUNC (on_toggle), chmod_dialog);	
	gtk_signal_connect (GTK_OBJECT (data->gr_check), "toggled",
						GTK_SIGNAL_FUNC (on_toggle), chmod_dialog);	
	gtk_signal_connect (GTK_OBJECT (data->gw_check), "toggled",
						GTK_SIGNAL_FUNC (on_toggle), chmod_dialog);	
	gtk_signal_connect (GTK_OBJECT (data->ge_check), "toggled",
						GTK_SIGNAL_FUNC (on_toggle), chmod_dialog);	
	gtk_signal_connect (GTK_OBJECT (data->wr_check), "toggled",
						GTK_SIGNAL_FUNC (on_toggle), chmod_dialog);	
	gtk_signal_connect (GTK_OBJECT (data->ww_check), "toggled",
						GTK_SIGNAL_FUNC (on_toggle), chmod_dialog);	
	gtk_signal_connect (GTK_OBJECT (data->we_check), "toggled",
						GTK_SIGNAL_FUNC (on_toggle), chmod_dialog);	
}




/***********************************
 * Public functions
 ***********************************/

GtkWidget*
gnome_cmd_chmod_dialog_new (GList *files)
{	
	GnomeCmdChmodDialog *dialog = gtk_type_new (gnome_cmd_chmod_dialog_get_type ());
	GnomeCmdChmodDialogPrivate *data = dialog->priv;
	
	data->files = gnome_cmd_file_list_copy (files);

	if (!data->files)
		return NULL;
	if (!data->files->data)
		return NULL;
	
	data->finfo = (GnomeCmdFile*)data->files->data;
	data->perm = vfsperm_to_mode (data->finfo->info->permissions);
	
	show_perms (dialog);
	
	return GTK_WIDGET (dialog);
}



GtkType
gnome_cmd_chmod_dialog_get_type         (void)
{
	static GtkType dlg_type = 0;

	if (dlg_type == 0)
	{
		GtkTypeInfo dlg_info =
		{
			"GnomeCmdChmodDialog",
			sizeof (GnomeCmdChmodDialog),
			sizeof (GnomeCmdChmodDialogClass),
			(GtkClassInitFunc) class_init,
			(GtkObjectInitFunc) init,
			/* reserved_1 */ NULL,
			/* reserved_2 */ NULL,
			(GtkClassInitFunc) NULL
		};

		dlg_type = gtk_type_unique (gnome_dialog_get_type (), &dlg_info);
	}
	return dlg_type;
}



