/*
 * Gmail. A Gnome email client.
 * Copyright (C) 1999 Wayne Schuller
 *
 * mysql.c - functions which involve talking to mysql server.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */

#include "main.h"

gboolean check_db(MYSQL *mysql);
gboolean open_db(MYSQL *mysql);
Message * load_vfolder_msg(gint id);
gboolean add_msg_to_db(Message *message, DTimer *timer);
GString * setup_db (gchar *ruser, gchar *rpass, gchar *user, gchar *pass, gchar *hostname, gchar *db);
gboolean uid_present(gchar *uid);
DTimer * msg_match_popup (void);
gint progress_timeout(DTimer *timer);
void destroy_dtimeout(GtkWidget *widget, DTimer *timer);
gboolean vfolder_msg_unread(gint id);
void mark_vfolder_unread_status_by_name(gchar *vfolder, gboolean unread_status);
void mark_vfolder_unread_status_by_node(GtkCTreeNode *node, gboolean unread_status);
gint message_count (void);
void free_message(Message *message);
void mark_readstatus(gint msg_id, gchar *val);
gboolean vfolder_contains_unread(Mailbox *mailbox);

extern _GmailApp_	*GmailApp;		/* Global vars live here. */
extern void check_vfolder_matches(gint id, DTimer *timer);

/* check_db - Check if the database connection is open, if it isn't
 * try and open one.
 * Returns TRUE for success, and FALSE for failure.
 */
gboolean
check_db (MYSQL *mysql)
{
	/* Check if the connection is open, and if so don't
	 * open it again.
	 */
	if (mysql_ping(mysql) == 0) {
		/* g_print("Maintaining database connection...\n"); */
		return(TRUE); /* server is alive. */
	} else {
		/* Close the dead connection. */
		mysql_close(mysql); 

		/* Open a new connection. */
		return(open_db(mysql)); 
	}
}

/* Open a database connection. */
gboolean
open_db (MYSQL *mysql)
{
	gchar *hostname, *username, *password, *dbname;

	hostname = gnome_config_get_string("/gmail/MysqlServer/hostname");
	username = gnome_config_get_string("/gmail/MysqlServer/username");
	password = gnome_config_private_get_string("/gmail/MysqlServer/password");
	dbname = gnome_config_get_string("/gmail/MysqlServer/dbname");

	/* We use null passwords if it is a blank field. */
	if (password != NULL && strlen(password) < 1) {
		g_free(password);
		password = NULL;
		}

	g_print(_("Connecting to database.\n"));

	/* Call mysql_init */
	if (mysql_init(mysql) == NULL) {
		GtkWidget *dialog;
		gchar *msg;
		msg = g_strdup_printf (_("MYSQL Init failed:\n%s\nYou cannot continue until this error is fixed.\n"), mysql_error(mysql));
		dialog = gnome_message_box_new(msg, GNOME_MESSAGE_BOX_ERROR, "doh!", 0);
 		gtk_window_set_position(GTK_WINDOW(dialog), GTK_WIN_POS_CENTER);
 		gtk_widget_show(dialog);
		g_free(msg);	
		return(FALSE);
		}

	/* g_print("settings:\n %s %s %s %s \n", GmailApp->hostname, GmailApp->username, GmailApp->password, GmailApp->dbname); */

	if (!mysql_real_connect(mysql, hostname, username, password, dbname, 0, NULL, 0)) {
		GtkWidget *dialog;
		gchar *msg;
		msg = g_strdup_printf (_("MYSQL Connection Failed:\n%s\nYou cannot continue until this error is fixed.\n"), mysql_error(mysql));
		dialog = gnome_message_box_new(msg, GNOME_MESSAGE_BOX_ERROR, "doh!", 0);
		gtk_window_set_title(GTK_WINDOW (dialog), "Gmail");
 		gtk_window_set_position(GTK_WINDOW(dialog), GTK_WIN_POS_CENTER);
 		gtk_widget_show(dialog);
		g_free(msg);	
		return(FALSE);
		} 

	g_free(hostname);
	g_free(username);
	g_free(password);
	g_free(dbname);

	return(TRUE);
}


/* add_msg_to_db - Add a message to the mysql database. 
 * Is passed a single Message type.
 * Returns TRUE for success, FALSE for failure.
 * Everything should have funny \r\n's filtered out but we need to
 * escape most of the fields cos nearly everything can have a ' or " in it.
 * Some people have their names bracketed with one or the other... annoying.
 *
 * It is up to the calling function to free the memory in the message.
 * This function doesn't free the memory used by the message.
 * It will free only the memory it has allocated for use.
 *
 * FIXME: do we need error checking on the message?
 * FIXME: There is some experimental code in here to count the length
 * of the query as we aresn't supposed to use strlen on it, but it works.
 */
gboolean
add_msg_to_db(Message *message, DTimer *timer)
{
	gint	tmp, id, len;
	gchar *query, *headers, *msg, *subject, *to, *from, *cc, *uid;
	gint q1len = 0, q2len = 0; /* Keep track of the 2 query lengths. */

	/* All of the following can't be NULL.
	 * date, cc, bcc can be NULL.
	 */
	g_return_val_if_fail(message != NULL, FALSE);
	g_return_val_if_fail(message->message != NULL, FALSE);
	g_return_val_if_fail(message->headers != NULL, FALSE);
	g_return_val_if_fail(message->subject != NULL, FALSE);
	g_return_val_if_fail(message->from != NULL, FALSE);
	g_return_val_if_fail(message->to != NULL, FALSE);
	g_return_val_if_fail(message->readstatus != NULL, FALSE);
	g_return_val_if_fail(message->direction != NULL, FALSE);

	/* g_print("add_msg_to_db split length is %i\n", message->message->len + message->headers->len); */

	/* Open Database. */
	if (!check_db(&GmailApp->mysql)) { return FALSE; };

	/* Escape the content of the message so newlines and ' symbols don't
	 * confuse our query.
	 * Mysql docs say we need length*2+1 reserved for an escaped string.
	 */
	msg = g_malloc(sizeof(gchar) * ((2 * message->message->len) + 1));
	q2len += mysql_escape_string(msg, message->message->str, message->message->len);

	headers = g_malloc(sizeof(gchar) * ((2 * message->headers->len) + 1));
	q2len += mysql_escape_string(headers, message->headers->str, message->headers->len);

	/* g_print("Escaped length: %i\n", q2len); */

	len = strlen(message->subject);
	subject = g_malloc(sizeof(gchar) * ((2 * len) + 1));
	q1len += mysql_escape_string(subject, message->subject, len);

	len = strlen(message->from);
	from = g_malloc(sizeof(gchar) * ((2 * len) + 1));
	q1len += mysql_escape_string(from, message->from, len);

	len = strlen(message->to);
	to = g_malloc(sizeof(gchar) * ((2 * len) + 1));
	q2len += mysql_escape_string(to, message->to, len);

	len = strlen(message->cc);
	cc = g_malloc(sizeof(gchar) * ((2 * len) + 1));
	q2len += mysql_escape_string(cc, message->cc, len);

	len = strlen(message->uid);
	uid = g_malloc(sizeof(gchar) * ((2 * len) + 1));
	q1len += mysql_escape_string(uid, message->uid, len);

	if (headers == NULL) headers = g_strdup("Parse Error");
	if (msg == NULL) msg = g_strdup("Parse Error");

	/* If we have the date put quotes around it, else use NOW() */
	if (message->date != NULL) {
		gchar *tmp;
		tmp = g_strdup_printf("'%s'", message->date);
		g_free(message->date);
		message->date = tmp;
	} else
		message->date = g_strdup("NOW()");

	q1len += strlen(message->date);
	q1len += strlen(message->readstatus);
	q1len += strlen(message->direction);

	/* Insert data into 'display' table. */
	query = g_strdup_printf("INSERT INTO display (date, subject, fromfield, readstatus, direction, uid) VALUES (%s, '%s' , '%s' , '%s', '%s', '%s')", 
					message->date,
					subject,
					from,
					message->readstatus,
					message->direction,
					uid);

	/* g_print("q1len %i strlen(query) %i\n", q1len, (guint) strlen(query)); */

	if (mysql_real_query(&GmailApp->mysql, query, (guint) strlen(query)) != 0) {
		g_print("query was: |%s|\n", query);
		g_warning("Query failed: %s\n", mysql_error(&GmailApp->mysql));
		return(FALSE);
		} 


	/* It _should_ have only affected one row. */	
	tmp = mysql_affected_rows(&GmailApp->mysql);
	if (tmp == 0) {
		/* Affecting zero rows is considered failure. */
		g_warning("add_msg_to_db affect %d rows...", tmp);
		g_free(query);
		return FALSE;
	} else if (tmp > 1) {
		/* Affecting more than one row on an INSERT is just 
		 * plain weird. Better to return success to make the 
		 * user think that open source software is reliable. 
		 */
		g_warning("add_msg_to_db affected %d rows. (It should have only effected one.", tmp);
	}	

	g_free(query);

	/* Find the database id of the message we just put in. */
	id = mysql_insert_id(&GmailApp->mysql);

	/* Insert data into 'details' table. */
	query = g_strdup_printf("INSERT INTO details (id, headers, message, tofield, ccfield) VALUES ('%d', '%s' , '%s' , '%s', '%s')", 
					id,
					headers,
					msg,
					to,
					cc);

	/* g_print("Inserting query length %i\n", (guint) strlen(query)); */

	if (mysql_real_query(&GmailApp->mysql, query, (guint) strlen(query)) != 0) {
		/* Don't print the query here to stdout as it is probably a 
		 * huge attachment.
		 * 
		 * FIXME: because there was a problem in the details table
		 * we should delete the entry from the display table that
		 * actually worked. Otherwise the user won't be allowed
		 * to download it again as the UID will exist.
		 */
		/* g_print("query was: |%s|\n", query); */
		g_warning("Query failed: %s (len %i)\n", mysql_error(&GmailApp->mysql), strlen(query));
		return(FALSE);
		} 


	/* It _should_ have only affected one row. */	
	tmp = mysql_affected_rows(&GmailApp->mysql);
	if (tmp == 0) {
		/* Affecting zero rows is considered failure. */
		g_warning("add_msg_to_db affect %d rows...", tmp);
		g_free(query);
		return FALSE;
	} else if (tmp > 1) {
		/* Affecting more than one row on an INSERT is just 
		 * plain weird. Better to return success to make the 
		 * user think that open source software is reliable. 
		 */
		g_warning("add_msg_to_db affected %d rows. (It should have only effected one.", tmp);
	}	

	g_free(msg);
	g_free(headers);
	g_free(subject);
	g_free(from);
	g_free(to);
	g_free(cc);
	g_free(uid);
	g_free(query);

	/* Update the matched index for this message. */
	check_vfolder_matches(id, timer); 

	return(TRUE);
}



/* load_vfolder_msg - Load a message based on a mysql id.
 * This doesn't display the message. You have to call display_message to do
 * that.
 * It returns the message, or NULL if there is a problem. 
 */
Message *
load_vfolder_msg(gint id)
{
	MYSQL_RES *res;
	MYSQL_ROW mysqlrow;
	gchar *query;
	Message *msg = NULL;

	/* Open Database. */
	if (!check_db(&GmailApp->mysql)) { return(NULL); };

	/* Initialise the message.*/
	msg = g_malloc(sizeof(Message));
	msg->id = id;
	msg->headers = NULL;
	msg->message = NULL;
	msg->date = NULL;
	msg->subject = NULL;
	msg->from = NULL;
	msg->to = NULL;
	msg->cc = NULL;
	msg->bcc = NULL;
	msg->uid = NULL;
	msg->readstatus = NULL;
	msg->direction = NULL;

	g_assert(msg != NULL);

	/* Get info from display table. */
	query = g_strdup_printf("SELECT fromfield,subject,date,direction FROM display WHERE id = %i", id);

	if (mysql_query(&GmailApp->mysql, query) != 0) {
		g_print("Query failed: %s\n", mysql_error(&GmailApp->mysql));
		g_print("Query was: %s\n", query);
		return(NULL);
	} 

	res = mysql_use_result(&GmailApp->mysql);

	if ((mysqlrow = mysql_fetch_row(res))) {

		/* FIXME: Do error checking on the result.. bad data causes core-dumps. */

		msg->from = g_mime_utils_8bit_header_decode(mysqlrow[0]);
		msg->subject = g_strdup(mysqlrow[1]);		
		msg->date = g_strdup(mysqlrow[2]);		
		msg->direction = g_strdup(mysqlrow[3]);		
	} else { 
		/* The query didn't have any results. 
		 * The msg_id isn't in the database.
		 */
		g_warning("load_vfolder_msg unable to load msg id: %i\n", id);
		return(NULL);
	}

	mysql_free_result(res); 

	g_free(query);

	/* Get info from details table. */
	query = g_strdup_printf("SELECT tofield,ccfield,message,headers FROM details WHERE id = %i", id);

	/* Query the database for the email. */
	if (mysql_query(&GmailApp->mysql, query) != 0) {
		g_print("Query failed: %s\n", mysql_error(&GmailApp->mysql));
		g_print("Query was: %s\n", query);
		return(NULL);
	} 

	res = mysql_use_result(&GmailApp->mysql);

	if ((mysqlrow = mysql_fetch_row(res))) {

		msg->to = g_strdup(mysqlrow[0]);		

		if ((mysqlrow[1] != NULL) && (strlen(mysqlrow[1]) > 2)) {
			msg->cc = g_strdup(mysqlrow[1]);		
			} else msg->cc = NULL;

		msg->message = g_string_new(mysqlrow[2]);
		msg->headers = g_string_new(mysqlrow[3]);

	} else { 
		/* The query didn't have any results. The msg_id isn't in the database. */	
		g_warning("load_vfolder_msg unable to load msg (details entry missing for id %i)...\n", id);
		return(NULL);
	}

	g_free(query);
	mysql_free_result(res); 

	return(msg);
}



/* setup_db - Setup the database for the user. 
 * It even puts in a welcome message.
 */

GString *
setup_db (gchar *ruser, gchar *rpass, gchar *user, gchar *pass, gchar *hostname, gchar *db)
{
	gchar *query;
	GString *result; /* Status info returned by the function. */

	result = g_string_new(NULL);

	mysql_init(&GmailApp->mysql);

	if (!mysql_real_connect(&GmailApp->mysql, hostname, ruser, rpass, NULL, 0, NULL, 0)) {
			g_string_append(result, _("Unable to open the database.\nAre the mysql root username and password settings correct?"));	
			return(result); }

	query = g_strdup_printf("create database %s", db);
	if (mysql_query(&GmailApp->mysql, query) != 0) {
		g_string_sprintfa(result, "Query failed: %s\n", mysql_error(&GmailApp->mysql));
		g_string_sprintfa(result, "Gmail was unable to create the database on the mysql server.\nYou may wish to try doing this yourself using the mysqladmin program.\n");

		return result;
		}

	/* Attempt to connect to the database. */
	if (mysql_select_db(&GmailApp->mysql, db) != 0) {
		g_string_sprintfa(result, "Open database (%s) failed: %s\n", db, mysql_error(&GmailApp->mysql));
		g_string_sprintfa(result, "Unable to select the database '%s' on the server.\n", db);
		return result;
		}

		/* IMPORTANT: Below is what gmail currently thinks the table design is. */

	/* 'display' table */
	query = g_strdup_printf("CREATE TABLE display (id MEDIUMINT(11) UNSIGNED NOT NULL AUTO_INCREMENT PRIMARY KEY, date DATETIME NOT NULL, index date (date), subject CHAR(255) NOT NULL, index subject (subject), fromfield CHAR(255) NOT NULL, index fromfield (fromfield), readstatus ENUM('Read', 'Unread', 'Sent', 'Queued') NOT NULL, index readstatus (readstatus), direction ENUM('Incoming', 'Outgoing') NOT NULL, index direction (direction), uid CHAR(255) NOT NULL, index uid (uid), matched SET(' ','Inbox','Outbox','Unread') NOT NULL, index matched (matched))");

	if (mysql_query(&GmailApp->mysql, query) != 0) {
			g_string_sprintfa(result, "Query failed: %s\n", mysql_error(&GmailApp->mysql));
			g_string_sprintfa(result, "Unable to create the table 'display' in database.\n");
			g_free(query);
			return result;
		}

	g_free(query);
	/* 'details' table */
	query = g_strdup_printf("CREATE TABLE details (id MEDIUMINT(11) UNSIGNED NOT NULL AUTO_INCREMENT PRIMARY KEY, headers TEXT NOT NULL, message LONGTEXT NOT NULL, tofield TEXT NOT NULL, ccfield TEXT, bccfield TEXT, attachments TEXT)");
	if (mysql_query(&GmailApp->mysql, query) != 0) {
			g_string_sprintfa(result, "Query failed: %s\n", mysql_error(&GmailApp->mysql));
			g_string_sprintfa(result, "Unable to create the table 'details' in database.\n");
			g_free(query);
			return result;
		}

	g_free(query);

	/* Create the user account */
	query = g_strdup_printf("GRANT ALL PRIVILEGES ON %s.* TO %s@localhost IDENTIFIED BY '%s' WITH GRANT OPTION", db, user, pass);
	if (mysql_query(&GmailApp->mysql, query) != 0) {
			g_string_sprintfa(result, "Query failed: %s\n", mysql_error(&GmailApp->mysql));
			g_string_sprintfa(result, "Database was created but unable to create a user for this account.\n");

			g_free(query);
			return(result);
			}

	/* FIXME: Check the result of this query. Should have modified 1 row. */


	g_free(query);

	query = g_strdup_printf("INSERT INTO display (date, subject, fromfield, readstatus, direction, uid, matched) VALUES (NOW(), 'Welcome to Gmail!', 'k_wayne@linuxpower.org', 'Unread', 'Incoming', '0', 'Inbox')"); 

	if (mysql_query(&GmailApp->mysql, query) != 0) {
		g_print("Couldn't provide welcome email: %s\n", mysql_error(&GmailApp->mysql));
		g_print("query was: %s\n", query);
		}

	g_free(query);

	query = g_strdup_printf("INSERT INTO details (tofield, message) VALUES ('%s@localhost', '\nDear Valued Customer,\n\nCongratulations on choosing Gmail.\nMake sure you read the online help. It will explain the basics of vfolders, they are like mail filters but more powerful.\n\nThe online help will explain clearly how to write vfolder queries. They are SQL based, and are quite simple.\n\nOnce you get a large number of vfolders with lots of messages, you will want to use the vfolder caching system. It allows for really fast speeds on huge mailboxes, but make sure you read the online help first.\n\nIf you are not satisfied with this software, please return it for a full refund.\n\nPlease reply to this message to send an email the author of gmail, me! I would be grateful for your comments, good or bad.\n\nthanks,\nWayne Schuller.')", user);

	if (mysql_query(&GmailApp->mysql, query) != 0) {
		g_print("Couldn't provide welcome email: %s\n", mysql_error(&GmailApp->mysql));
		g_print("query was: %s\n", query);
		}

	g_free(query);


	return(result);
}

/* uid_present - check for existing msg UID in database. 
 * Returns FALSE if UID not found in the database.
 * If the uid message is found we return TRUE.
 * If there is an error we return FALSE. 
 */
gboolean uid_present(gchar *uid)
{
	gint rows;
	MYSQL_RES *res;
	gchar *query, *new_uid;

	g_return_val_if_fail(uid != NULL, FALSE);
		
	/* Open Database. */
	if (!check_db(&GmailApp->mysql)) { return FALSE; };

	new_uid = g_malloc(sizeof(gchar) * ((2 * strlen(uid)) + 1));
	mysql_escape_string(new_uid, uid, strlen(uid));

	query = g_strdup_printf("select id from display where uid = \"%s\"", new_uid);
	g_free(new_uid);

	/* Query the database for the email. */
	if (mysql_query(&GmailApp->mysql, query) != 0) {
		g_warning("Query failed: %s\n", mysql_error(&GmailApp->mysql));
		g_print("query was: |%s|\n", query);
		/* Close the database connection. */
		g_free(query);
		return FALSE;
	} 

	g_free(query);

	res = mysql_store_result(&GmailApp->mysql);

	if (res == NULL) {
		g_warning("Store result failed: %s\n", mysql_error(&GmailApp->mysql));
		return FALSE;
	}

	rows = mysql_num_rows(res);
	mysql_free_result(res); 

	/* Did we not find it? */
	if (rows == 0) {
			return FALSE;
			} 

	return TRUE;	
}


/* msg_match_popup - Create popup window for matching and pop3 downloads. 
 * We are being tricky and using this popup window for two things:
 * a) pop3 downloads
 * b) manual matching runs. From Advanced menu "Rebuild Matched Values"
 * c) and for smtp sends?
 *
 */
DTimer *
msg_match_popup (void) {
 	GtkWidget *vbox;
 	GtkWidget *sw, *label;
	DTimer *timer;
	gchar *titles[2] = {"name", "matches"};
	
	timer = g_malloc(sizeof(DTimer));
	timer->value = 0;	
	timer->total = 1;
	timer->dialog = gnome_dialog_new (
                 "Gmail Popup",
                 NULL);
	if (GmailApp->app)
		gnome_dialog_set_parent (GNOME_DIALOG (timer->dialog), GTK_WINDOW (GmailApp->app));

	gtk_widget_set_usize(GTK_WIDGET(timer->dialog), 240, 400);	
	/* gtk_window_set_modal(GTK_WINDOW (timer->dialog), TRUE); */

	/* Use signal connect after, because we want to free our timer 
	 * structure as the very last thing we do.
	 */
	gtk_signal_connect_after(GTK_OBJECT (timer->dialog), "delete_event", GTK_SIGNAL_FUNC (destroy_dtimeout), timer);
	gtk_widget_show_all(timer->dialog);

	vbox = GNOME_DIALOG(timer->dialog)->vbox;

	timer->label = gtk_label_new("Looking up host...");
	gtk_box_pack_start (GTK_BOX (vbox), timer->label, FALSE, FALSE, 5);
	gtk_widget_show(timer->label);

	timer->progress = GTK_PROGRESS_BAR(gtk_progress_bar_new());

	/* The timeout time is large so when this is used for rebuilding
	 * the MI values the GUI won't slow it down too much.
	 */
	timer->timeout = gtk_timeout_add (500, (GtkFunction) progress_timeout, (gpointer) timer);
	gtk_progress_set_show_text (GTK_PROGRESS (timer->progress), TRUE);
	gtk_box_pack_start (GTK_BOX (vbox), GTK_WIDGET(timer->progress), FALSE, FALSE, 5);
	gtk_widget_show (GTK_WIDGET(timer->progress));

	timer->button = gtk_button_new_with_label(_("Abort"));
	gtk_box_pack_start (GTK_BOX (vbox), timer->button, FALSE, FALSE, 5);
	gtk_signal_connect_after (GTK_OBJECT (timer->button), "clicked",
                                   GTK_SIGNAL_FUNC (destroy_dtimeout),
                                   timer);
	gtk_widget_show(timer->button);

	label = gtk_label_new(_("Vfolder matches:"));
	gtk_box_pack_start (GTK_BOX (vbox), label, FALSE, FALSE, 5);
	gtk_widget_show(label);

	/* Create a scrolled window to hold the clist. */
	sw = gtk_scrolled_window_new (NULL, NULL);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (sw), GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);    
	gtk_box_pack_start (GTK_BOX (vbox), sw, TRUE, TRUE, 5);
	timer->clist = gtk_clist_new_with_titles (2, titles);
	gtk_clist_set_column_title(GTK_CLIST(timer->clist), 0, _("mailbox name"));
	gtk_clist_column_titles_show(GTK_CLIST(timer->clist));
	gtk_clist_column_titles_passive(GTK_CLIST(timer->clist));
	gtk_clist_set_column_width(GTK_CLIST(timer->clist), 0, 120);
	gtk_container_add (GTK_CONTAINER (sw), timer->clist);

	gtk_widget_show_all(timer->dialog);
	
	while (g_main_iteration(FALSE));
	return(timer);
}


/* Update the value of the progress bar so that we get
 * some movement */
gint progress_timeout(DTimer *timer)
{
	if (timer == NULL || timer->dialog == NULL || timer->progress == NULL || !GTK_IS_WIDGET(timer->dialog) || !GTK_IS_WIDGET(timer->progress)) {
		return(FALSE); /*  Let gtk destroy the timeout for us. */
		}

	/* g_print("updating %%%f (%i / %i) %f\n", percent, timer->value, timer->total, (gfloat) timer->value/timer->total);   */

 	gtk_progress_bar_update(timer->progress,  (gfloat) timer->value / timer->total);

	/* As this is a timeout function, return TRUE so that it
	 * continues to get called */
	return(TRUE);
}


/* Free the memory taken up by the timer.
 * Need to be careful, because this function can be called for two reasons:
 *    - the window is destroyed
 *    - the user has clicked on the button
 *
 * The second is the most common way.
 */
void destroy_dtimeout(GtkWidget *widget, DTimer *timer)
{
	if (timer == NULL) return;

	/* g_print("destroy_dtimeout! %i\n", timer->timeout); */

	/* Destroy the timeout if it is still here.
	 * This is important, because we are going to free the data
	 * structure, and if the timeout is still going it will segfault.
	 */
	gtk_timeout_remove(timer->timeout); /* remove the timeout */
	timer->timeout = -1;

	/* Let the gtk loop totally finish. We have to be 100% sure the
	 * timeout is destroyed.
	 */
	while (g_main_iteration(FALSE));

	if (GTK_IS_WIDGET(timer->dialog)) gnome_dialog_close(GNOME_DIALOG(timer->dialog));

	g_free(timer); /* free the data */

	timer = NULL;
}



/* Check if vfolder msg is unread or not. */
gboolean 
vfolder_msg_unread(gint id) {
	MYSQL_RES *res;
	gchar *query;
	gint affected_rows;

	/* Open the database. */
	if (!check_db(&GmailApp->mysql)) { return FALSE; };

	/* Select all from the database. FIXME: LIMIT 1 ? */
	query = g_strdup_printf("SELECT id FROM display WHERE (id = %d AND readstatus=\"Unread\")", id);

	/* Execute query.. */
	if (mysql_query(&GmailApp->mysql, query) != 0) {
		g_warning("Query failed: %s\n", mysql_error(&GmailApp->mysql));
		g_print("query was: |%s|\n", query);
		g_free(query);
		return FALSE;
	} 

	/* Check the result. */	
	res = mysql_store_result(&GmailApp->mysql);

	if (res == NULL) {
		g_warning("Query failed: %s\n", mysql_error(&GmailApp->mysql));
		g_warning("vfolder match query returned NULL. :(");
		g_print("query was: |%s|\n", query);
		g_free(query);
		return FALSE;
		}

	affected_rows = mysql_num_rows(res);

	mysql_free_result(res); 
	g_free(query);

	if (affected_rows == 1)
		return(TRUE); 
	else 
		return(FALSE); 
}

/* 
 * Mark a vfolder node as being read or unread and set it's font accordingly.
 *
 * This does not change the database.
 *
 * (note: this used to be called vfolder_contains_unread, but now there
 * is a different function with that name)
 *
 *
 */
void mark_vfolder_unread_status_by_name(gchar *vfolder, gboolean unread_status) {
	Mailbox	*mailbox;
	GtkCTreeNode *node;

	/* Iterate over the vfolder ctree to find our vfolder name. */
	node = gtk_ctree_node_nth(GTK_CTREE(GmailApp->mblist), 0);
	for (; node != NULL; node = GTK_CTREE_NODE_NEXT(node)) {
		mailbox = gtk_ctree_node_get_row_data(GTK_CTREE (GmailApp->mblist), GTK_CTREE_NODE (node));
		if (strcmp(mailbox->name, vfolder) == 0) {
			if (unread_status) 
				gtk_ctree_node_set_cell_style(GTK_CTREE (GmailApp->mblist), GTK_CTREE_NODE(node), 0, GmailApp->boldstyle);
				else gtk_ctree_node_set_cell_style(GTK_CTREE (GmailApp->mblist), GTK_CTREE_NODE(node), 0, GmailApp->style);
			break;
		}
	}
}

/* Same as above, but pass the ctree node if we already have it. 
 * This saves an interation over the ctree.
 */
void mark_vfolder_unread_status_by_node(GtkCTreeNode *node, gboolean unread_status) 
{
	if (unread_status) 
		gtk_ctree_node_set_cell_style(GTK_CTREE (GmailApp->mblist), GTK_CTREE_NODE(node), 0, GmailApp->boldstyle);
	else 
		gtk_ctree_node_set_cell_style(GTK_CTREE (GmailApp->mblist), GTK_CTREE_NODE(node), 0, GmailApp->style);
}




/* Count the number of messages in the database. */
gint message_count (void) {
	MYSQL_RES *res;
	gchar *query;
	gint affected_rows;

	/* Open the database. */
	if (!check_db(&GmailApp->mysql)) { return FALSE; };

	/* Select all from the database. */
	query = g_strdup_printf("SELECT id FROM display;");

	/* Execute query.. */
	if (mysql_query(&GmailApp->mysql, query) != 0) {
		g_warning("Query failed: %s\n", mysql_error(&GmailApp->mysql));
		g_print("query was: |%s|\n", query);
		return FALSE;
	} 

	/* Check the result. */	
	res = mysql_store_result(&GmailApp->mysql);

	if (res == NULL) {
		g_warning("Query failed: %s\n", mysql_error(&GmailApp->mysql));
		g_warning("vfolder match query returned NULL. :(");
		g_print("query was: |%s|\n", query);
		return FALSE;
		}

	affected_rows = mysql_num_rows(res);

	mysql_free_result(res); 
	g_free(query);
	return(affected_rows);
}

/* Free up the entire contents of a message structure. */
void free_message(Message *message)
{
	g_return_if_fail(message != NULL);
	
	if (message->headers != NULL) g_string_free(message->headers, TRUE);
	if (message->message != NULL) g_string_free(message->message, TRUE);
	if (message->date != NULL) g_free(message->date);
	if (message->subject != NULL) g_free(message->subject);
	if (message->from != NULL) g_free(message->from);
	if (message->to != NULL) g_free(message->to);
	if (message->cc != NULL) g_free(message->cc);
	if (message->bcc != NULL) g_free(message->bcc);
	if (message->readstatus != NULL) g_free(message->readstatus);
	if (message->direction != NULL) g_free(message->direction);
	if (message->uid != NULL) g_free(message->uid);
	g_free(message);
}

/* Mark the email with id msg_id as read/unread in the database.
 * Don't confuse with mark_msg_read and mark_msg_unread, these functions
 * actually bold and unbold a msg in the clist. They call this function.
 */
void
mark_readstatus(gint msg_id, gchar *val)
{
	gchar *query;

	/* Open Database. */
	if (!check_db(&GmailApp->mysql)) { return; };

	query = g_strdup_printf("UPDATE display SET readstatus=\"%s\" WHERE id = %i", val, msg_id);

	/* Execute query.. */
	if (mysql_query(&GmailApp->mysql, query) != 0) {
		g_warning("Query failed: %s\n", mysql_error(&GmailApp->mysql));
		g_print("Query was: %s\n", query);
		} 

	g_free(query);

}


/* Return true if the vfolder contains an unread message. 
 */
gboolean 
vfolder_contains_unread(Mailbox *mailbox) {
	MYSQL_RES *res;
	gchar *query;
	gint unread_total;

	g_return_val_if_fail(mailbox != NULL, FALSE);
	g_return_val_if_fail(mailbox->query != NULL, FALSE);
	g_return_val_if_fail(mailbox->name != NULL, FALSE);

	if (!check_db(&GmailApp->mysql)) { return(FALSE); };

	/* Obviously this is quicker with caching on.
	 * We use LIMIT 1, because we only need one unread message to 
	 * return TRUE. Helps the non-cache case especially.
	 */
	if (cache_vfolders()) {
		query = g_strdup_printf("SELECT display.id FROM display WHERE FIND_IN_SET('%s', matched) AND readstatus = \"Unread\" LIMIT 1", mailbox->name);  
	} else {
		query = g_strdup_printf("SELECT display.id FROM display,details WHERE display.id = details.id AND readstatus = \"Unread\" AND (%s) LIMIT 1", mailbox->query);  
	}

	/* Execute query.. */
	if (mysql_query(&GmailApp->mysql, query) != 0) {
		g_warning("Query failed: %s\n", mysql_error(&GmailApp->mysql));
		g_print("match query was: %s\n", query);

		g_free(query);
		return(FALSE);
	} 

	/* Check the result. */	
	res = mysql_store_result(&GmailApp->mysql);

	if (res == NULL) {
		g_warning("Query failed: %s\n", mysql_error(&GmailApp->mysql));
		g_warning("vfolder match query returned NULL. :(");
 		/* Abort this vfolder but keep going on the matching. */
		g_free(query);
		return(FALSE);
		}

	unread_total = mysql_num_rows(res);

	/* We're finished with the results now. */
	mysql_free_result(res); 

	g_free(query);

	/* g_print("row: %i, unread %i\n", row, unread_total); */
	if (unread_total > 0) {
		return(TRUE); 
	} else {
		return(FALSE);
	}
}
