/*  GKrellMMS: GKrellM XMMS Plugin
 |  Copyright (C) 2000 Sander Klein Lebbink
 |
 |  Author:  Sander Klein Lebbink <sander@cerberus.demon.nl>
 |  Latest versions might be found at:  http://gkrellm.luon.net/
 |
 |  This program is free software which I release under the GNU General Public
 |  License. You may redistribute and/or modify this program under the terms
 |  of that license as published by the Free Software Foundation; either
 |  version 2 of the License, or (at your option) any later version.
 |
 |  This program is distributed in the hope that it will be useful,
 |  but WITHOUT ANY WARRANTY; without even the implied warranty of
 |  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 |  GNU General Public License for more details.
 |
 |  To get a copy of the GNU General Public License,  write to the
 |  Free Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "gkrellmms.h"

/* Save/load functions */

static GtkWidget *pl_open_box;
static gboolean pl_open;

void save_playlist(gchar *filename, gint warn_tell)
{
  FILE *f;
  gint i;
  gint len;
  playlist_def *templ;
  f = fopen(filename, "w");
  if (f)
  {
    len = g_list_length(plist);
    for (i = 0; i < len; i++)
    {
      templ = (playlist_def *) g_list_nth(plist, i)->data;
      fprintf(f, "%s\n", templ->filename);
    }
    fclose(f);
  } else {
    if (warn_tell)
      gkrellm_message_window("GKrellMMS Error", "Couldn't save playlist. :(", NULL);
  }
}

void save_position(gint warn_tell)
{
  FILE *f;
  gint pos;
  if ((f = fopen(position_file, "w")) != NULL)
  {
    pos = xmms_pos;
    fprintf(f, "%d\n", pos);
    fclose(f);
  } else {
    if (warn_tell)
      gkrellm_message_window("GKrellMMS Error", "Couldn't save position. :(", NULL);
  }
}

void save_time(gint warn_tell)
{
  FILE *f;
  gint pos;
  if ((f = fopen(time_file, "w")) != NULL)
  {
    pos = xmms_remote_get_output_time(xmms_session);
    fprintf(f, "%d\n", pos);
    fclose(f);
  } else {
    if (warn_tell)
      gkrellm_message_window("GKrellMMS Error", "Couldn't save position. :(", NULL);
  }
}

void set_playlist_position(gint warn_tell)
{
  FILE *f;
  gint pos;
  pos = 0; 
  if ((f = fopen(position_file, "r")) != NULL)
  {
    fscanf(f, "%d", &pos);
    xmms_remote_set_playlist_pos(xmms_session, pos);
    fclose(f);
  } else {
    if (warn_tell)
      gkrellm_message_window("GKrellMMS Error", "Couldn't open position. :(", NULL);
  }
}  

void set_time_position(gint warn_tell)
{
  FILE *f;
  gint pos;
  gint timer;
  time_t lt;

  pos = 0;

  if ((f = fopen(time_file, "r")) != NULL)
  {
    fscanf(f, "%d", &pos);
    if (auto_play_start)
      xmms_remote_play(xmms_session);
    if (auto_seek && (pos != 0))
    {
      timer = localtime(&lt)->tm_sec;
      while (!xmms_remote_is_playing(xmms_session) &&
        ((localtime(&lt)->tm_sec - timer) < 10))
      {
        /* Do nothing, wait until xmms really plays;
         | stop waiting after 10 seconds.
         */
         usleep(0);
      }
      xmms_remote_jump_to_time(xmms_session, pos);
    }
    fclose(f);
  } else {
    if (warn_tell)
      gkrellm_message_window("GKrellMMS Error", "Couldn't open time position. :(", NULL);
  }
}

void save_playlist(gchar *, gint);
void save_position(gint);
void update_playlist(void);

void open_playlist(gchar *fn, gint warn_tell)
{
  FILE *fp;
  gchar *line, *blugh;
  gchar **files;
  gint linelen = 1024, i = 0, blaat = 0;
  gboolean is_playing = xmms_remote_is_playing(xmms_session);

  xmms_remote_stop(xmms_session);

  if ((fp = fopen(fn, "r")) == NULL)
  {
    if (warn_tell)
    {
      blugh = g_strdup_printf("Couldn't open file:\n%s", fn);
      gkrellm_message_window("GKrellMMS Error", blugh, NULL);
      g_free(blugh);
    }
    return;
  }

  files = g_malloc0(linelen * sizeof(gchar *));
  line = g_malloc(linelen);
  blugh = g_malloc(linelen);
  i = 0;
  while (fgets(blugh, linelen, fp))
  {
    blaat = sscanf(blugh, "%[^\n]", line);
    if (blaat)
      files[i++] = strdup(line);
  }
  fclose(fp);

  xmms_remote_playlist_clear(xmms_session);
  xmms_remote_playlist(xmms_session, files, i, TRUE);
  update_playlist();

  if (is_playing)
    xmms_remote_play(xmms_session);

  pl_open = FALSE;

  if (warn_tell)
  {
    save_playlist(playlist_file, 0);
    save_position(0);
    save_time(0);
  }
   
  g_free(line);
  g_free(blugh);
  for (i = 0; i < g_list_length(plist); i++)
    g_free(files[i]);
  g_free(files);
}
 
void pl_ok_sel (GtkWidget *w, GtkFileSelection *fs)
{
  open_playlist(gtk_file_selection_get_filename(GTK_FILE_SELECTION (fs)), 1);
  gtk_widget_destroy(pl_open_box);
}
 
void destroy_pl(GtkWidget *w, GtkFileSelection *fs)
{
  pl_open = FALSE;
  gtk_widget_destroy(pl_open_box);
}
 
void pl_open_func (GtkWidget *w, gpointer data)
{
 
  if (pl_open || !xmms_running)
    return;   
  else
    pl_open = TRUE;

  chdir(playlist_dir);

  pl_open_box = gtk_file_selection_new ("Open Playlist");
  
  gtk_signal_connect(GTK_OBJECT( GTK_FILE_SELECTION(pl_open_box)->ok_button),
                     "clicked", (GtkSignalFunc) pl_ok_sel, pl_open_box);

  gtk_signal_connect(GTK_OBJECT (pl_open_box), "destroy",
                     (GtkSignalFunc) destroy_pl, GTK_OBJECT(pl_open_box));

  gtk_signal_connect_object(GTK_OBJECT(GTK_FILE_SELECTION(pl_open_box)->cancel_button),
                            "clicked", (GtkSignalFunc) destroy_pl,
                            GTK_OBJECT(pl_open_box));
  gtk_widget_show(pl_open_box);
}

static gboolean pl_save_open;
static GtkWidget *pl_save_box;

void pl_save_ok_sel (GtkWidget *w, GtkFileSelection *fs)
{
  save_playlist(gtk_file_selection_get_filename(GTK_FILE_SELECTION (fs)), 1);
  gtk_widget_destroy(pl_save_box);
}
 
void destroy_save_pl(GtkWidget *w, GtkFileSelection *fs)
{
  pl_save_open = FALSE;
  gtk_widget_destroy(pl_save_box);
}
 
void pl_save_func (GtkWidget *w, gpointer data)
{
 
  if (pl_save_open || !xmms_running)
    return;   
  else
    pl_save_open = TRUE;

  chdir(playlist_dir);

  pl_save_box = gtk_file_selection_new ("Save Playlist");
  
  gtk_signal_connect(GTK_OBJECT( GTK_FILE_SELECTION(pl_save_box)->ok_button),
                     "clicked", (GtkSignalFunc) pl_save_ok_sel, pl_save_box);

  gtk_signal_connect(GTK_OBJECT (pl_save_box), "destroy",
                     (GtkSignalFunc) destroy_save_pl, GTK_OBJECT(pl_save_box));

  gtk_signal_connect_object(GTK_OBJECT(GTK_FILE_SELECTION(pl_save_box)->cancel_button),
                            "clicked", (GtkSignalFunc) destroy_save_pl,
                            GTK_OBJECT(pl_save_box));
  gtk_widget_show(pl_save_box);
}

void clear_playlist()
{
  playlist_def *templ;
  gint i, len;

  if (plist == NULL)
    return;
  
  len = g_list_length(plist);
  for (i = 0; i < len; i++)
  {
    templ = (playlist_def *) g_list_nth(plist, i)->data;
    if (templ->filename)
      g_free(templ->filename);
    if (templ->title)
      g_free(templ->title);
    g_free(templ);
  }
  g_list_free(plist);
  plist = NULL;
}

void update_playlist_window(void);

void update_playlist()
{
  static playlist_def *templ;
  gint i, len;
  
  if (plist != NULL)
    clear_playlist();

  total_plist_time = 0;

  len = xmms_remote_get_playlist_length(xmms_session);
  for (i = 0; i < len; i++)
  {
    templ = g_malloc(sizeof(playlist_def *));
    templ->filename = templ->title = '\0';

    while ((templ->filename == '\0') && xmms_remote_is_running(xmms_session))
      templ->filename = xmms_remote_get_playlist_file(xmms_session, i);

    if (always_load_info)
    {
      while ((templ->title == '\0') && xmms_remote_is_running(xmms_session))
        templ->title = xmms_remote_get_playlist_title(xmms_session, i);
      templ->time = xmms_remote_get_playlist_time(xmms_session, i);
      total_plist_time += templ->time;
    } else {
      templ->title = g_strdup("");
      templ->time = 0;
    }
    plist = g_list_insert(plist, templ, i);
  }
  xmms_pos = xmms_remote_get_playlist_pos(xmms_session);
  update_plist_statusbar(xmms_pos);
  update_playlist_window();
  if (plist != NULL)
    update_plist_window_row(-1, xmms_pos);
}

/* Playlist window */

static GtkWidget *play_clist;
static GtkWidget *status;
static GtkWidget *scrolled;

static GtkItemFactoryEntry gkrellmms_pl_popup[] =
{
  {"/-",                NULL, NULL,            0, "<Separator>"},
/* GKrellMMS 0.6.0
  {"/Delete Entry...",  NULL, NULL,            0, "<Item>"},
  {"/Move up...",       NULL, NULL,            0, "<Item>"},
  {"/Move down...",     NULL, NULL,            0, "<Item>"},
  {"/-",                NULL, NULL,            0, "<Separator>"},
*/
  {"/Open Playlist...", NULL, pl_open_func,    0, "<Item>"},
  {"/Save Playlist...", NULL, pl_save_func,    0, "<Item>"},
  {"/Reload...",        NULL, update_playlist, 0, "<Item>"},
  {"/-",                NULL, NULL,            0, "<Separator>"},
};

GtkItemFactory *playlist_menu_factory()
{
  int n;
  GtkItemFactory *music_factory;
  GtkAccelGroup *axel;
  axel = gtk_accel_group_new();
#if ((VERSION_MAJOR>1)||(VERION_MAJOR==1 && VERSION_MINOR==1 && VERSION_REV>3)||(VERSION_MAJOR==1 && VERSION_MINOR>1))
  gtk_window_add_accel_group(GTK_WINDOW(gkrellm_get_top_window()), axel);
#else
  gtk_window_add_accel_group(GTK_WINDOW(top_window), axel);
#endif
    
  music_factory = gtk_item_factory_new(GTK_TYPE_MENU, "<Main>", axel);

  n = sizeof(gkrellmms_pl_popup) / sizeof(GtkItemFactoryEntry);
  gtk_item_factory_create_items(music_factory, n, gkrellmms_pl_popup, NULL);
  
  return music_factory;
}

void pl_menu_popup(GdkEventButton *ev)
{
  gtk_menu_popup(GTK_MENU(pl_factory->widget),
                 NULL, NULL, NULL, NULL,
                 ev->button, ev->time);
}

void select_row_func(GtkWidget *clist, GdkEventButton *ev, gpointer data)
{
  gint row, column;
  if (!ev)
    return;

  gtk_clist_get_selection_info(GTK_CLIST(play_clist),
    ev->x, ev->y, &row, &column);
  if ((plist != NULL) && (row <= g_list_length(plist)))
  {
    gtk_clist_select_row(GTK_CLIST(play_clist), row, column);
    switch (ev->button)
    {
      case 1:
        if (ev->type == GDK_2BUTTON_PRESS)
          xmms_remote_set_playlist_pos(xmms_session, row);
        break;
      case 3:
        pl_menu_popup(ev);
        break;
    }
  }
}

void update_plist_statusbar(gint i)
{
  static gchar *status_text;
  static guint prev_status = -1;
  static playlist_def *templ;
  gint total;

  if (g_list_length(plist))
  {
    templ = (playlist_def *) g_list_nth(plist, i)->data;

    total_plist_time -= templ->time;
    templ->time = xmms_remote_get_playlist_time(xmms_session, i);
    total_plist_time += templ->time;

    g_free(templ->title);
    templ->title = xmms_remote_get_playlist_title(xmms_session, i);
  }

  if (!pl_window_open)
    return;

  if (prev_status != -1)
    gtk_statusbar_remove(GTK_STATUSBAR(status), 1, prev_status);

  if (!g_list_length(plist))
    prev_status = gtk_statusbar_push(GTK_STATUSBAR(status), 1, "No Playlist loaded.");
  else
  {
    total = total_plist_time / 1000;
    status_text = g_strdup_printf("Current: %d (%d:%02d)  -  total files: %d  -  total time: %d:%02d:%02d",
                  i + 1, templ->time / 60000, (templ->time / 1000) % 60,
                  g_list_length(plist), total / 3600,
                  (total / 60) % 60, total % 60);
    prev_status = gtk_statusbar_push(GTK_STATUSBAR(status), 1, status_text);
    g_free(status_text);
  }
}

void update_plist_window_row(gint prev_row, gint row)
{
  GdkColormap *cmap;
  static GdkColor *prev_color,
                  *color,
                  *prev_color_back,
                  *color_back;
  static playlist_def *templ;
  gchar *time;
  GtkAdjustment *v;

  if (!pl_window_open)
    return;

  cmap = gdk_colormap_get_system();
  prev_color = &GTK_WIDGET(play_clist)->style->fg[0];
  color = &GTK_WIDGET(play_clist)->style->fg[2];
  prev_color_back = &GTK_WIDGET(play_clist)->style->base[0];
  color_back = &GTK_WIDGET(play_clist)->style->bg[2];

  if (!gdk_color_alloc(cmap, color) || !g_list_length(plist))
    return;

  templ = (playlist_def *) g_list_nth(plist, row)->data;
  gtk_clist_freeze(GTK_CLIST(play_clist));
  v = gtk_scrolled_window_get_vadjustment(GTK_SCROLLED_WINDOW(scrolled));
  if (prev_row >= 0)
  {
    gtk_clist_set_foreground(GTK_CLIST(play_clist), prev_row, prev_color);
    gtk_clist_set_background(GTK_CLIST(play_clist), prev_row, prev_color_back);
  }
  gtk_clist_set_foreground(GTK_CLIST(play_clist), row, color);
  gtk_clist_set_background(GTK_CLIST(play_clist), row, color_back);
  gtk_clist_set_text(GTK_CLIST(play_clist), row, 1, templ->title);
  time = g_strdup_printf("%d:%02d", templ->time / 60000, (templ->time / 1000) % 60);
  gtk_clist_set_text(GTK_CLIST(play_clist), row, 2, time);
  g_free(time);
  gtk_clist_thaw(GTK_CLIST(play_clist));
  gtk_scrolled_window_set_vadjustment(GTK_SCROLLED_WINDOW(scrolled), v);
}
  
void update_playlist_window()
{
  static gchar *song_info[3] = {"", "", ""};
  gint i, len, nr_check, info_right;
  playlist_def *templ;
  GtkAdjustment *v;

  if (!pl_window_open)
    return;

  info_right = 0;

  gtk_clist_freeze(GTK_CLIST(play_clist));
  v = gtk_scrolled_window_get_vadjustment(GTK_SCROLLED_WINDOW(scrolled));
  gtk_clist_clear(GTK_CLIST(play_clist));
  for (i = 0; i < g_list_length(plist); i++)
  {
    info_right = 0;
    templ = (playlist_def *) g_list_nth(plist, i)->data;
    song_info[0] = g_strdup_printf("%d.", i + 1);
    if (strcmp(templ->title, "") != 0)
    {
      song_info[1] = g_strdup(templ->title);
    } else {
      song_info[1] = g_strdup(strrchr(templ->filename, '/'));
      if (!song_info[1])
      {
        g_free(song_info[1]);
        song_info[1] = g_strdup(templ->filename);
      } else {
        info_right = 1;
        song_info[1]++;
      }
    }
    len = templ->time;
    song_info[2] = g_strdup_printf("%d:%02d",
      len / 60000, (len / 1000) % 60);
    gtk_clist_append(GTK_CLIST(play_clist), song_info);
    if (i == (g_list_length(plist) - 1))
    {
      nr_check = gdk_string_width(GTK_WIDGET(play_clist)->style->font, song_info[0]);
      gtk_clist_set_column_width(GTK_CLIST(play_clist), 0, nr_check);
      gtk_clist_set_column_width(GTK_CLIST(play_clist), 1, 245 - nr_check);
    }
    g_free(song_info[0]);
    if (info_right)
      song_info[1]--;
    g_free(song_info[1]);
    g_free(song_info[2]);
  }
  gtk_clist_thaw(GTK_CLIST(play_clist));
  gtk_scrolled_window_set_vadjustment(GTK_SCROLLED_WINDOW(scrolled), v);
}

void pl_window_destroy(gpointer destroy_this)
{
  pl_window_open = FALSE;
  gtk_widget_destroy((GtkWidget *) destroy_this);
}

void pl_open_popup(GtkWidget *w, gpointer data)
{
  static GtkWidget *pl_window = NULL;
  static GtkWidget *vbox;
  static GtkWidget *hbox;
  static GtkWidget *refresh_button;
  static GtkWidget *open_button;
  static GtkWidget *save_button;
  static gchar *titles[3] = {"#", "title", "time"};
  gint i;

  if (pl_window_open)
    return;
  else
    pl_window_open = TRUE;

  pl_window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
  gtk_window_set_policy(GTK_WINDOW(pl_window), FALSE, FALSE, TRUE);
  gtk_window_set_title(GTK_WINDOW(pl_window), "GKrellMMS Playlist Editor");
  gtk_window_set_wmclass(GTK_WINDOW(pl_window), "GKrellMMS_playlist", "Gkrellm");
  gtk_signal_connect(GTK_OBJECT(pl_window), "destroy",
                     GTK_SIGNAL_FUNC(pl_window_destroy), (gpointer) pl_window);
  
  vbox = gtk_vbox_new(FALSE, 2);
  gtk_container_add(GTK_CONTAINER(pl_window), vbox);
  gtk_widget_show(vbox);

  scrolled = gtk_scrolled_window_new(NULL, NULL);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolled),
                                 GTK_POLICY_AUTOMATIC, GTK_POLICY_ALWAYS);
  gtk_widget_set_usize(GTK_WIDGET(scrolled), 320, 500);
  gtk_container_border_width(GTK_CONTAINER(scrolled), 2);
  gtk_box_pack_start(GTK_BOX(vbox), scrolled, TRUE, TRUE, 0);
  gtk_widget_show(scrolled);

  play_clist = gtk_clist_new_with_titles(3, titles);
  gtk_clist_set_shadow_type(GTK_CLIST(play_clist), GTK_SHADOW_OUT);
  
  gtk_clist_set_column_width(GTK_CLIST(play_clist), 0, 15);
  gtk_clist_set_column_width(GTK_CLIST(play_clist), 1, 230);

  gtk_clist_set_column_justification(GTK_CLIST(play_clist), 2, GTK_JUSTIFY_RIGHT);

  gtk_clist_set_selection_mode(GTK_CLIST(play_clist), GTK_SELECTION_BROWSE);
  gtk_signal_connect(GTK_OBJECT(play_clist), "button_press_event",
                     GTK_SIGNAL_FUNC(select_row_func), NULL);

  gtk_container_add(GTK_CONTAINER(scrolled), play_clist);
  gtk_widget_show(play_clist);

  hbox = gtk_hbox_new(FALSE, 0);
  open_button = gtk_button_new_with_label("Open Playlist");
  gtk_box_pack_start(GTK_BOX(hbox), open_button, TRUE, TRUE, 0);
  gtk_signal_connect_object(GTK_OBJECT(open_button), "clicked",
                            GTK_SIGNAL_FUNC(pl_open_func),
                            NULL);
  gtk_widget_show(open_button);

  save_button = gtk_button_new_with_label("Save Playlist");
  gtk_box_pack_start(GTK_BOX(hbox), save_button, TRUE, TRUE, 0);
  gtk_signal_connect_object(GTK_OBJECT(save_button), "clicked",
                            GTK_SIGNAL_FUNC(pl_save_func),
                            NULL);
  gtk_widget_show(save_button);

  refresh_button = gtk_button_new_with_label("Force reload");
  gtk_box_pack_start(GTK_BOX(hbox), refresh_button, TRUE, TRUE, 0);
  gtk_signal_connect_object(GTK_OBJECT(refresh_button), "clicked",
                            GTK_SIGNAL_FUNC(update_playlist),
                            NULL);
  gtk_widget_show(refresh_button);

  gtk_box_pack_start(GTK_BOX(vbox), hbox, TRUE, TRUE, 0);
  gtk_widget_show(hbox);

  status = gtk_statusbar_new();
  gtk_box_pack_start(GTK_BOX(vbox), status, TRUE, TRUE, 0);
  gtk_widget_show(status);

  gtk_widget_show(pl_window);

  i = xmms_pos;
  update_plist_statusbar(i);
  update_playlist_window();
  update_plist_window_row(-1, i);
}
