module StdIOCommon ( module StdIOCommon
                   , module StdIOBasic
                   , module StdKey
                   , module StdMaybe
                   , Id.Id, Id.RId, Id.R2Id, Id.rIdtoId, Id.r2IdtoId
                   ) where


--	********************************************************************************
--	Clean to Haskell Standard Object I/O library, version 1.2
--	
--	StdIOCommon defines common types and access functions for the I/O library.
--	********************************************************************************


import CleanExtras
import Id (Id, RId, R2Id, rIdtoId, r2IdtoId)
import StdIOBasic
import StdKey
import StdMaybe


{-	The SelectState and MarkState types.			-}

data	SelectState
	= Able | Unable
	deriving (Eq,Show)

enabled :: SelectState -> Bool					-- @1 == Able
enabled Able   = True
enabled unable = False

instance Toggle SelectState where				-- Able <-> Unable
	toggle Able = Unable
	toggle _    = Able


{-	The KeyboardState type.					-}

data	KeyboardState
	= CharKey    Char       KeyState			-- ASCII character input
	| SpecialKey SpecialKey KeyState Modifiers		-- Special key input
	| KeyLost						-- Key input lost while key was down
	deriving (Eq,Show)
data	KeyState
	= KeyDown    IsRepeatKey				-- Key is down
	| KeyUp							-- Key goes up
	deriving (Eq,Show)
type	IsRepeatKey						-- Flag on key down:
	= Bool							-- True iff key is repeating
data	Key
	= IsCharKey    Char
	| IsSpecialKey SpecialKey
type	KeyboardStateFilter					-- Predicate on KeyboardState:
	= KeyboardState -> Bool					-- evaluate KeyFunction only if True

getKeyboardStateKeyState:: KeyboardState -> KeyState		-- KeyUp   if KeyLost
getKeyboardStateKeyState (CharKey _ keyState)
	= keyState
getKeyboardStateKeyState (SpecialKey _ keyState _)
	= keyState
getKeyboardStateKeyState KeyLost
	= KeyUp

getKeyboardStateKey :: KeyboardState -> Maybe Key		-- Nothing if KeyLost
getKeyboardStateKey (CharKey char _)
	= Just (IsCharKey char)
getKeyboardStateKey (SpecialKey special _ _)
	= Just (IsSpecialKey special)
getKeyboardStateKey KeyLost
	= Nothing


{-	The UpdateState type.
-}
data	UpdateState
	= UpdateState
		{ oldFrame	:: !ViewFrame
		, newFrame	:: !ViewFrame
		, updArea	:: !UpdateArea
		}
	deriving (Show)
type	ViewDomain = Rectangle
type	ViewFrame  = Rectangle
type	UpdateArea = [ViewFrame]

rectangleToUpdateState :: Rectangle -> UpdateState
rectangleToUpdateState frame
	= UpdateState {oldFrame=frame,newFrame=frame,updArea=[frame]}

{-	viewDomainRange defines the minimum and maximum values for ViewDomains.
	viewFrameRange  defines the minimum and maximum values for ViewFrames.
	Values based on Clean maxSigned4ByteInt (see also Commondef.hs).
-}
viewDomainRange :: ViewDomain
viewDomainRange
	= Rectangle
		{ corner1 = Point2 {x = -1073741824,y = -1073741824}
		, corner2 = Point2 {x =  1073741824,y =  1073741824}
		}

viewFrameRange :: ViewFrame
viewFrameRange
	= Rectangle
		{ corner1 = Point2 {x = 2147483647,y = 2147483647}
		, corner2 = Point2 {x = 2147483647,y = 2147483647}
		}


{-	Modifiers indicates the meta keys that have been pressed (True) or not (False).
-}
data	Modifiers
	= Modifiers
		{ shiftDown	:: !Bool			-- True iff shift   down
		, optionDown	:: !Bool			-- True iff option  down
		, commandDown	:: !Bool			-- True iff command down
		, controlDown	:: !Bool			-- True iff control down
		, altDown	:: !Bool			-- True iff alt     down
		}
	deriving (Eq,Show)


--	Constants to check which of the Modifiers are down.

noModifiers = Modifiers {shiftDown = False, optionDown = False, commandDown = False, controlDown = False, altDown = False}
shiftOnly   = Modifiers {shiftDown = True,  optionDown = False, commandDown = False, controlDown = False, altDown = False}
optionOnly  = Modifiers {shiftDown = False, optionDown = True,  commandDown = False, controlDown = False, altDown = True }
commandOnly = Modifiers {shiftDown = False, optionDown = False, commandDown = True,  controlDown = True,  altDown = False}
controlOnly = Modifiers {shiftDown = False, optionDown = False, commandDown = True,  controlDown = True,  altDown = False}
altOnly     = Modifiers {shiftDown = False, optionDown = True,  commandDown = False, controlDown = False, altDown = True }


{-	The layout language used for windows and controls.	-}
type	ItemPos
	=	( ItemLoc
		, ItemOffset
		)
data	ItemLoc
 --	Absolute:
	= Fix
 --	Relative to corner:
	| LeftTop
	| RightTop
	| LeftBottom
	| RightBottom
 --	Relative in next line:
	| Left
	| Center
	| Right
 --	Relative to other item:
	| LeftOf  Id
	| RightTo Id
	| Above   Id
	| Below   Id
 --	Relative to previous item:
	| LeftOfPrev
	| RightToPrev
	| AbovePrev
	| BelowPrev
	deriving (Eq,Show)
type	ItemOffset
	= Vector2						-- A constant offset vector


{-	Document interface of interactive processes.	-}

data	DocumentInterface
	= NDI							-- No       Document Interface
	| SDI							-- Single   Document Interface
	| MDI							-- Multiple Document Interface
	deriving (Eq,Show)


{-	Common error report types.				-}

data	ErrorReport						-- Usual cause:
	= NoError						-- Everything went allright
	| ErrorViolateDI					-- Violation against DocumentInterface
	| ErrorIdsInUse						-- Object contains Ids that are bound
	| ErrorUnknownObject					-- Object can not be found
	| ErrorNotifierOpen					-- It was tried to open a second send notifier
	| OtherError !String					-- Other kind of error
	deriving (Eq,Show)
