module StdControl ( WState
                  , getWindow, getParentWindow
                  , controlSize
                  , setControlTexts, setControlText
                  , getControlTexts, getControlText
                  ) where


--	********************************************************************************
--	Clean to Haskell Standard Object I/O library, version 1.2
--	
--	StdControl specifies all control operations.
--	********************************************************************************


import CleanStdList
import CleanStdMisc
import Commondef
import Controlaccess
import Controlinternal
import Controllayout
import Id
import IOstate
import Ossystem
import StdControlClass
import Windowaccess
import Wstate
#include "preprocess.h"
         -- Compile with -cpp; set MVAR=1 for MVAR version


stdControlFatalError :: String -> String -> x
stdControlFatalError function error
	= dumpFatalError function "StdControl" error

{-	The function isOkControlId can be used to filter out the proper IdParent records.
-}
isOkControlId :: SystemId -> (x,Maybe IdParent) -> (Bool,(x,Id))
isOkControlId ioId (x,Just (IdParent {idpIOId=idpIOId,idpDevice=idpDevice,idpId=idpId}))
	= (ioId==idpIOId && idpDevice==WindowDevice,(x,idpId))
isOkControlId _ _
	= (False,undef)

--	Two locally used functions that retrieve the parent Id(s).
ioStGetIdParent :: Id -> GUI IF_MVAR(,ps) (Maybe IdParent)
ioStGetIdParent id
	= do {
		idtable <- ioStGetIdTable;
		return (getIdParent id idtable)
	  }

ioStGetIdParents :: [Id] -> GUI IF_MVAR(,ps) [Maybe IdParent]
ioStGetIdParents ids
	= do {
		idtable <- ioStGetIdTable;
		return (getIdParents ids idtable)
	  }


{-	gatherWindowIds collects all first Ids (ControlId) that belong to the same second Id (WindowId).
	gatherWindowIds' does the same, except that not only ControlIds are collected, but also their data item.
-}
gatherWindowIds :: [(Id,Id)] -> [([Id],Id)]
gatherWindowIds ((cId,wId) : ids)
	= (cId:cIds , wId) : cIds_wIds
	where
		(cIds,ids') = gatherControlsIds wId ids
		cIds_wIds   = gatherWindowIds ids'
		
		gatherControlsIds :: Id -> [(Id,Id)] -> ([Id],[(Id,Id)])
		gatherControlsIds wId ((cId,wId') : ids)
			| wId==wId'         = (cId:cIds, ids')
			| otherwise         = (cIds, (cId,wId'):ids')
			where
				(cIds,ids') = gatherControlsIds wId ids
		gatherControlsIds _ _
			= ([],[])
gatherWindowIds []
	= []

gatherWindowIds' :: [((Id,x),Id)] -> [([(Id,x)],Id)]
gatherWindowIds' (((cId,x),wId) : ids)
	= ( (cId,x):cIds , wId) : cIds_wIds
	where
		(cIds,ids') = gatherControlsIds wId ids
		cIds_wIds   = gatherWindowIds' ids'
		
		gatherControlsIds :: Id -> [((Id,x),Id)] -> ([(Id,x)],[((Id,x),Id)])
		gatherControlsIds wId (((cId,x),wId') : ids)
			| wId==wId'         = ( (cId,x):cIds ,ids')
			| otherwise         = (cIds, ((cId,x),wId'):ids' )
			where
				(cIds,ids') = gatherControlsIds wId ids
		gatherControlsIds _ _
			= ([],[])
gatherWindowIds' []
	= []


--	The WState window representation record:

data	WState
	= WState
		{ wIds     :: !WIDS
		, wRep     :: !WindowHandle'
		, wMetrics :: !OSWindowMetrics
		}


getWindow :: Id -> GUI IF_MVAR(,ps) (Maybe WState)
getWindow windowId
	= do {
		(found,wDevice) <- accIOEnv (ioStGetDevice WindowDevice);
		if   not found
		then return Nothing
		else 
		let  windows              = windowSystemStateGetWindowHandles wDevice
		     (found,wsH,windows1) = getWindowHandlesWindow (toWID windowId) windows
		in
		if   not found
		then appIOEnv (ioStSetDevice (WindowSystemState windows1)) >> return Nothing
		else do {
			(wsH',wsH1)      <- liftIO (retrieveWindowHandle' wsH);
			let (wids,wsH2)   = getWindowStateHandleWIDS wsH1
			    windows2      = setWindowHandlesWindow wsH2 windows1
			in  do {
				appIOEnv (ioStSetDevice (WindowSystemState windows2));
				wMetrics <- accIOEnv ioStGetOSWindowMetrics;
				return (Just (WState {wIds=wids,wRep=wsH',wMetrics=wMetrics}))
			    }
		      }
	  }

getParentWindow :: Id -> GUI IF_MVAR(,ps) (Maybe WState)
getParentWindow controlId
	= do {
		maybeParent <- ioStGetIdParent controlId;
		if   isNothing maybeParent
		then return Nothing
		else
		let  parent  = fromJust maybeParent
		in  do {
			ioId <- accIOEnv ioStGetIOId;
			if   ioId==idpIOId parent && idpDevice parent==WindowDevice
			then getWindow (idpId parent)
			else return Nothing
		    }
	  }

setWindow :: Id -> (WState -> IO WState) -> GUI IF_MVAR(,ps) ()
setWindow windowId f
	= do {
		(found,wDevice) <- accIOEnv (ioStGetDevice WindowDevice);
		if   not found
		then return ()
		else 
		let  windows                = windowSystemStateGetWindowHandles wDevice
		     (found,wsH,windows1)   = getWindowHandlesWindow (toWID windowId) windows
		in
		if   not found
		then appIOEnv (ioStSetDevice (WindowSystemState windows1)) >> return ()
		else do {
			wMetrics            <- accIOEnv ioStGetOSWindowMetrics;
			(wsH',wsH1)         <- liftIO (retrieveWindowHandle' wsH);
			let (wids,wsH2)      = getWindowStateHandleWIDS wsH1
			    wstate           = WState {wIds=wids,wRep=wsH',wMetrics=wMetrics}
			in  do {
				wstate1     <- liftIO (f wstate);
				let wsH''    = wRep wstate1
				    wsH3     = insertWindowHandle' wsH'' wsH2
				    windows2 = setWindowHandlesWindow wsH3 windows1
				in  appIOEnv (ioStSetDevice (WindowSystemState windows2))
			    }
		      }
	  }


{-	controlSize calculates the size of the given control.
-}
controlSize :: (Controls cdef) => cdef IF_MVAR(,ls ps) -> Bool -> Maybe (Int,Int) -> Maybe (Int,Int) -> Maybe (Int,Int) -> GUI IF_MVAR(,ps) Size
controlSize cdef isWindow hMargins vMargins itemSpaces
	= do {
		cs       <- controlToHandles cdef;
		wMetrics <- accIOEnv ioStGetOSWindowMetrics;
		let
			itemHs      = map controlStateToWElementHandle cs
			hMargins'   = case hMargins of
					Just (left,right) -> (max 0 left,max 0 right)
					_                 -> if isWindow then (0,0) else (osmHorMargin wMetrics,osmHorMargin wMetrics)
			vMargins'   = case vMargins of
					Just (top,bottom) -> (max 0 top,max 0 bottom)
					_                 -> if isWindow then (0,0) else (osmVerMargin wMetrics,osmVerMargin wMetrics)
			itemSpaces' = case itemSpaces of
					Just (hor,vert)   -> (max 0 hor,max 0 vert)
					_                 -> (osmHorItemSpace wMetrics,osmVerItemSpace wMetrics)
			domain      = viewDomainRange {corner1=zero}
		in liftIO (calcControlsSize wMetrics hMargins' vMargins' itemSpaces' zero zero [(domain,zero)] itemHs)
	  }


setControlTexts :: [(Id,String)] -> GUI IF_MVAR(,ps) ()
setControlTexts cid_texts
	= do {
		idtable <- ioStGetIdTable;
		ioId    <- accIOEnv ioStGetIOId;
		let  cid_texts_wIds  = gatherWindowIds' (filterMap (isOkControlId ioId) (zip cid_texts (getIdParents (fst (unzip cid_texts)) idtable)))
		in
		if   isEmpty cid_texts_wIds
		then return ()
		else sequence_ [setWindow wId (setControlTexts' cid_texts) | (cid_texts,wId)<-cid_texts_wIds]
	  }
	where
		setControlTexts' :: [(Id,String)] -> WState -> IO WState
		setControlTexts' texts wState@(WState {wIds=WIDS {wPtr=wPtr},wRep=wH,wMetrics=wMetrics})
			= setcontroltexts texts wMetrics wPtr wH >>= (\wH -> return wState {wRep=wH})

setControlText :: Id -> String -> GUI IF_MVAR(,ps) ()
setControlText id text
	= setControlTexts [(id,text)]



--	Access operations on WState:

getWStateControls :: WState -> [WElementHandle']
getWStateControls wstate
	= whItems' $ wRep $ wstate

snd3thd3 :: (a,b,c) -> (b,c)
snd3thd3 (_,t2,t3) = (t2,t3)


getControlTexts :: [Id] -> WState -> [(Bool,Maybe String)]
getControlTexts ids wstate
	= map snd3thd3 (snd (getcontrolstexts (getWStateControls wstate) (ids,[(id,defaultBool,defaultValue) | id<-ids])))
	where
		defaultBool  = False
		defaultValue = Nothing

getControlText :: Id -> WState -> (Bool,Maybe String)
getControlText id wstate = hd (getControlTexts [id] wstate)
