{-
HOpenGL - a binding of OpenGL and GLUT for Haskell.
Copyright (C) 2001  Sven Panne <Sven.Panne@BetaResearch.de>

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Library General Public
License as published by the Free Software Foundation; either
version 2 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Library General Public License for more details.

You should have received a copy of the GNU Library General Public
License along with this library (COPYING.LIB); if not, write to the Free
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

This module corresponds to section 4.1 (Per-Fragment Operations) of the
OpenGL 1.2.1 specs.
-}

module GL_PerFragment (
   scissor, ScissorTest(..),
   Comparison(..),
   unmarshalComparison,                 -- internal use only
   alphaFunc, AlphaTest(..),
   stencilFunc, StencilAction(..),
   unmarshalStencilAction,              -- internal use only
   stencilOp, StencilTest(..),
   depthFunc, DepthTest(..),
   blendColor, BlendEquation(..),       -- @GL_1_2@
   unmarshalBlendEquation,              -- @GL_1_2@, internal use only
   blendEquation,                       -- @GL_1_2@
   BlendFunc(..),
   unmarshalBlendFunc,                  -- internal use only
   blendFunc, Blend(..),
   Dither(..),
   LogicOp(..),
   unmarshalLogicOp,                    -- internal use only
   logicOp, ColIdxLogicOp(..)
) where

import GL_BasicTypes    (GLenum, GLint, GLuint, GLsizei, GLclampf,
                         Capability(..))
import GL_Constants     
import GL_VertexSpec    (Color4(..))

---------------------------------------------------------------------------
--  Section 4.1.2 (Scissor test)

foreign import "glScissor" unsafe scissor :: GLint -> GLint -> GLsizei -> GLsizei -> IO ()

data ScissorTest = ScissorTest deriving (Eq,Ord)

instance Capability ScissorTest where
   marshalCapability ScissorTest = gl_SCISSOR_TEST

---------------------------------------------------------------------------
--  Section 4.1.3 (Alpha test)

data Comparison =
     Never
   | Always
   | Less
   | Lequal
   | Equal
   | Gequal
   | Greater
   | Notequal
   deriving (Eq,Ord)

marshalComparison :: Comparison -> GLenum
marshalComparison Never    = gl_NEVER
marshalComparison Always   = gl_ALWAYS
marshalComparison Less     = gl_LESS
marshalComparison Lequal   = gl_LEQUAL
marshalComparison Equal    = gl_EQUAL
marshalComparison Gequal   = gl_GEQUAL
marshalComparison Greater  = gl_GREATER
marshalComparison Notequal = gl_NOTEQUAL

unmarshalComparison :: GLenum -> Comparison
unmarshalComparison c
   | c == gl_NEVER    = Never
   | c == gl_ALWAYS   = Always
   | c == gl_LESS     = Less
   | c == gl_LEQUAL   = Lequal
   | c == gl_EQUAL    = Equal
   | c == gl_GEQUAL   = Gequal
   | c == gl_GREATER  = Greater
   | c == gl_NOTEQUAL = Notequal
   | otherwise        = error "unmarshalComparison"

alphaFunc :: Comparison -> GLclampf -> IO ()
alphaFunc = glAlphaFunc . marshalComparison

foreign import "glAlphaFunc" unsafe glAlphaFunc :: GLenum -> GLclampf -> IO ()

data AlphaTest = AlphaTest deriving (Eq,Ord)

instance Capability AlphaTest where
   marshalCapability AlphaTest = gl_ALPHA_TEST

---------------------------------------------------------------------------
--  Section 4.1.4 (Stencil test)

stencilFunc :: Comparison -> GLint -> GLuint -> IO ()
stencilFunc = glStencilFunc . marshalComparison

foreign import "glStencilFunc" unsafe glStencilFunc ::
   GLenum -> GLint -> GLuint -> IO ()

-- GL_ZERO:    Collision with BlendFunc (resolved here)
-- GL_REPLACE: Collision with GL_Texturing.TextureFunction (resolved here)
-- GL_INVERT : Collision with LogicalOp (resolved here)
data StencilAction =
     Keep
   | Zero'
   | Replace'
   | Incr
   | Decr
   | Invert'
   deriving (Eq,Ord)

marshalStencilAction :: StencilAction -> GLenum
marshalStencilAction Keep     = gl_KEEP
marshalStencilAction Zero'    = gl_ZERO
marshalStencilAction Replace' = gl_REPLACE
marshalStencilAction Incr     = gl_INCR
marshalStencilAction Decr     = gl_DECR
marshalStencilAction Invert'  = gl_INVERT

unmarshalStencilAction :: GLenum -> StencilAction
unmarshalStencilAction a
   | a == gl_KEEP    = Keep
   | a == gl_ZERO    = Zero'
   | a == gl_REPLACE = Replace'
   | a == gl_INCR    = Incr
   | a == gl_DECR    = Decr
   | a == gl_INVERT  = Invert'
   | otherwise       = error "unmarshalStencilAction"

stencilOp :: StencilAction -> StencilAction -> StencilAction -> IO ()
stencilOp f zf zp = glStencilOp (marshalStencilAction f )
                                (marshalStencilAction zf)
                                (marshalStencilAction zp)

foreign import "glStencilOp" unsafe glStencilOp ::
   GLenum -> GLenum -> GLenum -> IO ()

data StencilTest = StencilTest deriving (Eq,Ord)

instance Capability StencilTest where
   marshalCapability StencilTest = gl_STENCIL_TEST

---------------------------------------------------------------------------
--  Section 4.1.5 (Depth buffer test)

depthFunc :: Comparison -> IO ()
depthFunc = glDepthFunc . marshalComparison

foreign import "glDepthFunc" unsafe glDepthFunc :: GLenum -> IO ()

data DepthTest = DepthTest deriving (Eq,Ord)

instance Capability DepthTest where
   marshalCapability DepthTest = gl_DEPTH_TEST

---------------------------------------------------------------------------
--  Section 4.1.6 (Blending)

-- @GL_1_2@
blendColor :: Color4 GLclampf -> IO ()
blendColor (Color4 r g b a) = glBlendColor r g b a

-- @GL_1_2@
foreign import "glBlendColor" unsafe glBlendColor ::
   GLclampf -> GLclampf -> GLclampf -> GLclampf -> IO ()

-- @GL_1_2@
data BlendEquation =
     FuncAdd
   | FuncSubtract
   | FuncReverseSubtract
   | Min
   | Max
   deriving (Eq,Ord)

-- @GL_1_2@
marshalBlendEquation :: BlendEquation -> GLenum
marshalBlendEquation FuncAdd             = gl_FUNC_ADD
marshalBlendEquation FuncSubtract        = gl_FUNC_SUBTRACT
marshalBlendEquation FuncReverseSubtract = gl_FUNC_REVERSE_SUBTRACT
marshalBlendEquation Min                 = gl_MIN
marshalBlendEquation Max                 = gl_MAX

-- @GL_1_2@
unmarshalBlendEquation :: GLenum -> BlendEquation
unmarshalBlendEquation e
   | e == gl_FUNC_ADD              = FuncAdd
   | e == gl_FUNC_SUBTRACT         = FuncSubtract
   | e == gl_FUNC_REVERSE_SUBTRACT = FuncReverseSubtract
   | e == gl_MIN                   = Min
   | e == gl_MAX                   = Max
   | otherwise                     = error "unmarshalBlendEquation"

-- @GL_1_2@
blendEquation :: BlendEquation -> IO ()
blendEquation = glBlendEquation . marshalBlendEquation

-- @GL_1_2@
foreign import "glBlendEquation" unsafe glBlendEquation :: GLenum -> IO ()

-- GL_ZERO: Collision with StencilAction (resolved there)
data BlendFunc =
     Zero
   | One
   | SrcColor
   | DstColor
   | OneMinusSrcColor
   | OneMinusDstColor
   | SrcAlpha
   | OneMinusSrcAlpha
   | DstAlpha
   | OneMinusDstAlpha
   | ConstantColor           -- @GL_1_2@
   | OneMinusConstantColor   -- @GL_1_2@
   | ConstantAlpha           -- @GL_1_2@
   | OneMinusConstantAlpha   -- @GL_1_2@
   | SrcAlphaSaturate
   deriving (Eq,Ord)

marshalBlendFunc :: BlendFunc -> GLenum
marshalBlendFunc Zero                  = gl_ZERO
marshalBlendFunc One                   = gl_ONE
marshalBlendFunc SrcColor              = gl_SRC_COLOR
marshalBlendFunc DstColor              = gl_DST_COLOR
marshalBlendFunc OneMinusSrcColor      = gl_ONE_MINUS_SRC_COLOR
marshalBlendFunc OneMinusDstColor      = gl_ONE_MINUS_DST_COLOR
marshalBlendFunc SrcAlpha              = gl_SRC_ALPHA
marshalBlendFunc OneMinusSrcAlpha      = gl_ONE_MINUS_SRC_ALPHA
marshalBlendFunc DstAlpha              = gl_DST_ALPHA
marshalBlendFunc OneMinusDstAlpha      = gl_ONE_MINUS_DST_ALPHA
marshalBlendFunc ConstantColor         = gl_CONSTANT_COLOR
marshalBlendFunc OneMinusConstantColor = gl_ONE_MINUS_CONSTANT_COLOR
marshalBlendFunc ConstantAlpha         = gl_CONSTANT_ALPHA
marshalBlendFunc OneMinusConstantAlpha = gl_ONE_MINUS_CONSTANT_ALPHA
marshalBlendFunc SrcAlphaSaturate      = gl_SRC_ALPHA_SATURATE

unmarshalBlendFunc :: GLenum ->  BlendFunc
unmarshalBlendFunc f
   | f == gl_ZERO                     = Zero
   | f == gl_ONE                      = One
   | f == gl_SRC_COLOR                = SrcColor
   | f == gl_DST_COLOR                = DstColor
   | f == gl_ONE_MINUS_SRC_COLOR      = OneMinusSrcColor
   | f == gl_ONE_MINUS_DST_COLOR      = OneMinusDstColor
   | f == gl_SRC_ALPHA                = SrcAlpha
   | f == gl_ONE_MINUS_SRC_ALPHA      = OneMinusSrcAlpha
   | f == gl_DST_ALPHA                = DstAlpha
   | f == gl_ONE_MINUS_DST_ALPHA      = OneMinusDstAlpha
   | f == gl_CONSTANT_COLOR           = ConstantColor
   | f == gl_ONE_MINUS_CONSTANT_COLOR = OneMinusConstantColor
   | f == gl_CONSTANT_ALPHA           = ConstantAlpha
   | f == gl_ONE_MINUS_CONSTANT_ALPHA = OneMinusConstantAlpha
   | f == gl_SRC_ALPHA_SATURATE       = SrcAlphaSaturate
   | otherwise                        = error "unmarshalBlendFunc"

blendFunc :: BlendFunc -> BlendFunc -> IO ()
blendFunc s d = glBlendFunc (marshalBlendFunc s) (marshalBlendFunc d)

foreign import "glBlendFunc" unsafe glBlendFunc :: GLenum -> GLenum -> IO ()

-- GL_BLEND: Collision with GL_Texturing.TextureFunction (resolved here)
data Blend = Blend' deriving (Eq,Ord)

instance Capability Blend where
   marshalCapability Blend' = gl_BLEND

---------------------------------------------------------------------------
--  Section 4.1.7 (Dithering)

data Dither = Dither deriving (Eq,Ord)

instance Capability Dither where
   marshalCapability Dither = gl_DITHER

---------------------------------------------------------------------------
--  Section 4.1.8 (Logical Operation)

-- GL_INVERT: Collision with StencilAction (resolved there)
data LogicOp =
     Clear
   | And
   | AndReverse
   | Copy
   | AndInverted
   | Noop
   | Xor
   | Or
   | Nor
   | Equiv
   | Invert
   | OrReverse
   | CopyInverted
   | OrInverted
   | Nand
   | Set
   deriving (Eq,Ord)

marshalLogicOp :: LogicOp -> GLenum
marshalLogicOp Clear        = gl_CLEAR
marshalLogicOp And          = gl_AND
marshalLogicOp AndReverse   = gl_AND_REVERSE
marshalLogicOp Copy         = gl_COPY
marshalLogicOp AndInverted  = gl_AND_INVERTED
marshalLogicOp Noop         = gl_NOOP
marshalLogicOp Xor          = gl_XOR
marshalLogicOp Or           = gl_OR
marshalLogicOp Nor          = gl_NOR
marshalLogicOp Equiv        = gl_EQUIV
marshalLogicOp Invert       = gl_INVERT
marshalLogicOp OrReverse    = gl_OR_REVERSE
marshalLogicOp CopyInverted = gl_COPY_INVERTED
marshalLogicOp OrInverted   = gl_OR_INVERTED
marshalLogicOp Nand         = gl_NAND
marshalLogicOp Set          = gl_SET

unmarshalLogicOp :: GLenum -> LogicOp
unmarshalLogicOp o
   | o == gl_CLEAR         = Clear
   | o == gl_AND           = And
   | o == gl_AND_REVERSE   = AndReverse
   | o == gl_COPY          = Copy
   | o == gl_AND_INVERTED  = AndInverted
   | o == gl_NOOP          = Noop
   | o == gl_XOR           = Xor
   | o == gl_OR            = Or
   | o == gl_NOR           = Nor
   | o == gl_EQUIV         = Equiv
   | o == gl_INVERT        = Invert
   | o == gl_OR_REVERSE    = OrReverse
   | o == gl_COPY_INVERTED = CopyInverted
   | o == gl_OR_INVERTED   = OrInverted
   | o == gl_NAND          = Nand
   | o == gl_SET           = Set
   | otherwise             = error "unmarshalLogicOp"

logicOp :: LogicOp -> IO ()
logicOp = glLogicOp . marshalLogicOp

foreign import "glLogicOp" unsafe glLogicOp :: GLenum -> IO ()

data ColIdxLogicOp =
     IndexLogicOp
   | ColorLogicOp
   deriving (Eq,Ord)

instance Capability ColIdxLogicOp where
   marshalCapability IndexLogicOp = gl_INDEX_LOGIC_OP
   marshalCapability ColorLogicOp = gl_COLOR_LOGIC_OP
