{-
HOpenGL - a binding of OpenGL and GLUT for Haskell.
Copyright (C) 2000  Sven Panne <Sven.Panne@BetaResearch.de>

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Library General Public
License as published by the Free Software Foundation; either
version 2 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Library General Public License for more details.

You should have received a copy of the GNU Library General Public
License along with this library (COPYING.LIB); if not, write to the Free
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

This module corresponds to section 2.10 (Coordinate Transformations) of
the OpenGL 1.2.1 specs.
-}

module GL_CoordTrans (
   depthRange, viewport,
   MatrixMode(..),
   unmarshalMatrixMode,        -- internal use only
   matrixMode, GLcolumn4(..), GLmatrix(..), Vector2(..), Vector3(..),
   MatrixElement(..), loadIdentity, frustum, ortho, pushMatrix, popMatrix,
   AutoNormalizeRescale(..),
   TextureCoordinate(..),
   marshalTextureCoordinate,   -- internal use only
   Plane(..), TexGenFunc(..), texGen, TextureGen(..)
) where

import Monad            ( zipWithM_ )
import Foreign          ( Ptr, Storable(..), castPtr, withObject )

import GL_Constants     ( gl_MODELVIEW, gl_PROJECTION, gl_TEXTURE, gl_COLOR,
                          gl_NORMALIZE, gl_RESCALE_NORMAL, gl_S, gl_T, gl_Q, gl_R,
                          gl_OBJECT_PLANE, gl_TEXTURE_GEN_MODE, gl_OBJECT_LINEAR,
                          gl_EYE_PLANE, gl_TEXTURE_GEN_MODE, gl_EYE_LINEAR,
                          gl_TEXTURE_GEN_MODE, gl_SPHERE_MAP, gl_TEXTURE_GEN_S,
                          gl_TEXTURE_GEN_T, gl_TEXTURE_GEN_R, gl_TEXTURE_GEN_Q )
import GL_BasicTypes    ( GLenum, GLint, GLsizei, GLfloat, GLdouble, GLclampd,
                          WindowPosition(..), WindowSize(..), Viewport(..),
                          Capability(..) )

---------------------------------------------------------------------------
-- Section 2.10.1 (Controlling the Viewport)

foreign import "glDepthRange" unsafe depthRange :: GLclampd -> GLclampd -> IO ()

viewport :: Viewport -> IO ()
viewport (Viewport (WindowPosition x y) (WindowSize w h)) = glViewport x y w h

foreign import "glViewport" unsafe glViewport :: GLint -> GLint -> GLsizei -> GLsizei -> IO ()

---------------------------------------------------------------------------
-- Section 2.10.2 (Matrices)

-- GL_Color: Collision with GL_MovePixels.PixelType (resolved here)
data MatrixMode =
     Modelview
   | Projection
   | Texture
   | Color'
   deriving (Eq,Ord)

marshalMatrixMode :: MatrixMode -> GLenum
marshalMatrixMode Modelview  = gl_MODELVIEW
marshalMatrixMode Projection = gl_PROJECTION
marshalMatrixMode Texture    = gl_TEXTURE
marshalMatrixMode Color'     = gl_COLOR

unmarshalMatrixMode :: GLenum -> MatrixMode
unmarshalMatrixMode mode
   | mode == gl_MODELVIEW  = Modelview
   | mode == gl_PROJECTION = Projection
   | mode == gl_TEXTURE    = Texture
   | mode == gl_COLOR      = Color'
   | otherwise             = error "unmarshalMatrixMode"

matrixMode :: MatrixMode -> IO ()
matrixMode = glMatrixMode . marshalMatrixMode

foreign import "glMatrixMode" unsafe glMatrixMode :: GLenum -> IO ()

-- Not perfect, but workable

data GLcolumn4 a = GLcolumn4 a a a a
data GLmatrix a = GLmatrix (GLcolumn4 a) (GLcolumn4 a) (GLcolumn4 a) (GLcolumn4 a)

instance Storable a => Storable (GLmatrix a) where
   sizeOf    ~(GLmatrix (GLcolumn4 x _ _ _) _ _ _) = 16 * sizeOf x
   alignment ~(GLmatrix (GLcolumn4 x _ _ _) _ _ _) = alignment x

   peek ptr = do
      [ a00, a01, a02, a03,
        a04, a05, a06, a07,
        a08, a09, a10, a11,
        a12, a13, a14, a15 ] <- mapM (peekElemOff (castPtr ptr)) [ 0 .. 15 ]
      return $ GLmatrix (GLcolumn4 a00 a01 a02 a03)
                        (GLcolumn4 a04 a05 a06 a07)
                        (GLcolumn4 a08 a09 a10 a11)
                        (GLcolumn4 a12 a13 a14 a15)

   poke ptr (GLmatrix (GLcolumn4 a00 a01 a02 a03)
                      (GLcolumn4 a04 a05 a06 a07)
                      (GLcolumn4 a08 a09 a10 a11)
                      (GLcolumn4 a12 a13 a14 a15)) =
      zipWithM_ (pokeElemOff (castPtr ptr)) [ 0 .. ]
                [ a00, a01, a02, a03,
                  a04, a05, a06, a07,
                  a08, a09, a10, a11,
                  a12, a13, a14, a15 ]

data Vector2 a = Vector2 a a

data Vector3 a = Vector3 a a a

class MatrixElement a where
   loadMatrix :: GLmatrix a     -> IO ()
   multMatrix :: GLmatrix a     -> IO ()
   rotate     :: a -> Vector3 a -> IO ()
   translate  :: Vector3 a      -> IO ()
   scale      :: a -> a -> a    -> IO ()

instance MatrixElement GLfloat where
   loadMatrix m              = withObject m glLoadMatrixf
   multMatrix m              = withObject m glMultMatrixf
   rotate a (Vector3 x y z)  = glRotatef a x y z
   translate (Vector3 x y z) = glTranslatef x y z
   scale                     = glScalef

instance MatrixElement GLdouble where
   loadMatrix m              = withObject m glLoadMatrixd
   multMatrix m              = withObject m glMultMatrixd
   rotate a (Vector3 x y z)  = glRotated a x y z
   translate (Vector3 x y z) = glTranslated x y z
   scale                     = glScaled

foreign import "glLoadMatrixf" unsafe glLoadMatrixf :: Ptr (GLmatrix GLfloat ) -> IO ()
foreign import "glLoadMatrixd" unsafe glLoadMatrixd :: Ptr (GLmatrix GLdouble) -> IO ()

foreign import "glMultMatrixf" unsafe glMultMatrixf :: Ptr (GLmatrix GLfloat ) -> IO ()
foreign import "glMultMatrixd" unsafe glMultMatrixd :: Ptr (GLmatrix GLdouble) -> IO ()

foreign import "glLoadIdentity" unsafe loadIdentity :: IO ()

foreign import "glRotatef" unsafe glRotatef :: GLfloat  -> GLfloat  -> GLfloat  -> GLfloat  -> IO ()
foreign import "glRotated" unsafe glRotated :: GLdouble -> GLdouble -> GLdouble -> GLdouble -> IO ()

foreign import "glTranslatef" unsafe glTranslatef :: GLfloat  -> GLfloat  -> GLfloat  -> IO ()
foreign import "glTranslated" unsafe glTranslated :: GLdouble -> GLdouble -> GLdouble -> IO ()

foreign import "glScalef" unsafe glScalef :: GLfloat  -> GLfloat -> GLfloat -> IO ()
foreign import "glScaled" unsafe glScaled :: GLdouble -> GLdouble -> GLdouble -> IO ()

foreign import "glFrustum" unsafe frustum ::
   GLdouble -> GLdouble -> GLdouble -> GLdouble -> GLdouble -> GLdouble -> IO ()

foreign import "glOrtho" unsafe ortho ::
   GLdouble -> GLdouble -> GLdouble -> GLdouble -> GLdouble -> GLdouble -> IO ()

foreign import "glPushMatrix" unsafe pushMatrix :: IO ()
foreign import "glPopMatrix"  unsafe popMatrix  :: IO ()

---------------------------------------------------------------------------
-- Section 2.10.3 (Normal Transformation)

data AutoNormalizeRescale =
     Normalize
   | RescaleNormal   -- @GL_1_2@
   deriving (Eq,Ord)

instance Capability AutoNormalizeRescale where
   marshalCapability Normalize     = gl_NORMALIZE
   marshalCapability RescaleNormal = gl_RESCALE_NORMAL

---------------------------------------------------------------------------
-- Section 2.10.4 (Generating Texture Coordinates)

data TextureCoordinate =
     S
   | T
   | R
   | Q
   deriving (Eq,Ord)

marshalTextureCoordinate :: TextureCoordinate -> GLenum
marshalTextureCoordinate S = gl_S
marshalTextureCoordinate T = gl_T
marshalTextureCoordinate Q = gl_Q
marshalTextureCoordinate R = gl_R

data Plane a = Plane a a a a

instance Storable a => Storable (Plane a) where
   sizeOf    ~(Plane p1 _ _ _) = 4 * sizeOf p1
   alignment ~(Plane p1 _ _ _) = alignment p1

   peek addr = do [ p1, p2, p3, p4 ] <- mapM (peekElemOff (castPtr addr)) [ 0 .. 3 ]
                  return $ Plane p1 p2 p3 p4

   poke addr (Plane p1 p2 p3 p4) = zipWithM_ (pokeElemOff (castPtr addr)) [ 0 .. ]
                                             [ p1, p2, p3, p4 ]

-- Note: GL_NV_texgen_reflection (nVidia) extension is missing here
-- (GL_NORMAL_MAP_NV, GL_REFLECTION_MAP_NV)
data TexGenFunc = ObjectLinear (Plane GLdouble)
                | EyeLinear    (Plane GLdouble)
                | SphereMap

texGen :: TextureCoordinate -> TexGenFunc -> IO ()
texGen coord (ObjectLinear plane) = do
   withObject plane $ glTexGendv (marshalTextureCoordinate coord) gl_OBJECT_PLANE
   glTexGeni (marshalTextureCoordinate coord) gl_TEXTURE_GEN_MODE (fromIntegral gl_OBJECT_LINEAR)
texGen coord (EyeLinear plane) = do
   withObject plane $ glTexGendv (marshalTextureCoordinate coord) gl_EYE_PLANE
   glTexGeni (marshalTextureCoordinate coord) gl_TEXTURE_GEN_MODE (fromIntegral gl_EYE_LINEAR)
texGen coord SphereMap =
   glTexGeni (marshalTextureCoordinate coord) gl_TEXTURE_GEN_MODE (fromIntegral gl_SPHERE_MAP)

foreign import "glTexGendv" unsafe glTexGendv :: GLenum -> GLenum -> Ptr (Plane GLdouble) -> IO ()
foreign import "glTexGeni"  unsafe glTexGeni  :: GLenum -> GLenum -> GLint                -> IO ()

data TextureGen =
     TextureGenS
   | TextureGenT
   | TextureGenR
   | TextureGenQ
   deriving (Eq,Ord)

instance Capability TextureGen where
   marshalCapability TextureGenS = gl_TEXTURE_GEN_S
   marshalCapability TextureGenT = gl_TEXTURE_GEN_T
   marshalCapability TextureGenR = gl_TEXTURE_GEN_R
   marshalCapability TextureGenQ = gl_TEXTURE_GEN_Q
