{-
HOpenGL - a binding of OpenGL and GLUT for Haskell.
Copyright (C) 2000  Sven Panne <Sven.Panne@BetaResearch.de>

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Library General Public
License as published by the Free Software Foundation; either
version 2 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Library General Public License for more details.

You should have received a copy of the GNU Library General Public
License along with this library (COPYING.LIB); if not, write to the Free
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

This module corresponds to chapter 4 (Matrix manipulation) of the GLU specs.
-}

module GLU_Matrix (
   ortho2D, perspective, lookAt, pickMatrix,
   project, unProject
) where

import Foreign          (Ptr, Storable(peek), withObject, alloca)

import GL_BasicTypes    (boolToGLint, GLdouble, GLint, Viewport)
import GL_CoordTrans    (GLmatrix)
import GL_VertexSpec    (Vertex3(..), Normal3(..))

---------------------------------------------------------------------------
-- matrix setup

foreign import "gluOrtho2D"     unsafe ortho2D     ::
   GLdouble -> GLdouble -> GLdouble -> GLdouble -> IO ()
foreign import "gluPerspective" unsafe perspective ::
   GLdouble -> GLdouble -> GLdouble -> GLdouble -> IO ()

lookAt :: Vertex3 GLdouble -> Vertex3 GLdouble -> Normal3 GLdouble -> IO ()
lookAt (Vertex3 eyeX eyeY eyeZ) (Vertex3 centerX centerY centerZ) (Normal3 upX upY upZ) =
   gluLookAt eyeX eyeY eyeZ centerX centerY centerZ upX upY upZ

foreign import "gluLookAt" unsafe gluLookAt ::
      GLdouble -> GLdouble -> GLdouble
   -> GLdouble -> GLdouble -> GLdouble
   -> GLdouble -> GLdouble -> GLdouble -> IO ()

pickMatrix :: (GLdouble, GLdouble) -> (GLdouble, GLdouble) -> Viewport -> IO ()
pickMatrix (x, y) (w, h) vp = withObject vp $ gluPickMatrix x y w h

foreign import "gluPickMatrix" unsafe gluPickMatrix ::
   GLdouble -> GLdouble -> GLdouble -> GLdouble -> Ptr Viewport -> IO ()

---------------------------------------------------------------------------
-- coordinate projection
-- TODO: Missing for GLU 1.3: gluUnProject4

type UnProjFunc =
      Vertex3 GLdouble
   -> GLmatrix Double -> GLmatrix Double -> Viewport
   -> IO (Maybe (Vertex3 GLdouble))

type UnProjFuncInternal =
      GLdouble -> GLdouble -> GLdouble
   -> Ptr (GLmatrix GLdouble) -> Ptr (GLmatrix GLdouble) -> Ptr Viewport
   -> Ptr GLdouble -> Ptr GLdouble -> Ptr GLdouble -> IO GLint

project :: UnProjFunc
project = projHelper gluProject

foreign import "gluProject" unsafe gluProject :: UnProjFuncInternal

unProject :: UnProjFunc
unProject = projHelper gluUnProject

foreign import "gluUnProject" unsafe gluUnProject :: UnProjFuncInternal

projHelper :: UnProjFuncInternal -> UnProjFunc
projHelper upf (Vertex3 objX objY objZ) model proj viewp =
   withObject model $ \modelBuf ->
   withObject proj  $ \projBuf  ->
   withObject viewp $ \viewBuf  ->
   alloca           $ \winXBuf  ->
   alloca           $ \winYBuf  ->
   alloca           $ \winZBuf  -> do
   success  <- upf objX objY objZ modelBuf projBuf viewBuf winXBuf winYBuf winZBuf
   if success == boolToGLint True
      then do winX <- peek winXBuf
              winY <- peek winYBuf
              winZ <- peek winZBuf
              return $ Just (Vertex3 winX winY winZ)
      else return Nothing
