#!/usr/bin/env ruby

require 'test/unit'
require 'geos'
require 'test_helper'

class TestGeosOperations < Test::Unit::TestCase
  def setup
    @test_helper = GeosTestHelper.new()
    @wkt_reader = Geos::WKTReader.new(@test_helper.geom_factory)
  end

  def run_operation(results, &block)
    geoms = @test_helper.create_geoms()

    geoms.each_with_index do |geom_in, i|
      # Get expected result
      expected = @wkt_reader.read(results[i])
      
      # Get the actual result
      actual = yield(geom_in)

      assert(actual.equals(expected),
             "Expected: #{expected}\nActual: #{actual}")
    end
  end

  def test_centroid()
    results = Array.new()
    results[0]  = "POINT (150.00 350.00)"
    results[1]  = "POINT (110.00 126.50)"
    results[2]  = "POINT (50.00 50.00)"
    results[3]  = "POINT (150.00 350.00)"
    results[4]  = "POINT (150.00 400.00)"
    results[5]  = "POINT (150.00 375.00)"
    results[6]  = "POINT (-0.00 -0.00)"
    results[7]  = "POINT (-0.00 0.00)"
    results[8]  = "POINT (-0.00 -0.00)"
    results[9]  = "POINT (-0.00 5.00)"
    results[10] = "POINT (8.50 16.00)"

    run_operation(results) do |geom|
      geom.get_centroid()
    end
  end

  def test_buffer()
    results = Array.new()
    results[0]  = "POLYGON ((140.00 348.00, 140.00 350.00, 140.00 352.00, 141.00 354.00, 141.50 355.50, 143.00 357.00, 144.50 358.50, 146.00 359.00, 148.00 360.00, 150.00 360.00, 152.00 360.00, 154.00 359.00, 155.50 358.50, 157.00 357.00, 158.50 355.50, 159.00 354.00, 160.00 352.00, 160.00 350.00, 160.00 348.00, 159.00 346.00, 158.50 344.50, 157.00 343.00, 155.50 341.50, 154.00 341.00, 152.00 340.00, 150.00 340.00, 148.00 340.00, 146.00 341.00, 144.50 341.50, 143.00 343.00, 141.50 344.50, 141.00 346.00, 140.00 348.00))"
    results[1]  = "POLYGON ((50.00 58.00, 50.00 60.00, 50.00 160.00, 50.00 162.00, 51.00 164.00, 51.50 165.50, 53.00 167.00, 54.50 168.50, 56.00 169.00, 58.00 170.00, 60.00 170.00, 160.00 170.00, 162.00 170.00, 164.00 169.00, 165.50 168.50, 167.00 167.00, 168.50 165.50, 169.00 164.00, 170.00 162.00, 170.00 160.00, 170.00 60.00, 170.00 58.00, 169.00 56.00, 168.50 54.50, 167.00 53.00, 165.50 51.50, 164.00 51.00, 162.00 50.00, 160.00 50.00, 158.00 50.00, 156.00 51.00, 154.50 51.50, 153.00 53.00, 151.50 54.50, 151.00 56.00, 150.00 58.00, 150.00 60.00, 150.00 150.00, 70.00 150.00, 70.00 60.00, 70.00 58.00, 69.00 56.00, 68.50 54.50, 67.00 53.00, 65.50 51.50, 64.00 51.00, 62.00 50.00, 60.00 50.00, 58.00 50.00, 56.00 51.00, 54.50 51.50, 53.00 53.00, 51.50 54.50, 51.00 56.00, 50.00 58.00))"
    results[2]  = "POLYGON ((-10.00 -2.00, -10.00 0.00, -10.00 2.00, -10.00 100.00, -10.00 102.00, -9.00 104.00, -8.50 105.50, -7.00 107.00, -5.50 108.50, -4.00 109.00, -2.00 110.00, 0.00 110.00, 100.00 110.00, 102.00 110.00, 104.00 109.00, 105.50 108.50, 107.00 107.00, 108.50 105.50, 109.00 104.00, 110.00 102.00, 110.00 100.00, 110.00 0.00, 110.00 -2.00, 109.00 -4.00, 108.50 -5.50, 107.00 -7.00, 105.50 -8.50, 104.00 -9.00, 102.00 -10.00, 100.00 -10.00, 2.00 -10.00, 0.00 -10.00, -2.00 -10.00, -4.00 -9.00, -5.50 -8.50, -7.00 -7.00, -8.50 -5.50, -9.00 -4.00, -10.00 -2.00), (10.00 10.00, 90.00 10.00, 90.00 90.00, 10.00 90.00, 10.00 10.00))"
    results[3]  = "POLYGON ((-10.00 198.00, -10.00 200.00, -10.00 500.00, -10.00 502.00, -9.00 504.00, -8.50 505.50, -7.00 507.00, -5.50 508.50, -4.00 509.00, -2.00 510.00, 0.00 510.00, 300.00 510.00, 302.00 510.00, 304.00 509.00, 305.50 508.50, 307.00 507.00, 308.50 505.50, 309.00 504.00, 310.00 502.00, 310.00 500.00, 310.00 200.00, 310.00 198.00, 309.00 196.00, 308.50 194.50, 307.00 193.00, 305.50 191.50, 304.00 191.00, 302.00 190.00, 300.00 190.00, 0.00 190.00, -2.00 190.00, -4.00 191.00, -5.50 191.50, -7.00 193.00, -8.50 194.50, -9.00 196.00, -10.00 198.00), (110.00 310.00, 190.00 310.00, 190.00 390.00, 110.00 390.00, 110.00 310.00))"
    results[4]  = "POLYGON ((-10.00 248.00, -10.00 250.00, -10.00 550.00, -10.00 552.00, -9.00 554.00, -8.50 555.50, -7.00 557.00, -5.50 558.50, -4.00 559.00, -2.00 560.00, 0.00 560.00, 300.00 560.00, 302.00 560.00, 304.00 559.00, 305.50 558.50, 307.00 557.00, 308.50 555.50, 309.00 554.00, 310.00 552.00, 310.00 550.00, 310.00 250.00, 310.00 248.00, 309.00 246.00, 308.50 244.50, 307.00 243.00, 305.50 241.50, 304.00 241.00, 302.00 240.00, 300.00 240.00, 0.00 240.00, -2.00 240.00, -4.00 241.00, -5.50 241.50, -7.00 243.00, -8.50 244.50, -9.00 246.00, -10.00 248.00), (110.00 360.00, 190.00 360.00, 190.00 440.00, 110.00 440.00, 110.00 360.00))"
    results[5]  = "MULTIPOLYGON (((-10.00 -2.00, -10.00 0.00, -10.00 2.00, -10.00 100.00, -10.00 102.00, -9.00 104.00, -8.50 105.50, -7.00 107.00, -5.50 108.50, -4.00 109.00, -2.00 110.00, 0.00 110.00, 50.00 110.00, 50.00 160.00, 50.00 162.00, 51.00 164.00, 51.50 165.50, 53.00 167.00, 54.50 168.50, 56.00 169.00, 58.00 170.00, 60.00 170.00, 160.00 170.00, 162.00 170.00, 164.00 169.00, 165.50 168.50, 167.00 167.00, 168.50 165.50, 169.00 164.00, 170.00 162.00, 170.00 160.00, 170.00 60.00, 170.00 58.00, 169.00 56.00, 168.50 54.50, 167.00 53.00, 165.50 51.50, 164.00 51.00, 162.00 50.00, 160.00 50.00, 158.00 50.00, 156.00 51.00, 154.50 51.50, 153.00 53.00, 151.50 54.50, 151.00 56.00, 150.00 58.00, 150.00 60.00, 150.00 150.00, 70.00 150.00, 70.00 110.00, 100.00 110.00, 102.00 110.00, 104.00 109.00, 105.50 108.50, 107.00 107.00, 108.50 105.50, 109.00 104.00, 110.00 102.00, 110.00 100.00, 110.00 0.00, 110.00 -2.00, 109.00 -4.00, 108.50 -5.50, 107.00 -7.00, 105.50 -8.50, 104.00 -9.00, 102.00 -10.00, 100.00 -10.00, 2.00 -10.00, 0.00 -10.00, -2.00 -10.00, -4.00 -9.00, -5.50 -8.50, -7.00 -7.00, -8.50 -5.50, -9.00 -4.00, -10.00 -2.00), (10.00 10.00, 90.00 10.00, 90.00 90.00, 70.00 90.00, 70.00 60.00, 70.00 58.00, 69.00 56.00, 68.50 54.50, 67.00 53.00, 65.50 51.50, 64.00 51.00, 62.00 50.00, 60.00 50.00, 58.00 50.00, 56.00 51.00, 54.50 51.50, 53.00 53.00, 51.50 54.50, 51.00 56.00, 50.00 58.00, 50.00 60.00, 50.00 90.00, 10.00 90.00, 10.00 10.00)), ((-10.00 198.00, -10.00 200.00, -10.00 248.00, -10.00 250.00, -10.00 500.00, -10.00 502.00, -10.00 550.00, -10.00 552.00, -9.00 554.00, -8.50 555.50, -7.00 557.00, -5.50 558.50, -4.00 559.00, -2.00 560.00, 0.00 560.00, 300.00 560.00, 302.00 560.00, 304.00 559.00, 305.50 558.50, 307.00 557.00, 308.50 555.50, 309.00 554.00, 310.00 552.00, 310.00 550.00, 310.00 502.00, 310.00 500.00, 310.00 250.00, 310.00 248.00, 310.00 200.00, 310.00 198.00, 309.00 196.00, 308.50 194.50, 307.00 193.00, 305.50 191.50, 304.00 191.00, 302.00 190.00, 300.00 190.00, 0.00 190.00, -2.00 190.00, -4.00 191.00, -5.50 191.50, -7.00 193.00, -8.50 194.50, -9.00 196.00, -10.00 198.00), (110.00 360.00, 148.00 360.00, 150.00 360.00, 152.00 360.00, 190.00 360.00, 190.00 390.00, 110.00 390.00, 110.00 360.00)))"
    results[6]  = "POLYGON ((-15.00 -2.50, -15.00 -2.00, -15.00 -1.50, -15.00 -0.50, -15.00 0.50, -15.00 1.50, -15.00 2.00, -15.00 2.50, -14.50 3.00, -14.50 3.50, -14.50 4.00, -14.50 4.50, -14.00 5.00, -14.00 6.00, -13.50 6.00, -13.50 6.50, -13.50 7.00, -13.00 7.50, -13.00 8.00, -12.50 8.50, -12.00 9.00, -12.00 9.50, -11.50 10.00, -11.00 10.00, -10.50 10.50, -10.00 11.00, -10.00 11.50, -9.50 12.00, -9.00 12.00, -8.50 12.50, -8.00 13.00, -7.50 13.00, -7.00 13.50, -6.50 13.50, -6.00 13.50, -6.00 14.00, -5.00 14.00, -4.50 14.50, -4.00 14.50, -3.50 14.50, -3.00 14.50, -2.50 15.00, -2.00 15.00, -1.50 15.00, -0.50 15.00, 0.50 15.00, 1.50 15.00, 2.00 15.00, 2.50 15.00, 3.00 14.50, 3.50 14.50, 4.00 14.50, 4.50 14.50, 5.00 14.00, 6.00 14.00, 6.00 13.50, 6.50 13.50, 7.00 13.50, 7.50 13.00, 8.00 13.00, 8.50 12.50, 9.00 12.00, 9.50 12.00, 10.00 11.50, 10.00 11.00, 10.50 10.50, 11.00 10.00, 11.50 10.00, 12.00 9.50, 12.00 9.00, 12.50 8.50, 13.00 8.00, 13.00 7.50, 13.50 7.00, 13.50 6.50, 13.50 6.00, 14.00 6.00, 14.00 5.00, 14.50 4.50, 14.50 4.00, 14.50 3.50, 14.50 3.00, 15.00 2.50, 15.00 2.00, 15.00 1.50, 15.00 0.50, 15.00 -0.50, 15.00 -1.50, 15.00 -2.00, 15.00 -2.50, 14.50 -3.00, 14.50 -3.50, 14.50 -4.00, 14.50 -4.50, 14.00 -5.00, 14.00 -6.00, 13.50 -6.00, 13.50 -6.50, 13.50 -7.00, 13.00 -7.50, 13.00 -8.00, 12.50 -8.50, 12.00 -9.00, 12.00 -9.50, 11.50 -10.00, 11.00 -10.00, 10.50 -10.50, 10.00 -11.00, 10.00 -11.50, 9.50 -12.00, 9.00 -12.00, 8.50 -12.50, 8.00 -13.00, 7.50 -13.00, 7.00 -13.50, 6.50 -13.50, 6.00 -13.50, 6.00 -14.00, 5.00 -14.00, 4.50 -14.50, 4.00 -14.50, 3.50 -14.50, 3.00 -14.50, 2.50 -15.00, 2.00 -15.00, 1.50 -15.00, 0.50 -15.00, -0.50 -15.00, -1.50 -15.00, -2.00 -15.00, -2.50 -15.00, -3.00 -14.50, -3.50 -14.50, -4.00 -14.50, -4.50 -14.50, -5.00 -14.00, -6.00 -14.00, -6.00 -13.50, -6.50 -13.50, -7.00 -13.50, -7.50 -13.00, -8.00 -13.00, -8.50 -12.50, -9.00 -12.00, -9.50 -12.00, -10.00 -11.50, -10.00 -11.00, -10.50 -10.50, -11.00 -10.00, -11.50 -10.00, -12.00 -9.50, -12.00 -9.00, -12.50 -8.50, -13.00 -8.00, -13.00 -7.50, -13.50 -7.00, -13.50 -6.50, -13.50 -6.00, -14.00 -6.00, -14.00 -5.00, -14.50 -4.50, -14.50 -4.00, -14.50 -3.50, -14.50 -3.00, -15.00 -2.50))"
    results[7]  = "POLYGON ((-14.00 -3.00, -14.00 -2.50, -14.00 -2.00, -14.00 -1.50, -14.00 -1.00, -14.00 -0.50, -14.00 -0.00, -14.00 0.50, -14.00 1.00, -14.00 1.50, -14.00 2.00, -14.00 2.50, -14.00 3.00, -13.50 3.50, -13.50 4.00, -13.50 4.50, -13.50 5.00, -13.00 5.50, -13.00 6.00, -13.00 6.50, -12.50 7.00, -12.50 7.50, -12.00 8.00, -12.00 8.50, -11.50 9.00, -11.00 9.50, -11.00 10.00, -10.50 10.50, -10.00 11.00, -10.00 11.50, -9.50 12.00, -9.00 12.50, -8.50 12.50, -8.00 13.00, -8.00 13.50, -7.00 14.00, -6.00 14.50, -5.00 15.00, -4.00 15.50, -3.00 15.50, -2.00 16.00, -1.50 16.00, -0.50 16.00, 0.50 16.00, 1.50 16.00, 2.00 16.00, 3.00 15.50, 4.00 15.50, 5.00 15.00, 6.00 14.50, 7.00 14.00, 8.00 13.50, 8.00 13.00, 8.50 12.50, 9.00 12.50, 9.50 12.00, 10.00 11.50, 10.00 11.00, 10.50 10.50, 11.00 10.00, 11.00 9.50, 11.50 9.00, 12.00 8.50, 12.00 8.00, 12.50 7.50, 12.50 7.00, 13.00 6.50, 13.00 6.00, 13.00 5.50, 13.50 5.00, 13.50 4.50, 13.50 4.00, 13.50 3.50, 14.00 3.00, 14.00 2.50, 14.00 2.00, 14.00 1.50, 14.00 1.00, 14.00 0.50, 14.00 0.00, 14.00 -0.50, 14.00 -1.00, 14.00 -1.50, 14.00 -2.00, 14.00 -2.50, 14.00 -3.00, 13.50 -3.50, 13.50 -4.00, 13.50 -4.50, 13.50 -5.00, 13.00 -5.50, 13.00 -6.00, 13.00 -6.50, 12.50 -7.00, 12.50 -7.50, 12.00 -8.00, 12.00 -8.50, 11.50 -9.00, 11.00 -9.50, 11.00 -10.00, 10.50 -10.50, 10.00 -11.00, 10.00 -11.50, 9.50 -12.00, 9.00 -12.50, 8.50 -12.50, 8.00 -13.00, 8.00 -13.50, 7.00 -14.00, 6.00 -14.50, 5.00 -15.00, 4.00 -15.50, 3.00 -15.50, 2.00 -16.00, 1.50 -16.00, 0.50 -16.00, -0.50 -16.00, -1.50 -16.00, -2.00 -16.00, -3.00 -15.50, -4.00 -15.50, -5.00 -15.00, -6.00 -14.50, -7.00 -14.00, -8.00 -13.50, -8.00 -13.00, -8.50 -12.50, -9.00 -12.50, -9.50 -12.00, -10.00 -11.50, -10.00 -11.00, -10.50 -10.50, -11.00 -10.00, -11.00 -9.50, -11.50 -9.00, -12.00 -8.50, -12.00 -8.00, -12.50 -7.50, -12.50 -7.00, -13.00 -6.50, -13.00 -6.00, -13.00 -5.50, -13.50 -5.00, -13.50 -4.50, -13.50 -4.00, -13.50 -3.50, -14.00 -3.00))"
    results[8]  = "POLYGON ((-15.00 -7.00, -15.00 -5.00, -15.00 5.00, -15.00 7.00, -14.00 9.00, -13.50 10.50, -12.00 12.00, -10.50 13.50, -9.00 14.00, -7.00 15.00, -5.00 15.00, 5.00 15.00, 7.00 15.00, 9.00 14.00, 10.50 13.50, 12.00 12.00, 13.50 10.50, 14.00 9.00, 15.00 7.00, 15.00 5.00, 15.00 -5.00, 15.00 -7.00, 14.00 -9.00, 13.50 -10.50, 12.00 -12.00, 10.50 -13.50, 9.00 -14.00, 7.00 -15.00, 5.00 -15.00, -5.00 -15.00, -7.00 -15.00, -9.00 -14.00, -10.50 -13.50, -12.00 -12.00, -13.50 -10.50, -14.00 -9.00, -15.00 -7.00))"
    results[9]  = "POLYGON ((-15.00 -7.00, -15.00 -5.00, -15.00 15.00, -15.00 17.00, -14.00 19.00, -13.50 20.50, -12.00 22.00, -10.50 23.50, -9.00 24.00, -7.00 25.00, -5.00 25.00, 5.00 25.00, 7.00 25.00, 9.00 24.00, 10.50 23.50, 12.00 22.00, 13.50 20.50, 14.00 19.00, 15.00 17.00, 15.00 15.00, 15.00 -5.00, 15.00 -7.00, 14.00 -9.00, 13.50 -10.50, 12.00 -12.00, 10.50 -13.50, 9.00 -14.00, 7.00 -15.00, 5.00 -15.00, -5.00 -15.00, -7.00 -15.00, -9.00 -14.00, -10.50 -13.50, -12.00 -12.00, -13.50 -10.50, -14.00 -9.00, -15.00 -7.00))"
    results[10] = "POLYGON ((-5.00 18.00, -5.00 20.00, -5.00 22.00, -4.00 24.00, -3.50 25.50, -2.00 27.00, -0.50 28.50, 1.00 29.00, 3.00 30.00, 5.00 30.00, 6.00 30.00, 8.00 30.00, 9.00 29.50, 10.00 29.00, 11.50 28.50, 12.00 28.00, 12.50 28.00, 13.00 27.50, 13.50 27.00, 14.00 26.50, 14.50 26.00, 15.00 25.50, 15.50 25.00, 16.00 24.50, 16.50 24.00, 17.00 23.00, 17.00 22.50, 17.00 22.00, 17.50 22.00, 18.00 20.50, 18.50 19.50, 19.00 18.50, 19.00 18.00, 19.50 17.00, 19.50 16.00, 20.00 15.00, 20.00 13.00, 20.00 10.00, 20.00 8.00, 19.00 6.00, 18.50 4.50, 17.00 3.00, 15.50 1.50, 14.00 1.00, 12.00 0.00, 10.00 0.00, 8.00 0.00, 6.00 1.00, 4.50 1.50, 3.00 3.00, 1.50 4.50, 1.00 6.00, 0.00 8.00, 0.00 10.00, -0.50 11.00, -0.50 11.50, -2.00 13.00, -3.50 14.50, -4.00 16.00, -5.00 18.00))"

    # Note - this test is quite slow
    run_operation(results) do |geom|
      geom.buffer(10)
    end
  end
  
  def test_convex_hull()
    results = Array.new()
    results[0]  = "POINT (150.00 350.00)"
    results[1]  = "POLYGON ((60.00 60.00, 60.00 160.00, 160.00 160.00, 160.00 60.00, 60.00 60.00))"
    results[2]  = "POLYGON ((0.00 0.00, 0.00 100.00, 100.00 100.00, 100.00 0.00, 0.00 0.00))"
    results[3]  = "POLYGON ((0.00 200.00, 0.00 500.00, 300.00 500.00, 300.00 200.00, 0.00 200.00))"
    results[4]  = "POLYGON ((0.00 250.00, 0.00 550.00, 300.00 550.00, 300.00 250.00, 0.00 250.00))"
    results[5]  = "POLYGON ((0.00 0.00, 0.00 550.00, 300.00 550.00, 300.00 200.00, 100.00 0.00, 0.00 0.00))"
    results[6]  = "POLYGON ((-5.00 0.00, -4.99 0.31, -4.96 0.63, -4.91 0.94, -4.84 1.24, -4.76 1.55, -4.65 1.84, -4.52 2.13, -4.38 2.41, -4.22 2.68, -4.05 2.94, -3.85 3.19, -3.64 3.42, -3.42 3.64, -3.19 3.85, -2.94 4.05, -2.68 4.22, -2.41 4.38, -2.13 4.52, -1.84 4.65, -1.55 4.76, -1.24 4.84, -0.94 4.91, -0.63 4.96, -0.31 4.99, 0.00 5.00, 0.31 4.99, 0.63 4.96, 0.94 4.91, 1.24 4.84, 1.55 4.76, 1.84 4.65, 2.13 4.52, 2.41 4.38, 2.68 4.22, 2.94 4.05, 3.19 3.85, 3.42 3.64, 3.64 3.42, 3.85 3.19, 4.05 2.94, 4.22 2.68, 4.38 2.41, 4.52 2.13, 4.65 1.84, 4.76 1.55, 4.84 1.24, 4.91 0.94, 4.96 0.63, 4.99 0.31, 5.00 0.00, 4.99 -0.31, 4.96 -0.63, 4.91 -0.94, 4.84 -1.24, 4.76 -1.55, 4.65 -1.84, 4.52 -2.13, 4.38 -2.41, 4.22 -2.68, 4.05 -2.94, 3.85 -3.19, 3.64 -3.42, 3.42 -3.64, 3.19 -3.85, 2.94 -4.05, 2.68 -4.22, 2.41 -4.38, 2.13 -4.52, 1.84 -4.65, 1.55 -4.76, 1.24 -4.84, 0.94 -4.91, 0.63 -4.96, 0.31 -4.99, -0.00 -5.00, -0.31 -4.99, -0.63 -4.96, -0.94 -4.91, -1.24 -4.84, -1.55 -4.76, -1.84 -4.65, -2.13 -4.52, -2.41 -4.38, -2.68 -4.22, -2.94 -4.05, -3.19 -3.85, -3.42 -3.64, -3.64 -3.42, -3.85 -3.19, -4.05 -2.94, -4.22 -2.68, -4.38 -2.41, -4.52 -2.13, -4.65 -1.84, -4.76 -1.55, -4.84 -1.24, -4.91 -0.94, -4.96 -0.63, -4.99 -0.31, -5.00 0.00))"
    results[7]  = "POLYGON ((-4.00 0.00, -3.99 0.38, -3.97 0.75, -3.93 1.12, -3.87 1.49, -3.80 1.85, -3.72 2.21, -3.62 2.55, -3.51 2.89, -3.38 3.21, -3.24 3.53, -3.08 3.82, -2.92 4.11, -2.74 4.37, -2.55 4.62, -2.35 4.85, -2.14 5.07, -1.93 5.26, -1.70 5.43, -1.47 5.58, -1.24 5.71, -0.99 5.81, -0.75 5.89, -0.50 5.95, -0.25 5.99, 0.00 6.00, 0.25 5.99, 0.50 5.95, 0.75 5.89, 0.99 5.81, 1.24 5.71, 1.47 5.58, 1.70 5.43, 1.93 5.26, 2.14 5.07, 2.35 4.85, 2.55 4.62, 2.74 4.37, 2.92 4.11, 3.08 3.82, 3.24 3.53, 3.38 3.21, 3.51 2.89, 3.62 2.55, 3.72 2.21, 3.80 1.85, 3.87 1.49, 3.93 1.12, 3.97 0.75, 3.99 0.38, 4.00 0.00, 3.99 -0.38, 3.97 -0.75, 3.93 -1.12, 3.87 -1.49, 3.80 -1.85, 3.72 -2.21, 3.62 -2.55, 3.51 -2.89, 3.38 -3.21, 3.24 -3.53, 3.08 -3.82, 2.92 -4.11, 2.74 -4.37, 2.55 -4.62, 2.35 -4.85, 2.14 -5.07, 1.93 -5.26, 1.70 -5.43, 1.47 -5.58, 1.24 -5.71, 0.99 -5.81, 0.75 -5.89, 0.50 -5.95, 0.25 -5.99, -0.00 -6.00, -0.25 -5.99, -0.50 -5.95, -0.75 -5.89, -0.99 -5.81, -1.24 -5.71, -1.47 -5.58, -1.70 -5.43, -1.93 -5.26, -2.14 -5.07, -2.35 -4.85, -2.55 -4.62, -2.74 -4.37, -2.92 -4.11, -3.08 -3.82, -3.24 -3.53, -3.38 -3.21, -3.51 -2.89, -3.62 -2.55, -3.72 -2.21, -3.80 -1.85, -3.87 -1.49, -3.93 -1.12, -3.97 -0.75, -3.99 -0.38, -4.00 0.00))"
    results[8]  = "POLYGON ((-5.00 -5.00, -5.00 5.00, 5.00 5.00, 5.00 -5.00, -5.00 -5.00))"
    results[9]  = "POLYGON ((-5.00 -5.00, -5.00 15.00, 5.00 15.00, 5.00 -5.00, -5.00 -5.00))"
    results[10] = "POLYGON ((5.00 20.00, 6.00 20.00, 7.00 19.50, 8.00 18.50, 9.00 16.50, 9.50 15.00, 10.00 13.00, 10.00 10.00, 5.00 20.00))"

    run_operation(results) do |geom|
      geom.convex_hull()
    end
  end
end