// ---------------------------------------------------------------------
// $Id: GeneticAlgorithm.hh 153 2008-06-04 05:49:38Z daaugusto $
//
//   GeneticAlgorithm.hh (created on Tue Nov 08 01:08:35 BRT 2005)
// 
//   Genetic Algorithm File Fitter (gaffitter)
//
//   Copyright (C) 2005-2008 Douglas A. Augusto
// 
// This file is part of gaffitter.
// 
// gaffitter is free software; you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by the
// Free Software Foundation; either version 3 of the License, or (at
// your option) any later version.
// 
// gaffitter is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with gaffitter; if not, see <http://www.gnu.org/licenses/>.
//
// ---------------------------------------------------------------------

#ifndef GeneticAlgorithm_hh
#define GeneticAlgorithm_hh

#include "../Input.hh"
#include "../util/Random.hh"
#include "../Optimizer.hh"

#include <vector>
#include <iterator>

using namespace std;

// ---------------------------------------------------------------------
/**
 * This class represents a Genome, i. e., an individual (candidate
 * solution) and its score.
 */
class Genome {
public:
   Genome(int size): m_chrom(size) {}

   virtual ~Genome() {}

   /** Chromosome size. */
   int Length() const { return m_chrom.size(); }

   /** Get the ith Gene of m_chrom. */
   bool Gene(int i) const { return m_chrom[i]; }
   /** Set the ith Gene of m_chrom. */
   void Gene(int i, bool gene) { m_chrom[i] = gene; }

   /** Returns true if the individual is bigger than target. */
   bool OverTarget() const { return m_diff < 0.0; }
   /** Returns true if the individual is smaller than target. */
   bool UnderTarget() const { return m_diff > 0.0; }
   /** Returns true if the individual is smaller (or equal) than target. */
   bool Feasible() const { return m_diff >= 0.0; }

   /**
    * Diff is the raw "distance" between the target and the sum of the
    * size of each file specified (gene = 1) by chromosome. The distance
    * maybe positive (smaller than target), zero (perfect fit) or
    * negative (greater than target). 
    */
   Params::Size_t Diff() const { return m_diff; }

   /** Sets the m_diff (\ref Diff()). */
   void Diff(Params::Size_t diff) { m_diff = diff; }

   /** Verify if the given size 's' can be inserted and update the \ref
    * m_diff. */
   void AddSize(int i, Params::Size_t s) 
   { 
      if (!Gene(i) && s<=Diff()) { Gene(i,true); m_diff -= s; }
   }
   /** 
    * Remove the 's' amount of size and update the \ref m_diff. */
   void DelSize(int i, Params::Size_t s) 
   { 
      if (Gene(i)) { Gene(i,false); m_diff += s; }
   }
   /** 
    * Writes the genome (vector containing '0's and '1's) in ostream
    * object (usually cout).
    */
   virtual ostream& Write(ostream& s) const
   {
      copy(m_chrom.begin(),m_chrom.end(),ostream_iterator<bool>(s));
      s << " [Size: " << m_chrom.size() << ", Diff: " << Diff() 
        << " bytes]"  << endl;

      return s;
   }

public:
   /**
    * Holds the list of selected files (candidate solution). '0' stands
    * for 'no selected file' and '1' stands for 'selected file'. Its
    * size is determined by the number of input files. The files are
    * sorted by size in reverse order (bigger to smaller).
    */
   vector<bool> m_chrom;

   /** The diff ("error") between the total size and the target. */
   Params::Size_t m_diff;
};

inline ostream& operator<< (ostream& s, const Genome& genome)
{
   return genome.Write(s);
}

// --------------------------------------------------------------------
/**
 * This class implements a Genetic Algorithm system as search algorithm.
 * This implementation aims to minimize the distance between a candidate
 * solution and the specified target.
 */
class GeneticAlgorithm: public Optimizer {
public:
   /**
    * Automatically adjust population size and num. of generations. Also
    * sets Random::Seed. These parameters are static (its adjustment)
    * over the iterations, however, \ref Initialize() is called before
    * each iteration.
    */
   GeneticAlgorithm(vector<SizeName>&, Params&);
   /**
    * Free the population.
    */
   ~GeneticAlgorithm();
private:
   /** Copies are not allowed. */
   GeneticAlgorithm(const GeneticAlgorithm&);
   /** Attributions are not allowed. */
   GeneticAlgorithm& operator= (const GeneticAlgorithm&);

public:

   /** Genetic Algorithm search.
    *
    * GA searches for optimal (not guaranteed) solutions by means of
    * Darwinian evolution.
    */
   void Evolve();

protected:
   /**
    * Initialize/adjust some parameters (per iteration) and prints some
    * information if required.
    */
   void Initialize();

   /**
    * Reset the GA system for a new iteration.
    */
   void Reset();

   /**
    * Create the population of individuals.
    *
    * The first created individuals are sparse (several '0' genes).
    * However the last individuals contain several '1' genes. This
    * improves the diversity and exploration of the search space.
    */
   void InitPopulation();

   /**
    * Runs the GA for one generation. Returns true if a perfect fit was
    * found; false otherwise.
    */
   bool Generation();

   /** Just for notation convenience. */
   Params::Size_t Score(int index) const { return Score(*m_genomes[index]); }

   /**
    * Returns how well is the given Genome. A perfect Genome is found
    * when its Score is equal to zero. Not so fitter returns a positive
    * (>0) score.
    */
   Params::Size_t Score(const Genome& g) const { return g.Diff(); }

   /** 
    * Copy the given gene with some mistakes (determined by the mutate
    * probability).
    */
   bool CopyMutate(bool gene) const
   { 
      return (Random::Probability(m_prob_mutation)) ? !gene : gene;
   }

   /**
    * This genetic operator performs a local optimization upon a Genome.
    *
    * First it makes the individual feasible, from left to right or
    * vice-versa. Finally, LocalOptimizer fills the individual to
    * minimize the difference (target - sum)
    */
   void LocalOptimizer(Genome&) const;
   /**
    * Select at random a gene and then change it to better fit the
    * target.
    *
    * If 'g' is feasible then change the gene to '0', otherwise try
    * change the gene to '1'.
    */
   void DirectedMutation(Genome& g) const;

   /** Select in [a,b] a random individual. */
   int Select(int a, int b) const { return Random::Int(a,b); }


   /** Select in [a,b] a random individual 'r' not equal 'p'. */
   int Select(int a, int b, int p) const 
   { 
      int r = Select(a,b);
      
      // equals?
      if (r==p) return r>a ? r-1 : a+1;

      return r; // Ok: 'b' not equal 'a'!
   }
   
   /** 
    * Promotes the crossover between two individuals, generating two
    * children.
    */
   void Reproduction(int, int, int, int) const;
   /** 
    * Choose an individual via tournament (need 2 or more competitors).
    * This is a two-competitor optimized version.
    *
    * \sa TournamentN()
    */
   pair<int,int> Tournament2(int, int) const;
   /** 
    * Choose an individual via tournament (need 2 or more competitors).
    * This is a three+ competitor version.
    *
    * \sa Tournament2()
    */
   pair<int,int> TournamentN(int, int) const;

   /**
    * Pointer to \ref Tournament2() or \ref TournamentN(), depending on
    * the tournament size.
    */
   pair<int,int> (GeneticAlgorithm::*Tournament)(int, int) const;

   /** The population (vector) of genomes/individuals. */
   vector<Genome*> m_genomes;

protected:
   /** 
    * Writes some information (like algorithm name and parameters) in
    * ostream object (usually cout).
    */
   ostream& Write(ostream&) const;

protected:
   int m_num_gen; /**< Number of generations. */
   int m_pop_size; /**< Population size. */
   int m_tournament_size; /**< Tournament size. */
   float m_prob_crossover; /**< Crossover probability. */
   float m_prob_mutation; /**< Mutation probability. */

private:
   /** Current population. */
   int m_cur_gen;

   /** 
    * This function makes the probability of choosing gene "1" varying
    * between 0% and 100%. However it is biased (non uniform), making
    * most probabilities near to 0.5 (50%).
    *
    * The first created individuals contain several '0's  (very sparse).
    * On the other hand, the last individuals contain several '1's.
    */
   float NonUniformFiller(float) const;
};

// --------------------------------------------------------------------
#endif
