/* 

                          Firewall Builder

                 Copyright (C) 2003 NetCitadel, LLC

  Author:  Vadim Kurland     vadim@fwbuilder.org

  $Id: ObjectManipulator.h,v 1.32 2004/08/24 04:41:21 vkurland Exp $

  This program is free software which we release under the GNU General Public
  License. You may redistribute and/or modify this program under the terms
  of that license as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
 
  To get a copy of the GNU General Public License, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/


#ifndef  __OBJECTMANIPULATOR_H_
#define  __OBJECTMANIPULATOR_H_

#include "config.h"
#include "global.h"

#include <qdialog.h>
#include <qlistview.h>
#include <qtooltip.h>

#include "objectmanipulator_q.h"

#include "fwbuilder/FWObject.h"
#include "fwbuilder/FWObjectDatabase.h"

#include <stack>

class ObjectTreeView;
class ObjectTreeViewItem;
class QComboBox;
class QPopupMenu;

class HistoryItem {
    ObjectTreeViewItem *itm;
    QString         objId;

 public:
    HistoryItem(ObjectTreeViewItem *oi,const QString &id) { itm=oi; objId=id; }
    ~HistoryItem();
    ObjectTreeViewItem* item() { return itm;   }
    QString         id()   { return objId; }
};

class ObjToolTip : public QToolTip {

    ObjectTreeView *otv;
 public:
    ObjToolTip(ObjectTreeView *widget);

    virtual void maybeTip(const QPoint &p);
};

class ObjectManipulator : public ObjectManipulator_q {

    Q_OBJECT

    std::vector<libfwbuilder::FWObject*>          idxToLibs;
    std::vector<QListView*>                       idxToTrees;
    
    std::stack<HistoryItem>                       history;

    libfwbuilder::FWObject                        *currentObj;
    ObjectTreeView                                *currentTreeView;
    
    int                                            treeWidth;
    int                                            treeHeight;

    bool                                           active;
    
/* this is a reverse idex of all objects in all trees. We use it to
 * quickly locate given object in the tree and open it
 */
    std::map<libfwbuilder::FWObject*, ObjectTreeViewItem*> allItems;

    
/* Under some circumstances, user may select several host or fw
 * objects so that their children objects are selected as well
 * (e.g. when shift-click is used). "Delete objects" or "group
 * objects" operations will work on all children objects, which leads
 * to unexpected results since it is not obvious to the user that
 * children objects were selected (since they are invisible). We need
 * to remove them from the list before we delete or perform other
 * actions.
 */
    void simplifySelection(std::vector<libfwbuilder::FWObject*> &selectedObjects);
    
    ObjectTreeViewItem* insertObject( ObjectTreeViewItem *itm,libfwbuilder::FWObject *obj );
    void insertSubtree( ObjectTreeViewItem *itm,libfwbuilder::FWObject *obj );

    void removeObjectFromTreeView(libfwbuilder::FWObject *obj );

    QString getTreeLabel( libfwbuilder::FWObject *obj );

    void addTreePage(libfwbuilder::FWObject *lib);
    void showObjectInTree(ObjectTreeViewItem *otvi);

    int  getIdxForLib(libfwbuilder::FWObject*);
    void addLib( libfwbuilder::FWObject *lib, QListView* otv=NULL);
    void removeLib(libfwbuilder::FWObject*);
    void removeLib(int idx);
    void updateCreateObjectMenu(libfwbuilder::FWObject* lib);
    void makeNameUnique(libfwbuilder::FWObject* p,libfwbuilder::FWObject* obj);

    libfwbuilder::FWObject* actuallyCreateObject(libfwbuilder::FWObject *parent,
                                                 const QString &objType,
                                                 const QString &objName,
                                                 libfwbuilder::FWObject *copyFrom=NULL);
    void autorename(libfwbuilder::FWObject *obj,bool ask=true);
    void autorename(libfwbuilder::FWObject *obj,
                    const std::string &objtype,
                    const std::string &namesuffix);
    
 public slots:
     virtual void libChanged(int l);

     void selectionChanged();

     void info();
     void edit(libfwbuilder::FWObject *obj);
     
     void contextMenu(QListViewItem *item, const QPoint &pos, int col);

     libfwbuilder::FWObject* createObject(const QString &objType,
                                          const QString &objName,
                                          libfwbuilder::FWObject *copyFrom=NULL);
     
     libfwbuilder::FWObject* createObject(libfwbuilder::FWObject *parent,
                                          const QString &objType,
                                          const QString &objName,
                                          libfwbuilder::FWObject *copyFrom=NULL);
     
     void newLibrary();
     void newObject();
     void newFirewall();
     void newHost();
     void newInterface();
     void newNetwork();
     void newAddress();
     void newPhysicalAddress();
     void newAddressRange();
     void newObjectGroup();
     
     void newCustom();
     void newIP();
     void newICMP();
     void newTCP();
     void newUDP();
     void newServiceGroup();
     
     void newInterval();

     void editObj();
     void duplicateObj(int libid);
     void duplicateObjUnderSameParent();
     void moveObj(int libid);
     void copyObj();
     void cutObj();
     void pasteObj();
     void deleteObj();
     void dumpObj();
     void compile();
     void install();

     void groupObjects();
     
     void openObject(QListViewItem *otvi);
     void openObject(libfwbuilder::FWObject *obj);

     void find();
     
     virtual void back();
    
 public:
    
    ObjectManipulator( QWidget *parent );
    void loadObjects();
    void loadObjects(libfwbuilder::FWObjectDatabase *db);
    void clearObjects();
    
    bool validateDialog();
    void invalidateDialog();

    void openObject(libfwbuilder::FWObject *obj, bool register_in_history);
    void openObject(ObjectTreeViewItem *otvi,    bool register_in_history);

    libfwbuilder::FWObject* duplicateObject(libfwbuilder::FWObject *target,
                                            libfwbuilder::FWObject *obj,
                                            const QString &name = QString::null,
                                            bool  askForAutorename=true);
    void moveObject(libfwbuilder::FWObject *target,
                    libfwbuilder::FWObject *obj);

    void moveObject(const QString &targetLibName,
                    libfwbuilder::FWObject *obj);
    
    libfwbuilder::FWObject* getOpened() { return currentObj; }

    void updateLibColor(libfwbuilder::FWObject *lib);
    void updateLibName(libfwbuilder::FWObject *lib);

    void updateObjName(libfwbuilder::FWObject *obj,
                       const QString &oldName,
                       bool  askForAutorename=true);
    void updateObjName(libfwbuilder::FWObject *obj,
                       const QString &oldName,
                       const QString &oldLabel,
                       bool  askForAutorename=true);
    
    ObjectTreeView* getCurrentObjectTree();

    /**
     *  this method opens given library in the tree
     */
    void openLib(libfwbuilder::FWObject *lib);

    /**
     * returns pointer at a library that is currently opened in the tree
     */
    libfwbuilder::FWObject*  getCurrentLib();

    /**
     *  this method makes sure the system library is NOT opened in the
     *  tree. If it is, it switches to the 'User' library. If one of
     *  the user's libraries is already opened, it does nothing.
     */
    void closeSystemLib();

    libfwbuilder::FWObject* pasteTo(libfwbuilder::FWObject *target,libfwbuilder::FWObject *obj,bool openobj=true);
    void delObj(libfwbuilder::FWObject *obj,bool openobj=true);

    /**
     * select whatever object is current in the tree (used to restore
     * selected state of the tree item after it was unselected)
     */
    void select();

    /**
     * unselect whatever object is currently selected
     */
    void unselect();

    /**
     * returns true if anything is selected in the tree
     */
    bool isSelected();

    /**
     * controls whether "Deleted Objects" library is shown
     */
    void showDeletedObjects(bool f);

 signals:
/**
 * the dialog class should have a slot that can load object's data
 * into dialog elements when ObjectManipulator emits this signal
 */
    void loadObject_sign(libfwbuilder::FWObject *);

/**
 * the dialog class should have a slot that can verify data entered by
 * user in the dialog elements when ObjectManipulator emits this
 * signal. The validation result is returned in variable "bool *res"
 */
    void validate_sign(bool *res);

/**
 * the dialog class should have a slot that can verify if the data in
 * dialog has been edited.
 */
    void isChanged_sign(bool *res);

/**
 * the dialog class should have a slot that applies changes made by
 * the user and saves data in the object.
 */
    void applyChanges_sign();

};

#endif
