/* 

                          Firewall Builder

                 Copyright (C) 2000 Vadim Kurland

  Author:  Vadim Kurland     vadim@vk.crocodile.org

  $Id: PrintDialog.cc,v 1.26 2001/12/29 10:06:35 vkurland Exp $


  This program is free software which we release under the GNU General Public
  License. You may redistribute and/or modify this program under the terms
  of that license as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
 
  To get a copy of the GNU General Public License, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/

#include "config.h"

#include <list>
#include <unistd.h>
#include <stdio.h>
#include <sys/types.h>
#include <dirent.h>
#include <iostream.h>

#include <gtk--.h>


#include "StockButton.hh"
#include "PrintDialog.hh"
#include "fwbuilder/XMLTools.hh"
#include "fwbuilder/FWObjectDatabase.hh"
#include "FileSel.hh"
#include "Preferences.hh"

using namespace libfwbuilder;

const string PrintDialog::FILTERS_DIR = TEMPLATE_DIR  "/filters";

PrintDialog::PrintDialog(const gchar *id):Gtk::Dialog(),
                           data_box(false,10),
                           labels_box(false,0),
                           entry_box(false,0),
                           rb_printer("Printer"),
                           rb_file("File"),
                           dst_radio_box(false,10),
                           dst_label("Destinatoin:"),
                           pcmd_label("Print Command:"),
                           pcmd_field(80),
                           pfname_box(false,10),
                           pfname_label("File Name:"),
                           pfname_field(80),
                           pfname_button("Browse..."),
                           flt_label("Filter:"),
                           rb_all("All Tree"),
                           rb_current("Current Node"),
                           src_radio_box(false,10),
                           src_label("Print:")

{
    set_position(GTK_WIN_POS_CENTER);

    current_node = id;

    rb_file.set_group(rb_printer.group());
    rb_all.set_group(rb_current.group());

    // Init GUI using prefs data
    loadPrefs();

    rb_all.set_active(true);

    rb_file.set_active(to_file);
    rb_printer.set_active(!to_file);
    on_dst_changed();

    pcmd_field.set_text(command);
    pfname_field.set_text(filename);

    bool flt_found = false;
    list<string> gl;
    DIR *dir=opendir(FILTERS_DIR.c_str());
    if(dir)
    {
        struct dirent *de;
        string suffix = ".xsl";
        while((de=readdir(dir)))
        {
            string fn=de->d_name;
            if(fn.length()>suffix.length() &&
               fn.substr(fn.length()-suffix.length()).compare(suffix)==0)
                gl.push_back(fn);
            if(!flt_found && fn==filter)
                flt_found=true;
        }
        closedir(dir);
    }
    flt_list.set_popdown_strings(gl);
    flt_list.get_entry()->set_editable(false);
    if(flt_found)
    {
        //previously used filter found - make it current
        flt_list.get_entry()->set_text(filter);
    } 
    
    // Layout
    labels_box.pack_start(dst_label );

    dst_radio_box.pack_start(rb_file   );
    dst_radio_box.pack_start(rb_printer);
    entry_box.pack_start(dst_radio_box);

    labels_box.pack_start(src_label );

    src_radio_box.pack_start(rb_all  );
    src_radio_box.pack_start(rb_current);
    entry_box.pack_start(src_radio_box);

    labels_box.pack_start(pcmd_label);
    entry_box.pack_start(pcmd_field);
    
    labels_box.pack_start(pfname_label);
    pfname_box.pack_start(pfname_field);
    pfname_box.pack_start(pfname_button, TRUE, FALSE);
    entry_box.pack_start(pfname_box);

    labels_box.pack_start(flt_label);
    entry_box.pack_start(flt_list);


    ok_button     = manage(new StockButton(StockButton::OK));
    cancel_button = manage(new StockButton(StockButton::CANCEL));
    ok_button->set_usize(80, 30);
    cancel_button->set_usize(80, 30);

    Gtk::HBox *actionarea = get_action_area();
    g_assert(actionarea!=NULL);
    actionarea->set_border_width(10);
    actionarea->set_homogeneous(true);
    actionarea->set_spacing(5);
    actionarea->pack_start(*ok_button, false, false, 0);
    actionarea->pack_start(*cancel_button, false, false, 0);

    data_box.pack_start(labels_box);
    data_box.pack_start(entry_box);

    Gtk::VBox *dataarea = get_vbox();
    g_assert(dataarea!=NULL);
    dataarea->set_border_width(10);
    dataarea->set_spacing(5);
    dataarea->pack_start(data_box);

    set_usize(350, 250);
    set_title("Print");
    set_policy(true, true, false);

    rb_file.toggled.connect(slot(this,&PrintDialog::on_dst_changed));
    pfname_button.clicked.connect(SigC::slot(this, &PrintDialog::on_browse_clicked));
    ok_button->clicked.connect(SigC::slot(this, &PrintDialog::on_ok_clicked));
    cancel_button->clicked.connect(SigC::slot(this, &PrintDialog::on_cancel_clicked));

    show_all();   
}

void PrintDialog::on_dst_changed()
{
    bool f = rb_file.get_active();
    pfname_field.set_editable(f);
    pcmd_field.set_editable(!f);
    pfname_button.set_sensitive(f); 
}

void PrintDialog::on_browse_clicked()
{
    FileSel *fs=new FileSel("Select file",
			    Preferences::global_prefs->getWdir(),
			    "");
    string df=fs->run();
    delete fs;
    if(df!="")
        pfname_field.set_text(df);
}

PrintDialog::~PrintDialog()
{
}

void PrintDialog::savePrefs()
{
    // We do not save last file name into preferences!
    Preferences::global_prefs->setOpt("/FWBuilderPreferences/Print/Command"   , command );
    Preferences::global_prefs->setOpt("/FWBuilderPreferences/Print/Filter"    , filter  );
    Preferences::global_prefs->setOptBool("/FWBuilderPreferences/Print/ToFile", to_file );
    try
    {
        Preferences::global_prefs->savePrefs();
    } catch(FWException &ex)
    {
        cerr << "Error saving preferences: " << ex.toString() << endl ;
    }
}

void PrintDialog::loadPrefs()
{
    command = Preferences::global_prefs->getOpt("/FWBuilderPreferences/Print/Command"    );
    filter  = Preferences::global_prefs->getOpt("/FWBuilderPreferences/Print/Filter"     );
    to_file = Preferences::global_prefs->getOptBool("/FWBuilderPreferences/Print/ToFile" );
    filename = "";
}

void PrintDialog::on_ok_clicked()
{
    choice = true;
    Gtk::Main::quit();
}

void PrintDialog::on_cancel_clicked()
{
    choice = false;
    Gtk::Main::quit();
}

void PrintDialog::run() throw(FWException)
{
    Gtk::Main::grab_add(*this);
    Gtk::Main::run();
    Gtk::Main::grab_remove(*this);
    hide();
    
    if(!choice) 
        return; // cancel
    
    command  = pcmd_field.get_text()   ;
    filename = pfname_field.get_text() ;
    to_file  = rb_file.get_active()    ;
    filter   = flt_list.get_entry()->get_text() ;
    
    savePrefs();
        
    xmlDocPtr doc=xmlNewDoc(TOXMLCAST("1.0"));
    try
    {
        FWObjectDatabase::db->saveXML(doc);
    } catch(FWException &ex)
    {
        xmlFreeDoc(doc);  
        throw;
    }

    const char *params[3];
    string p1;
    if(rb_current.get_active() && current_node)
    {
        // Printing only current node.
        // pass ID of the current node
        // to XSLT processor
        p1=string("'") + current_node + "'";
    } else
    {
        // printing all tree
        // pass ID of the root node
        // to XSLT processor
        p1=string("'") + FWObjectDatabase::db->getId() + "'";
    } 
    params[0]="PrintFrom";
    params[1]=p1.c_str();
    params[2]=NULL;

    string dst_file;
    if(to_file)
    {
        dst_file=filename;
    } else
    {
        char fnbuf[L_tmpnam+2];
        dst_file=tmpnam(fnbuf);
    }

    try
    {
        XMLTools::transformDocumentToFile(doc, 
                                          FILTERS_DIR+"/"+filter, 
                                          params,
                                          dst_file
        );
    } catch(FWException &ex)
    {
        xmlFreeDoc(doc);
        throw;
    }
    xmlFreeDoc(doc);
  
    int rc;
    if(!to_file)
    {
        string cmd=command+" < "+ dst_file;
        rc=system(cmd.c_str());
        unlink(dst_file.c_str());
    } else
    {
        rc=0;
    }

    if(rc)
        throw FWException("Execution of print command failed with code: "+rc);
    
}




