/*
 * Copyright (C) 2000-2001 Chris Ross and Evan Webb
 * Copyright (C) 1999-2000 Chris Ross
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to
 * deal in the Software without restriction, including without limitation the
 * rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
 * sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *   
 * The above copyright notice and this permission notice shall be included in
 * all copies of the Software, its documentation and marketing & publicity 
 * materials, and acknowledgment shall be given in the documentation, materials
 * and software packages that this Software was used.
 *    
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
 * THE AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER 
 * IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */
#include "ferite.h"
#include <pthread.h>

/*!
 * The locking mutex for the Garbage Collector
 */
pthread_mutex_t     __ferite_gc_lock = PTHREAD_MUTEX_INITIALIZER;
#define LOCK_GC     pthread_mutex_lock( &__ferite_gc_lock )
#define UNLOCK_GC   pthread_mutex_unlock( &__ferite_gc_lock )

/*!
 * The size of the Garbage Collextor
 */
int __ferite_gc_size = FE_GC_INITIAL_SIZE;

/*!
 * \fn void __ferite_init_gc()
 * \brief Initialise the garbage collector
 */
void __ferite_init_gc()
{
   int i;
   
   FE_ENTER_FUNCTION;
   genv->gc = fmalloc( sizeof(FeriteObject) * __ferite_gc_size );
   for( i = 0; i < __ferite_gc_size; i++ )
     genv->gc[i] = NULL;
   FE_LEAVE_FUNCTION( NOWT );
}

/*!
 * \fn void __ferite_deinit_gc()
 * \brief Clean the garbage collector up and shut it down
 */
void __ferite_deinit_gc()
{
   int i = 0;
   
   FE_ENTER_FUNCTION;
   FUD(("GC: +---------------------+\n"));
   FUD(("GC: | CLEANING UP GC      |\n"));
   FUD(("GC: +---------------------+\n"));
   for( i = 0; i < __ferite_gc_size; i++ )
   {
      if( genv->gc[i] != NULL )
      {
	 FUD(("Deleting object %s [%d]\n", genv->gc[i]->name, genv->gc[i]->refcount ));
	__ferite_delete_class_object( genv->gc[i] );
      }
   }
   ffree( genv->gc );
   FE_LEAVE_FUNCTION( NOWT );
}

/*!
 * \fn void __ferite_add_to_gc( FeriteObject *obj )
 * \brief Add an object to the GC
 * \param obj The FeriteObject to add
 * 
 * This function will automaticaly grow the gc if it runs out of space
 */
void __ferite_add_to_gc( FeriteObject *obj )
{
   int i, j;
   FE_ENTER_FUNCTION;
   
   LOCK_GC;
   for( i = 0; i < __ferite_gc_size; i++ )
   {
      if( genv->gc[i] == NULL )
      {
	 FUD(( "GC: Object \"%s\" (%d)\n", obj->name, i ));
	 genv->gc[i] = obj;
	 UNLOCK_GC;
	 FE_LEAVE_FUNCTION( NOWT );
      }
   }
   FUD(("GC: ERROR: no more space in the gc\n"));
   FUD(("GC: Increasing size\n" ));
   i = __ferite_gc_size;
   j = __ferite_gc_size;
   __ferite_gc_size *= 2;
   genv->gc = frealloc( genv->gc, sizeof( FeriteObject ) * __ferite_gc_size );
   for( ; j < __ferite_gc_size; j++ )
     genv->gc[j] = NULL;
   genv->gc[i] = obj;
   __ferite_check_gc(); /* force gc checking if we run put of space */
   UNLOCK_GC;
   FE_LEAVE_FUNCTION( NOWT );
}

/*!
 * \fn void __ferite_check_gc()
 * \brief Check the GC and free up any objects that have yet to be cleaned up
 */
void __ferite_check_gc()
{
   int i;
   FE_ENTER_FUNCTION;

   LOCK_GC;
   FUD(( "GC: Scanning gc for unreferenced objects\n" ));
   for( i = 0; i < __ferite_gc_size; i++ )
   {
      if( genv->gc[i] != NULL && genv->gc[i]->refcount <= 0 )
      {
		 FUD(( "GC: Destroying object \"%s\"\n", genv->gc[i]->name ));
		 __ferite_delete_class_object( genv->gc[i] );
		 genv->gc[i] = NULL;
      }
   }
   UNLOCK_GC;
   FE_LEAVE_FUNCTION( NOWT );
}
