/*
 * Copyright (C) 2000-2001 Chris Ross and Evan Webb
 * Copyright (C) 1999-2000 Chris Ross
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to
 * deal in the Software without restriction, including without limitation the
 * rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
 * sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies of the Software, its documentation and marketing & publicity
 * materials, and acknowledgment shall be given in the documentation, materials
 * and software packages that this Software was used.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
 * THE AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
 * IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */
#include "ferite.h"
#include "builder_compile.h"
#include <setjmp.h>

typedef struct _ferite_compile_record
{
   FeriteFunction     *function;
   FeriteVariableHash *variables;
   FeriteClass        *cclass;
   FeriteScript       *script;
   FeriteNamespace    *ns;
}
BuilderCompileRecord;

/* these are used to keep track of verious bit's and pieces, and makes
 * it possible to compile ferite in one pass rather than several passes
 * like other langyages :) */
FeriteStack  *fwd_look_stack;
FeriteStack  *bck_look_stack;
FeriteStack  *break_look_stack;

/* mwhahhaa, we can keep track of these things :) */
FeriteStack  *compile_stack;
BuilderCompileRecord *current_compile;

/* make life easier :) */
#define CURRENT_NAMESPACE current_compile->ns
#define CURRENT_FUNCTION  current_compile->function
#define CURRENT_CLASS     current_compile->cclass
#define CURRENT_VARS      current_compile->variables
#define CURRENT_SCRIPT    current_compile->script

/* stolen from ferite_scanner.l -> so we can report errors on files */
extern char   *__builder_scanner_file;
extern int     __builder_scanner_lineno;
int            __builder_compile_error = 0;
jmp_buf        __builder_compiler_jmpback;
extern char header[8096];

FeriteBkRequest *__builder_create_request( FeriteOp *op, int type )
{
   FeriteBkRequest *ptr;

   FE_ENTER_FUNCTION;
   ptr = fmalloc( sizeof( FeriteBkRequest ) );
   ptr->reqop = op;
   ptr->type = type;
   FE_LEAVE_FUNCTION( ptr );
}

void __builder_destroy_request( FeriteBkRequest *ptr )
{
   FE_ENTER_FUNCTION;
   ffree( ptr );
   FE_LEAVE_FUNCTION( NOWT );
}

void __builder_delete_request_stack( FeriteStack *stack )
{
   int i;
   FE_ENTER_FUNCTION;
   for( i = 0; i <= stack->stack_ptr; i++ )
   {
      if( stack->stack[i] != NULL )
		ffree( stack->stack[i] );
   }
   ffree( stack->stack );
   ffree( stack );
   FE_LEAVE_FUNCTION( NOWT );
}

void __builder_clean_compiler()
{
   FE_ENTER_FUNCTION;
   builder_clean_parser();
   __builder_delete_request_stack( fwd_look_stack );
   __builder_delete_request_stack( bck_look_stack );
   __builder_delete_request_stack( break_look_stack );
   __builder_delete_request_stack( compile_stack );
   ffree( current_compile );
   fwd_look_stack = NULL;
   bck_look_stack = NULL;
   break_look_stack = NULL;
   compile_stack = NULL;
   FE_LEAVE_FUNCTION( NOWT );
}

FeriteScript *builder_script_compile( char *filename )
{
   FeriteScript *ptr;
   FeriteScript *compiled_script;

   FE_ENTER_FUNCTION;
   ptr = ferite_new_script();
   ferite_script_load( ptr, filename );
   if( ptr->scripttext != NULL )
   {
      builder_set_filename( filename );
      compiled_script = __builder_compile_string( ptr->scripttext );
      if( compiled_script != NULL )
      {
		 __ferite_delete_namespace( compiled_script, ptr->mainns );
		 ptr->mainns = compiled_script->mainns;
		 __ferite_delete_stack( compiled_script->exec_stack );
		 ffree( compiled_script );
		 FE_LEAVE_FUNCTION(ptr);
      }
      else
      {
		 ferite_error( CURRENT_SCRIPT, " -> \"%s\"\n", filename );
		 ferite_script_delete( ptr );
		 FE_LEAVE_FUNCTION( NULL );
      }
   }
   else
   {
      __ferite_delete_stack( ptr->exec_stack );
      __ferite_delete_namespace( ptr, ptr->mainns );
      ffree( ptr );
      ferite_error( NULL, "Can't open script %s\n", filename );
      FE_LEAVE_FUNCTION( NULL );
   }
   FE_LEAVE_FUNCTION( NULL );
}

FeriteScript *__builder_compile_string( char *str )
{
   FeriteScript *new_script = NULL;

   FE_ENTER_FUNCTION;

   new_script = ferite_new_script();
   current_compile = fmalloc( sizeof( BuilderCompileRecord ) );
   CURRENT_SCRIPT = new_script;
   CURRENT_CLASS = NULL;
   CURRENT_VARS = NULL;
   CURRENT_FUNCTION = NULL;
   CURRENT_NAMESPACE = new_script->mainns;

   fwd_look_stack = __ferite_create_stack( 32 );
   bck_look_stack = __ferite_create_stack( 32 );
   break_look_stack = __ferite_create_stack( 32 );
   compile_stack = __ferite_create_stack( 32 );

   FUD(("Setting up parser\n"));
   builder_prepare_parser( str );
   FUD(("Running parser\n"));
   if( setjmp( __builder_compiler_jmpback ) == 0 )
   {  /* we compiled correctly */
      builder_parse();
      FUD(("Cleaning Up Parser\n"));
      __builder_clean_compiler();
      FE_LEAVE_FUNCTION( new_script );
   }
   else
   {  /* uh, *cough* we didn't */
      ferite_error( NULL, "Error compiling script\n" );
      __builder_clean_compiler();
      ferite_script_delete( new_script );
      FE_LEAVE_FUNCTION( NULL );
   }
   FE_LEAVE_FUNCTION( NULL );
}

void __builder_do_load_module( char *name )
{
   FE_ENTER_FUNCTION;
   FUD(( "COMPILER: Request to load module %s\n", name ));

   printf( "Add a module loader to the module initlisation code" );

   if( __builder_compile_error )
   {
      ferite_error( CURRENT_SCRIPT, "Can't load module \"%s\" into the ferite engine, make sure it installed.\n" );
      ffree( name );
      longjmp( __builder_compiler_jmpback, 1 );
   }
   FE_LEAVE_FUNCTION( NOWT );
}

void __builder_do_load_script( char *name )
{
   FeriteScript *script;
   char *scripttext;
   jmp_buf old___builder_compiler_jmpback;

   FE_ENTER_FUNCTION;
   FUD(( "COMPILER: Request to load script %s\n", name ));

   /* save the older enviroment */
   memcpy( &old___builder_compiler_jmpback, &__builder_compiler_jmpback, sizeof(jmp_buf) );

   ferite_save_lexer();
   script = fmalloc( sizeof(FeriteScript) );
   ferite_script_load( script, name );
   ferite_set_filename( name );
   FUD(( "COMPILER: Setting up parser (__builder_do_load_script)\n"));
   scripttext = script->scripttext;
   if( script->filename != NULL )
     ffree( script->filename );
   ffree( script );

   if( scripttext != NULL )
     ferite_prepare_parser( scripttext );
   else
   {
      ferite_error( CURRENT_SCRIPT, "Can't include file \"%s\" (check that the paths are correct)\n", name );
      ferite_restore_lexer();
      __builder_compile_error = 1;
      ffree( name ); /* free up the reseource given to us - as the lexer/parser wont in this case */
      memcpy( &__builder_compiler_jmpback, &old___builder_compiler_jmpback, sizeof(jmp_buf) );
      longjmp( __builder_compiler_jmpback, 1 );
   }

   if( setjmp( __builder_compiler_jmpback ) == 0 )
   {
      FUD(( "COMPILER: Running parser (__builder_do_load_script)\n" ));
      ferite_parse();
      FUD(( "COMPILER: Cleaning Up Parser (__builder_do_load_script)\n" ));
      ferite_clean_parser();
      ferite_restore_lexer();
      ffree( scripttext );
      /* we need to do this as we is not wantin to av a start function */
      __ferite_delete_namespace_element_from_namespace( CURRENT_SCRIPT, CURRENT_NAMESPACE, "_start" );
      /* restore the enviroment */
      memcpy( &__builder_compiler_jmpback, &old___builder_compiler_jmpback, sizeof(jmp_buf) );
   }
   else
   {
      ferite_error( CURRENT_SCRIPT, "Can't include file \"%s\" (Due to compliation errors)\n", name );
      ferite_clean_parser();
      ferite_restore_lexer();
      __builder_compile_error = 1;
      ffree( name ); /* free up the reseource given to us - as the lexer/parser wont in this case */
      ffree( scripttext );
      memcpy( &__builder_compiler_jmpback, &old___builder_compiler_jmpback, sizeof(jmp_buf) );
      longjmp( __builder_compiler_jmpback, 1 );
   }
   FE_LEAVE_FUNCTION(NOWT);
}

void __builder_do_function_header( char *name, int is_static )
{
   FeriteFunction *new_function;
   BuilderCompileRecord *rec;
   int i = 0;

   FE_ENTER_FUNCTION;

   new_function = fmalloc( sizeof( FeriteFunction ) );
   new_function->name = fstrdup( name );
   new_function->type = FNC_IS_EXTRL;
   new_function->localvars = NULL;
   new_function->ccode = NULL;
   new_function->signature = fmalloc( sizeof( FeriteParameterRecord* ) * 10 );
   for( i = 0; i < 10; i++ )
     new_function->signature[i] = NULL;
   new_function->arg_count = 0;
   new_function->returnt = NULL;
   new_function->next = NULL;
   new_function->is_static = is_static;
   new_function->fncPtr = NULL;

   rec = current_compile;
   __ferite_stack_push( compile_stack, current_compile );
   current_compile = fmalloc( sizeof( BuilderCompileRecord ) );
   CURRENT_SCRIPT = NULL;
   CURRENT_FUNCTION = new_function;
   CURRENT_VARS = new_function->localvars;
   CURRENT_NAMESPACE = rec->ns;
   CURRENT_CLASS = rec->cclass;
   FE_LEAVE_FUNCTION( NOWT );
}

void __builder_do_c_code( char *ccode )
{
   FUD(( "Found ccode = \"%s\"\n", ccode ));
   CURRENT_FUNCTION->fncPtr = (void*)fstrdup( ccode );
}

void __builder_do_function_footer()
{
   FE_ENTER_FUNCTION;
   if( CURRENT_CLASS == NULL )
   { /* add to a script */
      if( __ferite_namespace_element_exists( CURRENT_SCRIPT, CURRENT_NAMESPACE, CURRENT_FUNCTION->name ) != NULL )
      {
		 ferite_error( CURRENT_SCRIPT, "Compile Error: function %s already exists in script", CURRENT_FUNCTION->name );
		 __builder_compile_error = 1;
		 __ferite_delete_function_list( CURRENT_SCRIPT, CURRENT_FUNCTION );
		 longjmp( __builder_compiler_jmpback, 1 );
      }
      __ferite_register_ns_function( CURRENT_SCRIPT, CURRENT_NAMESPACE, CURRENT_FUNCTION );
   }
   else
   { /* add to a class */
      if( __ferite_hash_get( CURRENT_SCRIPT, CURRENT_CLASS->functions, CURRENT_FUNCTION->name ) != NULL )
      {
		 ferite_error( CURRENT_SCRIPT, "Compile Error: function %s already exists in class %s", CURRENT_FUNCTION->name, CURRENT_CLASS->name );
		 __builder_compile_error = 1;
		 __ferite_delete_function_list( CURRENT_SCRIPT, CURRENT_FUNCTION );
		 longjmp( __builder_compiler_jmpback, 1 );
      }
      ferite_register_class_function( CURRENT_SCRIPT, CURRENT_CLASS, CURRENT_FUNCTION );
   }
   ffree( current_compile );
   current_compile = __ferite_stack_pop( compile_stack );
   FE_LEAVE_FUNCTION( NOWT );
}

void __builder_do_class_header( char *name, char *extends )
{
   FeriteClass *classtmpl;
   FeriteClass *ptr = NULL;

   FE_ENTER_FUNCTION;

   /*classtmpl = ferite_register_inherited_class( CURRENT_NAMESPACE, name, extends );*/
   /* taken from src/ferite_class.c : ferite_register_inherited_class */
   if( (__ferite_namespace_element_exists( CURRENT_SCRIPT, CURRENT_NAMESPACE, name )) != NULL )
   {
      ferite_warning( CURRENT_SCRIPT, "Class %s already exists can't register\n", name );
      FE_LEAVE_FUNCTION( NOWT );
   }
   classtmpl = fmalloc( sizeof( FeriteClass ) );
   classtmpl->name = fstrdup( name );
   classtmpl->variables = __ferite_variable_hash_alloc( CURRENT_SCRIPT, 50 );
   classtmpl->functions = __ferite_create_hash( CURRENT_SCRIPT, 15 );
   classtmpl->id = 0;
   classtmpl->parent = ( extends == NULL ? NULL : (void *)fstrdup( extends ) );
   classtmpl->next = NULL;
   __ferite_register_ns_class( CURRENT_SCRIPT, CURRENT_NAMESPACE, classtmpl );
   /* end code taken from src/ferite_class.c : ferite_register_inherited_class */

   __ferite_stack_push( compile_stack, current_compile );
   current_compile = fmalloc( sizeof( BuilderCompileRecord ) );
   CURRENT_VARS = classtmpl->variables;
   CURRENT_CLASS = classtmpl;
   CURRENT_SCRIPT = NULL;
   CURRENT_FUNCTION = NULL;
   FE_LEAVE_FUNCTION( NOWT );
}

void __builder_do_class_footer()
{
   FE_ENTER_FUNCTION;
   ffree( current_compile );
   current_compile = __ferite_stack_pop( compile_stack );
   FE_LEAVE_FUNCTION( NOWT );
}

void __builder_do_namespace_header( char *name )
{
   FeriteNamespace *ns;
   FE_ENTER_FUNCTION;
   if( __ferite_namespace_element_exists( CURRENT_SCRIPT, CURRENT_NAMESPACE, name ) == NULL )
   {
      FUD(( "registering namespace %s\n", name ));
      ns = __ferite_register_namespace( CURRENT_SCRIPT, name, CURRENT_NAMESPACE );
      __ferite_stack_push( compile_stack, current_compile );
      current_compile = fmalloc( sizeof( BuilderCompileRecord ) );
      CURRENT_VARS = NULL;
      CURRENT_CLASS = NULL;
      CURRENT_SCRIPT = NULL;
      CURRENT_FUNCTION = NULL;
      CURRENT_NAMESPACE = ns;
   }
   else
   {
      ferite_warning( CURRENT_SCRIPT, "Namespace %s already exists\n", name );
      ffree( name );
      longjmp( __builder_compiler_jmpback, 1 );
   }
   FE_LEAVE_FUNCTION( NOWT );
}

void __builder_do_namespace_footer()
{
   FE_ENTER_FUNCTION;
   ffree( current_compile );
   current_compile = __ferite_stack_pop( compile_stack );
   FE_LEAVE_FUNCTION( NOWT );
}

#define CHECK_FUNCTION_CODE( blah ) \
   if( CURRENT_FUNCTION == NULL ) { \
      ferite_error( CURRENT_SCRIPT, "Compile Error: on line %d, in %s\n", __builder_scanner_lineno, __builder_scanner_file ); \
      __builder_compile_error = 1; \
      blah; \
      longjmp( __builder_compiler_jmpback, 1 ); \
   }

void __builder_do_add_variable_to_paramlist( char *name, char *type )
{
   FeriteVariable *new_variable;

   FE_ENTER_FUNCTION;
   CHECK_FUNCTION_CODE(
					 {
						ffree(name); ffree(type);
					 }
					   );
   /* now lets add the paramter */
   if( strcmp( type, "number" ) == 0 )
     new_variable = __ferite_create_number_long_variable( name, 0 );
   if( strcmp( type, "string" ) == 0 )
     new_variable = __ferite_create_string_variable( name, "" );
   if( strcmp( type, "object" ) == 0 )
     new_variable = __ferite_create_object_variable( name );
   if( strcmp( type, "array" ) == 0 )
     new_variable = __ferite_create_uarray_variable( name, 0 );
   if( strcmp( type, "void" ) == 0 )
     new_variable = __ferite_create_void_variable( name );
   CURRENT_FUNCTION->signature[CURRENT_FUNCTION->arg_count] = fmalloc( sizeof( FeriteParameterRecord* ) );
   CURRENT_FUNCTION->signature[CURRENT_FUNCTION->arg_count]->variable = new_variable;
   CURRENT_FUNCTION->signature[CURRENT_FUNCTION->arg_count]->has_default_value = 0;
   CURRENT_FUNCTION->arg_count++;
   FE_LEAVE_FUNCTION( NOWT );
}

void __builder_do_flags_c( char *code )
{
   FE_ENTER_FUNCTION;
   if( __ferite_namespace_element_exists( CURRENT_SCRIPT, CURRENT_SCRIPT->mainns, "'module_flags" ) != NULL) {
      ferite_warning(NULL, "Found another flags block in the .fec file. You can only have one flags block.\n");
      FE_LEAVE_FUNCTION( NOWT );
   }

   __ferite_register_ns_variable( CURRENT_SCRIPT, CURRENT_SCRIPT->mainns, fe_new_str("'module_flags", code ) );
   FE_LEAVE_FUNCTION( NOWT );
}

void __builder_do_header_c( char *code )
{
   FE_ENTER_FUNCTION;
   if( __ferite_namespace_element_exists( CURRENT_SCRIPT, CURRENT_SCRIPT->mainns, "'module_header" ) != NULL ){
	  ferite_warning( NULL, "Found another header block in the .fec file. You can only have one header block.\n" );
	  FE_LEAVE_FUNCTION( NOWT );
   }
   __ferite_register_ns_variable( CURRENT_SCRIPT, CURRENT_SCRIPT->mainns, fe_new_str( "'module_header", code ) );
   FE_LEAVE_FUNCTION( NOWT );
}

void __builder_do_module_init_c( char *code )
{
   FE_ENTER_FUNCTION;
   if( __ferite_namespace_element_exists( CURRENT_SCRIPT, CURRENT_SCRIPT->mainns, "'module_init" ) != NULL ){
	  ferite_warning( NULL, "Found another initialisation block in the .fec file. You can only have initialisation block.\n" );
	  FE_LEAVE_FUNCTION( NOWT );
   }
   __ferite_register_ns_variable( CURRENT_SCRIPT, CURRENT_SCRIPT->mainns, fe_new_str( "'module_init", code ) );
   FE_LEAVE_FUNCTION( NOWT );
}

void __builder_do_module_deinit_c( char *code )
{
   FE_ENTER_FUNCTION;
   if( __ferite_namespace_element_exists( CURRENT_SCRIPT, CURRENT_SCRIPT->mainns, "'module_deinit" ) != NULL ){
	  ferite_warning( NULL, "Found another deinitialisation block in the .fec file. You can only have deinitialisation block.\n" );
	  FE_LEAVE_FUNCTION( NOWT );
   }
   __ferite_register_ns_variable( CURRENT_SCRIPT, CURRENT_SCRIPT->mainns, fe_new_str( "'module_deinit", code ) );
   FE_LEAVE_FUNCTION( NOWT );
}

void __builder_do_add_variable( char *name, char *type, int is_global, int is_final, int is_static )
{
   FeriteVariable *new_variable = NULL;
   FeriteVariableHash *current_hash;

   FE_ENTER_FUNCTION;
   if( is_global )
     current_hash = CURRENT_NAMESPACE->space;
   else
   {
      if( CURRENT_VARS != NULL )
		current_hash = CURRENT_VARS;
      else
		current_hash = CURRENT_NAMESPACE->space;
   }
   if( (new_variable = __ferite_get_variable_from_hash( CURRENT_SCRIPT, current_hash, name )) == NULL )
   {
      if( strcmp( type, "number" ) == 0 )
		new_variable = __ferite_create_number_long_variable( name, 0 );
      if( strcmp( type, "string" ) == 0 )
		new_variable = __ferite_create_string_variable( name, "" );
      if( strcmp( type, "object" ) == 0 )
		new_variable = __ferite_create_object_variable( name );
      if( strcmp( type, "array" ) == 0 )
		new_variable = __ferite_create_uarray_variable( name, 0);
      if( strcmp( type, "void" ) == 0 )
		new_variable = __ferite_create_void_variable( name );
      if( new_variable != NULL )
      {
		 if( is_final )
		 {
			FUD(( "Marking variable %s as final.\n", name ));
			new_variable->flags.constant = 1;
		 }
		 new_variable->flags.is_static = is_static;
		 if( is_global )
		   __ferite_register_ns_variable( CURRENT_SCRIPT, CURRENT_NAMESPACE, new_variable );
		 else
		 {
			if( CURRENT_VARS != NULL )
			  __ferite_add_variable_to_hash( CURRENT_SCRIPT, CURRENT_VARS, new_variable );
			else
			  __ferite_register_ns_variable( CURRENT_SCRIPT, CURRENT_NAMESPACE, new_variable );
		 }
		 FE_LEAVE_FUNCTION( NOWT );
      }
      ferite_error( CURRENT_SCRIPT, "Compile Error: Trying to declare variable of unknown type \"%s\"", type );
      ffree( name );
      ffree( type );
      __builder_compile_error = 1;
      longjmp( __builder_compiler_jmpback, 1 );
      FE_LEAVE_FUNCTION( NOWT );
   }
   else
   {
      ferite_error( CURRENT_SCRIPT, "Compile Error: Variable \"%s\" already exist's", name );
      ffree( name );
      ffree( type );
      __builder_compile_error = 1;
      longjmp( __builder_compiler_jmpback, 1 );
   }
}
