/* $Id: simsetup_expect.l,v 1.11 2009-10-28 12:17:10 potyra Exp $
 *
 * Scanner for simulation.setup files.
 *
 * Copyright (C) 2005-2009 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

%option nounput

%{
#include "config.h"

#include <assert.h>
#include <ctype.h>
#include <stdlib.h>
#include <search.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>

#include "env.h"
#include "memory.h"
#include "list.h"
#include "log.h"
#include "memory.h"
#include "simsetup.h"
#include "simsetup_expect.h"
#include "structure.h"

static int line = 1;
static const char *file_name;

static struct structure_component *current = NULL;

#define YY_SKIP_YYWRAP
#define	YY_DECL	int yylex(const char *current_config_dir)

static int
simsetup_expect_wrap(void)
{
	return 1;
}

%}

%x	MAPPING
%x	OPTIONS

ws			[ \t]
letter			[a-zA-Z_]
alpha			({letter}|[0-9])
path_key		(\/|{alpha}|,|\.|\-|\_|#|$|\*|:)
ip_address		[0-9]+\.[0-9]+\.[0-9]+\.[0-9]+

%%

<*>#.*$ {}
<*>{ws}+ {}

<*>^\[options\] {
	BEGIN(OPTIONS);
}

<OPTIONS>{letter}+((({ws}+)|({ws}*=)){ws}*{path_key}+)? {
	char	*option, *value;

	option=yytext;
	for (value=yytext; value<&yytext[yyleng]; value++) {
		if (*value==' ' || *value == '\t' || *value == '=') {
			*value++ = '\0';
			break;
		}
		if (*value == 0) {
			break;
		}
	}
	if (*value) {
		for (; value<&yytext[yyleng]; value++) {
			if (*value!=' ' && *value != '\t' && *value != '=')
				break;
		}
	}
	value = env_replace(value);
	if (! strcmp(option, "index")) {
		simsetup.index = atoi(value);
	} else if (! strcmp(option, "vhdl_model")) {
		simsetup.vhdl_model = mem_new_string(value);
	} else if (! strcmp(option, "base_entity")) {
		simsetup.base_entity = mem_new_string(value);
	} else if (! strcmp(option, "deterministic")) {
		simsetup.deterministic = atoi(value);
	} else if (! strcmp(option, "interactive")) {
		simsetup.interactive = atoi(value);
	} else if (! strcmp(option, "cow")) {
		simsetup.cow = atoi(value);
	} else if (! strcmp(option, "sparse")) {
		simsetup.sparse = atoi(value);
	} else if (! strcmp(option, "create")) {
		simsetup.create = atoi(value);
	} else if (! strcmp(option, "gui")) {
		simsetup.gui = mem_new_string(value);
	} else {
		fatal_error("illegal option %s in config file", option);
	}
}

<*>^\[\/{path_key}+\] {
	BEGIN(MAPPING);
	yytext[yyleng - 1] = 0;	/* delete ] from string */
	logit(LOG_DEBUG, "*** adding configuration for node %s", yytext + 1);
	list_Foreach(&simsetup.comps, current) {
		if (strcmp(current->name, yytext + 1) == 0) {
			break;
		}
	}
	if (! current) {
		char *name;

		name = mem_new_string(yytext + 1);
		current = MEM_NEW(struct structure_component);
		memset(current, 0, sizeof(*current));
		current->name = name;
		list_push(&simsetup.comps, current);
	}
}

<MAPPING>{path_key}+(({ws}+)|({ws}*=)){ws}*(({path_key}+)|(\"[^\"]*\")) {
	char		*key, *value;
	char		*replaced;
	  
	key=yytext;
	for (value=yytext; value<&yytext[yyleng]; value++) {
		if (*value==' ' || *value == '\t' || *value == '=') {
			*value = 0;
			break;
		}
	}
	value++;
	for (; value<&yytext[yyleng]; value++) {
		if (*value!=' ' && *value != '\t' && *value != '=')
			break;
	}

	if (*value == '"') {
		replaced = value + 1;
		assert(replaced[strlen(replaced) - 1] == '"');
		replaced[strlen(replaced) - 1] = 0;
	} else {
		replaced = env_replace(value);
	}

	if (0 < strlen(replaced)) {
		char *nkey, *nval;
		struct structure_keyval	*m;

		nkey = mem_new_string(key);
		nval = mem_new_string(replaced);
		m = MEM_NEW(struct structure_keyval);
		m->name = nkey;
		m->value = nval;
		list_push(&current->simsetups, m);
		logit(LOG_DEBUG, "***  setting param %s to %s", key, value);
	} else {
		logit(LOG_DEBUG, "***  ignored param %s without any entry", key);
	}
}

<*>\n	{ line++; }

<*>. {
	fatal_error("illegal character (%s) in config-file for directory\n\n\t%s, line %d\n",
		yytext, current_config_dir, line);
}

<*><<EOF>> {
	return 0;
}

%%

void
simsetup_init(const char *filename)
{
	FILE *input;

	/*
	 * get simulation setup: vhdl-file name, base entity, index,
	 * host mappings (setup node list before).
	 */
	input = fopen(filename, "r");
	if (! input) {
		fatal_error("can't open config file '%s'\n",
				filename);
	}

	simsetup.index = 0x4000;

	list_init(&simsetup.comps);

	simsetup.vhdl_model = NULL;
	simsetup.base_entity = NULL;
	simsetup.deterministic = 0;
	simsetup.interactive = 0;
	simsetup.cow = 0;
	simsetup.sparse = 0;
	simsetup.create = 0;
	simsetup.gui = NULL;

	file_name = filename;

	yyrestart(input);
	if (yylex(filename)) {
		fatal_error("parsing config file %s failed", filename);
	}

	fclose(input);
}
